/*
 * typeEvaluator.ts
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT license.
 * Author: Eric Traut
 *
 * Module that evaluates types of parse tree nodes within
 * a program.
 *
 * Note: This is a gargantuan module - much larger than I would
 * normally create. It is written this way primarily for performance,
 * with the internal methods having access to the full closure of
 * the createTypeEvaluator function. This is the same approach
 * taken by the TypeScript compiler.
 */

import { CancellationToken } from 'vscode-languageserver';

import { Commands } from '../commands/commands';
import { throwIfCancellationRequested } from '../common/cancellationUtils';
import { appendArray } from '../common/collectionUtils';
import { DiagnosticLevel } from '../common/configOptions';
import { ConsoleInterface } from '../common/console';
import { assert, assertNever, fail } from '../common/debug';
import { AddMissingOptionalToParamAction, DiagnosticAddendum } from '../common/diagnostic';
import { DiagnosticRule } from '../common/diagnosticRules';
import { convertOffsetsToRange, convertOffsetToPosition } from '../common/positionUtils';
import { PythonVersion } from '../common/pythonVersion';
import { TextRange } from '../common/textRange';
import { Localizer, ParameterizedString } from '../localization/localize';
import {
    ArgumentCategory,
    AssignmentNode,
    AugmentedAssignmentNode,
    AwaitNode,
    BinaryOperationNode,
    CallNode,
    CaseNode,
    ClassNode,
    ConstantNode,
    DecoratorNode,
    DictionaryNode,
    ExceptNode,
    ExpressionNode,
    FormatStringNode,
    ForNode,
    FunctionNode,
    ImportAsNode,
    ImportFromAsNode,
    ImportFromNode,
    IndexNode,
    isExpressionNode,
    LambdaNode,
    ListComprehensionForIfNode,
    ListComprehensionNode,
    ListNode,
    MatchNode,
    MemberAccessNode,
    NameNode,
    NumberNode,
    ParameterCategory,
    ParameterNode,
    ParseNode,
    ParseNodeType,
    RaiseNode,
    SetNode,
    SliceNode,
    StringListNode,
    StringNode,
    TernaryNode,
    TupleNode,
    TypeAliasNode,
    TypeAnnotationNode,
    TypeParameterCategory,
    TypeParameterListNode,
    TypeParameterNode,
    UnaryOperationNode,
    UnpackNode,
    WithItemNode,
    YieldFromNode,
    YieldNode,
} from '../parser/parseNodes';
import { ParseOptions, Parser } from '../parser/parser';
import { KeywordType, OperatorType, StringTokenFlags } from '../parser/tokenizerTypes';
import { AnalyzerFileInfo, ImportLookup, isAnnotationEvaluationPostponed } from './analyzerFileInfo';
import * as AnalyzerNodeInfo from './analyzerNodeInfo';
import { CodeFlowAnalyzer, FlowNodeTypeOptions, FlowNodeTypeResult, getCodeFlowEngine } from './codeFlowEngine';
import {
    CodeFlowReferenceExpressionNode,
    createKeyForReference,
    FlowNode,
    isCodeFlowSupportedForReference,
    wildcardImportReferenceKey,
} from './codeFlowTypes';
import { assignTypeToTypeVar, populateTypeVarContextBasedOnExpectedType } from './constraintSolver';
import { applyConstructorTransform, hasConstructorTransform } from './constructorTransform';
import {
    applyDataClassClassBehaviorOverrides,
    applyDataClassDecorator,
    applyDataClassDefaultBehaviors,
    getDataclassDecoratorBehaviors,
    synthesizeDataClassMethods,
    validateDataClassTransformDecorator,
} from './dataClasses';
import {
    ClassDeclaration,
    Declaration,
    DeclarationType,
    FunctionDeclaration,
    ModuleLoaderActions,
} from './declaration';
import {
    createSynthesizedAliasDeclaration,
    getDeclarationsWithUsesLocalNameRemoved,
    getNameNodeForDeclaration,
    resolveAliasDeclaration as resolveAliasDeclarationUtil,
    ResolvedAliasInfo,
} from './declarationUtils';
import {
    createEnumType,
    getEnumAutoValueType,
    getTypeOfEnumMember,
    isDeclInEnumClass,
    isKnownEnumType,
    transformTypeForPossibleEnumClass,
} from './enums';
import { applyFunctionTransform } from './functionTransform';
import { createNamedTupleType } from './namedTuples';
import {
    getParameterListDetails,
    ParameterListDetails,
    ParameterSource,
    VirtualParameterDetails,
} from './parameterUtils';
import * as ParseTreeUtils from './parseTreeUtils';
import { assignTypeToPatternTargets, checkForUnusedPattern, narrowTypeBasedOnPattern } from './patternMatching';
import {
    assignProperty,
    clonePropertyWithDeleter,
    clonePropertyWithSetter,
    createProperty,
    validatePropertyMethod,
} from './properties';
import { assignClassToProtocol, assignModuleToProtocol } from './protocols';
import { Scope, ScopeType, SymbolWithScope } from './scope';
import * as ScopeUtils from './scopeUtils';
import { evaluateStaticBoolExpression } from './staticExpressions';
import { indeterminateSymbolId, Symbol, SymbolFlags } from './symbol';
import { isConstantName, isPrivateName, isPrivateOrProtectedName } from './symbolNameUtils';
import { getLastTypedDeclaredForSymbol } from './symbolUtils';
import { SpeculativeTypeTracker } from './typeCacheUtils';
import {
    assignToTypedDict,
    assignTypedDictToTypedDict as assignTypedDictToTypedDict,
    createTypedDictType,
    createTypedDictTypeInlined,
    getTypedDictMembersForClass,
    getTypeOfIndexedTypedDict,
    synthesizeTypedDictClassMethods,
} from './typedDicts';
import {
    AbstractMethod,
    AnnotationTypeOptions,
    ArgResult,
    CallResult,
    CallSignature,
    CallSignatureInfo,
    ClassTypeResult,
    DeclaredSymbolTypeInfo,
    EffectiveTypeResult,
    EvaluatorFlags,
    EvaluatorUsage,
    ExpectedTypeOptions,
    ExpectedTypeResult,
    FunctionArgument,
    FunctionTypeResult,
    maxSubtypesForInferredType,
    PrintTypeOptions,
    TypeEvaluator,
    TypeResult,
    TypeResultWithNode,
    ValidateArgTypeParams,
} from './typeEvaluatorTypes';
import * as TypePrinter from './typePrinter';
import {
    AnyType,
    ClassType,
    ClassTypeFlags,
    combineTypes,
    DataClassBehaviors,
    findSubtype,
    FunctionParameter,
    FunctionType,
    FunctionTypeFlags,
    InheritanceChain,
    isAny,
    isAnyOrUnknown,
    isClass,
    isClassInstance,
    isFunction,
    isInstantiableClass,
    isModule,
    isNever,
    isNoneInstance,
    isNoneTypeClass,
    isOverloadedFunction,
    isParamSpec,
    isTypeSame,
    isTypeVar,
    isUnbound,
    isUnion,
    isUnknown,
    isUnpacked,
    isUnpackedClass,
    isUnpackedVariadicTypeVar,
    isVariadicTypeVar,
    LiteralValue,
    maxTypeRecursionCount,
    ModuleType,
    NeverType,
    NoneType,
    OverloadedFunctionType,
    removeFromUnion,
    removeNoneFromUnion,
    removeUnbound,
    TupleTypeArgument,
    Type,
    TypeBase,
    TypeCategory,
    TypeCondition,
    TypedDictEntry,
    TypeVarScopeId,
    TypeVarScopeType,
    TypeVarType,
    UnboundType,
    UnionType,
    UnknownType,
    Variance,
    WildcardTypeVarScopeId,
} from './types';
import {
    addConditionToType,
    addTypeVarsToListIfUnique,
    applySolvedTypeVars,
    areTypesSame,
    AssignTypeFlags,
    buildTypeVarContextFromSpecializedClass,
    ClassMember,
    ClassMemberLookupFlags,
    combineSameSizedTuples,
    combineVariances,
    computeMroLinearization,
    containsLiteralType,
    containsUnknown,
    convertParamSpecValueToType,
    convertToInstance,
    convertToInstantiable,
    convertTypeToParamSpecValue,
    derivesFromClassRecursive,
    doForEachSubtype,
    ensureFunctionSignaturesAreUnique,
    explodeGenericClass,
    getContainerDepth,
    getDeclaredGeneratorReturnType,
    getGeneratorTypeArgs,
    getGeneratorYieldType,
    getLiteralTypeClassName,
    getSpecializedTupleType,
    getTypeCondition,
    getTypeVarArgumentsRecursive,
    getTypeVarScopeId,
    getUnionSubtypeCount,
    InferenceContext,
    isDescriptorInstance,
    isEffectivelyInstantiable,
    isEllipsisType,
    isIncompleteUnknown,
    isLiteralType,
    isMaybeDescriptorInstance,
    isMetaclassInstance,
    isOptionalType,
    isPartlyUnknown,
    isProperty,
    isTupleClass,
    isTypeAliasPlaceholder,
    isTypeAliasRecursive,
    isTypeVarLimitedToCallable,
    isTypeVarSame,
    isUnboundedTupleClass,
    isUnionableType,
    isVarianceOfTypeArgumentCompatible,
    lookUpClassMember,
    lookUpObjectMember,
    makeInferenceContext,
    mapSubtypes,
    partiallySpecializeType,
    preserveUnknown,
    removeParamSpecVariadicsFromFunction,
    removeParamSpecVariadicsFromSignature,
    requiresSpecialization,
    requiresTypeArguments,
    setTypeArgumentsRecursive,
    sortTypes,
    specializeClassType,
    specializeForBaseClass,
    specializeTupleClass,
    synthesizeTypeVarForSelfCls,
    transformPossibleRecursiveTypeAlias,
    UniqueSignatureTracker,
    validateTypeVarDefault,
} from './typeUtils';
import { TypeVarContext, TypeVarSignatureContext } from './typeVarContext';

const enum MemberAccessFlags {
    None = 0,

    // By default, member accesses are assumed to access the attributes
    // of a class instance. By setting this flag, only attributes of
    // the class are considered.
    AccessClassMembersOnly = 1 << 0,

    // By default, members of base classes are also searched.
    // Set this flag to consider only the specified class' members.
    SkipBaseClasses = 1 << 1,

    // Do not include the "object" base class in the search.
    SkipObjectBaseClass = 1 << 2,

    // Consider writes to symbols flagged as ClassVars as an error.
    DisallowClassVarWrites = 1 << 3,

    // Normally __new__ is treated as a static method, but when
    // it is invoked implicitly through a constructor call, it
    // acts like a class method instead.
    TreatConstructorAsClassMethod = 1 << 4,

    // By default, class member lookups start with the class itself
    // and fall back on the metaclass if it's not found. This option
    // skips the first check.
    ConsiderMetaclassOnly = 1 << 5,

    // If an attribute cannot be found when looking for instance
    // members, normally an attribute access override method
    // (__getattr__, etc.) may provide the missing attribute type.
    // This disables this check.
    SkipAttributeAccessOverride = 1 << 6,

    // Do not include the class itself, only base classes.
    SkipOriginalClass = 1 << 7,

    // Do not include the "type" base class in the search.
    SkipTypeBaseClass = 1 << 8,
}

interface ValidateTypeArgsOptions {
    allowEmptyTuple?: boolean;
    allowVariadicTypeVar?: boolean;
    allowParamSpec?: boolean;
    allowTypeArgList?: boolean;
    allowUnpackedTuples?: boolean;
}

interface GetTypeArgsOptions {
    isAnnotatedClass?: boolean;
    hasCustomClassGetItem?: boolean;
    isFinalAnnotation?: boolean;
    isClassVarAnnotation?: boolean;
    supportsTypedDictTypeArg?: boolean;
}

interface MatchArgsToParamsResult {
    overload: FunctionType;
    overloadIndex: number;

    argumentErrors: boolean;
    isTypeIncomplete: boolean;
    argParams: ValidateArgTypeParams[];
    activeParam?: FunctionParameter | undefined;
    paramSpecTarget?: TypeVarType | undefined;
    paramSpecArgList?: FunctionArgument[] | undefined;

    // A higher relevance means that it should be considered
    // first, before lower relevance overloads.
    relevance: number;
}

interface ClassMemberLookup {
    symbol: Symbol | undefined;

    // Type of symbol.
    type: Type;
    isTypeIncomplete: boolean;

    // True if class member, false otherwise.
    isClassMember: boolean;

    // The class that declares the accessed member.
    classType?: ClassType | UnknownType;

    // True if the member is explicitly declared as ClassVar
    // within a Protocol.
    isClassVar: boolean;

    // Is member a descriptor object that is asymmetric with respect
    // to __get__ and __set__ types?
    isAsymmetricDescriptor: boolean;
}

export interface DescriptorTypeResult {
    type: Type;
    isAsymmetricDescriptor: boolean;
}

interface ScopedTypeVarResult {
    type: TypeVarType;
    isRescoped: boolean;
    foundInterveningClass: boolean;
}

interface AliasMapEntry {
    alias: string;
    module: 'builtins' | 'collections' | 'self';
}

interface ParamAssignmentInfo {
    argsNeeded: number;
    argsReceived: number;
    isPositionalOnly: boolean;
}

interface MatchedOverloadInfo {
    overload: FunctionType;
    matchResults: MatchArgsToParamsResult;
    typeVarContext: TypeVarContext;
    argResults: ArgResult[];
    returnType: Type;
}

// Maps binary operators to the magic methods that implement them.
const binaryOperatorMap: { [operator: number]: [string, string] } = {
    [OperatorType.Add]: ['__add__', '__radd__'],
    [OperatorType.Subtract]: ['__sub__', '__rsub__'],
    [OperatorType.Multiply]: ['__mul__', '__rmul__'],
    [OperatorType.FloorDivide]: ['__floordiv__', '__rfloordiv__'],
    [OperatorType.Divide]: ['__truediv__', '__rtruediv__'],
    [OperatorType.Mod]: ['__mod__', '__rmod__'],
    [OperatorType.Power]: ['__pow__', '__rpow__'],
    [OperatorType.MatrixMultiply]: ['__matmul__', '__rmatmul__'],
    [OperatorType.BitwiseAnd]: ['__and__', '__rand__'],
    [OperatorType.BitwiseOr]: ['__or__', '__ror__'],
    [OperatorType.BitwiseXor]: ['__xor__', '__rxor__'],
    [OperatorType.LeftShift]: ['__lshift__', '__rlshift__'],
    [OperatorType.RightShift]: ['__rshift__', '__rrshift__'],
    [OperatorType.Equals]: ['__eq__', '__eq__'],
    [OperatorType.NotEquals]: ['__ne__', '__ne__'],
    [OperatorType.LessThan]: ['__lt__', '__gt__'],
    [OperatorType.LessThanOrEqual]: ['__le__', '__ge__'],
    [OperatorType.GreaterThan]: ['__gt__', '__lt__'],
    [OperatorType.GreaterThanOrEqual]: ['__ge__', '__le__'],
};

// Map of operators that always return a bool result.
const booleanOperatorMap: { [operator: number]: true } = {
    [OperatorType.And]: true,
    [OperatorType.Or]: true,
    [OperatorType.Is]: true,
    [OperatorType.IsNot]: true,
    [OperatorType.In]: true,
    [OperatorType.NotIn]: true,
};

// This table contains the names of several built-in types that
// are not subscriptable at runtime on older versions of Python.
// It lists the first version of Python where subscripting is
// allowed.
const nonSubscriptableBuiltinTypes: Map<string, PythonVersion> = new Map([
    ['asyncio.futures.Future', PythonVersion.V3_9],
    ['asyncio.tasks.Task', PythonVersion.V3_9],
    ['builtins.dict', PythonVersion.V3_9],
    ['builtins.frozenset', PythonVersion.V3_9],
    ['builtins.list', PythonVersion.V3_9],
    ['builtins._PathLike', PythonVersion.V3_9],
    ['builtins.set', PythonVersion.V3_9],
    ['builtins.tuple', PythonVersion.V3_9],
    ['collections.ChainMap', PythonVersion.V3_9],
    ['collections.Counter', PythonVersion.V3_9],
    ['collections.defaultdict', PythonVersion.V3_9],
    ['collections.DefaultDict', PythonVersion.V3_9],
    ['collections.deque', PythonVersion.V3_9],
    ['collections.OrderedDict', PythonVersion.V3_9],
    ['queue.Queue', PythonVersion.V3_9],
]);

// Some types that do not inherit from others are still considered
// "compatible" based on the Python spec. These are sometimes referred
// to as "type promotions".
const typePromotions: Map<string, string[]> = new Map([
    ['builtins.float', ['builtins.int']],
    ['builtins.complex', ['builtins.float', 'builtins.int']],
    ['builtins.bytes', ['builtins.bytearray', 'builtins.memoryview']],
]);

interface SymbolResolutionStackEntry {
    // The symbol ID and declaration being resolved.
    symbolId: number;
    declaration: Declaration;

    // Initially true, it's set to false if a recursion
    // is detected.
    isResultValid: boolean;

    // Some limited forms of recursion are allowed. In these
    // cases, a partially-constructed type can be registered.
    partialType?: Type | undefined;
}

interface ReturnTypeInferenceContext {
    functionNode: FunctionNode;
    codeFlowAnalyzer: CodeFlowAnalyzer;
}

// How many levels deep should we attempt to infer return
// types based on call-site argument types? The deeper we go,
// the more types we may be able to infer, but the worse the
// performance.
const maxReturnTypeInferenceStackSize = 2;

// What is the max number of input arguments we should allow
// for call-site return type inference? We've found that large,
// complex functions with many arguments can take too long to
// analyze.
const maxReturnTypeInferenceArgumentCount = 6;

// What is the max complexity of the code flow graph that
// we will analyze to determine the return type of a function
// when its parameters are unannotated? We want to keep this
// pretty low because this can be very costly.
const maxReturnTypeInferenceCodeFlowComplexity = 32;

// What is the max complexity of the code flow graph for
// call-site type inference? This is very expensive, so we
// want to keep this very low.
const maxReturnCallSiteTypeInferenceCodeFlowComplexity = 8;

// What is the max number of return types cached per function
// when using call-site inference?
const maxCallSiteReturnTypeCacheSize = 8;

// How many entries in a list, set, or dict should we examine
// when inferring the type? We need to cut it off at some point
// to avoid excessive computation.
const maxEntriesToUseForInference = 64;

// How many assignments to an unannotated variable should be used
// when inferring its type? We need to cut it off at some point
// to avoid excessive computation.
const maxDeclarationsToUseForInference = 64;

// Maximum number of times to attempt effective type evaluation
// of a variable that has no type declaration.
const maxEffectiveTypeEvaluationAttempts = 16;

// Maximum number of combinatoric union type expansions allowed
// when resolving an overload.
const maxOverloadUnionExpansionCount = 64;

// Maximum number of recursive function return type inference attempts
// that can be concurrently pending before we give up.
const maxInferFunctionReturnRecursionCount = 12;

// In certain loops, it's possible to construct arbitrarily-deep containers
// (tuples, lists, sets, or dicts) which can lead to infinite type analysis.
// This limits the depth.
const maxInferredContainerDepth = 8;

// Maximum recursion amount when comparing two recursive type aliases.
// Increasing this can greatly increase the time required to evaluate
// two recursive type aliases that have the same definition. Decreasing
// it can increase the chance of false negatives for such recursive
// type aliases.
const maxRecursiveTypeAliasRecursionCount = 10;

// This switch enables a special debug mode that attempts to catch
// bugs due to inconsistent evaluation flags used when reading types
// from the type cache.
const verifyTypeCacheEvaluatorFlags = false;

// This debugging option prints each expression and its evaluated type.
const printExpressionTypes = false;

// If the number of subtypes starts to explode when applying "literal math",
// cut off the literal union and fall back to the non-literal supertype.
const maxLiteralMathSubtypeCount = 64;

// The following number is chosen somewhat arbitrarily. We need to cut
// off code flow analysis at some point for code flow graphs that are too
// complex. Otherwise we risk overflowing the stack or incurring extremely
// long analysis times. This number has been tuned empirically.
export const maxCodeComplexity = 768;

export interface EvaluatorOptions {
    printTypeFlags: TypePrinter.PrintTypeFlags;
    logCalls: boolean;
    minimumLoggingThreshold: number;
    evaluateUnknownImportsAsAny: boolean;
    verifyTypeCacheEvaluatorFlags: boolean;
}

// Describes a "callback hook" that is called when a class type is
// fully created and the "PartiallyEvaluated" flag has just been cleared.
// This allows us to properly compute information like the MRO which
// depends on a full understanding of base classes.
interface ClassTypeHook {
    dependency: ClassType;
    callback: () => void;
}

interface TypeCacheEntry {
    typeResult: TypeResult;
    incompleteGenerationCount: number;
    flags: EvaluatorFlags | undefined;
}

interface BinaryOperationOptions {
    isLiteralMathAllowed?: boolean;
    isTupleAddAllowed?: boolean;
}

export function createTypeEvaluator(importLookup: ImportLookup, evaluatorOptions: EvaluatorOptions): TypeEvaluator {
    const symbolResolutionStack: SymbolResolutionStackEntry[] = [];
    const asymmetricDescriptorAssignmentCache = new Set<number>();
    const speculativeTypeTracker = new SpeculativeTypeTracker();
    const suppressedNodeStack: ParseNode[] = [];

    let functionRecursionMap = new Set<number>();
    let codeFlowAnalyzerCache = new Map<number, CodeFlowAnalyzer>();
    let typeCache = new Map<number, TypeCacheEntry>();
    let effectiveTypeCache = new Map<number, Map<string, EffectiveTypeResult>>();
    let expectedTypeCache = new Map<number, Type>();
    let classTypeHooks: ClassTypeHook[] = [];
    let cancellationToken: CancellationToken | undefined;
    let isBasicTypesInitialized = false;
    let noneType: Type | undefined;
    let unionType: Type | undefined;
    let objectType: Type | undefined;
    let typeClassType: Type | undefined;
    let functionObj: Type | undefined;
    let tupleClassType: Type | undefined;
    let boolClassType: Type | undefined;
    let intClassType: Type | undefined;
    let strClassType: Type | undefined;
    let dictClassType: Type | undefined;
    let typedDictClassType: Type | undefined;
    let typedDictPrivateClassType: Type | undefined;
    let printExpressionSpaceCount = 0;
    let incompleteGenerationCount = 0;

    const returnTypeInferenceContextStack: ReturnTypeInferenceContext[] = [];
    let returnTypeInferenceTypeCache: Map<number, TypeCacheEntry> | undefined;

    function runWithCancellationToken<T>(token: CancellationToken, callback: () => T): T {
        try {
            cancellationToken = token;
            return callback();
        } finally {
            cancellationToken = undefined;
        }
    }

    function checkForCancellation() {
        if (cancellationToken) {
            throwIfCancellationRequested(cancellationToken);
        }
    }

    function getTypeCacheEntryCount(): number {
        return typeCache.size;
    }

    // This function should be called immediately prior to discarding
    // the type evaluator. It forcibly replacing existing cache maps
    // with empty equivalents. This shouldn't be necessary, but there
    // is apparently a bug in the v8 GC where it is unable to detect
    // circular references in complex data structures, so it fails
    // to clean up the objects if we don't help it out.
    function disposeEvaluator() {
        functionRecursionMap = new Set<number>();
        codeFlowAnalyzerCache = new Map<number, CodeFlowAnalyzer>();
        typeCache = new Map<number, TypeCacheEntry>();
        effectiveTypeCache = new Map<number, Map<string, EffectiveTypeResult>>();
        expectedTypeCache = new Map<number, Type>();
    }

    function readTypeCacheEntry(node: ParseNode) {
        // Should we use a temporary cache associated with a contextual
        // analysis of a function, contextualized based on call-site argument types?
        if (returnTypeInferenceTypeCache && isNodeInReturnTypeInferenceContext(node)) {
            return returnTypeInferenceTypeCache.get(node.id);
        } else {
            return typeCache.get(node.id);
        }
    }

    function isTypeCached(node: ParseNode) {
        const cacheEntry = readTypeCacheEntry(node);
        if (!cacheEntry) {
            return false;
        }

        return (
            !cacheEntry.typeResult.isIncomplete || cacheEntry.incompleteGenerationCount === incompleteGenerationCount
        );
    }

    function readTypeCache(node: ParseNode, flags: EvaluatorFlags | undefined): Type | undefined {
        const cacheEntry = readTypeCacheEntry(node);
        if (!cacheEntry || cacheEntry.typeResult.isIncomplete) {
            return undefined;
        }

        if (evaluatorOptions.verifyTypeCacheEvaluatorFlags || verifyTypeCacheEvaluatorFlags) {
            if (flags !== undefined) {
                const expectedFlags = cacheEntry.flags;

                if (expectedFlags !== undefined && flags !== expectedFlags) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                    const position = convertOffsetToPosition(node.start, fileInfo.lines);

                    const message =
                        `Type cache flag mismatch for node type ${node.nodeType} ` +
                        `(parent ${node.parent?.nodeType ?? 'none'}): ` +
                        `cached flags = ${expectedFlags}, access flags = ${flags}, ` +
                        `file = {${fileInfo.filePath} [${position.line + 1}:${position.character + 1}]}`;
                    if (evaluatorOptions.verifyTypeCacheEvaluatorFlags) {
                        fail(message);
                    } else {
                        console.log(message);
                    }
                }
            }
        }

        return cacheEntry.typeResult.type;
    }

    function writeTypeCache(
        node: ParseNode,
        typeResult: TypeResult,
        flags: EvaluatorFlags | undefined,
        inferenceContext?: InferenceContext,
        allowSpeculativeCaching = false
    ) {
        // Should we use a temporary cache associated with a contextual
        // analysis of a function, contextualized based on call-site argument types?
        const typeCacheToUse =
            returnTypeInferenceTypeCache && isNodeInReturnTypeInferenceContext(node)
                ? returnTypeInferenceTypeCache
                : typeCache;

        if (!typeResult.isIncomplete) {
            incompleteGenerationCount++;
        } else {
            const oldValue = typeCacheToUse.get(node.id);
            if (oldValue !== undefined && !isTypeSame(typeResult.type, oldValue.typeResult.type)) {
                incompleteGenerationCount++;
            }
        }

        typeCacheToUse.set(node.id, { typeResult, flags, incompleteGenerationCount: incompleteGenerationCount });

        // If the entry is located within a part of the parse tree that is currently being
        // "speculatively" evaluated, track it so we delete the cached entry when we leave
        // this speculative context.
        if (speculativeTypeTracker.isSpeculative(node)) {
            speculativeTypeTracker.trackEntry(typeCacheToUse, node.id);
            if (allowSpeculativeCaching) {
                speculativeTypeTracker.addSpeculativeType(
                    node,
                    typeResult,
                    incompleteGenerationCount,
                    inferenceContext?.expectedType
                );
            }
        }
    }

    function setTypeForNode(node: ParseNode, type: Type = UnknownType.create(), flags = EvaluatorFlags.None) {
        writeTypeCache(node, { type }, flags);
    }

    function setAsymmetricDescriptorAssignment(node: ParseNode) {
        if (speculativeTypeTracker.isSpeculative(/* node */ undefined)) {
            return;
        }

        asymmetricDescriptorAssignmentCache.add(node.id);
    }

    function isAsymmetricDescriptorAssignment(node: ParseNode) {
        return asymmetricDescriptorAssignmentCache.has(node.id);
    }

    // Determines whether the specified node is contained within
    // the function node corresponding to the function that we
    // are currently analyzing in the context of parameter types
    // defined by a call site.
    function isNodeInReturnTypeInferenceContext(node: ParseNode) {
        const stackSize = returnTypeInferenceContextStack.length;
        if (stackSize === 0) {
            return false;
        }

        const contextNode = returnTypeInferenceContextStack[stackSize - 1];

        let curNode: ParseNode | undefined = node;
        while (curNode) {
            if (curNode === contextNode.functionNode) {
                return true;
            }
            curNode = curNode.parent;
        }

        return false;
    }

    function getCodeFlowAnalyzerForReturnTypeInferenceContext() {
        const stackSize = returnTypeInferenceContextStack.length;
        assert(stackSize > 0);
        const contextNode = returnTypeInferenceContextStack[stackSize - 1];
        return contextNode.codeFlowAnalyzer;
    }

    function getIndexOfSymbolResolution(symbol: Symbol, declaration: Declaration) {
        return symbolResolutionStack.findIndex(
            (entry) => entry.symbolId === symbol.id && entry.declaration === declaration
        );
    }

    function pushSymbolResolution(symbol: Symbol, declaration: Declaration) {
        const index = getIndexOfSymbolResolution(symbol, declaration);
        if (index >= 0) {
            // Mark all of the entries between these two as invalid.
            for (let i = index + 1; i < symbolResolutionStack.length; i++) {
                symbolResolutionStack[i].isResultValid = false;
            }
            return false;
        }

        symbolResolutionStack.push({
            symbolId: symbol.id,
            declaration,
            isResultValid: true,
        });
        return true;
    }

    function popSymbolResolution(symbol: Symbol) {
        const poppedEntry = symbolResolutionStack.pop()!;
        assert(poppedEntry.symbolId === symbol.id);
        return poppedEntry.isResultValid;
    }

    function setSymbolResolutionPartialType(symbol: Symbol, declaration: Declaration, type: Type) {
        const index = getIndexOfSymbolResolution(symbol, declaration);
        if (index >= 0) {
            symbolResolutionStack[index].partialType = type;
        }
    }

    function getSymbolResolutionPartialType(symbol: Symbol, declaration: Declaration): Type | undefined {
        const index = getIndexOfSymbolResolution(symbol, declaration);
        if (index >= 0) {
            return symbolResolutionStack[index].partialType;
        }

        return undefined;
    }

    // Determines the type of the specified node by evaluating it in
    // context, logging any errors in the process. This may require the
    // type of surrounding statements to be evaluated.
    function getType(node: ExpressionNode): Type | undefined {
        return evaluateTypeForSubnode(node, () => {
            evaluateTypesForExpressionInContext(node);
        })?.type;
    }

    function getTypeResult(node: ExpressionNode): TypeResult | undefined {
        return evaluateTypeForSubnode(node, () => {
            evaluateTypesForExpressionInContext(node);
        });
    }

    // Reads the type of the node from the cache.
    function getCachedType(node: ExpressionNode): Type | undefined {
        return readTypeCache(node, EvaluatorFlags.None);
    }

    // Determines the expected type of a specified node based on surrounding
    // context. For example, if it's a subexpression of an argument expression,
    // the associated parameter type might inform the expected type.
    function getExpectedType(node: ExpressionNode): ExpectedTypeResult | undefined {
        // Scan up the parse tree to find the top-most expression node
        // so we can evaluate the entire expression.
        let topExpression = node;
        let curNode: ParseNode | undefined = node;
        while (curNode) {
            if (isExpressionNode(curNode)) {
                topExpression = curNode;
            }

            curNode = curNode.parent;
        }

        // Evaluate the expression. This will have the side effect of
        // storing an expected type in the expected type cache.
        evaluateTypesForExpressionInContext(topExpression);

        // Look for the resulting expected type by scanning up the parse tree.
        curNode = node;
        while (curNode) {
            const expectedType = expectedTypeCache.get(curNode.id);
            if (expectedType) {
                return {
                    type: expectedType,
                    node: curNode,
                };
            }

            if (curNode === topExpression) {
                break;
            }

            curNode = curNode.parent;
        }

        return undefined;
    }

    function initializedBasicTypes(node: ParseNode) {
        if (!isBasicTypesInitialized) {
            // Some of these types have cyclical dependencies on each other,
            // so don't re-enter this block once we start executing it.
            isBasicTypesInitialized = true;

            objectType = getBuiltInObject(node, 'object');
            typeClassType = getBuiltInType(node, 'type');
            functionObj = getBuiltInObject(node, 'function');

            // Initialize and cache "Collection" to break a cyclical dependency
            // that occurs when resolving tuple below.
            getTypingType(node, 'Collection');

            noneType = getTypeshedType(node, 'NoneType') || AnyType.create();
            tupleClassType = getBuiltInType(node, 'tuple');
            boolClassType = getBuiltInType(node, 'bool');
            intClassType = getBuiltInType(node, 'int');
            strClassType = getBuiltInType(node, 'str');
            dictClassType = getBuiltInType(node, 'dict');
            typedDictClassType = getTypingType(node, 'TypedDict');
            typedDictPrivateClassType = getTypingType(node, '_TypedDict');
        }
    }

    function getTypeOfExpression(
        node: ExpressionNode,
        flags = EvaluatorFlags.None,
        inferenceContext?: InferenceContext
    ): TypeResult {
        // Is this type already cached?
        const cacheEntry = readTypeCacheEntry(node);
        if (
            cacheEntry &&
            (!cacheEntry.typeResult.isIncomplete || cacheEntry.incompleteGenerationCount === incompleteGenerationCount)
        ) {
            if (printExpressionTypes) {
                console.log(
                    `${getPrintExpressionTypesSpaces()}${ParseTreeUtils.printExpression(node)} (${getLineNum(
                        node
                    )}): Cached ${printType(cacheEntry.typeResult.type)} ${
                        cacheEntry.typeResult.typeErrors ? ' Errors' : ''
                    }`
                );
            }
            return cacheEntry.typeResult;
        } else {
            // Is it cached in the speculative type cache?
            const cacheEntry = speculativeTypeTracker.getSpeculativeType(node, inferenceContext?.expectedType);
            if (
                cacheEntry &&
                (!cacheEntry.typeResult.isIncomplete ||
                    cacheEntry.incompleteGenerationCount === incompleteGenerationCount)
            ) {
                if (printExpressionTypes) {
                    console.log(
                        `${getPrintExpressionTypesSpaces()}${ParseTreeUtils.printExpression(node)} (${getLineNum(
                            node
                        )}): Speculative ${printType(cacheEntry.typeResult.type)}`
                    );
                }
                return cacheEntry.typeResult;
            }
        }

        if (printExpressionTypes) {
            console.log(
                `${getPrintExpressionTypesSpaces()}${ParseTreeUtils.printExpression(node)} (${getLineNum(node)}): Pre`
            );
            printExpressionSpaceCount++;
        }

        // This is a frequently-called routine, so it's a good place to call
        // the cancellation check. If the operation is canceled, an exception
        // will be thrown at this point.
        checkForCancellation();

        if (inferenceContext) {
            inferenceContext.expectedType = transformPossibleRecursiveTypeAlias(inferenceContext.expectedType);
        }

        // If we haven't already fetched some core type definitions from the
        // typeshed stubs, do so here. It would be better to fetch this when it's
        // needed in assignType, but we don't have access to the parse tree
        // at that point.
        initializedBasicTypes(node);

        let typeResult: TypeResult | undefined;
        let reportExpectingTypeErrors = (flags & EvaluatorFlags.ExpectingType) !== 0;

        switch (node.nodeType) {
            case ParseNodeType.Name: {
                typeResult = getTypeOfName(node, flags);
                break;
            }

            case ParseNodeType.MemberAccess: {
                typeResult = getTypeOfMemberAccess(node, flags);
                break;
            }

            case ParseNodeType.Index: {
                typeResult = getTypeOfIndex(node, flags);
                break;
            }

            case ParseNodeType.Call: {
                typeResult = getTypeOfCall(node, inferenceContext, flags);
                break;
            }

            case ParseNodeType.Tuple: {
                typeResult = getTypeOfTuple(node, inferenceContext, flags);
                break;
            }

            case ParseNodeType.Constant: {
                typeResult = getTypeOfConstant(node, flags);
                break;
            }

            case ParseNodeType.StringList: {
                const isExpectingType =
                    (flags & EvaluatorFlags.EvaluateStringLiteralAsType) !== 0 && !isAnnotationLiteralValue(node);

                if (isExpectingType) {
                    // Don't report expecting type errors again. We will have already
                    // reported them when analyzing the contents of the string.
                    reportExpectingTypeErrors = false;
                }

                typeResult = getTypeOfStringList(node, flags, isExpectingType);
                break;
            }

            case ParseNodeType.Number: {
                typeResult = getTypeOfNumber(node, typeResult);
                break;
            }

            case ParseNodeType.Ellipsis: {
                typeResult = getTypeOfEllipsis(flags, typeResult, node);
                break;
            }

            case ParseNodeType.UnaryOperation: {
                typeResult = getTypeOfUnaryOperation(node, inferenceContext);
                break;
            }

            case ParseNodeType.BinaryOperation: {
                typeResult = getTypeOfBinaryOperation(node, inferenceContext, flags);
                break;
            }

            case ParseNodeType.AugmentedAssignment: {
                typeResult = getTypeOfAugmentedAssignment(node, inferenceContext);
                break;
            }

            case ParseNodeType.List:
            case ParseNodeType.Set: {
                typeResult = getTypeOfListOrSet(node, inferenceContext);
                break;
            }

            case ParseNodeType.Slice: {
                typeResult = getTypeOfSlice(node);
                break;
            }

            case ParseNodeType.Await: {
                typeResult = getTypeOfAwaitOperator(node, flags, inferenceContext);
                break;
            }

            case ParseNodeType.Ternary: {
                typeResult = getTypeOfTernary(node, flags, inferenceContext);
                break;
            }

            case ParseNodeType.ListComprehension: {
                typeResult = getTypeOfListComprehension(node, inferenceContext);
                break;
            }

            case ParseNodeType.Dictionary: {
                typeResult = getTypeOfDictionary(node, inferenceContext);
                break;
            }

            case ParseNodeType.Lambda: {
                typeResult = getTypeOfLambda(node, inferenceContext);
                break;
            }

            case ParseNodeType.Assignment: {
                typeResult = getTypeOfExpression(node.rightExpression);
                assignTypeToExpression(
                    node.leftExpression,
                    typeResult.type,
                    !!typeResult.isIncomplete,
                    node.rightExpression,
                    /* ignoreEmptyContainers */ true,
                    /* allowAssignmentToFinalVar */ true
                );
                break;
            }

            case ParseNodeType.AssignmentExpression: {
                typeResult = getTypeOfExpression(node.rightExpression);
                assignTypeToExpression(
                    node.name,
                    typeResult.type,
                    !!typeResult.isIncomplete,
                    node.rightExpression,
                    /* ignoreEmptyContainers */ true
                );
                break;
            }

            case ParseNodeType.Yield: {
                typeResult = getTypeOfYield(node);
                break;
            }

            case ParseNodeType.YieldFrom: {
                typeResult = getTypeOfYieldFrom(node);
                break;
            }

            case ParseNodeType.Unpack: {
                typeResult = getTypeOfUnpackOperator(node, flags, inferenceContext);
                break;
            }

            case ParseNodeType.TypeAnnotation: {
                typeResult = getTypeOfExpression(
                    node.typeAnnotation,
                    EvaluatorFlags.ExpectingType |
                        EvaluatorFlags.ExpectingTypeAnnotation |
                        EvaluatorFlags.EvaluateStringLiteralAsType |
                        EvaluatorFlags.DisallowParamSpec |
                        EvaluatorFlags.DisallowTypeVarTuple |
                        EvaluatorFlags.VariableTypeAnnotation
                );
                break;
            }

            case ParseNodeType.String:
            case ParseNodeType.FormatString: {
                typeResult = getTypeOfString(node);
                break;
            }

            case ParseNodeType.Error: {
                // Evaluate the child expression as best we can so the
                // type information is cached for the completion handler.
                suppressDiagnostics(node, () => {
                    if (node.child) {
                        getTypeOfExpression(node.child);
                    }
                });
                typeResult = { type: UnknownType.create() };
                break;
            }

            default:
                assertNever(node, `Illegal node type: ${(node as any).nodeType}`);
        }

        if (!typeResult) {
            // We shouldn't get here. If we do, report an error.
            fail(`Unhandled expression type '${ParseTreeUtils.printExpression(node)}'`);
        }

        if (reportExpectingTypeErrors && !typeResult.isIncomplete) {
            if (flags & EvaluatorFlags.DisallowTypeVarTuple) {
                if (isVariadicTypeVar(typeResult.type) && !typeResult.type.isVariadicInUnion) {
                    addError(Localizer.Diagnostic.typeVarTupleContext(), node);
                    typeResult.type = UnknownType.create();
                }
            }

            if (!isEffectivelyInstantiable(typeResult.type)) {
                const isEmptyVariadic =
                    isClassInstance(typeResult.type) &&
                    ClassType.isTupleClass(typeResult.type) &&
                    typeResult.type.tupleTypeArguments?.length === 0;

                if (!isEmptyVariadic) {
                    addExpectedClassDiagnostic(typeResult.type, node);
                    typeResult.type = UnknownType.create();
                    typeResult.typeErrors = true;
                }
            }
        }

        writeTypeCache(node, typeResult, flags, inferenceContext, /* allowSpeculativeCaching */ true);

        if (
            inferenceContext &&
            !isAnyOrUnknown(inferenceContext.expectedType) &&
            !isNever(inferenceContext.expectedType)
        ) {
            expectedTypeCache.set(node.id, inferenceContext.expectedType);

            if (!typeResult.isIncomplete && !typeResult.expectedTypeDiagAddendum) {
                const diag = new DiagnosticAddendum();

                // Make sure the resulting type is assignable to the expected type.
                // Use the "solve for scopes" of the associated typeVarContext if
                // it is provided.
                if (
                    !assignType(
                        inferenceContext.expectedType,
                        typeResult.type,
                        diag,
                        new TypeVarContext(inferenceContext.typeVarContext?.getSolveForScopes())
                    )
                ) {
                    typeResult.typeErrors = true;
                    typeResult.expectedTypeDiagAddendum = diag;
                    diag.addTextRange(node);
                }
            }
        }

        if (printExpressionTypes) {
            printExpressionSpaceCount--;
            console.log(
                `${getPrintExpressionTypesSpaces()}${ParseTreeUtils.printExpression(node)} (${getLineNum(
                    node
                )}): Post ${printType(typeResult.type)}${typeResult.isIncomplete ? ' Incomplete' : ''}`
            );
        }

        return typeResult;
    }

    function getTypeOfAwaitOperator(node: AwaitNode, flags: EvaluatorFlags, inferenceContext?: InferenceContext) {
        const effectiveExpectedType = inferenceContext
            ? createAwaitableReturnType(node, inferenceContext.expectedType, /* isGenerator */ false)
            : undefined;

        const exprTypeResult = getTypeOfExpression(node.expression, flags, makeInferenceContext(effectiveExpectedType));
        const typeResult: TypeResult = {
            type: getTypeOfAwaitable(exprTypeResult.type, node.expression),
        };

        if (exprTypeResult.isIncomplete) {
            typeResult.isIncomplete = true;
        }
        return typeResult;
    }

    function getTypeOfEllipsis(flags: EvaluatorFlags, typeResult: TypeResult | undefined, node: ExpressionNode) {
        if ((flags & EvaluatorFlags.ConvertEllipsisToAny) !== 0) {
            typeResult = { type: AnyType.create(/* isEllipsis */ true) };
        } else if ((flags & EvaluatorFlags.ConvertEllipsisToUnknown) !== 0) {
            typeResult = { type: UnknownType.create() };
        } else {
            const ellipsisType = getBuiltInObject(node, 'ellipsis') || AnyType.create();
            typeResult = { type: ellipsisType };
        }
        return typeResult;
    }

    function getTypeOfNumber(node: NumberNode, typeResult: TypeResult | undefined) {
        if (node.isImaginary) {
            typeResult = { type: getBuiltInObject(node, 'complex') };
        } else if (node.isInteger) {
            typeResult = { type: cloneBuiltinObjectWithLiteral(node, 'int', node.value) };
        } else {
            typeResult = { type: getBuiltInObject(node, 'float') };
        }
        return typeResult;
    }

    function getTypeOfUnpackOperator(node: UnpackNode, flags: EvaluatorFlags, inferenceContext?: InferenceContext) {
        let typeResult: TypeResult | undefined;
        let iterExpectedType: Type | undefined;

        if (inferenceContext) {
            const iterableType = getBuiltInType(node, 'Iterable');
            if (iterableType && isInstantiableClass(iterableType)) {
                iterExpectedType = ClassType.cloneAsInstance(
                    ClassType.cloneForSpecialization(
                        iterableType,
                        [inferenceContext.expectedType],
                        /* isTypeArgumentExplicit */ true
                    )
                );
            }
        }

        const iterTypeResult = getTypeOfExpression(node.expression, flags, makeInferenceContext(iterExpectedType));
        const iterType = iterTypeResult.type;
        if (
            (flags & EvaluatorFlags.DisallowTypeVarTuple) === 0 &&
            isVariadicTypeVar(iterType) &&
            !iterType.isVariadicUnpacked
        ) {
            typeResult = { type: TypeVarType.cloneForUnpacked(iterType) };
        } else {
            if (
                (flags & EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple) !== 0 &&
                isInstantiableClass(iterType) &&
                ClassType.isBuiltIn(iterType, 'tuple')
            ) {
                typeResult = { type: ClassType.cloneForUnpacked(iterType) };
            } else {
                const iteratorTypeResult = getTypeOfIterator(iterTypeResult, /* isAsync */ false, node) ?? {
                    type: UnknownType.create(!!iterTypeResult.isIncomplete),
                    isIncomplete: iterTypeResult.isIncomplete,
                };
                typeResult = {
                    type: iteratorTypeResult.type,
                    unpackedType: iterType,
                    isIncomplete: iteratorTypeResult.isIncomplete,
                };
            }
        }
        return typeResult;
    }

    function getTypeOfStringList(node: StringListNode, flags: EvaluatorFlags, isExpectingType: boolean) {
        let typeResult: TypeResult | undefined;

        if (isExpectingType) {
            let updatedFlags = flags | EvaluatorFlags.AllowForwardReferences | EvaluatorFlags.ExpectingType;

            // In most cases, annotations within a string are not parsed by the interpreter.
            // There are a few exceptions (e.g. the "bound" value for a TypeVar constructor).
            if ((flags & EvaluatorFlags.InterpreterParsesStringLiteral) === 0) {
                updatedFlags |= EvaluatorFlags.NotParsedByInterpreter;
            }

            if (node.typeAnnotation) {
                typeResult = getTypeOfExpression(node.typeAnnotation, updatedFlags);
            } else if (!node.typeAnnotation && node.strings.length === 1) {
                // We didn't know at parse time that this string node was going
                // to be evaluated as a forward-referenced type. We need
                // to re-invoke the parser at this stage.
                const expr = parseStringAsTypeAnnotation(node);
                if (expr) {
                    typeResult = getTypeOfExpression(expr, updatedFlags);
                }
            }

            if (!typeResult) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.expectedTypeNotString(),
                    node
                );
                typeResult = { type: UnknownType.create() };
            }
        } else {
            // Evaluate the format string expressions in this context.
            let isLiteralString = true;
            let isIncomplete = false;

            node.strings.forEach((expr) => {
                const typeResult = getTypeOfString(expr);

                if (typeResult.isIncomplete) {
                    isIncomplete = true;
                }

                let isExprLiteralString = false;
                if (isClassInstance(typeResult.type)) {
                    if (ClassType.isBuiltIn(typeResult.type, 'str') && typeResult.type.literalValue !== undefined) {
                        isExprLiteralString = true;
                    } else if (ClassType.isBuiltIn(typeResult?.type, 'LiteralString')) {
                        isExprLiteralString = true;
                    }
                }

                if (!isExprLiteralString) {
                    isLiteralString = false;
                }
            });

            const isBytes = (node.strings[0].token.flags & StringTokenFlags.Bytes) !== 0;

            // Don't create a literal type if it's an f-string.
            if (node.strings.some((str) => str.nodeType === ParseNodeType.FormatString)) {
                if (isLiteralString) {
                    const literalStringType = getTypingType(node, 'LiteralString');
                    if (literalStringType && isInstantiableClass(literalStringType)) {
                        typeResult = { type: ClassType.cloneAsInstance(literalStringType) };
                    }
                }

                if (!typeResult) {
                    typeResult = {
                        type: getBuiltInObject(node, isBytes ? 'bytes' : 'str'),
                        isIncomplete,
                    };
                }
            } else {
                typeResult = {
                    type: cloneBuiltinObjectWithLiteral(
                        node,
                        isBytes ? 'bytes' : 'str',
                        node.strings.map((s) => s.value).join('')
                    ),
                    isIncomplete,
                };
            }
        }

        return typeResult;
    }

    function getTypeOfString(node: StringNode | FormatStringNode): TypeResult {
        const isBytes = (node.token.flags & StringTokenFlags.Bytes) !== 0;
        let typeResult: TypeResult | undefined;

        // Don't create a literal type if it's an f-string.
        if (node.nodeType === ParseNodeType.FormatString) {
            let isLiteralString = true;

            // If all of the format expressions are of type LiteralString, then
            // the resulting formatted string is also LiteralString.
            node.expressions.forEach((expr) => {
                const exprType = getTypeOfExpression(expr).type;

                doForEachSubtype(exprType, (exprSubtype) => {
                    if (!isClassInstance(exprSubtype)) {
                        isLiteralString = false;
                        return;
                    }

                    if (ClassType.isBuiltIn(exprSubtype, 'LiteralString')) {
                        return;
                    }

                    if (ClassType.isBuiltIn(exprSubtype, 'str') && exprSubtype.literalValue !== undefined) {
                        return;
                    }

                    isLiteralString = false;
                });
            });

            if (!isBytes && isLiteralString) {
                const literalStringType = getTypingType(node, 'LiteralString');
                if (literalStringType && isInstantiableClass(literalStringType)) {
                    typeResult = { type: ClassType.cloneAsInstance(literalStringType) };
                }
            }

            if (!typeResult) {
                typeResult = {
                    type: getBuiltInObject(node, isBytes ? 'bytes' : 'str'),
                };
            }
        } else {
            typeResult = {
                type: cloneBuiltinObjectWithLiteral(node, isBytes ? 'bytes' : 'str', node.value),
            };
        }

        return typeResult;
    }

    function stripLiteralValue(type: Type): Type {
        return mapSubtypes(type, (subtype) => {
            if (isClass(subtype)) {
                if (subtype.literalValue !== undefined) {
                    return ClassType.cloneWithLiteral(subtype, /* value */ undefined);
                }

                if (ClassType.isBuiltIn(subtype, 'LiteralString')) {
                    // Handle "LiteralString" specially.
                    if (strClassType && isInstantiableClass(strClassType)) {
                        let strInstance = ClassType.cloneAsInstance(strClassType);

                        if (subtype.condition) {
                            strInstance = TypeBase.cloneForCondition(strInstance, getTypeCondition(subtype));
                        }

                        return strInstance;
                    }
                }
            }

            return subtype;
        });
    }

    function getTypeOfParameterAnnotation(paramTypeNode: ExpressionNode, paramCategory: ParameterCategory) {
        return getTypeOfAnnotation(paramTypeNode, {
            associateTypeVarsWithScope: true,
            allowTypeVarTuple: paramCategory === ParameterCategory.VarArgList,
            allowUnpackedTypedDict: paramCategory === ParameterCategory.VarArgDictionary,
            allowUnpackedTuple: paramCategory === ParameterCategory.VarArgList,
        });
    }

    function getTypeOfAnnotation(node: ExpressionNode, options?: AnnotationTypeOptions): Type {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        // Special-case the typing.pyi file, which contains some special
        // types that the type analyzer needs to interpret differently.
        if (fileInfo.isTypingStubFile || fileInfo.isTypingExtensionsStubFile) {
            const specialType = handleTypingStubTypeAnnotation(node);
            if (specialType) {
                return specialType;
            }
        }

        let evaluatorFlags =
            EvaluatorFlags.ExpectingType |
            EvaluatorFlags.ExpectingTypeAnnotation |
            EvaluatorFlags.ConvertEllipsisToAny |
            EvaluatorFlags.EvaluateStringLiteralAsType;

        if (options?.isVariableAnnotation) {
            evaluatorFlags |= EvaluatorFlags.VariableTypeAnnotation;
        }

        if (!options?.allowFinal) {
            evaluatorFlags |= EvaluatorFlags.DisallowFinal;
        }

        if (!options?.allowClassVar) {
            evaluatorFlags |= EvaluatorFlags.DisallowClassVar;
        }

        if (!options?.allowTypeVarTuple) {
            evaluatorFlags |= EvaluatorFlags.DisallowTypeVarTuple;
        } else {
            evaluatorFlags |= EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple;
        }

        if (!options?.allowParamSpec) {
            evaluatorFlags |= EvaluatorFlags.DisallowParamSpec;
        }

        if (options?.associateTypeVarsWithScope) {
            evaluatorFlags |= EvaluatorFlags.AssociateTypeVarsWithCurrentScope;
        } else {
            evaluatorFlags |= EvaluatorFlags.DisallowTypeVarsWithoutScopeId;
        }

        if (options?.allowUnpackedTypedDict) {
            evaluatorFlags |= EvaluatorFlags.AllowUnpackedTypedDict;
        }

        if (options?.allowUnpackedTuple) {
            evaluatorFlags |= EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple;
        }

        if (options?.notParsedByInterpreter) {
            evaluatorFlags |= EvaluatorFlags.NotParsedByInterpreter;
        }

        if (options?.allowRequired) {
            evaluatorFlags |= EvaluatorFlags.AllowRequired;
        }

        if (isAnnotationEvaluationPostponed(fileInfo)) {
            evaluatorFlags |= EvaluatorFlags.AllowForwardReferences;
        }

        // If the annotation is part of a comment, allow forward references
        // even if it's not enclosed in quotes.
        if (node?.parent?.nodeType === ParseNodeType.Assignment && node.parent.typeAnnotationComment === node) {
            evaluatorFlags |= EvaluatorFlags.AllowForwardReferences | EvaluatorFlags.NotParsedByInterpreter;
        } else if (node?.parent?.nodeType === ParseNodeType.FunctionAnnotation) {
            if (node.parent.returnTypeAnnotation === node || node.parent.paramTypeAnnotations.some((n) => n === node)) {
                evaluatorFlags |= EvaluatorFlags.AllowForwardReferences | EvaluatorFlags.NotParsedByInterpreter;
            }
        } else if (node?.parent?.nodeType === ParseNodeType.Parameter) {
            if (node.parent.typeAnnotationComment === node) {
                evaluatorFlags |= EvaluatorFlags.AllowForwardReferences | EvaluatorFlags.NotParsedByInterpreter;
            }
        }

        const annotationType = getTypeOfExpression(node, evaluatorFlags).type;

        if (isModule(annotationType)) {
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.moduleAsType(),
                node
            );
        }

        return convertToInstance(annotationType);
    }

    function getTypeOfDecorator(node: DecoratorNode, functionOrClassType: Type): Type {
        // Evaluate the type of the decorator expression.
        let flags = AnalyzerNodeInfo.getFileInfo(node).isStubFile
            ? EvaluatorFlags.AllowForwardReferences
            : EvaluatorFlags.None;
        if (node.expression.nodeType !== ParseNodeType.Call) {
            flags |= EvaluatorFlags.DoNotSpecialize;
        }

        const decoratorTypeResult = getTypeOfExpression(node.expression, flags);

        // Special-case the combination of a classmethod decorator applied
        // to a property. This is allowed in Python 3.9, but it's not reflected
        // in the builtins.pyi stub for classmethod.
        if (
            isInstantiableClass(decoratorTypeResult.type) &&
            ClassType.isBuiltIn(decoratorTypeResult.type, 'classmethod') &&
            isProperty(functionOrClassType)
        ) {
            return functionOrClassType;
        }

        const argList: FunctionArgument[] = [
            {
                argumentCategory: ArgumentCategory.Simple,
                typeResult: { type: functionOrClassType },
            },
        ];

        const returnType =
            validateCallArguments(
                node.expression,
                argList,
                decoratorTypeResult,
                /* typeVarContext */ undefined,
                /* skipUnknownArgCheck */ true
            ).returnType || UnknownType.create();

        // If the return type is a function that has no annotations
        // and just *args and **kwargs parameters, assume that it
        // preserves the type of the input function.
        if (isFunction(returnType) && !returnType.details.declaredReturnType) {
            if (
                !returnType.details.parameters.some((param, index) => {
                    // Don't allow * or / separators or params with declared types.
                    if (!param.name || param.hasDeclaredType) {
                        return true;
                    }

                    // Allow *args or **kwargs parameters.
                    if (param.category !== ParameterCategory.Simple) {
                        return false;
                    }

                    // Allow inferred "self" or "cls" parameters.
                    return index !== 0 || !param.isTypeInferred;
                })
            ) {
                return functionOrClassType;
            }
        }

        // If the decorator is completely unannotated and the return type
        // includes unknowns, assume that it preserves the type of the input
        // function.
        if (isPartlyUnknown(returnType)) {
            if (isFunction(decoratorTypeResult.type)) {
                if (
                    !decoratorTypeResult.type.details.parameters.find((param) => param.typeAnnotation !== undefined) &&
                    decoratorTypeResult.type.details.declaredReturnType === undefined
                ) {
                    return functionOrClassType;
                }
            }
        }

        return returnType;
    }

    function canBeFalsy(type: Type, recursionCount = 0): boolean {
        type = makeTopLevelTypeVarsConcrete(type);

        if (recursionCount > maxTypeRecursionCount) {
            return true;
        }
        recursionCount++;

        switch (type.category) {
            case TypeCategory.Unbound:
            case TypeCategory.Unknown:
            case TypeCategory.Any:
            case TypeCategory.Never:
            case TypeCategory.None: {
                return true;
            }

            case TypeCategory.Union: {
                return findSubtype(type, (subtype) => canBeFalsy(subtype, recursionCount)) !== undefined;
            }

            case TypeCategory.Function:
            case TypeCategory.OverloadedFunction:
            case TypeCategory.Module:
            case TypeCategory.TypeVar: {
                return false;
            }

            case TypeCategory.Class: {
                if (TypeBase.isInstantiable(type)) {
                    return false;
                }

                // Handle tuples specially.
                if (isTupleClass(type) && type.tupleTypeArguments) {
                    return isUnboundedTupleClass(type) || type.tupleTypeArguments.length === 0;
                }

                // Handle subclasses of tuple, such as NamedTuple.
                const tupleBaseClass = type.details.mro.find(
                    (mroClass) => !isClass(mroClass) || isTupleClass(mroClass)
                );
                if (tupleBaseClass && isClass(tupleBaseClass) && tupleBaseClass.tupleTypeArguments) {
                    return isUnboundedTupleClass(tupleBaseClass) || tupleBaseClass.tupleTypeArguments.length === 0;
                }

                // Check for Literal[False] and Literal[True].
                if (ClassType.isBuiltIn(type, 'bool') && type.literalValue !== undefined) {
                    return type.literalValue === false;
                }

                // If this is a protocol class, don't make any assumptions about the absence
                // of specific methods. These could be provided by a class that conforms
                // to the protocol.
                if (ClassType.isProtocolClass(type)) {
                    return true;
                }

                const lenMethod = lookUpObjectMember(type, '__len__');
                if (lenMethod) {
                    return true;
                }

                const boolMethod = lookUpObjectMember(type, '__bool__');
                if (boolMethod) {
                    const boolMethodType = getTypeOfMember(boolMethod);

                    // If the __bool__ function unconditionally returns True, it can never be falsy.
                    if (isFunction(boolMethodType) && boolMethodType.details.declaredReturnType) {
                        const returnType = boolMethodType.details.declaredReturnType;
                        if (
                            isClassInstance(returnType) &&
                            ClassType.isBuiltIn(returnType, 'bool') &&
                            returnType.literalValue === true
                        ) {
                            return false;
                        }
                    }

                    return true;
                }

                return false;
            }
        }
    }

    function canBeTruthy(type: Type, recursionCount = 0): boolean {
        type = makeTopLevelTypeVarsConcrete(type);

        if (recursionCount > maxTypeRecursionCount) {
            return true;
        }
        recursionCount++;

        switch (type.category) {
            case TypeCategory.Unknown:
            case TypeCategory.Function:
            case TypeCategory.OverloadedFunction:
            case TypeCategory.Module:
            case TypeCategory.TypeVar:
            case TypeCategory.Never:
            case TypeCategory.Any: {
                return true;
            }

            case TypeCategory.Union: {
                return findSubtype(type, (subtype) => canBeTruthy(subtype, recursionCount)) !== undefined;
            }

            case TypeCategory.Unbound:
            case TypeCategory.None: {
                return false;
            }

            case TypeCategory.Class: {
                if (TypeBase.isInstantiable(type)) {
                    return true;
                }

                // Check for Tuple[()] (an empty tuple).
                if (isTupleClass(type)) {
                    if (type.tupleTypeArguments && type.tupleTypeArguments!.length === 0) {
                        return false;
                    }
                }

                // Check for Literal[False], Literal[0], Literal[""].
                if (
                    type.literalValue === false ||
                    type.literalValue === 0 ||
                    type.literalValue === BigInt(0) ||
                    type.literalValue === ''
                ) {
                    return false;
                }

                // If this is a protocol class, don't make any assumptions about the absence
                // of specific methods. These could be provided by a class that conforms
                // to the protocol.
                if (ClassType.isProtocolClass(type)) {
                    return true;
                }

                const boolMethod = lookUpObjectMember(type, '__bool__');
                if (boolMethod) {
                    const boolMethodType = getTypeOfMember(boolMethod);

                    // If the __bool__ function unconditionally returns False, it can never be truthy.
                    if (isFunction(boolMethodType) && boolMethodType.details.declaredReturnType) {
                        const returnType = boolMethodType.details.declaredReturnType;
                        if (
                            isClassInstance(returnType) &&
                            ClassType.isBuiltIn(returnType, 'bool') &&
                            returnType.literalValue === false
                        ) {
                            return false;
                        }
                    }
                }

                return true;
            }
        }
    }

    // Filters a type such that that no part of it is definitely
    // truthy. For example, if a type is a union of None
    // and a custom class "Foo" that has no __len__ or __nonzero__
    // method, this method would strip off the "Foo"
    // and return only the "None".
    function removeTruthinessFromType(type: Type): Type {
        return mapSubtypes(type, (subtype) => {
            const concreteSubtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isClassInstance(concreteSubtype)) {
                if (concreteSubtype.literalValue !== undefined) {
                    // If the object is already definitely falsy, it's fine to
                    // include, otherwise it should be removed.
                    return !concreteSubtype.literalValue ? subtype : undefined;
                }

                // If the object is a bool, make it "false", since
                // "true" is a truthy value.
                if (ClassType.isBuiltIn(concreteSubtype, 'bool')) {
                    return ClassType.cloneWithLiteral(concreteSubtype, /* value */ false);
                }

                // If the object is an int, str or bytes, narrow to a literal type.
                // This is slightly unsafe in that someone could subclass `int`, `str`
                // or `bytes` and override the `__bool__` method to change its behavior,
                // but this is extremely unlikely (and ill advised).
                if (ClassType.isBuiltIn(concreteSubtype, 'int')) {
                    return ClassType.cloneWithLiteral(concreteSubtype, /* value */ 0);
                } else if (ClassType.isBuiltIn(concreteSubtype, ['str', 'bytes'])) {
                    return ClassType.cloneWithLiteral(concreteSubtype, /* value */ '');
                }
            }

            // If it's possible for the type to be falsy, include it.
            if (canBeFalsy(subtype)) {
                return subtype;
            }

            return undefined;
        });
    }

    // Filters a type such that that no part of it is definitely
    // falsy. For example, if a type is a union of None
    // and an "int", this method would strip off the "None"
    // and return only the "int".
    function removeFalsinessFromType(type: Type): Type {
        return mapSubtypes(type, (subtype) => {
            const concreteSubtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isClassInstance(concreteSubtype)) {
                if (concreteSubtype.literalValue !== undefined) {
                    // If the object is already definitely truthy, it's fine to
                    // include, otherwise it should be removed.
                    return concreteSubtype.literalValue ? subtype : undefined;
                }

                // If the object is a bool, make it "true", since
                // "false" is a falsy value.
                if (ClassType.isBuiltIn(concreteSubtype, 'bool')) {
                    return ClassType.cloneWithLiteral(concreteSubtype, /* value */ true);
                }
            }

            // If it's possible for the type to be truthy, include it.
            if (canBeTruthy(subtype)) {
                return subtype;
            }

            return undefined;
        });
    }

    // Gets a member type from an object and if it's a function binds
    // it to the object. If bindToClass is undefined, the binding is done
    // using the objectType parameter. Callers can specify these separately
    // to handle the case where we're fetching the object member from a
    // metaclass but binding to the class.
    function getTypeOfObjectMember(
        errorNode: ExpressionNode,
        objectType: ClassType,
        memberName: string,
        usage: EvaluatorUsage = { method: 'get' },
        diag: DiagnosticAddendum | undefined = undefined,
        memberAccessFlags = MemberAccessFlags.None,
        bindToType?: ClassType | TypeVarType
    ): TypeResult | undefined {
        const memberInfo = getTypeOfClassMemberName(
            errorNode,
            ClassType.cloneAsInstantiable(objectType),
            /* isAccessedThroughObject */ true,
            memberName,
            usage,
            diag,
            memberAccessFlags | MemberAccessFlags.DisallowClassVarWrites,
            bindToType
        );

        if (memberInfo) {
            return {
                type: memberInfo.type,
                classType: memberInfo.classType,
                isIncomplete: !!memberInfo.isTypeIncomplete,
                isAsymmetricDescriptor: memberInfo.isAsymmetricDescriptor,
            };
        }
        return undefined;
    }

    // Gets a member type from a class and if it's a function binds
    // it to the class.
    function getTypeOfClassMember(
        errorNode: ExpressionNode,
        classType: ClassType,
        memberName: string,
        usage: EvaluatorUsage = { method: 'get' },
        diag: DiagnosticAddendum | undefined = undefined,
        memberAccessFlags = MemberAccessFlags.None,
        bindToType?: ClassType | TypeVarType
    ): TypeResult | undefined {
        let memberInfo: ClassMemberLookup | undefined;
        const classDiag = diag ? new DiagnosticAddendum() : undefined;
        const metaclassDiag = diag ? new DiagnosticAddendum() : undefined;

        if (ClassType.isPartiallyEvaluated(classType)) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.classDefinitionCycle().format({ name: classType.details.name }),
                errorNode
            );
            return { type: UnknownType.create() };
        }

        if ((memberAccessFlags & MemberAccessFlags.ConsiderMetaclassOnly) === 0) {
            memberInfo = getTypeOfClassMemberName(
                errorNode,
                classType,
                /* isAccessedThroughObject */ false,
                memberName,
                usage,
                classDiag,
                memberAccessFlags | MemberAccessFlags.AccessClassMembersOnly,
                bindToType
            );
        }

        // If this is a protocol class X and we're accessing a non ClassVar,
        // emit an error.
        if (
            memberInfo &&
            memberInfo.classType &&
            memberInfo.symbol &&
            isClass(memberInfo.classType) &&
            ClassType.isProtocolClass(memberInfo.classType)
        ) {
            const primaryDecl = getLastTypedDeclaredForSymbol(memberInfo.symbol);
            if (primaryDecl && primaryDecl.type === DeclarationType.Variable && !memberInfo.isClassVar) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.protocolMemberNotClassVar().format({
                        memberName,
                        className: memberInfo.classType.details.name,
                    }),
                    errorNode
                );
            }
        }

        const isMemberPresentOnClass = memberInfo?.classType !== undefined;

        // If it wasn't found on the class, see if it's part of the metaclass.
        if (!memberInfo) {
            const metaclass = classType.details.effectiveMetaclass;
            if (metaclass && isInstantiableClass(metaclass) && !ClassType.isSameGenericClass(metaclass, classType)) {
                memberInfo = getTypeOfClassMemberName(
                    errorNode,
                    metaclass,
                    /* isAccessedThroughObject */ true,
                    memberName,
                    usage,
                    metaclassDiag,
                    memberAccessFlags,
                    classType
                );
            }
        }

        if (memberInfo) {
            return {
                type: memberInfo.type,
                isIncomplete: !!memberInfo.isTypeIncomplete,
                isAsymmetricDescriptor: memberInfo.isAsymmetricDescriptor,
            };
        }

        // Determine whether to use the class or metaclass diagnostic addendum.
        const subDiag = isMemberPresentOnClass ? classDiag : metaclassDiag;
        if (diag && subDiag) {
            diag.addAddendum(subDiag);
        }

        return undefined;
    }

    function getBoundMethod(
        classType: ClassType,
        memberName: string,
        recursionCount = 0,
        treatConstructorAsClassMember = false
    ): FunctionType | OverloadedFunctionType | undefined {
        const memberInfo = lookUpClassMember(classType, memberName, ClassMemberLookupFlags.SkipInstanceVariables);

        if (memberInfo) {
            const unboundMethodType = getTypeOfMember(memberInfo);
            if (isFunction(unboundMethodType) || isOverloadedFunction(unboundMethodType)) {
                const boundMethod = bindFunctionToClassOrObject(
                    ClassType.cloneAsInstance(classType),
                    unboundMethodType,
                    /* memberClass */ undefined,
                    /* errorNode */ undefined,
                    recursionCount,
                    treatConstructorAsClassMember
                );

                if (boundMethod) {
                    return boundMethod;
                }
            } else if (isAnyOrUnknown(unboundMethodType)) {
                const unknownFunction = FunctionType.createSynthesizedInstance(
                    '',
                    FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck
                );
                FunctionType.addDefaultParameters(unknownFunction);
                return unknownFunction;
            }
        }

        return undefined;
    }

    // Returns the signature(s) associated with a call node that contains
    // the specified node. It also returns the index of the argument
    // that contains the node.
    function getCallSignatureInfo(
        callNode: CallNode,
        activeIndex: number,
        activeOrFake: boolean
    ): CallSignatureInfo | undefined {
        const exprNode = callNode.leftExpression;
        const callType = getType(exprNode);
        if (callType === undefined) {
            return undefined;
        }

        const argList: FunctionArgument[] = [];
        let previousCategory = ArgumentCategory.Simple;

        // Empty arguments do not enter the AST as nodes, but instead are left blank.
        // Instead, we detect when we appear to be between two known arguments or at the
        // end of the argument list and insert a fake argument of an unknown type to have
        // something to match later.
        function addFakeArg() {
            argList.push({
                argumentCategory: previousCategory,
                typeResult: { type: UnknownType.create() },
                active: true,
            });
        }

        callNode.arguments.forEach((arg, index) => {
            let active = false;
            if (index === activeIndex) {
                if (activeOrFake) {
                    active = true;
                } else {
                    addFakeArg();
                }
            }

            previousCategory = arg.argumentCategory;

            argList.push({
                valueExpression: arg.valueExpression,
                argumentCategory: arg.argumentCategory,
                name: arg.name,
                active: active,
            });
        });

        if (callNode.arguments.length < activeIndex) {
            addFakeArg();
        }

        const signatures: CallSignature[] = [];

        function addOneFunctionToSignature(type: FunctionType) {
            let callResult: CallResult | undefined;

            useSpeculativeMode(callNode!, () => {
                callResult = validateFunctionArguments(
                    exprNode,
                    argList,
                    { type },
                    new TypeVarContext(getTypeVarScopeId(type)),
                    /* skipUnknownArgCheck */ true
                );
            });

            signatures.push({
                type,
                activeParam: callResult?.activeParam,
            });
        }

        function addFunctionToSignature(type: FunctionType | OverloadedFunctionType) {
            if (isFunction(type)) {
                addOneFunctionToSignature(type);
            } else {
                OverloadedFunctionType.getOverloads(type).forEach((func) => {
                    addOneFunctionToSignature(func);
                });
            }
        }

        doForEachSubtype(callType, (subtype) => {
            switch (subtype.category) {
                case TypeCategory.Function:
                case TypeCategory.OverloadedFunction: {
                    addFunctionToSignature(subtype);
                    break;
                }

                case TypeCategory.Class: {
                    if (TypeBase.isInstantiable(subtype)) {
                        let methodType: FunctionType | OverloadedFunctionType | undefined;

                        // Try to get the `__init__` method first because it typically has more
                        // type information than `__new__`.
                        methodType = getBoundMethod(subtype, '__init__');

                        // Is this the __init__ method provided by the object class?
                        const isObjectInit =
                            !!methodType &&
                            isFunction(methodType) &&
                            methodType.details.fullName === 'builtins.object.__init__';
                        const isSkipConstructor =
                            !!methodType && isFunction(methodType) && FunctionType.isSkipConstructorCheck(methodType);
                        const isDefaultParams =
                            methodType && isFunction(methodType) && FunctionType.hasDefaultParameters(methodType);

                        // If there was no `__init__` or the only `__init__` that was found was from
                        // the `object` class or accepts only default parameters(* args, ** kwargs),
                        // see if we can find a better signature from the `__new__` method.
                        if (!methodType || isObjectInit || isSkipConstructor || isDefaultParams) {
                            const constructorType = getBoundMethod(
                                subtype,
                                '__new__',
                                /* recursionCount */ undefined,
                                /* treatConstructorAsClassMember */ true
                            );

                            if (constructorType) {
                                // Is this the __new__ method provided by the object class?
                                const isObjectNew =
                                    isFunction(constructorType) &&
                                    constructorType.details.fullName === 'builtins.object.__new__';

                                if (!isObjectNew) {
                                    methodType = constructorType;
                                }
                            }
                        }

                        if (methodType) {
                            addFunctionToSignature(methodType);
                        }
                    } else {
                        const methodType = getBoundMethod(subtype, '__call__');
                        if (methodType) {
                            addFunctionToSignature(methodType);
                        }
                    }
                    break;
                }
            }
        });

        if (signatures.length === 0) {
            return undefined;
        }

        return {
            callNode,
            signatures,
        };
    }

    // Determines whether the specified expression is an explicit TypeAlias declaration.
    function isDeclaredTypeAlias(expression: ExpressionNode): boolean {
        if (expression.nodeType === ParseNodeType.TypeAnnotation) {
            if (expression.valueExpression.nodeType === ParseNodeType.Name) {
                const symbolWithScope = lookUpSymbolRecursive(
                    expression,
                    expression.valueExpression.value,
                    /* honorCodeFlow */ false
                );
                if (symbolWithScope) {
                    const symbol = symbolWithScope.symbol;
                    return symbol.getDeclarations().find((decl) => isExplicitTypeAliasDeclaration(decl)) !== undefined;
                }
            }
        }

        return false;
    }

    // Determines whether the specified expression is a symbol with a declared type.
    function getDeclaredTypeForExpression(expression: ExpressionNode, usage?: EvaluatorUsage): Type | undefined {
        let symbol: Symbol | undefined;
        let classOrObjectBase: ClassType | undefined;
        let memberAccessClass: Type | undefined;
        let bindFunction = true;

        switch (expression.nodeType) {
            case ParseNodeType.Name: {
                const symbolWithScope = lookUpSymbolRecursive(expression, expression.value, /* honorCodeFlow */ true);
                if (symbolWithScope) {
                    symbol = symbolWithScope.symbol;

                    // Handle the case where the symbol is a class-level variable
                    // where the type isn't declared in this class but is in
                    // a parent class.
                    if (
                        !getDeclaredTypeOfSymbol(symbol, expression)?.type &&
                        symbolWithScope.scope.type === ScopeType.Class
                    ) {
                        const enclosingClass = ParseTreeUtils.getEnclosingClassOrFunction(expression);
                        if (enclosingClass && enclosingClass.nodeType === ParseNodeType.Class) {
                            const classTypeInfo = getTypeOfClass(enclosingClass);
                            if (classTypeInfo) {
                                const classMemberInfo = lookUpClassMember(
                                    classTypeInfo.classType,
                                    expression.value,
                                    ClassMemberLookupFlags.SkipInstanceVariables |
                                        ClassMemberLookupFlags.DeclaredTypesOnly
                                );
                                if (classMemberInfo) {
                                    symbol = classMemberInfo.symbol;
                                }
                            }
                        }
                    }
                }
                break;
            }

            case ParseNodeType.TypeAnnotation: {
                return getDeclaredTypeForExpression(expression.valueExpression, usage);
            }

            case ParseNodeType.MemberAccess: {
                const baseType = makeTopLevelTypeVarsConcrete(
                    getTypeOfExpression(expression.leftExpression, EvaluatorFlags.DoNotSpecialize).type
                );
                let classMemberInfo: ClassMember | undefined;

                if (isClassInstance(baseType)) {
                    classMemberInfo = lookUpObjectMember(
                        baseType,
                        expression.memberName.value,
                        ClassMemberLookupFlags.DeclaredTypesOnly
                    );
                    classOrObjectBase = baseType;
                    memberAccessClass = classMemberInfo?.classType;

                    // If this is an instance member (e.g. a dataclass field), don't
                    // bind it to the object if it's a function.
                    if (classMemberInfo?.isInstanceMember) {
                        bindFunction = false;
                    }
                } else if (isInstantiableClass(baseType)) {
                    classMemberInfo = lookUpClassMember(
                        baseType,
                        expression.memberName.value,
                        ClassMemberLookupFlags.SkipInstanceVariables | ClassMemberLookupFlags.DeclaredTypesOnly
                    );
                    classOrObjectBase = baseType;
                    memberAccessClass = classMemberInfo?.classType;
                }

                if (classMemberInfo) {
                    symbol = classMemberInfo.symbol;
                }
                break;
            }

            case ParseNodeType.Index: {
                const baseType = makeTopLevelTypeVarsConcrete(
                    getTypeOfExpression(expression.baseExpression, EvaluatorFlags.DoNotSpecialize).type
                );

                if (baseType && isClassInstance(baseType)) {
                    const setItemMember = lookUpClassMember(baseType, '__setitem__');
                    if (setItemMember) {
                        const setItemType = getTypeOfMember(setItemMember);
                        if (isFunction(setItemType)) {
                            const boundFunction = bindFunctionToClassOrObject(
                                baseType,
                                setItemType,
                                isInstantiableClass(setItemMember.classType) ? setItemMember.classType : undefined,
                                expression,
                                /* recursionCount */ undefined,
                                /* treatConstructorAsClassMember */ false
                            );
                            if (boundFunction && isFunction(boundFunction)) {
                                if (boundFunction.details.parameters.length >= 2) {
                                    const paramType = FunctionType.getEffectiveParameterType(boundFunction, 1);
                                    if (!isAnyOrUnknown(paramType)) {
                                        return paramType;
                                    }
                                }
                            }
                        }
                    } else if (ClassType.isTypedDictClass(baseType)) {
                        const typeFromTypedDict = getTypeOfIndexedTypedDict(
                            evaluatorInterface,
                            expression,
                            baseType,
                            usage || { method: 'get' }
                        );
                        if (typeFromTypedDict) {
                            return typeFromTypedDict.type;
                        }
                    }
                }
                break;
            }
        }

        if (symbol) {
            let declaredType = getDeclaredTypeOfSymbol(symbol)?.type;
            if (declaredType) {
                // If it's a descriptor, we need to get the setter type.
                if (isClassInstance(declaredType)) {
                    const setterInfo = lookUpClassMember(declaredType, '__set__');
                    const setter = setterInfo ? getTypeOfMember(setterInfo) : undefined;
                    if (setter && isFunction(setter) && setter.details.parameters.length >= 3) {
                        declaredType = setter.details.parameters[2].type;

                        if (isAnyOrUnknown(declaredType)) {
                            return undefined;
                        }
                    }
                }

                if (classOrObjectBase) {
                    if (memberAccessClass && isInstantiableClass(memberAccessClass)) {
                        declaredType = partiallySpecializeType(declaredType, memberAccessClass);
                    }

                    if (isFunction(declaredType) || isOverloadedFunction(declaredType)) {
                        if (bindFunction) {
                            declaredType = bindFunctionToClassOrObject(
                                classOrObjectBase,
                                declaredType,
                                /* memberClass */ undefined,
                                expression
                            );
                        }
                    }
                }

                return declaredType;
            }
        }

        return undefined;
    }

    // Applies an "await" operation to the specified type and returns
    // the result. According to PEP 492, await operates on an Awaitable
    // (object that provides an __await__ that returns a generator object).
    // If errorNode is undefined, no errors are reported.
    function getTypeOfAwaitable(type: Type, errorNode?: ExpressionNode): Type {
        return mapSubtypes(type, (subtype) => {
            subtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isAnyOrUnknown(subtype)) {
                return subtype;
            }

            if (isClassInstance(subtype)) {
                const awaitReturnType = getSpecializedReturnType(subtype, '__await__', [], errorNode);
                if (awaitReturnType) {
                    if (isAnyOrUnknown(awaitReturnType)) {
                        return awaitReturnType;
                    }

                    if (isClassInstance(awaitReturnType)) {
                        const iterReturnType = getSpecializedReturnType(awaitReturnType, '__iter__', [], errorNode);

                        if (iterReturnType) {
                            const generatorReturnType = getReturnTypeFromGenerator(awaitReturnType);
                            if (generatorReturnType) {
                                return generatorReturnType;
                            }
                        }
                    }
                }
            }

            if (errorNode) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeNotAwaitable().format({ type: printType(subtype) }),
                    errorNode
                );
            }

            return UnknownType.create();
        });
    }

    // Validates that the type is an iterator and returns the iterated type
    // (i.e. the type returned from the '__next__' or '__anext__' method).
    function getTypeOfIterator(
        typeResult: TypeResult,
        isAsync: boolean,
        errorNode: ExpressionNode | undefined
    ): TypeResult | undefined {
        const iterMethodName = isAsync ? '__aiter__' : '__iter__';
        const nextMethodName = isAsync ? '__anext__' : '__next__';
        let isValidIterator = true;

        let type = transformPossibleRecursiveTypeAlias(typeResult.type);
        type = makeTopLevelTypeVarsConcrete(type);

        if (isOptionalType(type)) {
            if (errorNode && !typeResult.isIncomplete) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportOptionalIterable,
                    DiagnosticRule.reportOptionalIterable,
                    Localizer.Diagnostic.noneNotIterable(),
                    errorNode
                );
            }
            type = removeNoneFromUnion(type);
        }

        const iterableType = mapSubtypes(type, (subtype) => {
            subtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isAnyOrUnknown(subtype)) {
                return subtype;
            }

            const diag = new DiagnosticAddendum();
            if (isClass(subtype)) {
                let iterReturnType: Type | undefined;

                if (TypeBase.isInstance(subtype)) {
                    // Handle an empty tuple specially.
                    if (
                        isTupleClass(subtype) &&
                        subtype.tupleTypeArguments &&
                        subtype.tupleTypeArguments.length === 0
                    ) {
                        return NeverType.createNever();
                    }

                    iterReturnType = getSpecializedReturnType(subtype, iterMethodName, [], errorNode);
                } else if (
                    TypeBase.isInstantiable(subtype) &&
                    subtype.details.effectiveMetaclass &&
                    isInstantiableClass(subtype.details.effectiveMetaclass)
                ) {
                    iterReturnType = getSpecializedReturnType(
                        ClassType.cloneAsInstance(subtype.details.effectiveMetaclass),
                        iterMethodName,
                        [],
                        errorNode,
                        subtype
                    );
                }

                if (!iterReturnType) {
                    // There was no __iter__. See if we can fall back to
                    // the __getitem__ method instead.
                    if (!isAsync && isClassInstance(subtype)) {
                        const getItemReturnType = getSpecializedReturnType(
                            subtype,
                            '__getitem__',
                            [
                                {
                                    argumentCategory: ArgumentCategory.Simple,
                                    typeResult: {
                                        type:
                                            intClassType && isInstantiableClass(intClassType)
                                                ? ClassType.cloneAsInstance(intClassType)
                                                : UnknownType.create(),
                                    },
                                },
                            ],
                            errorNode
                        );
                        if (getItemReturnType) {
                            return getItemReturnType;
                        }
                    }

                    diag.addMessage(Localizer.Diagnostic.methodNotDefined().format({ name: iterMethodName }));
                } else {
                    const iterReturnTypeDiag = new DiagnosticAddendum();

                    const returnType = mapSubtypesExpandTypeVars(
                        iterReturnType,
                        /* conditionFilter */ undefined,
                        (subtype) => {
                            if (isAnyOrUnknown(subtype)) {
                                return subtype;
                            }

                            if (isClassInstance(subtype)) {
                                let nextReturnType = getSpecializedReturnType(subtype, nextMethodName, [], errorNode);

                                if (!nextReturnType) {
                                    iterReturnTypeDiag.addMessage(
                                        Localizer.Diagnostic.methodNotDefinedOnType().format({
                                            name: nextMethodName,
                                            type: printType(subtype!),
                                        })
                                    );
                                } else {
                                    // Convert any unpacked TypeVarTuples into object instances. We don't
                                    // know anything more about them.
                                    nextReturnType = mapSubtypes(nextReturnType, (returnSubtype) => {
                                        if (isTypeVar(returnSubtype) && isUnpackedVariadicTypeVar(returnSubtype)) {
                                            return objectType ?? UnknownType.create();
                                        }

                                        return returnSubtype;
                                    });

                                    if (!isAsync) {
                                        return nextReturnType;
                                    }

                                    // If it's an async iteration, there's an implicit
                                    // 'await' operator applied.
                                    return getTypeOfAwaitable(nextReturnType, errorNode);
                                }
                            } else {
                                iterReturnTypeDiag.addMessage(
                                    Localizer.Diagnostic.methodReturnsNonObject().format({ name: iterMethodName })
                                );
                            }

                            return undefined;
                        }
                    );

                    if (iterReturnTypeDiag.isEmpty()) {
                        return returnType;
                    }

                    diag.addAddendum(iterReturnTypeDiag);
                }
            }

            if (errorNode && !typeResult.isIncomplete) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeNotIterable().format({ type: printType(subtype) }) + diag.getString(),
                    errorNode
                );
            }

            isValidIterator = false;
            return undefined;
        });

        return isValidIterator ? { type: iterableType, isIncomplete: typeResult.isIncomplete } : undefined;
    }

    // Validates that the type is an iterable and returns the iterable type argument.
    function getTypeOfIterable(
        typeResult: TypeResult,
        isAsync: boolean,
        errorNode: ExpressionNode | undefined
    ): TypeResult | undefined {
        const iterMethodName = isAsync ? '__aiter__' : '__iter__';
        let isValidIterable = true;

        let type = makeTopLevelTypeVarsConcrete(typeResult.type);

        if (isOptionalType(type)) {
            if (errorNode && !typeResult.isIncomplete) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportOptionalIterable,
                    DiagnosticRule.reportOptionalIterable,
                    Localizer.Diagnostic.noneNotIterable(),
                    errorNode
                );
            }
            type = removeNoneFromUnion(type);
        }

        const iterableType = mapSubtypes(type, (subtype) => {
            if (isAnyOrUnknown(subtype)) {
                return subtype;
            }

            if (isClass(subtype)) {
                let iterReturnType: Type | undefined;

                if (TypeBase.isInstance(subtype)) {
                    iterReturnType = getSpecializedReturnType(subtype, iterMethodName, [], errorNode);
                } else if (
                    TypeBase.isInstantiable(subtype) &&
                    subtype.details.effectiveMetaclass &&
                    isInstantiableClass(subtype.details.effectiveMetaclass)
                ) {
                    iterReturnType = getSpecializedReturnType(
                        ClassType.cloneAsInstance(subtype.details.effectiveMetaclass),
                        iterMethodName,
                        [],
                        errorNode,
                        subtype
                    );
                }

                if (iterReturnType) {
                    return makeTopLevelTypeVarsConcrete(iterReturnType);
                }
            }

            if (errorNode) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeNotIterable().format({ type: printType(subtype) }),
                    errorNode
                );
            }

            isValidIterable = false;
            return undefined;
        });

        return isValidIterable ? { type: iterableType, isIncomplete: typeResult.isIncomplete } : undefined;
    }

    function isTypeHashable(type: Type): boolean {
        let isTypeHashable = true;

        doForEachSubtype(makeTopLevelTypeVarsConcrete(type), (subtype) => {
            if (isClassInstance(subtype)) {
                // Assume the class is hashable.
                let isObjectHashable = true;

                // Have we already computed and cached the hashability?
                if (subtype.details.isInstanceHashable !== undefined) {
                    isObjectHashable = subtype.details.isInstanceHashable;
                } else {
                    const hashMember = lookUpObjectMember(
                        subtype,
                        '__hash__',
                        ClassMemberLookupFlags.SkipObjectBaseClass
                    );

                    if (hashMember && hashMember.isTypeDeclared) {
                        const decls = hashMember.symbol.getTypedDeclarations();
                        const synthesizedType = hashMember.symbol.getSynthesizedType();

                        // Handle the case where the type is synthesized (used for
                        // dataclasses).
                        if (synthesizedType) {
                            isObjectHashable = !isNoneInstance(synthesizedType);
                        } else {
                            // Assume that if '__hash__' is declared as a variable, it is
                            // not hashable. If it's declared as a function, it is. We'll
                            // skip evaluating its full type because that's not needed in
                            // this case.
                            if (decls.every((decl) => decl.type === DeclarationType.Variable)) {
                                isObjectHashable = false;
                            }
                        }
                    }

                    // Cache the hashability for next time.
                    subtype.details.isInstanceHashable = isObjectHashable;
                }

                if (!isObjectHashable) {
                    isTypeHashable = false;
                }
            }
        });

        return isTypeHashable;
    }

    function getTypedDictClassType() {
        return typedDictPrivateClassType;
    }

    function getTupleClassType() {
        return tupleClassType;
    }

    function getObjectType() {
        return objectType;
    }

    function getTypingType(node: ParseNode, symbolName: string): Type | undefined {
        return (
            getTypeOfModule(node, symbolName, ['typing']) ?? getTypeOfModule(node, symbolName, ['typing_extensions'])
        );
    }

    function getTypeshedType(node: ParseNode, symbolName: string): Type | undefined {
        return getTypeOfModule(node, symbolName, ['_typeshed']);
    }

    function getTypeOfModule(node: ParseNode, symbolName: string, nameParts: string[]) {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const lookupResult = importLookup({ nameParts, importingFilePath: fileInfo.filePath });

        if (!lookupResult) {
            return undefined;
        }

        const symbol = lookupResult.symbolTable.get(symbolName);
        if (!symbol) {
            return undefined;
        }

        return getEffectiveTypeOfSymbol(symbol);
    }

    function checkCodeFlowTooComplex(node: ParseNode): boolean {
        const scopeNode = node.nodeType === ParseNodeType.Function ? node : ParseTreeUtils.getExecutionScopeNode(node);
        const codeComplexity = AnalyzerNodeInfo.getCodeFlowComplexity(scopeNode);

        if (codeComplexity > maxCodeComplexity) {
            let errorRange: TextRange = scopeNode;
            if (scopeNode.nodeType === ParseNodeType.Function) {
                errorRange = scopeNode.name;
            } else if (scopeNode.nodeType === ParseNodeType.Module) {
                errorRange = { start: 0, length: 0 };
            }

            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            addDiagnosticForTextRange(
                fileInfo,
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.codeTooComplexToAnalyze(),
                errorRange
            );

            return true;
        }

        return false;
    }

    function isNodeReachable(node: ParseNode, sourceNode?: ParseNode): boolean {
        if (checkCodeFlowTooComplex(node)) {
            return true;
        }

        const flowNode = AnalyzerNodeInfo.getFlowNode(node);
        if (!flowNode) {
            if (node.parent) {
                return isNodeReachable(node.parent, sourceNode);
            }
            return false;
        }

        const sourceFlowNode = sourceNode ? AnalyzerNodeInfo.getFlowNode(sourceNode) : undefined;

        return codeFlowEngine.isFlowNodeReachable(flowNode, sourceFlowNode);
    }

    function isAfterNodeReachable(node: ParseNode): boolean {
        const returnFlowNode = AnalyzerNodeInfo.getAfterFlowNode(node);
        if (!returnFlowNode) {
            return false;
        }

        if (checkCodeFlowTooComplex(node)) {
            return true;
        }

        if (!codeFlowEngine.isFlowNodeReachable(returnFlowNode)) {
            return false;
        }

        if (!isFlowNodeReachableUsingNeverNarrowing(node, returnFlowNode)) {
            return false;
        }

        return true;
    }

    // Although isFlowNodeReachable indicates that the node is reachable, it
    // may not be reachable if we apply "never narrowing".
    function isFlowNodeReachableUsingNeverNarrowing(node: ParseNode, flowNode: FlowNode) {
        const analyzer = getCodeFlowAnalyzerForNode(node.id);

        if (checkCodeFlowTooComplex(node)) {
            return true;
        }

        const codeFlowResult = analyzer.getTypeFromCodeFlow(
            flowNode,
            /* reference */ undefined,
            /* targetSymbolId */ undefined,
            /* typeAtStart */ UnboundType.create(),
            {
                skipNoReturnCallAnalysis: true,
            }
        );

        return codeFlowResult.type !== undefined && !isNever(codeFlowResult.type);
    }

    // Determines whether there is a code flow path from sourceNode to sinkNode.
    function isFlowPathBetweenNodes(sourceNode: ParseNode, sinkNode: ParseNode, allowSelf = true) {
        if (checkCodeFlowTooComplex(sourceNode)) {
            return true;
        }

        const sourceFlowNode = AnalyzerNodeInfo.getFlowNode(sourceNode);
        const sinkFlowNode = AnalyzerNodeInfo.getFlowNode(sinkNode);
        if (!sourceFlowNode || !sinkFlowNode) {
            return false;
        }
        if (sourceFlowNode === sinkFlowNode) {
            return allowSelf;
        }

        return codeFlowEngine.isFlowNodeReachable(sinkFlowNode, sourceFlowNode, /* ignoreNoReturn */ true);
    }

    // Determines whether the specified string literal is part
    // of a Literal['xxx'] statement. If so, we will not treat
    // the string as a normal forward-declared type annotation.
    function isAnnotationLiteralValue(node: StringListNode): boolean {
        if (node.parent && node.parent.nodeType === ParseNodeType.Index) {
            const baseType = getTypeOfExpression(node.parent.baseExpression).type;
            if (baseType && isInstantiableClass(baseType)) {
                if (ClassType.isSpecialBuiltIn(baseType, 'Literal')) {
                    return true;
                }
            }
        }

        return false;
    }

    function addInformation(message: string, node: ParseNode, range?: TextRange) {
        return addDiagnosticWithSuppressionCheck('information', message, node, range);
    }

    function addWarning(message: string, node: ParseNode, range?: TextRange) {
        return addDiagnosticWithSuppressionCheck('warning', message, node, range);
    }

    function addError(message: string, node: ParseNode, range?: TextRange) {
        return addDiagnosticWithSuppressionCheck('error', message, node, range);
    }

    function addUnusedCode(node: ParseNode, textRange: TextRange) {
        if (!isDiagnosticSuppressedForNode(node)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            fileInfo.diagnosticSink.addUnusedCodeWithTextRange(Localizer.Diagnostic.unreachableCode(), textRange);
        }
    }

    function addUnreachableCode(node: ParseNode, textRange: TextRange) {
        if (!isDiagnosticSuppressedForNode(node)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            fileInfo.diagnosticSink.addUnreachableCodeWithTextRange(Localizer.Diagnostic.unreachableCode(), textRange);
        }
    }

    function addDeprecated(message: string, node: ParseNode) {
        if (!isDiagnosticSuppressedForNode(node)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            fileInfo.diagnosticSink.addDeprecatedWithTextRange(message, node);
        }
    }

    function addDiagnosticWithSuppressionCheck(
        diagLevel: DiagnosticLevel,
        message: string,
        node: ParseNode,
        range?: TextRange
    ) {
        if (!isDiagnosticSuppressedForNode(node) && isNodeReachable(node)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            return fileInfo.diagnosticSink.addDiagnosticWithTextRange(diagLevel, message, range || node);
        }

        return undefined;
    }

    function isDiagnosticSuppressedForNode(node: ParseNode) {
        return (
            suppressedNodeStack.some((suppressedNode) => ParseTreeUtils.isNodeContainedWithin(node, suppressedNode)) ||
            speculativeTypeTracker.isSpeculative(node)
        );
    }

    function addDiagnostic(
        diagLevel: DiagnosticLevel,
        rule: string,
        message: string,
        node: ParseNode,
        range?: TextRange
    ) {
        if (diagLevel === 'none') {
            return undefined;
        }

        // Should we suppress this diagnostic because it's within an unannotated function?
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        if (!fileInfo.diagnosticRuleSet.analyzeUnannotatedFunctions) {
            const containingFunction = ParseTreeUtils.getEnclosingFunction(node);

            // Is the target node within the body of the function? If so, suppress the diagnostic.
            if (
                containingFunction &&
                ParseTreeUtils.isUnannotatedFunction(containingFunction) &&
                ParseTreeUtils.isNodeContainedWithin(node, containingFunction.suite)
            ) {
                return undefined;
            }
        }

        const diagnostic = addDiagnosticWithSuppressionCheck(diagLevel, message, node, range);
        if (diagnostic) {
            diagnostic.setRule(rule);
        }

        return diagnostic;
    }

    function addDiagnosticForTextRange(
        fileInfo: AnalyzerFileInfo,
        diagLevel: DiagnosticLevel,
        rule: string,
        message: string,
        range: TextRange
    ) {
        if (diagLevel === 'none') {
            return undefined;
        }

        const diagnostic = fileInfo.diagnosticSink.addDiagnosticWithTextRange(diagLevel, message, range);
        if (rule) {
            diagnostic.setRule(rule);
        }

        return diagnostic;
    }

    function addExpectedClassDiagnostic(type: Type, node: ParseNode) {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const diag = new DiagnosticAddendum();
        if (isUnion(type)) {
            doForEachSubtype(type, (subtype) => {
                if (!isEffectivelyInstantiable(subtype)) {
                    diag.addMessage(Localizer.DiagnosticAddendum.typeNotClass().format({ type: printType(subtype) }));
                }
            });
        }

        addDiagnostic(
            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
            DiagnosticRule.reportGeneralTypeIssues,
            Localizer.Diagnostic.typeExpectedClass().format({ type: printType(type) }) + diag.getString(),
            node
        );
    }

    function assignTypeToNameNode(
        nameNode: NameNode,
        type: Type,
        isTypeIncomplete: boolean,
        ignoreEmptyContainers: boolean,
        srcExpression?: ParseNode,
        allowAssignmentToFinalVar = false,
        expectedTypeDiagAddendum?: DiagnosticAddendum
    ) {
        const nameValue = nameNode.value;

        const symbolWithScope = lookUpSymbolRecursive(nameNode, nameValue, /* honorCodeFlow */ false);
        if (!symbolWithScope) {
            // This can happen when we are evaluating a piece of code that was
            // determined to be unreachable by the binder.
            return;
        }

        const declarations = symbolWithScope.symbol.getDeclarations();
        let declaredType = getDeclaredTypeOfSymbol(symbolWithScope.symbol)?.type;
        const fileInfo = AnalyzerNodeInfo.getFileInfo(nameNode);

        // If this is a class scope and there is no type declared for this class variable,
        // see if a parent class has a type declared.
        if (declaredType === undefined && symbolWithScope.scope.type === ScopeType.Class) {
            const containingClass = ParseTreeUtils.getEnclosingClass(nameNode);
            if (containingClass) {
                const classType = getTypeOfClass(containingClass);
                if (classType) {
                    const memberInfo = lookUpClassMember(
                        classType.classType,
                        nameNode.value,
                        ClassMemberLookupFlags.SkipOriginalClass
                    );
                    if (memberInfo?.isTypeDeclared) {
                        declaredType = getTypeOfMember(memberInfo);
                    }
                }
            }
        }

        // We found an existing declared type. Make sure the type is assignable.
        let destType = type;
        const isTypeAlias =
            !!declaredType && isClassInstance(declaredType) && ClassType.isBuiltIn(declaredType, 'TypeAlias');

        if (declaredType && !isTypeAlias) {
            let diagAddendum = new DiagnosticAddendum();

            if (!assignType(declaredType, type, diagAddendum)) {
                // If there was an expected type mismatch, use that diagnostic
                // addendum because it will be more informative.
                if (expectedTypeDiagAddendum) {
                    diagAddendum = expectedTypeDiagAddendum;
                }

                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeAssignmentMismatch().format({
                        sourceType: printType(type),
                        destType: printType(declaredType),
                    }) + diagAddendum.getString(),
                    srcExpression ?? nameNode,
                    diagAddendum.getEffectiveTextRange() ?? srcExpression ?? nameNode
                );

                // Replace the assigned type with the (unnarrowed) declared type.
                destType = declaredType;
            } else {
                // Constrain the resulting type to match the declared type.
                destType = narrowTypeBasedOnAssignment(declaredType, type);
            }
        } else {
            // If this is a member name (within a class scope) and the member name
            // appears to be a constant, use the strict source type. If it's a member
            // variable that can be overridden by a child class, use the more general
            // version by stripping off the literal.
            const scope = ScopeUtils.getScopeForNode(nameNode);
            if (scope?.type === ScopeType.Class) {
                if (
                    TypeBase.isInstance(destType) &&
                    !isConstantName(nameValue) &&
                    !isFinalVariable(symbolWithScope.symbol)
                ) {
                    destType = stripLiteralValue(destType);
                }
            }
        }

        const varDecl: Declaration | undefined = declarations.find((decl) => decl.type === DeclarationType.Variable);

        if (varDecl && varDecl.type === DeclarationType.Variable) {
            if (varDecl.isConstant) {
                // A constant variable can be assigned only once. If this
                // isn't the first assignment, generate an error.
                if (nameNode !== getNameNodeForDeclaration(declarations[0])) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportConstantRedefinition,
                        DiagnosticRule.reportConstantRedefinition,
                        Localizer.Diagnostic.constantRedefinition().format({ name: nameValue }),
                        nameNode
                    );
                }
            } else if (isFinalVariableDeclaration(varDecl) && !allowAssignmentToFinalVar) {
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.finalReassigned().format({ name: nameValue }),
                    nameNode
                );
            }
        }

        if (!isTypeIncomplete) {
            reportPossibleUnknownAssignment(
                fileInfo.diagnosticRuleSet.reportUnknownVariableType,
                DiagnosticRule.reportUnknownVariableType,
                nameNode,
                destType,
                nameNode,
                ignoreEmptyContainers
            );
        }

        writeTypeCache(nameNode, { type: destType, isIncomplete: isTypeIncomplete }, EvaluatorFlags.None);
    }

    function assignTypeToMemberAccessNode(
        target: MemberAccessNode,
        type: Type,
        isTypeIncomplete: boolean,
        srcExpr?: ExpressionNode,
        expectedTypeDiagAddendum?: DiagnosticAddendum
    ) {
        const baseTypeResult = getTypeOfExpression(target.leftExpression, EvaluatorFlags.DoNotSpecialize);
        const baseType = makeTopLevelTypeVarsConcrete(baseTypeResult.type);

        // Handle member accesses (e.g. self.x or cls.y).
        if (target.leftExpression.nodeType === ParseNodeType.Name) {
            // Determine whether we're writing to a class or instance member.
            const enclosingClassNode = ParseTreeUtils.getEnclosingClass(target);

            if (enclosingClassNode) {
                const classTypeResults = getTypeOfClass(enclosingClassNode);

                if (classTypeResults && isInstantiableClass(classTypeResults.classType)) {
                    if (isClassInstance(baseType)) {
                        if (ClassType.isSameGenericClass(baseType, classTypeResults.classType)) {
                            assignTypeToMemberVariable(
                                target,
                                type,
                                isTypeIncomplete,
                                /* isInstanceMember */ true,
                                srcExpr
                            );
                        }
                    } else if (isInstantiableClass(baseType)) {
                        if (ClassType.isSameGenericClass(baseType, classTypeResults.classType)) {
                            assignTypeToMemberVariable(
                                target,
                                type,
                                isTypeIncomplete,
                                /* isInstanceMember */ false,
                                srcExpr
                            );
                        }
                    }

                    // Assignments to instance or class variables through "self" or "cls" is not
                    // allowed for protocol classes unless it is also declared within the class.
                    if (ClassType.isProtocolClass(classTypeResults.classType)) {
                        const memberSymbol = classTypeResults.classType.details.fields.get(target.memberName.value);
                        if (memberSymbol) {
                            const classLevelDecls = memberSymbol.getDeclarations().filter((decl) => {
                                return !ParseTreeUtils.getEnclosingFunction(decl.node);
                            });
                            if (classLevelDecls.length === 0) {
                                addError(Localizer.Diagnostic.assignmentInProtocol(), target.memberName);
                            }
                        }
                    }
                }
            }
        }

        const setTypeResult = getTypeOfMemberAccessWithBaseType(
            target,
            baseTypeResult,
            {
                method: 'set',
                setType: { type, isIncomplete: isTypeIncomplete },
                setErrorNode: srcExpr,
                setExpectedTypeDiag: expectedTypeDiagAddendum,
            },
            EvaluatorFlags.None
        );

        if (setTypeResult.isAsymmetricDescriptor) {
            setAsymmetricDescriptorAssignment(target);
        }

        writeTypeCache(target.memberName, { type, isIncomplete: isTypeIncomplete }, EvaluatorFlags.None);
        writeTypeCache(target, { type, isIncomplete: isTypeIncomplete }, EvaluatorFlags.None);
    }

    function assignTypeToMemberVariable(
        node: MemberAccessNode,
        srcType: Type,
        isTypeIncomplete: boolean,
        isInstanceMember: boolean,
        srcExprNode?: ExpressionNode
    ) {
        const memberName = node.memberName.value;
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        const classDef = ParseTreeUtils.getEnclosingClass(node);
        if (!classDef) {
            return;
        }

        const classTypeInfo = getTypeOfClass(classDef);
        if (classTypeInfo && isInstantiableClass(classTypeInfo.classType)) {
            let memberInfo = lookUpClassMember(
                classTypeInfo.classType,
                memberName,
                isInstanceMember ? ClassMemberLookupFlags.Default : ClassMemberLookupFlags.SkipInstanceVariables
            );

            const memberFields = classTypeInfo.classType.details.fields;
            if (memberInfo) {
                // Are we accessing an existing member on this class, or is
                // it a member on a parent class?
                const memberClass = isInstantiableClass(memberInfo.classType) ? memberInfo.classType : undefined;
                const isThisClass = memberClass && ClassType.isSameGenericClass(classTypeInfo.classType, memberClass);

                // Check for an attempt to write to an instance variable that is
                // not defined by __slots__.
                if (isThisClass && isInstanceMember) {
                    if (memberClass?.details.inheritedSlotsNames && memberClass.details.localSlotsNames) {
                        // Skip this check if the local slots is specified but empty because this pattern
                        // is used in a legitimate manner for mix-in classes.
                        if (
                            memberClass.details.localSlotsNames.length > 0 &&
                            !memberClass.details.inheritedSlotsNames.some((name) => name === memberName)
                        ) {
                            // Determine whether the assignment corresponds to a descriptor
                            // that was assigned as a class variable. If so, then slots will not
                            // apply in this case.
                            const classMemberDetails = lookUpClassMember(
                                memberClass,
                                memberName,
                                ClassMemberLookupFlags.SkipInstanceVariables
                            );
                            let isPotentiallyDescriptor = false;

                            if (classMemberDetails) {
                                const classMemberSymbolType = getEffectiveTypeOfSymbol(classMemberDetails.symbol);
                                if (
                                    isAnyOrUnknown(classMemberSymbolType) ||
                                    isUnbound(classMemberSymbolType) ||
                                    isMaybeDescriptorInstance(classMemberSymbolType)
                                ) {
                                    isPotentiallyDescriptor = true;
                                }
                            }

                            if (!isPotentiallyDescriptor) {
                                addDiagnostic(
                                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.slotsAttributeError().format({ name: memberName }),
                                    node.memberName
                                );
                            }
                        }
                    }
                }

                if (isThisClass && memberInfo.isInstanceMember === isInstanceMember) {
                    const symbol = memberFields.get(memberName)!;
                    assert(symbol !== undefined);

                    const typedDecls = symbol.getDeclarations();

                    // Check for an attempt to overwrite a constant member variable.
                    if (
                        typedDecls.length > 0 &&
                        typedDecls[0].type === DeclarationType.Variable &&
                        srcExprNode &&
                        node.memberName !== typedDecls[0].node
                    ) {
                        if (typedDecls[0].isConstant) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportConstantRedefinition,
                                DiagnosticRule.reportConstantRedefinition,
                                Localizer.Diagnostic.constantRedefinition().format({ name: node.memberName.value }),
                                node.memberName
                            );
                        }
                    }
                } else {
                    // Is the target a property?
                    const declaredType = getDeclaredTypeOfSymbol(memberInfo.symbol)?.type;
                    if (declaredType && !isProperty(declaredType)) {
                        // Handle the case where there is a class variable defined with the same
                        // name, but there's also now an instance variable introduced. Combine the
                        // type of the class variable with that of the new instance variable.
                        if (!memberInfo.isInstanceMember && isInstanceMember) {
                            // The class variable is accessed in this case.
                            setSymbolAccessed(fileInfo, memberInfo.symbol, node.memberName);
                            const memberType = getTypeOfMember(memberInfo);
                            srcType = combineTypes([srcType, memberType]);
                        }
                    }
                }
            }

            // Look up the member info again, now that we've potentially updated it.
            memberInfo = lookUpClassMember(
                classTypeInfo.classType,
                memberName,
                ClassMemberLookupFlags.DeclaredTypesOnly
            );

            if (!memberInfo && srcExprNode && !isTypeIncomplete) {
                reportPossibleUnknownAssignment(
                    fileInfo.diagnosticRuleSet.reportUnknownMemberType,
                    DiagnosticRule.reportUnknownMemberType,
                    node.memberName,
                    srcType,
                    node,
                    /* ignoreEmptyContainers */ true
                );
            }
        }
    }

    function assignTypeToTupleOrListNode(
        target: TupleNode | ListNode,
        type: Type,
        isTypeIncomplete: boolean,
        srcExpr: ExpressionNode
    ) {
        const targetExpressions = target.nodeType === ParseNodeType.List ? target.entries : target.expressions;

        // Initialize the array of target types, one for each target.
        const targetTypes: Type[][] = new Array(targetExpressions.length);
        for (let i = 0; i < targetExpressions.length; i++) {
            targetTypes[i] = [];
        }
        const targetUnpackIndex = targetExpressions.findIndex((expr) => expr.nodeType === ParseNodeType.Unpack);

        // Do any of the targets use an unpack operator? If so, it will consume all of the
        // entries at that location.
        const unpackIndex = targetExpressions.findIndex((expr) => expr.nodeType === ParseNodeType.Unpack);

        type = makeTopLevelTypeVarsConcrete(type);

        const diagAddendum = new DiagnosticAddendum();

        doForEachSubtype(type, (subtype) => {
            // Is this subtype a tuple?
            const tupleType = getSpecializedTupleType(subtype);
            if (tupleType && tupleType.tupleTypeArguments) {
                const sourceEntryTypes = tupleType.tupleTypeArguments.map((t) =>
                    addConditionToType(t.type, getTypeCondition(subtype))
                );

                const unboundedIndex = tupleType.tupleTypeArguments.findIndex((t) => t.isUnbounded);

                if (unboundedIndex >= 0) {
                    if (sourceEntryTypes.length < targetTypes.length) {
                        const typeToReplicate =
                            sourceEntryTypes.length > 0 ? sourceEntryTypes[unboundedIndex] : AnyType.create();

                        // Add elements to make the count match the target count.
                        while (sourceEntryTypes.length < targetTypes.length) {
                            sourceEntryTypes.splice(unboundedIndex, 0, typeToReplicate);
                        }
                    }
                }

                // If there's an unpack operator in the target and we have too many source elements,
                // combine them to assign to the unpacked target.
                if (targetUnpackIndex >= 0) {
                    if (sourceEntryTypes.length > targetTypes.length) {
                        const removedEntries = sourceEntryTypes.splice(
                            targetUnpackIndex,
                            sourceEntryTypes.length - targetTypes.length + 1
                        );
                        let combinedTypes = combineTypes(removedEntries);
                        if (target.nodeType === ParseNodeType.List) {
                            combinedTypes = stripLiteralValue(combinedTypes);
                        }
                        sourceEntryTypes.splice(targetUnpackIndex, 0, combinedTypes);
                    } else if (sourceEntryTypes.length === targetTypes.length - 1) {
                        sourceEntryTypes.splice(targetUnpackIndex, 0, NeverType.createNever());
                    }
                }

                sourceEntryTypes.forEach((type, targetIndex) => {
                    if (targetIndex < targetTypes.length) {
                        targetTypes[targetIndex].push(type);
                    }
                });

                // Have we accounted for all of the targets and sources? If not, we have a size mismatch.
                if (sourceEntryTypes.length !== targetExpressions.length) {
                    const expectedEntryCount =
                        unpackIndex >= 0 ? targetExpressions.length - 1 : targetExpressions.length;
                    const subDiag = diagAddendum.createAddendum();
                    subDiag.addMessage(
                        (target.nodeType === ParseNodeType.List
                            ? Localizer.DiagnosticAddendum.listAssignmentMismatch()
                            : Localizer.DiagnosticAddendum.tupleAssignmentMismatch()
                        ).format({
                            type: printType(subtype),
                        })
                    );
                    subDiag.createAddendum().addMessage(
                        Localizer.DiagnosticAddendum.tupleSizeMismatch().format({
                            expected: expectedEntryCount,
                            received: sourceEntryTypes.length,
                        })
                    );
                }
            } else {
                // The assigned expression isn't a tuple, so it had better
                // be some iterable type.
                const iterableType =
                    getTypeOfIterator({ type: subtype, isIncomplete: isTypeIncomplete }, /* isAsync */ false, srcExpr)
                        ?.type ?? UnknownType.create();
                for (let index = 0; index < targetExpressions.length; index++) {
                    targetTypes[index].push(addConditionToType(iterableType, getTypeCondition(subtype)));
                }
            }
        });

        if (!diagAddendum.isEmpty()) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(target);
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                (target.nodeType === ParseNodeType.List
                    ? Localizer.Diagnostic.listAssignmentMismatch()
                    : Localizer.Diagnostic.tupleAssignmentMismatch()
                ).format({
                    type: printType(type),
                }) + diagAddendum.getString(),
                target
            );
        }

        // Assign the resulting types to the individual names in the tuple
        // or list target expression.
        targetExpressions.forEach((expr, index) => {
            const typeList = targetTypes[index];
            const targetType = typeList.length === 0 ? UnknownType.create() : combineTypes(typeList);

            assignTypeToExpression(expr, targetType, isTypeIncomplete, srcExpr, /* ignoreEmptyContainers */ true);
        });

        writeTypeCache(target, { type, isIncomplete: isTypeIncomplete }, EvaluatorFlags.None);
    }

    // Replaces all of the top-level TypeVars (as opposed to TypeVars
    // used as type arguments in other types) with their concrete form.
    // If conditionFilter is specified and the TypeVar is a constrained
    // TypeVar, only the conditions that match the filter will be included.
    function makeTopLevelTypeVarsConcrete(
        type: Type,
        makeParamSpecsConcrete = false,
        conditionFilter?: TypeCondition[]
    ): Type {
        return mapSubtypes(type, (subtype) => {
            if (isParamSpec(subtype)) {
                if (subtype.paramSpecAccess === 'args') {
                    if (
                        tupleClassType &&
                        isInstantiableClass(tupleClassType) &&
                        objectType &&
                        isClassInstance(objectType)
                    ) {
                        return ClassType.cloneAsInstance(
                            specializeTupleClass(tupleClassType, [{ type: objectType, isUnbounded: true }])
                        );
                    }

                    return UnknownType.create();
                } else if (subtype.paramSpecAccess === 'kwargs') {
                    if (
                        dictClassType &&
                        isInstantiableClass(dictClassType) &&
                        strClassType &&
                        isInstantiableClass(strClassType) &&
                        objectType &&
                        isClassInstance(objectType)
                    ) {
                        return ClassType.cloneAsInstance(
                            ClassType.cloneForSpecialization(
                                dictClassType,
                                [convertToInstance(strClassType), objectType],
                                /* isTypeArgumentExplicit */ true
                            )
                        );
                    }

                    return UnknownType.create();
                }
            }

            // If this is a function that contains only a ParamSpec (no additional
            // parameters), convert it to a concrete type of (*args: Any, **kwargs: Any).
            if (
                makeParamSpecsConcrete &&
                isFunction(subtype) &&
                subtype.details.parameters.length === 0 &&
                subtype.details.paramSpec
            ) {
                const concreteFunction = FunctionType.createInstance(
                    '',
                    '',
                    '',
                    FunctionTypeFlags.SynthesizedMethod | FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck
                );
                FunctionType.addDefaultParameters(concreteFunction);

                return FunctionType.cloneForParamSpec(subtype, concreteFunction);
            }

            // If this is a TypeVarTuple *Ts, convert it to an unpacked tuple
            // *tuple[*Ts].
            if (isVariadicTypeVar(subtype)) {
                if (tupleClassType && isInstantiableClass(tupleClassType)) {
                    return convertToInstance(
                        specializeTupleClass(
                            tupleClassType,
                            [{ type: subtype, isUnbounded: false }],
                            /* isTypeArgumentExplicit */ true,
                            /* isUnpackedTuple */ true
                        )
                    );
                }
            }

            if (isTypeVar(subtype) && !subtype.details.recursiveTypeAliasName) {
                if (subtype.details.boundType) {
                    const boundType = TypeBase.isInstantiable(subtype)
                        ? convertToInstantiable(subtype.details.boundType)
                        : subtype.details.boundType;

                    return subtype.details.isSynthesized
                        ? boundType
                        : addConditionToType(boundType, [
                              {
                                  typeVarName: TypeVarType.getNameWithScope(subtype),
                                  constraintIndex: 0,
                                  isConstrainedTypeVar: false,
                              },
                          ]);
                }

                // If this is a recursive type alias placeholder
                // that hasn't yet been resolved, return it as is.
                if (subtype.details.recursiveTypeAliasName) {
                    return subtype;
                }

                if (subtype.details.constraints.length > 0) {
                    const typesToCombine: Type[] = [];

                    // Expand the list of constrained subtypes, filtering out any that are
                    // disallowed by the conditionFilter.
                    subtype.details.constraints.forEach((constraintType, constraintIndex) => {
                        if (conditionFilter) {
                            const typeVarName = TypeVarType.getNameWithScope(subtype);
                            const applicableConstraint = conditionFilter.find(
                                (filter) => filter.typeVarName === typeVarName
                            );

                            // If this type variable is being constrained to a single index,
                            // don't include the other indices.
                            if (applicableConstraint && applicableConstraint.constraintIndex !== constraintIndex) {
                                return;
                            }
                        }

                        if (TypeBase.isInstantiable(subtype)) {
                            constraintType = convertToInstantiable(constraintType);
                        }

                        typesToCombine.push(
                            addConditionToType(constraintType, [
                                {
                                    typeVarName: TypeVarType.getNameWithScope(subtype),
                                    constraintIndex,
                                    isConstrainedTypeVar: true,
                                },
                            ])
                        );
                    });

                    return combineTypes(typesToCombine);
                }

                if (subtype.details.isExemptFromBoundCheck) {
                    return AnyType.create();
                }

                // Convert to an "object" or "type" instance depending on whether
                // it's instantiable.
                if (TypeBase.isInstantiable(subtype)) {
                    if (typeClassType && isInstantiableClass(typeClassType)) {
                        return subtype.details.isSynthesized
                            ? typeClassType
                            : addConditionToType(ClassType.cloneAsInstance(typeClassType), [
                                  {
                                      typeVarName: TypeVarType.getNameWithScope(subtype),
                                      constraintIndex: 0,
                                      isConstrainedTypeVar: false,
                                  },
                              ]);
                    }
                } else if (objectType) {
                    return subtype.details.isSynthesized
                        ? objectType
                        : addConditionToType(objectType, [
                              {
                                  typeVarName: TypeVarType.getNameWithScope(subtype),
                                  constraintIndex: 0,
                                  isConstrainedTypeVar: false,
                              },
                          ]);
                }

                return AnyType.create();
            }

            return subtype;
        });
    }

    // Creates a new type by mapping an existing type (which could be a union)
    // to another type or types. The callback is called for each subtype.
    // Top-level TypeVars are expanded (e.g. a bound TypeVar is expanded to
    // its bound type and a constrained TypeVar is expanded to its individual
    // constrained types). If conditionFilter is specified, conditions that
    // do not match will be ignored.
    function mapSubtypesExpandTypeVars(
        type: Type,
        conditionFilter: TypeCondition[] | undefined,
        callback: (expandedSubtype: Type, unexpandedSubtype: Type) => Type | undefined
    ): Type {
        const newSubtypes: Type[] = [];
        let typeChanged = false;

        const expandSubtype = (unexpandedType: Type) => {
            let expandedType = isUnion(unexpandedType) ? unexpandedType : makeTopLevelTypeVarsConcrete(unexpandedType);

            expandedType = transformPossibleRecursiveTypeAlias(expandedType);

            doForEachSubtype(expandedType, (subtype) => {
                if (conditionFilter) {
                    if (!TypeCondition.isCompatible(getTypeCondition(subtype), conditionFilter)) {
                        return undefined;
                    }
                }

                let transformedType = callback(subtype, unexpandedType);
                if (transformedType !== unexpandedType) {
                    typeChanged = true;
                }
                if (transformedType) {
                    // Apply the type condition if it's associated with a constrained TypeVar.
                    const typeCondition = getTypeCondition(subtype)?.filter(
                        (condition) => condition.isConstrainedTypeVar
                    );
                    if (typeCondition && typeCondition.length > 0) {
                        transformedType = addConditionToType(transformedType, typeCondition);
                    }

                    newSubtypes.push(transformedType);
                }
                return undefined;
            });
        };

        if (isUnion(type)) {
            type.subtypes.forEach((subtype) => {
                expandSubtype(subtype);
            });
        } else {
            expandSubtype(type);
        }

        if (!typeChanged) {
            return type;
        }

        const newType = combineTypes(newSubtypes);

        // Do our best to retain type aliases.
        if (newType.category === TypeCategory.Union) {
            UnionType.addTypeAliasSource(newType, type);
        }
        return newType;
    }

    function markNamesAccessed(node: ParseNode, names: string[]) {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const scope = ScopeUtils.getScopeForNode(node);

        if (scope) {
            names.forEach((symbolName) => {
                const symbolInScope = scope.lookUpSymbolRecursive(symbolName);
                if (symbolInScope) {
                    setSymbolAccessed(fileInfo, symbolInScope.symbol, node);
                }
            });
        }
    }

    function assignTypeToExpression(
        target: ExpressionNode,
        type: Type,
        isTypeIncomplete: boolean,
        srcExpr: ExpressionNode,
        ignoreEmptyContainers = false,
        allowAssignmentToFinalVar = false,
        expectedTypeDiagAddendum?: DiagnosticAddendum
    ) {
        // Is the source expression a TypeVar() call?
        if (isTypeVar(type)) {
            if (srcExpr && srcExpr.nodeType === ParseNodeType.Call) {
                const callType = getTypeOfExpression(srcExpr.leftExpression, EvaluatorFlags.DoNotSpecialize).type;
                if (
                    isInstantiableClass(callType) &&
                    (ClassType.isBuiltIn(callType, 'TypeVar') ||
                        ClassType.isBuiltIn(callType, 'TypeVarTuple') ||
                        ClassType.isBuiltIn(callType, 'ParamSpec'))
                ) {
                    if (target.nodeType !== ParseNodeType.Name || target.value !== type.details.name) {
                        addError(
                            type.details.isParamSpec
                                ? Localizer.Diagnostic.paramSpecAssignedName().format({
                                      name: TypeVarType.getReadableName(type),
                                  })
                                : Localizer.Diagnostic.typeVarAssignedName().format({
                                      name: TypeVarType.getReadableName(type),
                                  }),
                            target
                        );
                    }
                }
            }
        }

        // If the type was partially unbound, an error will have already been logged.
        // Remove the unbound before assigning to the target expression so the unbound
        // error doesn't propagate.
        type = removeUnbound(type);

        switch (target.nodeType) {
            case ParseNodeType.Name: {
                assignTypeToNameNode(
                    target,
                    type,
                    isTypeIncomplete,
                    ignoreEmptyContainers,
                    srcExpr,
                    allowAssignmentToFinalVar,
                    expectedTypeDiagAddendum
                );
                break;
            }

            case ParseNodeType.MemberAccess: {
                assignTypeToMemberAccessNode(target, type, isTypeIncomplete, srcExpr, expectedTypeDiagAddendum);
                break;
            }

            case ParseNodeType.Index: {
                const baseTypeResult = getTypeOfExpression(target.baseExpression, EvaluatorFlags.DoNotSpecialize);

                getTypeOfIndexWithBaseType(
                    target,
                    baseTypeResult,
                    {
                        method: 'set',
                        setType: { type, isIncomplete: isTypeIncomplete },
                        setErrorNode: srcExpr,
                        setExpectedTypeDiag: expectedTypeDiagAddendum,
                    },
                    EvaluatorFlags.None
                );

                writeTypeCache(target, { type, isIncomplete: isTypeIncomplete }, EvaluatorFlags.None);
                break;
            }

            case ParseNodeType.List:
            case ParseNodeType.Tuple: {
                assignTypeToTupleOrListNode(target, type, isTypeIncomplete, srcExpr);
                break;
            }

            case ParseNodeType.TypeAnnotation: {
                const annotationType: Type | undefined = getTypeOfAnnotation(target.typeAnnotation, {
                    isVariableAnnotation: true,
                    allowFinal: ParseTreeUtils.isFinalAllowedForAssignmentTarget(target.valueExpression),
                    allowClassVar: ParseTreeUtils.isClassVarAllowedForAssignmentTarget(target.valueExpression),
                });

                // Handle a bare "Final" or "ClassVar" in a special manner.
                const isBareFinalOrClassVar =
                    isClassInstance(annotationType) &&
                    (ClassType.isBuiltIn(annotationType, 'Final') || ClassType.isBuiltIn(annotationType, 'ClassVar'));

                if (!isBareFinalOrClassVar) {
                    const isTypeAliasAnnotation =
                        isClassInstance(annotationType) && ClassType.isBuiltIn(annotationType, 'TypeAlias');

                    if (!isTypeAliasAnnotation) {
                        if (assignType(annotationType, type)) {
                            // Don't attempt to narrow based on the annotated type if the type
                            // is a enum because the annotated type in an enum doesn't reflect
                            // the type of the symbol.
                            if (!isClassInstance(type) || !ClassType.isEnumClass(type)) {
                                type = narrowTypeBasedOnAssignment(annotationType, type);
                            }
                        }
                    }
                }

                assignTypeToExpression(
                    target.valueExpression,
                    type,
                    /* isIncomplete */ false,
                    srcExpr,
                    ignoreEmptyContainers,
                    allowAssignmentToFinalVar,
                    expectedTypeDiagAddendum
                );
                break;
            }

            case ParseNodeType.Unpack: {
                if (target.expression.nodeType === ParseNodeType.Name) {
                    assignTypeToNameNode(
                        target.expression,
                        getBuiltInObject(target.expression, 'list', [type]),
                        /* isIncomplete */ false,
                        ignoreEmptyContainers,
                        srcExpr
                    );
                }
                break;
            }

            case ParseNodeType.Error: {
                // Evaluate the child expression as best we can so the
                // type information is cached for the completion handler.
                if (target.child) {
                    suppressDiagnostics(target.child, () => {
                        getTypeOfExpression(target.child!);
                    });
                }
                break;
            }

            default: {
                addError(Localizer.Diagnostic.assignmentTargetExpr(), target);
                break;
            }
        }
    }

    function verifyRaiseExceptionType(node: RaiseNode) {
        const baseExceptionType = getBuiltInType(node, 'BaseException');

        if (node.typeExpression) {
            const exceptionType = getTypeOfExpression(node.typeExpression).type;

            // Validate that the argument of "raise" is an exception object or class.
            // If it is a class, validate that the class's constructor accepts zero
            // arguments.
            if (exceptionType && baseExceptionType && isInstantiableClass(baseExceptionType)) {
                const diagAddendum = new DiagnosticAddendum();

                doForEachSubtype(exceptionType, (subtype) => {
                    const concreteSubtype = makeTopLevelTypeVarsConcrete(subtype);

                    if (!isAnyOrUnknown(concreteSubtype)) {
                        if (isInstantiableClass(concreteSubtype) && concreteSubtype.literalValue === undefined) {
                            if (
                                !derivesFromClassRecursive(
                                    concreteSubtype,
                                    baseExceptionType,
                                    /* ignoreUnknown */ false
                                )
                            ) {
                                diagAddendum.addMessage(
                                    Localizer.Diagnostic.exceptionTypeIncorrect().format({
                                        type: printType(subtype),
                                    })
                                );
                            } else {
                                let callResult: CallResult | undefined;
                                suppressDiagnostics(node.typeExpression!, () => {
                                    callResult = validateConstructorArguments(
                                        node.typeExpression!,
                                        [],
                                        concreteSubtype,
                                        /* skipUnknownArgCheck */ false,
                                        /* inferenceContext */ undefined
                                    );
                                });

                                if (callResult && callResult.argumentErrors) {
                                    diagAddendum.addMessage(
                                        Localizer.Diagnostic.exceptionTypeNotInstantiable().format({
                                            type: printType(subtype),
                                        })
                                    );
                                }
                            }
                        } else if (isClassInstance(concreteSubtype)) {
                            if (
                                !derivesFromClassRecursive(
                                    ClassType.cloneAsInstantiable(concreteSubtype),
                                    baseExceptionType,
                                    /* ignoreUnknown */ false
                                )
                            ) {
                                diagAddendum.addMessage(
                                    Localizer.Diagnostic.exceptionTypeIncorrect().format({
                                        type: printType(subtype),
                                    })
                                );
                            }
                        } else {
                            diagAddendum.addMessage(
                                Localizer.Diagnostic.exceptionTypeIncorrect().format({
                                    type: printType(subtype),
                                })
                            );
                        }
                    }
                });

                if (!diagAddendum.isEmpty()) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.expectedExceptionClass() + diagAddendum.getString(),
                        node.typeExpression
                    );
                }
            }
        }
    }

    function verifyDeleteExpression(node: ExpressionNode) {
        switch (node.nodeType) {
            case ParseNodeType.Name: {
                // Get the type to evaluate whether it's bound
                // and to mark it accessed.
                getTypeOfExpression(node);
                break;
            }

            case ParseNodeType.MemberAccess: {
                const baseTypeResult = getTypeOfExpression(node.leftExpression, EvaluatorFlags.DoNotSpecialize);
                const memberType = getTypeOfMemberAccessWithBaseType(
                    node,
                    baseTypeResult,
                    { method: 'del' },
                    EvaluatorFlags.None
                );
                writeTypeCache(node.memberName, { type: memberType.type }, EvaluatorFlags.None);
                writeTypeCache(node, { type: memberType.type }, EvaluatorFlags.None);
                break;
            }

            case ParseNodeType.Index: {
                const baseTypeResult = getTypeOfExpression(node.baseExpression, EvaluatorFlags.DoNotSpecialize);
                getTypeOfIndexWithBaseType(node, baseTypeResult, { method: 'del' }, EvaluatorFlags.None);
                writeTypeCache(node, { type: UnboundType.create() }, EvaluatorFlags.None);
                break;
            }

            case ParseNodeType.Tuple: {
                node.expressions.forEach((expr) => {
                    verifyDeleteExpression(expr);
                });
                break;
            }

            case ParseNodeType.Error: {
                // Evaluate the child expression as best we can so the
                // type information is cached for the completion handler.
                if (node.child) {
                    suppressDiagnostics(node.child, () => {
                        getTypeOfExpression(node.child!);
                    });
                }
                break;
            }

            default: {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.delTargetExpr(),
                    node
                );
                break;
            }
        }
    }

    function setSymbolAccessed(fileInfo: AnalyzerFileInfo, symbol: Symbol, node: ParseNode) {
        if (!speculativeTypeTracker.isSpeculative(node)) {
            fileInfo.accessedSymbolSet.add(symbol.id);
        }
    }

    function getReturnTypeFromGenerator(type: Type): Type | undefined {
        if (isAnyOrUnknown(type)) {
            return type;
        }

        if (isClassInstance(type)) {
            // Is this a Generator? If so, return the third
            // type argument, which is the await response type.
            if (ClassType.isBuiltIn(type, 'Generator')) {
                const typeArgs = type.typeArguments;
                if (typeArgs && typeArgs.length >= 3) {
                    return typeArgs[2];
                }
            }
        }

        return undefined;
    }

    function getSpecializedReturnType(
        objType: ClassType,
        memberName: string,
        argList: FunctionArgument[],
        errorNode: ExpressionNode | undefined,
        bindToClass?: ClassType
    ) {
        const classMember = lookUpObjectMember(objType, memberName, ClassMemberLookupFlags.SkipInstanceVariables);
        if (!classMember) {
            return undefined;
        }

        const memberTypeResult = getTypeOfMemberInternal(classMember, objType);
        if (!memberTypeResult) {
            return undefined;
        }

        const memberType = memberTypeResult.type;
        if (isAnyOrUnknown(memberType)) {
            return memberType;
        }

        if (isFunction(memberType) || isOverloadedFunction(memberType)) {
            const methodType = bindFunctionToClassOrObject(
                bindToClass || objType,
                memberType,
                classMember && isInstantiableClass(classMember.classType) ? classMember.classType : undefined,
                errorNode,
                /* recursionCount */ undefined,
                /* treatConstructorAsClassMember */ false,
                /* firstParamType */ bindToClass
            );
            if (methodType) {
                if (isOverloadedFunction(methodType)) {
                    if (errorNode) {
                        const bestOverload = getBestOverloadForArguments(
                            errorNode,
                            { type: methodType, isIncomplete: memberTypeResult.isIncomplete },
                            argList
                        );

                        if (bestOverload) {
                            return getFunctionEffectiveReturnType(bestOverload);
                        }
                    }
                } else {
                    return getFunctionEffectiveReturnType(methodType);
                }
            }
        }

        return undefined;
    }

    function getTypeOfName(node: NameNode, flags: EvaluatorFlags): TypeResult {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const name = node.value;
        let symbol: Symbol | undefined;
        let type: Type | undefined;
        let isIncomplete = false;
        const allowForwardReferences = (flags & EvaluatorFlags.AllowForwardReferences) !== 0 || fileInfo.isStubFile;

        // Does this name refer to a PEP 695-style type parameter?
        const typeParamSymbol = AnalyzerNodeInfo.getTypeParameterSymbol(node);
        if (typeParamSymbol) {
            symbol = typeParamSymbol;
            assert(symbol.getDeclarations().length === 1);
            const decl = getLastTypedDeclaredForSymbol(symbol);
            assert(decl?.type === DeclarationType.TypeParameter);
            type = getTypeOfTypeParameter(decl.node);
            setSymbolAccessed(fileInfo, symbol, node);
        } else {
            // Look for the scope that contains the value definition and
            // see if it has a declared type.
            let symbolWithScope = lookUpSymbolRecursive(
                node,
                name,
                !allowForwardReferences,
                allowForwardReferences && (flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0
            );

            if (!symbolWithScope) {
                // If the node is part of a "from X import Y as Z" statement and the node
                // is the "Y" (non-aliased) name, we need to look up the alias symbol
                // since the non-aliased name is not in the symbol table.
                const alias = getAliasFromImport(node);
                if (alias) {
                    symbolWithScope = lookUpSymbolRecursive(
                        alias,
                        alias.value,
                        !allowForwardReferences,
                        allowForwardReferences && (flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0
                    );
                }
            }

            if (symbolWithScope) {
                let useCodeFlowAnalysis = !allowForwardReferences;

                // If the symbol is implicitly imported from the builtin
                // scope, there's no need to use code flow analysis.
                if (symbolWithScope.scope.type === ScopeType.Builtin) {
                    useCodeFlowAnalysis = false;
                }

                symbol = symbolWithScope.symbol;
                setSymbolAccessed(fileInfo, symbol, node);

                // If we're not supposed to be analyzing this function, skip the remaining work
                // to determine the name's type. Simply evaluate its type as Any.
                if (!fileInfo.diagnosticRuleSet.analyzeUnannotatedFunctions) {
                    const containingFunction = ParseTreeUtils.getEnclosingFunction(node);
                    if (containingFunction && ParseTreeUtils.isUnannotatedFunction(containingFunction)) {
                        return {
                            type: AnyType.create(),
                            isIncomplete: false,
                        };
                    }
                }

                // Get the effective type (either the declared type or the inferred type).
                // If we're using code flow analysis, pass the usage node so we consider
                // only the assignment nodes that are reachable from this usage.
                const effectiveTypeInfo = getEffectiveTypeOfSymbolForUsage(
                    symbol,
                    useCodeFlowAnalysis ? node : undefined
                );
                let effectiveType = transformPossibleRecursiveTypeAlias(effectiveTypeInfo.type);

                if (effectiveTypeInfo.isIncomplete) {
                    if (isUnbound(effectiveType)) {
                        effectiveType = UnknownType.create(/* isIncomplete */ true);
                    }
                    isIncomplete = true;
                }

                if (effectiveTypeInfo.isRecursiveDefinition && isNodeReachable(node)) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.recursiveDefinition().format({ name }),
                        node
                    );
                }

                const isSpecialBuiltIn =
                    !!effectiveType && isInstantiableClass(effectiveType) && ClassType.isSpecialBuiltIn(effectiveType);

                type = effectiveType;
                if (useCodeFlowAnalysis && !isSpecialBuiltIn) {
                    // See if code flow analysis can tell us anything more about the type.
                    // If the symbol is declared outside of our execution scope, use its effective
                    // type. If it's declared inside our execution scope, it generally starts
                    // as unbound at the start of the code flow.
                    let typeAtStart = effectiveType;
                    if (!symbolWithScope.isBeyondExecutionScope && symbol.isInitiallyUnbound()) {
                        typeAtStart = UnboundType.create();

                        // Is this a module-level scope? If so, see if it's an alias of a builtin.
                        if (symbolWithScope.scope.type === ScopeType.Module) {
                            assert(symbolWithScope.scope.parent);
                            const builtInSymbol = symbolWithScope.scope.parent.lookUpSymbol(name);
                            if (builtInSymbol) {
                                const builtInEffectiveType = getEffectiveTypeOfSymbolForUsage(builtInSymbol);
                                typeAtStart = builtInEffectiveType.type;
                            }
                        }
                    }

                    const codeFlowTypeResult = getFlowTypeOfReference(
                        node,
                        symbol.id,
                        typeAtStart,
                        /* startNode */ undefined,
                        {
                            skipConditionalNarrowing: (flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0,
                        }
                    );
                    if (codeFlowTypeResult.type) {
                        type = codeFlowTypeResult.type;
                    }

                    if (codeFlowTypeResult.isIncomplete) {
                        isIncomplete = true;
                    }

                    if (!codeFlowTypeResult.type && symbolWithScope.isBeyondExecutionScope) {
                        const outerScopeTypeResult = getCodeFlowTypeForCapturedVariable(
                            node,
                            symbolWithScope,
                            effectiveType
                        );

                        if (outerScopeTypeResult?.type) {
                            type = outerScopeTypeResult.type;
                        }

                        if (outerScopeTypeResult?.isIncomplete) {
                            isIncomplete = true;
                        }
                    }
                }

                // Detect, report, and fill in missing type arguments if appropriate.
                type = reportMissingTypeArguments(node, type, flags);

                if ((flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0) {
                    // Verify that the name does not refer to a (non type alias) variable.
                    if (effectiveTypeInfo.includesVariableDecl && !type.typeAliasInfo) {
                        let isAllowedTypeForVariable = isTypeVar(type) || isTypeAliasPlaceholder(type);
                        if (isClass(type) && !type.includeSubclasses) {
                            // This check exempts class types that are created by calling
                            // NewType, NamedTuple, and by invoking a metaclass directly.
                            isAllowedTypeForVariable = true;
                        }

                        // Disable for assignments in the typings.pyi file, since it defines special forms.
                        if (!isAllowedTypeForVariable && !fileInfo.isTypingStubFile) {
                            // This might be a union that was previously a type alias
                            // but was reconstituted in such a way that we lost the
                            // typeAliasInfo. Avoid the false positive error by suppressing
                            // the error when it looks like a plausible type alias type.
                            if (
                                effectiveTypeInfo.includesIllegalTypeAliasDecl ||
                                !TypeBase.isInstantiable(type) ||
                                (flags & EvaluatorFlags.DoNotSpecialize) !== 0
                            ) {
                                addDiagnostic(
                                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.typeAnnotationVariable(),
                                    node
                                );
                                type = UnknownType.create();
                            }
                        }
                    }
                }
            } else {
                // Handle the special case of "reveal_type" and "reveal_locals".
                if (name === 'reveal_type' || name === 'reveal_locals') {
                    type = AnyType.create();
                } else {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportUndefinedVariable,
                        DiagnosticRule.reportUndefinedVariable,
                        Localizer.Diagnostic.symbolIsUndefined().format({ name }),
                        node
                    );

                    type = UnknownType.create();
                }
            }
        }

        if (isParamSpec(type)) {
            if (flags & EvaluatorFlags.DisallowParamSpec) {
                addError(Localizer.Diagnostic.paramSpecContext(), node);
                type = UnknownType.create();
            }
        }

        if (
            isTypeVar(type) &&
            !type.details.isParamSpec &&
            !type.isVariadicInUnion &&
            (flags & EvaluatorFlags.ExpectingType) === 0 &&
            type.details.name === name
        ) {
            // Handle the special case of a PEP 604 union. These can appear within
            // an implied type alias where we are not expecting a type.
            const isPep604Union =
                node.parent?.nodeType === ParseNodeType.BinaryOperation &&
                node.parent.operator === OperatorType.BitwiseOr;

            if (!isPep604Union) {
                // A TypeVar in contexts where we're not expecting a type is
                // simply a TypeVar or TypeVarTuple object.
                const typeVarType = type.details.isVariadic
                    ? getTypingType(node, 'TypeVarTuple')
                    : getTypingType(node, 'TypeVar');
                if (typeVarType && isInstantiableClass(typeVarType)) {
                    type = ClassType.cloneAsInstance(typeVarType);
                } else {
                    type = UnknownType.create();
                }
            }
        }

        if ((flags & EvaluatorFlags.ExpectingType) !== 0) {
            if ((flags & EvaluatorFlags.AllowGenericClassType) === 0) {
                if (isInstantiableClass(type) && ClassType.isBuiltIn(type, 'Generic')) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.genericNotAllowed(),
                        node
                    );
                }
            }
        }

        if (isTypeVar(type) && !type.details.isSynthesized) {
            type = validateTypeVarUsage(node, type, flags);
        }

        return { type, isIncomplete };
    }

    // Handles the case where a variable or parameter is defined in an outer
    // scope and captured by an inner scope (either a function or a lambda).
    function getCodeFlowTypeForCapturedVariable(
        node: NameNode,
        symbolWithScope: SymbolWithScope,
        effectiveType: Type
    ): FlowNodeTypeResult | undefined {
        // This function applies only to variables, parameters, and imports, not to other
        // types of symbols.
        if (
            !symbolWithScope.symbol
                .getDeclarations()
                .every(
                    (decl) =>
                        decl.type === DeclarationType.Variable ||
                        decl.type === DeclarationType.Parameter ||
                        decl.type === DeclarationType.Alias
                )
        ) {
            return undefined;
        }

        // If the symbol is a variable captured by an inner function
        // or lambda, see if we can infer the type from the outer scope.
        const scopeHierarchy = ScopeUtils.getScopeHierarchy(node, symbolWithScope.scope);

        // Handle the case where all of the nested scopes are functions,
        // lambdas and modules. Don't allow other types of scopes.
        if (
            scopeHierarchy &&
            scopeHierarchy.length >= 2 &&
            scopeHierarchy.every((s) => s.type === ScopeType.Function || s.type === ScopeType.Module)
        ) {
            // Find the parse node associated with the scope that is just inside of the
            // scope that declares the captured variable.
            const innerScopeNode = ScopeUtils.findTopNodeInScope(node, scopeHierarchy[scopeHierarchy.length - 2]);
            if (
                innerScopeNode &&
                (innerScopeNode.nodeType === ParseNodeType.Function || innerScopeNode.nodeType === ParseNodeType.Lambda)
            ) {
                const innerScopeCodeFlowNode = AnalyzerNodeInfo.getFlowNode(innerScopeNode);
                if (innerScopeCodeFlowNode) {
                    // See if any of the assignments of the symbol are reachable
                    // from this node. If so, we cannot apply any narrowing because
                    // the type could change after the capture.
                    if (
                        symbolWithScope.symbol.getDeclarations().every((decl) => {
                            // Parameter declarations always start life at the beginning
                            // of the execution scope, so they are always safe to narrow.
                            if (decl.type === DeclarationType.Parameter) {
                                return true;
                            }

                            // Assume alias declarations are also always safe to narrow.
                            if (decl.type === DeclarationType.Alias) {
                                return true;
                            }

                            const declCodeFlowNode = AnalyzerNodeInfo.getFlowNode(decl.node);
                            if (!declCodeFlowNode) {
                                return false;
                            }

                            return !codeFlowEngine.isFlowNodeReachable(
                                declCodeFlowNode,
                                innerScopeCodeFlowNode,
                                /* ignoreNoReturn */ true
                            );
                        })
                    ) {
                        return getFlowTypeOfReference(node, symbolWithScope.symbol.id, effectiveType, innerScopeNode);
                    }
                }
            }
        }

        return undefined;
    }

    // Validates that a TypeVar is valid in this context. If so, it clones it
    // and provides a scope ID defined by its containing scope (class, function
    // or type alias). If not, it emits errors indicating why the TypeVar
    // cannot be used in this location.
    function validateTypeVarUsage(node: ExpressionNode, type: TypeVarType, flags: EvaluatorFlags) {
        if (TypeBase.isInstantiable(type) && !type.scopeId && !isTypeAliasPlaceholder(type)) {
            const scopedTypeVarInfo = findScopedTypeVar(node, type);
            type = scopedTypeVarInfo.type;

            if ((flags & EvaluatorFlags.DisallowTypeVarsWithScopeId) !== 0 && type.scopeId !== undefined) {
                if (!type.details.isSynthesized && !type.details.isParamSpec) {
                    // This TypeVar already has a scope ID assigned to it. See if it
                    // originates from type parameter syntax. If so, allow it.
                    if (type.details.isTypeParamSyntax) {
                        return type;
                    }

                    // If this type variable expression is used within a generic class,
                    // function, or type alias that uses type parameter syntax, there is
                    // no need to report an error here.
                    const typeVarScopeNode = ParseTreeUtils.getTypeVarScopeNode(node);
                    if (
                        typeVarScopeNode &&
                        typeVarScopeNode.typeParameters &&
                        !typeVarScopeNode.typeParameters.parameters.some((t) => t.name === node)
                    ) {
                        return type;
                    }

                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarUsedByOuterScope().format({ name: type.details.name }),
                        node
                    );
                }
            } else if ((flags & EvaluatorFlags.AssociateTypeVarsWithCurrentScope) !== 0) {
                if (type.scopeId === undefined) {
                    if (!scopedTypeVarInfo.foundInterveningClass) {
                        let enclosingScope = ParseTreeUtils.getEnclosingClassOrFunction(node);

                        // Handle P.args and P.kwargs as a special case for inner functions.
                        if (
                            enclosingScope &&
                            node.parent?.nodeType === ParseNodeType.MemberAccess &&
                            node.parent.leftExpression === node
                        ) {
                            const memberName = node.parent.memberName.value;
                            if (memberName === 'args' || memberName === 'kwargs') {
                                const outerFunctionScope = ParseTreeUtils.getEnclosingClassOrFunction(enclosingScope);

                                if (outerFunctionScope?.nodeType === ParseNodeType.Function) {
                                    if (scopedTypeVarInfo.isRescoped) {
                                        addDiagnostic(
                                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet
                                                .reportGeneralTypeIssues,
                                            DiagnosticRule.reportGeneralTypeIssues,
                                            Localizer.Diagnostic.paramSpecScopedToReturnType().format({
                                                name: type.details.name,
                                            }),
                                            node
                                        );
                                    } else {
                                        enclosingScope = outerFunctionScope;
                                    }
                                } else if (!scopedTypeVarInfo.type.scopeId) {
                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.paramSpecNotUsedByOuterScope().format({
                                            name: type.details.name,
                                        }),
                                        node
                                    );
                                }
                            }
                        }

                        if (enclosingScope) {
                            // If the enclosing scope is using type parameter syntax, traditional
                            // type variables can't be used in this context.
                            if (
                                enclosingScope.typeParameters &&
                                !enclosingScope.typeParameters.parameters.some(
                                    (param) => param.name.value === type.details.name
                                )
                            ) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.typeParameterNotDeclared().format({
                                        name: type.details.name,
                                        container: enclosingScope.name.value,
                                    }),
                                    node
                                );
                            }

                            type = TypeVarType.cloneForScopeId(
                                type,
                                getScopeIdForNode(enclosingScope),
                                enclosingScope.name.value,
                                enclosingScope.nodeType === ParseNodeType.Function
                                    ? TypeVarScopeType.Function
                                    : TypeVarScopeType.Class
                            );
                        } else {
                            fail('AssociateTypeVarsWithCurrentScope flag was set but enclosing scope not found');
                        }
                    } else {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeVarUsedByOuterScope().format({ name: type.details.name }),
                            node
                        );
                    }
                }
            } else if ((flags & EvaluatorFlags.DisallowTypeVarsWithoutScopeId) !== 0) {
                if (
                    (type.scopeId === undefined || scopedTypeVarInfo.foundInterveningClass) &&
                    !type.details.isSynthesized
                ) {
                    let message: ParameterizedString<{ name: string }>;
                    if (scopedTypeVarInfo.isRescoped) {
                        message = isParamSpec(type)
                            ? Localizer.Diagnostic.paramSpecScopedToReturnType()
                            : Localizer.Diagnostic.typeVarScopedToReturnType();
                    } else {
                        message = isParamSpec(type)
                            ? Localizer.Diagnostic.paramSpecNotUsedByOuterScope()
                            : Localizer.Diagnostic.typeVarNotUsedByOuterScope();
                    }
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        message.format({ name: type.details.name }),
                        node
                    );
                }
            }
        }

        // If this type var is variadic, the name refers to the packed form. It
        // must be unpacked in most contexts.
        if (isUnpackedVariadicTypeVar(type)) {
            type = TypeVarType.cloneForPacked(type);
        }

        return type;
    }

    // Determines if the type is a generic class or type alias with missing
    // type arguments. If so, it fills in these type arguments with Unknown
    // and optionally reports an error.
    function reportMissingTypeArguments(node: ExpressionNode, type: Type, flags: EvaluatorFlags): Type {
        if ((flags & EvaluatorFlags.DoNotSpecialize) !== 0) {
            return type;
        }

        // Is this a generic class that needs to be specialized?
        if (isInstantiableClass(type)) {
            if ((flags & EvaluatorFlags.ExpectingType) !== 0) {
                if (requiresTypeArguments(type) && !type.typeArguments) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportMissingTypeArgument,
                        DiagnosticRule.reportMissingTypeArgument,
                        Localizer.Diagnostic.typeArgsMissingForClass().format({
                            name: type.aliasName || type.details.name,
                        }),
                        node
                    );
                }
            }
            if (!type.typeArguments) {
                type = createSpecializedClassType(type, /* typeArgs */ undefined, flags, node)?.type;
            }
        }

        // Is this a generic type alias that needs to be specialized?
        if (
            (flags & EvaluatorFlags.ExpectingType) !== 0 &&
            type.typeAliasInfo &&
            type.typeAliasInfo.typeParameters &&
            type.typeAliasInfo.typeParameters.length > 0 &&
            !type.typeAliasInfo.typeArguments
        ) {
            let reportMissingTypeArguments = false;
            const defaultTypeArgs: Type[] = [];
            const typeVarContext = new TypeVarContext(type.typeAliasInfo.typeVarScopeId);

            type.typeAliasInfo.typeParameters.forEach((param) => {
                if (!param.details.defaultType) {
                    reportMissingTypeArguments = true;
                }

                let defaultType: Type;
                if (param.details.defaultType || param.details.isParamSpec) {
                    defaultType = applySolvedTypeVars(param, typeVarContext, { unknownIfNotFound: true });
                } else {
                    defaultType = UnknownType.create();
                }

                defaultTypeArgs.push(defaultType);
                typeVarContext.setTypeVarType(param, defaultType);
            });

            if (reportMissingTypeArguments) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportMissingTypeArgument,
                    DiagnosticRule.reportMissingTypeArgument,
                    Localizer.Diagnostic.typeArgsMissingForAlias().format({
                        name: type.typeAliasInfo.name,
                    }),
                    node
                );
            }

            type = TypeBase.cloneForTypeAlias(
                applySolvedTypeVars(type, typeVarContext, { unknownIfNotFound: true }),
                type.typeAliasInfo.name,
                type.typeAliasInfo.fullName,
                type.typeAliasInfo.typeVarScopeId,
                type.typeAliasInfo.typeParameters,
                defaultTypeArgs
            );
        }

        return type;
    }

    // Creates an ID that identifies this parse node in a way that will
    // not change each time the file is parsed (unless, of course, the
    // file contents change).
    function getScopeIdForNode(node: ParseNode): string {
        let name = '';
        if (node.nodeType === ParseNodeType.Class) {
            name = node.name.value;
        } else if (node.nodeType === ParseNodeType.Function) {
            name = node.name.value;
        }

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        return `${fileInfo.filePath}.${node.start.toString()}-${name}`;
    }

    // Walks up the parse tree and finds all scopes that can provide
    // a context for a TypeVar and returns the scope ID for each.
    function getTypeVarScopesForNode(node: ParseNode): TypeVarScopeId[] {
        const scopeIds: TypeVarScopeId[] = [];

        let curNode: ParseNode | undefined = node;
        while (curNode) {
            curNode = ParseTreeUtils.getTypeVarScopeNode(curNode);
            if (!curNode) {
                break;
            }

            scopeIds.push(getScopeIdForNode(curNode));
            curNode = curNode.parent;
        }

        return scopeIds;
    }

    // Walks up the parse tree to find a function, class, or type alias
    // declaration that provides the context for a type variable.
    function findScopedTypeVar(node: ExpressionNode, type: TypeVarType): ScopedTypeVarResult {
        let curNode: ParseNode | undefined = node;
        let nestedClassCount = 0;

        assert(TypeBase.isInstantiable(type));

        while (curNode) {
            curNode = ParseTreeUtils.getTypeVarScopeNode(curNode);
            if (!curNode) {
                break;
            }

            let typeParametersForScope: TypeVarType[] | undefined;
            let scopeUsesTypeParameterSyntax = false;

            if (curNode.nodeType === ParseNodeType.Class) {
                const classTypeInfo = getTypeOfClass(curNode);
                if (classTypeInfo && !ClassType.isPartiallyEvaluated(classTypeInfo.classType)) {
                    typeParametersForScope = classTypeInfo.classType.details.typeParameters;
                }

                scopeUsesTypeParameterSyntax = !!curNode.typeParameters;
                nestedClassCount++;
            } else if (curNode.nodeType === ParseNodeType.Function) {
                const functionTypeInfo = getTypeOfFunction(curNode);
                if (functionTypeInfo) {
                    const functionDetails = functionTypeInfo.functionType.details;
                    typeParametersForScope = functionDetails.typeParameters;

                    // Was this type parameter "rescoped" to a callable found within the
                    // return type annotation? If so, it is not available for use within
                    // the function body.
                    if (functionDetails.rescopedTypeParameters?.some((tp) => tp.details.name === type.details.name)) {
                        return { type, isRescoped: true, foundInterveningClass: false };
                    }
                }

                scopeUsesTypeParameterSyntax = !!curNode.typeParameters;
            } else if (curNode.nodeType === ParseNodeType.TypeAlias) {
                scopeUsesTypeParameterSyntax = !!curNode.typeParameters;
            }

            if (typeParametersForScope) {
                const match = typeParametersForScope.find((typeVar) => typeVar.details.name === type.details.name);

                if (match?.scopeId) {
                    // Use the scoped version of the TypeVar rather than the (unscoped) original type.
                    type = TypeVarType.cloneForScopeId(type, match.scopeId, match.scopeName!, match.scopeType!);
                    return {
                        type,
                        isRescoped: false,
                        foundInterveningClass: nestedClassCount > 1 && !scopeUsesTypeParameterSyntax,
                    };
                }
            }

            curNode = curNode.parent;
        }

        // See if this is part of an assignment statement that is defining a type alias.
        curNode = node;
        while (curNode) {
            let leftType: Type | undefined;
            let typeAliasNode: TypeAliasNode | undefined = undefined;

            if (curNode.nodeType === ParseNodeType.TypeAlias) {
                leftType = readTypeCache(curNode.name, EvaluatorFlags.None);
                typeAliasNode = curNode;
            } else if (curNode.nodeType === ParseNodeType.Assignment) {
                leftType = readTypeCache(curNode.leftExpression, EvaluatorFlags.None);
            }

            if (leftType) {
                // Is this a placeholder that was temporarily written to the cache for
                // purposes of resolving type aliases?
                if (
                    leftType &&
                    isTypeVar(leftType) &&
                    leftType.details.recursiveTypeAliasScopeId &&
                    leftType.details.recursiveTypeAliasName
                ) {
                    // Type alias statements cannot be used with old-style type variables.
                    if (typeAliasNode && !type.details.isTypeParamSyntax) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeParameterNotDeclared().format({
                                name: type.details.name,
                                container: typeAliasNode.name.value,
                            }),
                            node
                        );
                    }

                    return {
                        type: TypeVarType.cloneForScopeId(
                            type,
                            leftType.details.recursiveTypeAliasScopeId,
                            leftType.details.recursiveTypeAliasName,
                            TypeVarScopeType.TypeAlias
                        ),
                        isRescoped: false,
                        foundInterveningClass: false,
                    };
                }
            }

            curNode = curNode.parent;
        }

        // Return the original type.
        return { type, isRescoped: false, foundInterveningClass: false };
    }

    function getTypeOfMemberAccess(node: MemberAccessNode, flags: EvaluatorFlags): TypeResult {
        const baseTypeFlags =
            EvaluatorFlags.DoNotSpecialize |
            (flags &
                (EvaluatorFlags.ExpectingTypeAnnotation |
                    EvaluatorFlags.VariableTypeAnnotation |
                    EvaluatorFlags.AllowForwardReferences |
                    EvaluatorFlags.NotParsedByInterpreter |
                    EvaluatorFlags.DisallowTypeVarsWithScopeId |
                    EvaluatorFlags.DisallowTypeVarsWithoutScopeId |
                    EvaluatorFlags.AssociateTypeVarsWithCurrentScope));
        const baseTypeResult = getTypeOfExpression(node.leftExpression, baseTypeFlags);

        if (isTypeAliasPlaceholder(baseTypeResult.type)) {
            return {
                type: UnknownType.create(/* isIncomplete */ true),
                isIncomplete: true,
            };
        }

        const typeResult = getTypeOfMemberAccessWithBaseType(
            node,
            baseTypeResult,
            { method: 'get' },
            flags | EvaluatorFlags.DoNotSpecialize
        );

        if (isCodeFlowSupportedForReference(node)) {
            // Before performing code flow analysis, update the cache to prevent recursion.
            writeTypeCache(node, { ...typeResult, isIncomplete: true }, flags);
            writeTypeCache(node.memberName, { ...typeResult, isIncomplete: true }, flags);

            // If the type is initially unbound, see if there's a parent class that
            // potentially initialized the value.
            let typeAtStart = typeResult.type;
            let isTypeAtStartIncomplete = !!typeResult.isIncomplete;
            if (isUnbound(typeAtStart)) {
                const baseType = makeTopLevelTypeVarsConcrete(baseTypeResult.type);

                let classMemberInfo: ClassMember | undefined;
                if (isInstantiableClass(baseType)) {
                    classMemberInfo = lookUpClassMember(
                        baseType,
                        node.memberName.value,
                        ClassMemberLookupFlags.SkipOriginalClass
                    );
                } else if (isClassInstance(baseType)) {
                    classMemberInfo = lookUpObjectMember(
                        baseType,
                        node.memberName.value,
                        ClassMemberLookupFlags.SkipOriginalClass
                    );
                }

                if (classMemberInfo) {
                    typeAtStart = getTypeOfMember(classMemberInfo);
                    isTypeAtStartIncomplete = false;
                }
            }

            // See if we can refine the type based on code flow analysis.
            const codeFlowTypeResult = getFlowTypeOfReference(
                node,
                indeterminateSymbolId,
                typeAtStart,
                /* startNode */ undefined,
                {
                    isTypeAtStartIncomplete,
                    skipConditionalNarrowing: (flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0,
                }
            );

            if (codeFlowTypeResult.type) {
                typeResult.type = codeFlowTypeResult.type;
            }

            if (codeFlowTypeResult.isIncomplete) {
                typeResult.isIncomplete = true;
            }

            // Detect, report, and fill in missing type arguments if appropriate.
            typeResult.type = reportMissingTypeArguments(node, typeResult.type, flags);
        }

        if (baseTypeResult.isIncomplete) {
            typeResult.isIncomplete = true;
        }

        // Cache the type information in the member name node.
        writeTypeCache(node.memberName, typeResult, flags);

        return typeResult;
    }

    function getTypeOfMemberAccessWithBaseType(
        node: MemberAccessNode,
        baseTypeResult: TypeResult,
        usage: EvaluatorUsage,
        flags: EvaluatorFlags
    ): TypeResult {
        let baseType = baseTypeResult.type;
        const memberName = node.memberName.value;
        let diag = new DiagnosticAddendum();
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        let type: Type | undefined;
        let isIncomplete = !!baseTypeResult.isIncomplete;
        let isAsymmetricDescriptor: boolean | undefined;
        const isRequired = false;
        const isNotRequired = false;

        // If the base type was incomplete and unbound, don't proceed
        // because false positive errors will be generated.
        if (baseTypeResult.isIncomplete && isUnbound(baseTypeResult.type)) {
            return { type: UnknownType.create(/* isIncomplete */ true), isIncomplete: true };
        }

        // Handle the special case where the expression is an actual
        // UnionType special form.
        if (isUnion(baseType) && TypeBase.isSpecialForm(baseType)) {
            if (objectType) {
                baseType = objectType;
            }
        }

        const getTypeOfNoneBase = (subtype: NoneType) => {
            if (noneType && isInstantiableClass(noneType)) {
                if (TypeBase.isInstance(subtype)) {
                    return getTypeOfObjectMember(
                        node.memberName,
                        ClassType.cloneAsInstance(noneType),
                        memberName,
                        usage,
                        diag
                    );
                } else {
                    return getTypeOfClassMember(node.memberName, noneType, memberName, usage, diag);
                }
            }
            return undefined;
        };

        if (isParamSpec(baseType) && baseType.paramSpecAccess) {
            baseType = makeTopLevelTypeVarsConcrete(baseType);
        }

        switch (baseType.category) {
            case TypeCategory.Any:
            case TypeCategory.Unknown:
            case TypeCategory.Never: {
                type = baseType;
                break;
            }

            case TypeCategory.TypeVar: {
                if (baseType.details.isParamSpec) {
                    if (memberName === 'args') {
                        const paramNode = ParseTreeUtils.getEnclosingParameter(node);
                        if (!paramNode || paramNode.category !== ParameterCategory.VarArgList) {
                            addError(Localizer.Diagnostic.paramSpecArgsUsage(), node);
                            return { type: UnknownType.create(isIncomplete), isIncomplete };
                        }
                        return { type: TypeVarType.cloneForParamSpecAccess(baseType, 'args'), isIncomplete };
                    }

                    if (memberName === 'kwargs') {
                        const paramNode = ParseTreeUtils.getEnclosingParameter(node);
                        if (!paramNode || paramNode.category !== ParameterCategory.VarArgDictionary) {
                            addError(Localizer.Diagnostic.paramSpecKwargsUsage(), node);
                            return { type: UnknownType.create(isIncomplete), isIncomplete };
                        }
                        return { type: TypeVarType.cloneForParamSpecAccess(baseType, 'kwargs'), isIncomplete };
                    }

                    if (!isIncomplete) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.paramSpecUnknownMember().format({ name: memberName }),
                            node
                        );
                    }
                    return { type: UnknownType.create(isIncomplete), isIncomplete };
                }

                if (flags & EvaluatorFlags.ExpectingType) {
                    if (!isIncomplete) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeVarNoMember().format({
                                type: printType(baseType),
                                name: memberName,
                            }),
                            node.leftExpression
                        );
                    }

                    return { type: UnknownType.create(isIncomplete), isIncomplete };
                }

                if (baseType.details.recursiveTypeAliasName) {
                    return { type: UnknownType.create(/* isIncomplete */ true), isIncomplete: true };
                }

                return getTypeOfMemberAccessWithBaseType(
                    node,
                    {
                        type: makeTopLevelTypeVarsConcrete(baseType),
                        bindToType: baseType,
                        isIncomplete,
                    },
                    usage,
                    EvaluatorFlags.None
                );
            }

            case TypeCategory.Class: {
                if (TypeBase.isInstantiable(baseType)) {
                    const typeResult = getTypeOfClassMember(
                        node.memberName,
                        baseType,
                        memberName,
                        usage,
                        diag,
                        MemberAccessFlags.None,
                        baseTypeResult.bindToType
                    );

                    type = typeResult?.type;
                    if (typeResult?.isIncomplete) {
                        isIncomplete = true;
                    }

                    if (typeResult?.isAsymmetricDescriptor) {
                        isAsymmetricDescriptor = true;
                    }
                } else {
                    // Handle the special case of 'name' and 'value' members within an enum.
                    const enumMemberResult = getTypeOfEnumMember(
                        evaluatorInterface,
                        node,
                        baseType,
                        memberName,
                        isIncomplete
                    );
                    if (enumMemberResult) {
                        return enumMemberResult;
                    }

                    const typeResult = getTypeOfObjectMember(
                        node.memberName,
                        baseType,
                        memberName,
                        usage,
                        diag,
                        /* memberAccessFlags */ undefined,
                        baseTypeResult.bindToType
                    );

                    if (typeResult) {
                        type = addConditionToType(typeResult.type, getTypeCondition(baseType));
                    }

                    if (typeResult?.isIncomplete) {
                        isIncomplete = true;
                    }

                    if (typeResult?.isAsymmetricDescriptor) {
                        isAsymmetricDescriptor = true;
                    }
                }
                break;
            }

            case TypeCategory.Module: {
                const symbol = ModuleType.getField(baseType, memberName);
                if (symbol && !symbol.isExternallyHidden()) {
                    if (usage.method === 'get') {
                        setSymbolAccessed(AnalyzerNodeInfo.getFileInfo(node), symbol, node.memberName);
                    }

                    type = getEffectiveTypeOfSymbolForUsage(
                        symbol,
                        /* usageNode */ undefined,
                        /* useLastDecl */ true
                    ).type;

                    if (isTypeVar(type)) {
                        type = validateTypeVarUsage(node, type, flags);
                    }

                    // If the type resolved to "unbound", treat it as "unknown" in
                    // the case of a module reference because if it's truly unbound,
                    // that error will be reported within the module and should not
                    // leak into other modules that import it.
                    if (isUnbound(type)) {
                        type = UnknownType.create(/* isIncomplete */ true);
                    }

                    if (symbol.isPrivateMember()) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportPrivateUsage,
                            DiagnosticRule.reportPrivateUsage,
                            Localizer.Diagnostic.privateUsedOutsideOfModule().format({
                                name: memberName,
                            }),
                            node.memberName
                        );
                    }

                    if (symbol.isPrivatePyTypedImport()) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportPrivateImportUsage,
                            DiagnosticRule.reportPrivateImportUsage,
                            Localizer.Diagnostic.privateImportFromPyTypedModule().format({
                                name: memberName,
                                module: baseType.moduleName,
                            }),
                            node.memberName
                        );
                    }
                } else {
                    // Does the module export a top-level __getattr__ function?
                    if (usage.method === 'get') {
                        const getAttrSymbol = ModuleType.getField(baseType, '__getattr__');
                        if (getAttrSymbol) {
                            const isModuleGetAttrSupported =
                                fileInfo.executionEnvironment.pythonVersion >= PythonVersion.V3_7 ||
                                getAttrSymbol
                                    .getDeclarations()
                                    .some((decl) => decl.path.toLowerCase().endsWith('.pyi'));

                            if (isModuleGetAttrSupported) {
                                const getAttrTypeResult = getEffectiveTypeOfSymbolForUsage(getAttrSymbol);
                                if (isFunction(getAttrTypeResult.type)) {
                                    type = getFunctionEffectiveReturnType(getAttrTypeResult.type);
                                    if (getAttrTypeResult.isIncomplete) {
                                        isIncomplete = true;
                                    }
                                }
                            }
                        }
                    }

                    // If the field was not found and the module type is marked
                    // such that all fields should be Any/Unknown, return that type.
                    if (!type && baseType.notPresentFieldType) {
                        type = baseType.notPresentFieldType;
                    }

                    if (!type) {
                        if (!isIncomplete) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.moduleUnknownMember().format({
                                    memberName,
                                    moduleName: baseType.moduleName,
                                }),
                                node.memberName
                            );
                        }
                        type = evaluatorOptions.evaluateUnknownImportsAsAny ? AnyType.create() : UnknownType.create();
                    }
                }
                break;
            }

            case TypeCategory.Union: {
                type = mapSubtypes(baseType, (subtype) => {
                    if (isNoneInstance(subtype)) {
                        const typeResult = getTypeOfNoneBase(subtype);
                        if (typeResult) {
                            type = addConditionToType(typeResult.type, getTypeCondition(baseType));
                            if (typeResult.isIncomplete) {
                                isIncomplete = true;
                            }
                            return type;
                        } else {
                            if (!isIncomplete) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportOptionalMemberAccess,
                                    DiagnosticRule.reportOptionalMemberAccess,
                                    Localizer.Diagnostic.noneUnknownMember().format({ name: memberName }),
                                    node.memberName
                                );
                            }
                            return undefined;
                        }
                    } else if (isUnbound(subtype)) {
                        // Don't do anything if it's unbound. The error will already
                        // be reported elsewhere.
                        return undefined;
                    } else {
                        const typeResult = getTypeOfMemberAccessWithBaseType(
                            node,
                            {
                                type: subtype,
                                isIncomplete: baseTypeResult.isIncomplete,
                            },
                            usage,
                            EvaluatorFlags.None
                        );
                        if (typeResult.isIncomplete) {
                            isIncomplete = true;
                        }
                        return typeResult.type;
                    }
                });
                break;
            }

            case TypeCategory.Function:
            case TypeCategory.OverloadedFunction: {
                if (memberName === '__defaults__') {
                    // The "__defaults__" member is not currently defined in the "function"
                    // class, so we'll special-case it here.
                    type = AnyType.create();
                } else if (memberName === '__self__') {
                    // The "__self__" member is not currently defined in the "function"
                    // class, so we'll special-case it here.
                    const functionType = isFunction(baseType) ? baseType : baseType.overloads[0];
                    if (
                        functionType.preBoundFlags !== undefined &&
                        (functionType.preBoundFlags & FunctionTypeFlags.StaticMethod) === 0
                    ) {
                        type = functionType.boundToType;
                    }
                } else {
                    if (!functionObj) {
                        type = AnyType.create();
                    } else {
                        type = getTypeOfMemberAccessWithBaseType(node, { type: functionObj }, usage, flags).type;
                    }
                }
                break;
            }

            case TypeCategory.None: {
                const typeResult = getTypeOfNoneBase(baseType);
                if (typeResult) {
                    type = addConditionToType(typeResult.type, getTypeCondition(baseType));
                    if (typeResult.isIncomplete) {
                        isIncomplete = true;
                    }
                }
                break;
            }

            default:
                diag.addMessage(Localizer.DiagnosticAddendum.typeUnsupported().format({ type: printType(baseType) }));
                break;
        }

        if (!type) {
            const isFunctionRule =
                isFunction(baseType) ||
                isOverloadedFunction(baseType) ||
                (isClassInstance(baseType) && ClassType.isBuiltIn(baseType, 'function'));

            if (!baseTypeResult.isIncomplete) {
                let diagMessage = Localizer.Diagnostic.memberAccess();
                if (usage.method === 'set') {
                    diagMessage = Localizer.Diagnostic.memberSet();
                } else if (usage.method === 'del') {
                    diagMessage = Localizer.Diagnostic.memberDelete();
                }

                // If there is an expected type diagnostic addendum (used for assignments),
                // use that rather than the local diagnostic addendum because it will be
                // more informative.
                if (usage.setExpectedTypeDiag) {
                    diag = usage.setExpectedTypeDiag;
                }

                const [ruleSet, rule] = isFunctionRule
                    ? [fileInfo.diagnosticRuleSet.reportFunctionMemberAccess, DiagnosticRule.reportFunctionMemberAccess]
                    : [fileInfo.diagnosticRuleSet.reportGeneralTypeIssues, DiagnosticRule.reportGeneralTypeIssues];

                addDiagnostic(
                    ruleSet,
                    rule,
                    diagMessage.format({ name: memberName, type: printType(baseType) }) + diag.getString(),
                    node.memberName,
                    diag.getEffectiveTextRange() ?? node.memberName
                );
            }

            // If this is member access on a function, use "Any" so if the
            // reportFunctionMemberAccess rule is disabled, we don't trigger
            // additional reportUnknownMemberType diagnostics.
            type = isFunctionRule ? AnyType.create() : UnknownType.create();
        }

        // Should we specialize the class?
        if ((flags & EvaluatorFlags.DoNotSpecialize) === 0) {
            if (isInstantiableClass(type) && !type.typeArguments) {
                type = createSpecializedClassType(type, /* typeArgs */ undefined, flags, node)?.type;
            }
        }

        if (usage.method === 'get') {
            let skipPartialUnknownCheck = isIncomplete;

            // Don't report an error if the type is a partially-specialized
            // class being passed as an argument. This comes up frequently in
            // cases where a type is passed as an argument (e.g. "defaultdict(list)").
            // It can also come up in cases like "isinstance(x, (list, dict))".
            if (isInstantiableClass(type)) {
                const argNode = ParseTreeUtils.getParentNodeOfType(node, ParseNodeType.Argument);
                if (argNode && argNode?.parent?.nodeType === ParseNodeType.Call) {
                    skipPartialUnknownCheck = true;
                }
            }

            if (!skipPartialUnknownCheck) {
                reportPossibleUnknownAssignment(
                    fileInfo.diagnosticRuleSet.reportUnknownMemberType,
                    DiagnosticRule.reportUnknownMemberType,
                    node.memberName,
                    type,
                    node,
                    /* ignoreEmptyContainers */ false
                );
            }
        }

        return { type, isIncomplete, isAsymmetricDescriptor, isRequired, isNotRequired };
    }

    function getTypeOfClassMemberName(
        errorNode: ExpressionNode,
        classType: ClassType,
        isAccessedThroughObject: boolean,
        memberName: string,
        usage: EvaluatorUsage,
        diag: DiagnosticAddendum | undefined,
        flags: MemberAccessFlags,
        bindToType?: ClassType | TypeVarType
    ): ClassMemberLookup | undefined {
        let classLookupFlags = ClassMemberLookupFlags.Default;
        if (flags & MemberAccessFlags.AccessClassMembersOnly) {
            classLookupFlags |= ClassMemberLookupFlags.SkipInstanceVariables;
        }
        if (flags & MemberAccessFlags.SkipBaseClasses) {
            classLookupFlags |= ClassMemberLookupFlags.SkipBaseClasses;
        }
        if (flags & MemberAccessFlags.SkipObjectBaseClass) {
            classLookupFlags |= ClassMemberLookupFlags.SkipObjectBaseClass;
        }
        if (flags & MemberAccessFlags.SkipTypeBaseClass) {
            classLookupFlags |= ClassMemberLookupFlags.SkipTypeBaseClass;
        }
        if (flags & MemberAccessFlags.SkipOriginalClass) {
            classLookupFlags |= ClassMemberLookupFlags.SkipOriginalClass;
        }

        // Always look for a member with a declared type first.
        let memberInfo = lookUpClassMember(
            classType,
            memberName,
            classLookupFlags | ClassMemberLookupFlags.DeclaredTypesOnly
        );

        // If we couldn't find a symbol with a declared type, use
        // a symbol with an inferred type.
        if (!memberInfo) {
            memberInfo = lookUpClassMember(classType, memberName, classLookupFlags);
        }

        if (memberInfo) {
            let type: Type | undefined;
            let isTypeIncomplete = false;

            if (memberInfo.symbol.isInitVar()) {
                diag?.addMessage(Localizer.DiagnosticAddendum.memberIsInitVar().format({ name: memberName }));
                return undefined;
            }

            if (usage.method !== 'get') {
                // If the usage indicates a 'set' or 'delete' and the access is within the
                // class definition itself, use only the declared type to avoid circular
                // type evaluation.
                const containingClass = ParseTreeUtils.getEnclosingClass(errorNode);
                if (containingClass) {
                    const containingClassType = getTypeOfClass(containingClass)?.classType;
                    if (
                        containingClassType &&
                        isInstantiableClass(containingClassType) &&
                        ClassType.isSameGenericClass(containingClassType, classType)
                    ) {
                        type = getDeclaredTypeOfSymbol(memberInfo.symbol)?.type;
                        if (type && isInstantiableClass(memberInfo.classType)) {
                            type = partiallySpecializeType(type, memberInfo.classType);
                        }

                        // If we're setting a class variable via a write through an object,
                        // this is normally considered a type violation. But it is allowed
                        // if the class variable is a descriptor object. In this case, we will
                        // clear the flag that causes an error to be generated.
                        if (usage.method === 'set' && memberInfo.symbol.isClassVar() && isAccessedThroughObject) {
                            const selfClass = bindToType || memberName === '__new__' ? undefined : classType;
                            const typeResult = getTypeOfMemberInternal(memberInfo, selfClass);

                            if (typeResult) {
                                if (isDescriptorInstance(typeResult.type, /* requireSetter */ true)) {
                                    type = typeResult.type;
                                    flags &= MemberAccessFlags.DisallowClassVarWrites;
                                }
                            }
                        }

                        if (!type) {
                            type = UnknownType.create();
                        }
                    }
                }
            }

            if (!type) {
                // Determine whether to replace Self variables with a specific
                // class. Avoid doing this if there's a "bindToType" specified
                // because that case is used for super() calls where we want
                // to leave the Self type generic (not specialized). We'll also
                // skip this for __new__ methods because they are not bound
                // to the class but rather assume the type of the cls argument.
                const selfClass = bindToType || memberName === '__new__' ? undefined : classType;

                const typeResult = getTypeOfMemberInternal(memberInfo, selfClass);

                if (typeResult) {
                    type = typeResult.type;
                    if (typeResult.isIncomplete) {
                        isTypeIncomplete = true;
                    }
                } else {
                    type = UnknownType.create();
                }
            }

            // Don't include variables within typed dict classes.
            if (isClass(memberInfo.classType) && ClassType.isTypedDictClass(memberInfo.classType)) {
                const typedDecls = memberInfo.symbol.getTypedDeclarations();
                if (typedDecls.length > 0 && typedDecls[0].type === DeclarationType.Variable) {
                    diag?.addMessage(Localizer.DiagnosticAddendum.memberUnknown().format({ name: memberName }));
                    return undefined;
                }
            }

            if (usage.method === 'get') {
                // Mark the member accessed if it's not coming from a parent class.
                if (
                    isInstantiableClass(memberInfo.classType) &&
                    ClassType.isSameGenericClass(memberInfo.classType, classType)
                ) {
                    setSymbolAccessed(AnalyzerNodeInfo.getFileInfo(errorNode), memberInfo.symbol, errorNode);
                }
            }

            const descriptorResult = applyDescriptorAccessMethod(
                type,
                memberInfo,
                classType,
                bindToType,
                isAccessedThroughObject,
                flags,
                errorNode,
                memberName,
                usage,
                diag
            );

            if (!descriptorResult) {
                return undefined;
            }
            type = descriptorResult.type;

            if (usage.method === 'set' && usage.setType) {
                // Verify that the assigned type is compatible.
                if (!assignType(type, usage.setType.type, diag?.createAddendum())) {
                    if (!usage.setType.isIncomplete) {
                        diag?.addMessage(
                            Localizer.DiagnosticAddendum.memberAssignment().format({
                                type: printType(usage.setType.type),
                                name: memberName,
                                classType: printObjectTypeForClass(classType),
                            })
                        );
                    }
                    return undefined;
                }

                if (
                    isInstantiableClass(memberInfo.classType) &&
                    ClassType.isFrozenDataClass(memberInfo.classType) &&
                    isAccessedThroughObject
                ) {
                    diag?.addMessage(
                        Localizer.DiagnosticAddendum.dataClassFrozen().format({
                            name: printType(ClassType.cloneAsInstance(memberInfo.classType)),
                        })
                    );
                    return undefined;
                }
            }

            return {
                symbol: memberInfo.symbol,
                type,
                isTypeIncomplete,
                isClassMember: !memberInfo.isInstanceMember,
                isClassVar: memberInfo.isClassVar,
                classType: memberInfo.classType,
                isAsymmetricDescriptor: descriptorResult.isAsymmetricDescriptor,
            };
        }

        // No attribute of that name was found. If this is a member access
        // through an object, see if there's an attribute access override
        // method ("__getattr__", etc.).
        if (
            (flags & (MemberAccessFlags.AccessClassMembersOnly | MemberAccessFlags.SkipAttributeAccessOverride)) ===
            0
        ) {
            const generalAttrType = applyAttributeAccessOverride(classType, errorNode, usage, memberName);
            if (generalAttrType) {
                return {
                    symbol: undefined,
                    type: generalAttrType,
                    isTypeIncomplete: false,
                    isClassMember: false,
                    isClassVar: false,
                    isAsymmetricDescriptor: false,
                };
            }
        }

        diag?.addMessage(Localizer.DiagnosticAddendum.memberUnknown().format({ name: memberName }));

        return undefined;
    }

    // Applies descriptor access methods "__get__", "__set__", or "__delete__"
    // if they apply. Also binds methods to the class/object through which it
    // is accessed.
    function applyDescriptorAccessMethod(
        type: Type,
        memberInfo: ClassMember | undefined,
        baseTypeClass: ClassType,
        bindToType: ClassType | TypeVarType | undefined,
        isAccessedThroughObject: boolean,
        flags: MemberAccessFlags,
        errorNode: ExpressionNode,
        memberName: string,
        usage: EvaluatorUsage,
        diag: DiagnosticAddendum | undefined
    ): DescriptorTypeResult | undefined {
        const treatConstructorAsClassMember = (flags & MemberAccessFlags.TreatConstructorAsClassMethod) !== 0;
        let isTypeValid = true;
        let isAsymmetricDescriptor = false;

        type = mapSubtypes(type, (subtype) => {
            const concreteSubtype = makeTopLevelTypeVarsConcrete(subtype);
            const isClassMember = !memberInfo || memberInfo.isClassMember;

            if (isClass(concreteSubtype) && isClassMember) {
                // If it's an object, use its class to lookup the descriptor. If it's a class,
                // use its metaclass instead.
                let lookupClass: ClassType | undefined = concreteSubtype;
                let isAccessedThroughMetaclass = false;
                if (TypeBase.isInstantiable(concreteSubtype)) {
                    if (
                        concreteSubtype.details.effectiveMetaclass &&
                        isInstantiableClass(concreteSubtype.details.effectiveMetaclass)
                    ) {
                        // When accessing a class member that is a class whose metaclass implements
                        // a descriptor protocol, only 'get' operations are allowed. If it's accessed
                        // through the object, all access methods are supported.
                        if (isAccessedThroughObject || usage.method === 'get') {
                            lookupClass = convertToInstance(concreteSubtype.details.effectiveMetaclass) as ClassType;
                            isAccessedThroughMetaclass = true;
                        } else {
                            lookupClass = undefined;
                        }
                    } else {
                        lookupClass = undefined;
                    }
                }

                if (lookupClass) {
                    let accessMethodName: string;

                    if (usage.method === 'get') {
                        accessMethodName = '__get__';
                    } else if (usage.method === 'set') {
                        accessMethodName = '__set__';
                    } else {
                        accessMethodName = '__delete__';
                    }

                    const accessMethod = lookUpClassMember(
                        lookupClass,
                        accessMethodName,
                        ClassMemberLookupFlags.SkipInstanceVariables
                    );

                    // Handle properties specially.
                    if (ClassType.isPropertyClass(lookupClass)) {
                        if (usage.method === 'set') {
                            if (!accessMethod) {
                                diag?.addMessage(
                                    Localizer.DiagnosticAddendum.propertyMissingSetter().format({
                                        name: memberName,
                                    })
                                );
                                isTypeValid = false;
                                return undefined;
                            }
                        } else if (usage.method === 'del') {
                            if (!accessMethod) {
                                diag?.addMessage(
                                    Localizer.DiagnosticAddendum.propertyMissingDeleter().format({
                                        name: memberName,
                                    })
                                );
                                isTypeValid = false;
                                return undefined;
                            }
                        }
                    }

                    if (accessMethod) {
                        let accessMethodType = getTypeOfMember(accessMethod);
                        const argList: FunctionArgument[] = [
                            {
                                // Provide "obj" argument.
                                argumentCategory: ArgumentCategory.Simple,
                                typeResult: {
                                    type: ClassType.isClassProperty(lookupClass)
                                        ? baseTypeClass
                                        : isAccessedThroughObject
                                        ? bindToType || ClassType.cloneAsInstance(baseTypeClass)
                                        : NoneType.createInstance(),
                                },
                            },
                        ];

                        if (usage.method === 'get') {
                            // Provide "objtype" argument.
                            argList.push({
                                argumentCategory: ArgumentCategory.Simple,
                                typeResult: { type: baseTypeClass },
                            });
                        } else if (usage.method === 'set') {
                            // Provide "value" argument.
                            argList.push({
                                argumentCategory: ArgumentCategory.Simple,
                                typeResult: {
                                    type: usage.setType?.type ?? UnknownType.create(),
                                    isIncomplete: !!usage.setType?.isIncomplete,
                                },
                            });
                        }

                        if (
                            ClassType.isPropertyClass(lookupClass) &&
                            memberInfo &&
                            isInstantiableClass(memberInfo!.classType)
                        ) {
                            // This specialization is required specifically for properties, which should be
                            // generic but are not defined that way. Because of this, we use type variables
                            // in the synthesized methods (e.g. __get__) for the property class that are
                            // defined in the class that declares the fget method.

                            // Infer return types before specializing. Otherwise a generic inferred
                            // return type won't be properly specialized.
                            inferReturnTypeIfNecessary(accessMethodType);

                            accessMethodType = partiallySpecializeType(accessMethodType, memberInfo.classType);

                            // If the property is being accessed from a protocol class (not an instance),
                            // flag this as an error because a property within a protocol is meant to be
                            // interpreted as a read-only attribute rather than a protocol, so accessing
                            // it directly from the class has an ambiguous meaning.
                            if (
                                (flags & MemberAccessFlags.AccessClassMembersOnly) !== 0 &&
                                ClassType.isProtocolClass(baseTypeClass)
                            ) {
                                diag?.addMessage(Localizer.DiagnosticAddendum.propertyAccessFromProtocolClass());
                                isTypeValid = false;
                            }
                        }

                        if (
                            accessMethodType &&
                            (isFunction(accessMethodType) || isOverloadedFunction(accessMethodType))
                        ) {
                            const methodType = accessMethodType;

                            // Don't emit separate diagnostics for these method calls because
                            // they will be redundant.
                            const returnType = suppressDiagnostics(errorNode, () => {
                                // Bind the accessor to the base object type.
                                let bindToClass: ClassType | undefined;

                                // The "bind-to" class depends on whether the descriptor is defined
                                // on the metaclass or the class. We handle properties specially here
                                // because of the way we model the __get__ logic in the property class.
                                if (ClassType.isPropertyClass(concreteSubtype) && !isAccessedThroughMetaclass) {
                                    if (memberInfo && isInstantiableClass(memberInfo.classType)) {
                                        bindToClass = memberInfo.classType;
                                    }
                                } else {
                                    if (isInstantiableClass(accessMethod.classType)) {
                                        bindToClass = accessMethod.classType;
                                    }
                                }

                                let boundMethodType = bindFunctionToClassOrObject(
                                    lookupClass,
                                    methodType,
                                    bindToClass,
                                    errorNode,
                                    /* recursionCount */ undefined,
                                    /* treatConstructorAsClassMember */ undefined,
                                    isAccessedThroughMetaclass ? concreteSubtype : undefined
                                );

                                // The synthesized access method for the property may contain
                                // type variables associated with the "bindToClass", so we need
                                // to specialize those here.
                                if (
                                    boundMethodType &&
                                    (isFunction(boundMethodType) || isOverloadedFunction(boundMethodType))
                                ) {
                                    const typeVarContext = new TypeVarContext(getTypeVarScopeId(boundMethodType));
                                    if (bindToClass) {
                                        const specializedBoundType = partiallySpecializeType(
                                            boundMethodType,
                                            bindToClass,
                                            baseTypeClass
                                        );
                                        if (specializedBoundType) {
                                            if (
                                                isFunction(specializedBoundType) ||
                                                isOverloadedFunction(specializedBoundType)
                                            ) {
                                                boundMethodType = specializedBoundType;
                                            }
                                        }
                                    }

                                    const callResult = validateCallArguments(
                                        errorNode,
                                        argList,
                                        { type: boundMethodType },
                                        typeVarContext,
                                        /* skipUnknownArgCheck */ true
                                    );

                                    if (callResult.argumentErrors) {
                                        if (usage.method === 'set') {
                                            if (
                                                usage.setType &&
                                                isFunction(boundMethodType) &&
                                                boundMethodType.details.parameters.length >= 2 &&
                                                !usage.setType.isIncomplete
                                            ) {
                                                const setterType = FunctionType.getEffectiveParameterType(
                                                    boundMethodType,
                                                    1
                                                );

                                                diag?.addMessage(
                                                    Localizer.DiagnosticAddendum.typeIncompatible().format({
                                                        destType: printType(setterType),
                                                        sourceType: printType(usage.setType.type),
                                                    })
                                                );
                                            } else if (isOverloadedFunction(boundMethodType)) {
                                                diag?.addMessage(
                                                    Localizer.Diagnostic.noOverload().format({ name: accessMethodName })
                                                );
                                            }
                                        }

                                        isTypeValid = false;
                                        return AnyType.create();
                                    }

                                    // For set or delete, always return Any.
                                    return usage.method === 'get'
                                        ? callResult.returnType ?? UnknownType.create()
                                        : AnyType.create();
                                }

                                return undefined;
                            });

                            // Determine if we're calling __set__ on an asymmetric descriptor or property.
                            if (usage.method === 'set' && isClass(accessMethod.classType)) {
                                if (isAsymmetricDescriptorClass(accessMethod.classType)) {
                                    isAsymmetricDescriptor = true;
                                }
                            }

                            if (returnType) {
                                return returnType;
                            }
                        }
                    }
                }
            } else if (isFunction(concreteSubtype) || isOverloadedFunction(concreteSubtype)) {
                // Check for an attempt to overwrite a final method.
                if (usage.method === 'set') {
                    let isFinal = false;
                    if (isFunction(concreteSubtype)) {
                        isFinal = FunctionType.isFinal(concreteSubtype);
                    } else {
                        const impl = OverloadedFunctionType.getImplementation(concreteSubtype);
                        if (impl) {
                            isFinal = FunctionType.isFinal(impl);
                        }
                    }

                    if (isFinal && memberInfo && isClass(memberInfo.classType)) {
                        diag?.addMessage(
                            Localizer.Diagnostic.finalMethodOverride().format({
                                name: memberName,
                                className: memberInfo.classType.details.name,
                            })
                        );
                        isTypeValid = false;
                        return undefined;
                    }
                }

                // If this function is an instance member (e.g. a lambda that was
                // assigned to an instance variable), don't perform any binding.
                if (!isAccessedThroughObject || (memberInfo && !memberInfo.isInstanceMember)) {
                    // Skip binding if the class appears to be a metaclass (i.e. a subclass of
                    // `type`) because the first parameter of instance methods in a metaclass
                    // are not `self` instances.
                    const isMetaclass =
                        !isAccessedThroughObject &&
                        isClass(baseTypeClass) &&
                        (flags & MemberAccessFlags.TreatConstructorAsClassMethod) === 0 &&
                        baseTypeClass.details.mro.some(
                            (mroType) => isClass(mroType) && ClassType.isBuiltIn(mroType, 'type')
                        );

                    if (isMetaclass) {
                        return concreteSubtype;
                    }

                    return bindFunctionToClassOrObject(
                        isAccessedThroughObject ? ClassType.cloneAsInstance(baseTypeClass) : baseTypeClass,
                        concreteSubtype,
                        memberInfo && isInstantiableClass(memberInfo.classType) ? memberInfo.classType : undefined,
                        errorNode,
                        /* recursionCount */ undefined,
                        treatConstructorAsClassMember,
                        bindToType
                    );
                }
            }

            if (usage.method === 'set') {
                if (memberInfo?.symbol.isClassVar()) {
                    if (flags & MemberAccessFlags.DisallowClassVarWrites) {
                        diag?.addMessage(Localizer.DiagnosticAddendum.memberSetClassVar().format({ name: memberName }));
                        isTypeValid = false;
                        return undefined;
                    }
                }

                // Check for an attempt to overwrite a final member variable.
                const finalVarTypeDecl = memberInfo?.symbol
                    .getDeclarations()
                    .find((decl) => isFinalVariableDeclaration(decl));

                if (finalVarTypeDecl && !ParseTreeUtils.isNodeContainedWithin(errorNode, finalVarTypeDecl.node)) {
                    // If a Final instance variable is declared in the class body but is
                    // being assigned within an __init__ method, it's allowed.
                    const enclosingFunctionNode = ParseTreeUtils.getEnclosingFunction(errorNode);
                    if (!enclosingFunctionNode || enclosingFunctionNode.name.value !== '__init__') {
                        diag?.addMessage(Localizer.Diagnostic.finalReassigned().format({ name: memberName }));
                        isTypeValid = false;
                        return undefined;
                    }
                }

                // Check for an attempt to overwrite an instance variable that is
                // read-only (e.g. in a named tuple).
                if (
                    memberInfo?.isInstanceMember &&
                    isClass(memberInfo.classType) &&
                    ClassType.isReadOnlyInstanceVariables(memberInfo.classType)
                ) {
                    diag?.addMessage(Localizer.DiagnosticAddendum.readOnlyAttribute().format({ name: memberName }));
                    isTypeValid = false;
                    return undefined;
                }

                let enforceTargetType = false;

                if (memberInfo && memberInfo.symbol.hasTypedDeclarations()) {
                    // If the member has a declared type, we will enforce it.
                    enforceTargetType = true;
                } else {
                    // If the member has no declared type, we will enforce it
                    // if this assignment isn't within the enclosing class. If
                    // it is within the enclosing class, the assignment is used
                    // to infer the type of the member.
                    if (memberInfo && !memberInfo.symbol.getDeclarations().some((decl) => decl.node === errorNode)) {
                        enforceTargetType = true;
                    }
                }

                if (enforceTargetType) {
                    let effectiveType = subtype;

                    // If the code is patching a method (defined on the class)
                    // with an object-level function, strip the "self" parameter
                    // off the original type. This is sometimes done for test
                    // purposes to override standard behaviors of specific methods.
                    if (isAccessedThroughObject) {
                        if (!memberInfo!.isInstanceMember && isFunction(concreteSubtype)) {
                            if (
                                FunctionType.isClassMethod(concreteSubtype) ||
                                FunctionType.isInstanceMethod(concreteSubtype)
                            ) {
                                effectiveType = FunctionType.clone(concreteSubtype, /* stripFirstParam */ true);
                            }
                        }
                    }

                    return effectiveType;
                }
            }

            return subtype;
        });

        if (!isTypeValid) {
            return undefined;
        }

        return { type, isAsymmetricDescriptor };
    }

    function isAsymmetricDescriptorClass(classType: ClassType): boolean {
        // If the value has already been cached in this type, return the cached value.
        if (classType.isAsymmetricDescriptor !== undefined) {
            return classType.isAsymmetricDescriptor;
        }

        let isAsymmetric = false;

        const getterSymbolResult = lookUpClassMember(classType, '__get__', ClassMemberLookupFlags.SkipBaseClasses);
        const setterSymbolResult = lookUpClassMember(classType, '__set__', ClassMemberLookupFlags.SkipBaseClasses);

        if (!getterSymbolResult || !setterSymbolResult) {
            isAsymmetric = false;
        } else {
            const getterType = getEffectiveTypeOfSymbol(getterSymbolResult.symbol);
            const setterType = getEffectiveTypeOfSymbol(setterSymbolResult.symbol);

            // If either the setter or getter is an overload (or some other non-function type),
            // conservatively assume that it's not asymmetric.
            if (isFunction(getterType) && isFunction(setterType)) {
                // If there's no declared return type on the getter, assume it's symmetric.
                if (setterType.details.parameters.length >= 3 && getterType.details.declaredReturnType) {
                    const setterValueType = FunctionType.getEffectiveParameterType(setterType, 2);
                    const getterReturnType = FunctionType.getSpecializedReturnType(getterType) ?? UnknownType.create();

                    if (!isTypeSame(setterValueType, getterReturnType)) {
                        isAsymmetric = true;
                    }
                }
            }
        }

        // Cache the value for next time.
        classType.isAsymmetricDescriptor = isAsymmetric;
        return isAsymmetric;
    }

    // Applies the __getattr__, __setattr__ or __delattr__ method if present.
    function applyAttributeAccessOverride(
        classType: ClassType,
        errorNode: ExpressionNode,
        usage: EvaluatorUsage,
        memberName: string
    ): Type | undefined {
        const getAttributeAccessMember = (name: string) => {
            // See if the class has a "__getattribute__" or "__getattr__" method.
            // If so, arbitrary members are supported.
            return getTypeOfClassMember(
                errorNode,
                classType,
                name,
                { method: 'get' },
                /* diag */ undefined,
                MemberAccessFlags.SkipObjectBaseClass | MemberAccessFlags.SkipAttributeAccessOverride
            )?.type;
        };

        let accessMemberType: Type | undefined;
        if (usage.method === 'get') {
            accessMemberType = getAttributeAccessMember('__getattribute__') ?? getAttributeAccessMember('__getattr__');
        } else if (usage.method === 'set') {
            accessMemberType = getAttributeAccessMember('__setattr__');
        } else {
            assert(usage.method === 'del');
            accessMemberType = getAttributeAccessMember('__delattr__');
        }

        if (accessMemberType) {
            let nameLiteralType: Type = AnyType.create();
            if (strClassType && isInstantiableClass(strClassType)) {
                nameLiteralType = ClassType.cloneWithLiteral(ClassType.cloneAsInstance(strClassType), memberName);
            }

            const argList: FunctionArgument[] = [
                {
                    // Provide "self" argument.
                    argumentCategory: ArgumentCategory.Simple,
                    typeResult: { type: ClassType.cloneAsInstance(classType) },
                },
                {
                    // Provide "name" argument.
                    argumentCategory: ArgumentCategory.Simple,
                    typeResult: { type: nameLiteralType },
                },
            ];

            if (usage.method === 'set') {
                argList.push({
                    // Provide "value" argument.
                    argumentCategory: ArgumentCategory.Simple,
                    typeResult: {
                        type: usage.setType?.type ?? UnknownType.create(),
                        isIncomplete: !!usage.setType?.isIncomplete,
                    },
                });
            }

            if (isFunction(accessMemberType) || isOverloadedFunction(accessMemberType)) {
                const boundMethodType = bindFunctionToClassOrObject(classType, accessMemberType, classType, errorNode);

                if (boundMethodType && (isFunction(boundMethodType) || isOverloadedFunction(boundMethodType))) {
                    const typeVarContext = new TypeVarContext(getTypeVarScopeId(boundMethodType));
                    const callResult = validateCallArguments(
                        errorNode,
                        argList,
                        { type: boundMethodType },
                        typeVarContext,
                        /* skipUnknownArgCheck */ true
                    );

                    return callResult.returnType ?? UnknownType.create();
                }
            }
        }

        return undefined;
    }

    function getTypeOfIndex(node: IndexNode, flags = EvaluatorFlags.None): TypeResult {
        const baseTypeResult = getTypeOfExpression(node.baseExpression, flags | EvaluatorFlags.DoNotSpecialize);

        // If this is meant to be a type and the base expression is a string expression,
        // emit an error because this will generate a runtime exception in Python versions
        // less than 3.10.
        if (flags & EvaluatorFlags.ExpectingType) {
            if (node.baseExpression.nodeType === ParseNodeType.StringList) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                if (!fileInfo.isStubFile && fileInfo.executionEnvironment.pythonVersion < PythonVersion.V3_10) {
                    addError(Localizer.Diagnostic.stringNotSubscriptable(), node.baseExpression);
                }
            }
        }

        // Check for builtin classes that will generate runtime exceptions if subscripted.
        if ((flags & EvaluatorFlags.AllowForwardReferences) === 0) {
            // We can skip this check if the class is used within a PEP 526 variable
            // type annotation within a class or function. For some undocumented reason,
            // they don't result in runtime exceptions when used in this manner.
            let skipSubscriptCheck = (flags & EvaluatorFlags.VariableTypeAnnotation) !== 0;
            if (skipSubscriptCheck) {
                const scopeNode = ParseTreeUtils.getExecutionScopeNode(node);
                if (scopeNode?.nodeType === ParseNodeType.Module) {
                    skipSubscriptCheck = false;
                }
            }

            if (!skipSubscriptCheck) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                if (
                    isInstantiableClass(baseTypeResult.type) &&
                    ClassType.isBuiltIn(baseTypeResult.type) &&
                    !baseTypeResult.type.aliasName
                ) {
                    const minPythonVersion = nonSubscriptableBuiltinTypes.get(baseTypeResult.type.details.fullName);
                    if (
                        minPythonVersion !== undefined &&
                        fileInfo.executionEnvironment.pythonVersion < minPythonVersion &&
                        !fileInfo.isStubFile
                    ) {
                        addError(
                            Localizer.Diagnostic.classNotRuntimeSubscriptable().format({
                                name: baseTypeResult.type.aliasName || baseTypeResult.type.details.name,
                            }),
                            node.baseExpression
                        );
                    }
                }
            }
        }

        const indexTypeResult = getTypeOfIndexWithBaseType(node, baseTypeResult, { method: 'get' }, flags);

        if (isCodeFlowSupportedForReference(node)) {
            // We limit type narrowing for index expressions to built-in types that are
            // known to have symmetric __getitem__ and __setitem__ methods (i.e. the value
            // passed to __setitem__ is the same type as the value returned by __getitem__).
            let baseTypeSupportsIndexNarrowing = true;
            mapSubtypesExpandTypeVars(baseTypeResult.type, /* conditionFilter */ undefined, (subtype) => {
                if (
                    !isClassInstance(subtype) ||
                    !(ClassType.isBuiltIn(subtype) || ClassType.isTypedDictClass(subtype))
                ) {
                    baseTypeSupportsIndexNarrowing = false;
                }

                return undefined;
            });

            if (baseTypeSupportsIndexNarrowing) {
                // Before performing code flow analysis, update the cache to prevent recursion.
                writeTypeCache(node, indexTypeResult, flags);

                // See if we can refine the type based on code flow analysis.
                const codeFlowTypeResult = getFlowTypeOfReference(
                    node,
                    indeterminateSymbolId,
                    indexTypeResult.type,
                    /* startNode */ undefined,
                    {
                        isTypeAtStartIncomplete: !!baseTypeResult.isIncomplete || !!indexTypeResult.isIncomplete,
                        skipConditionalNarrowing: (flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0,
                    }
                );
                if (codeFlowTypeResult.type) {
                    indexTypeResult.type = codeFlowTypeResult.type;
                }

                if (codeFlowTypeResult.isIncomplete) {
                    indexTypeResult.isIncomplete = true;
                }
            }
        }

        if (baseTypeResult.isIncomplete) {
            indexTypeResult.isIncomplete = true;
        }

        return indexTypeResult;
    }

    function adjustTypeArgumentsForVariadicTypeVar(
        typeArgs: TypeResultWithNode[],
        typeParameters: TypeVarType[],
        errorNode: ExpressionNode
    ): TypeResultWithNode[] {
        const variadicIndex = typeParameters.findIndex((param) => isVariadicTypeVar(param));

        // Do we need to adjust the type arguments to map to a variadic type
        // param at the end of the list?
        if (variadicIndex >= 0) {
            const variadicTypeVar = typeParameters[variadicIndex];

            if (tupleClassType && isInstantiableClass(tupleClassType)) {
                if (variadicIndex < typeArgs.length) {
                    const variadicTypeResults = typeArgs.slice(
                        variadicIndex,
                        variadicIndex + 1 + typeArgs.length - typeParameters.length
                    );

                    // If the type args consist of a lone variadic type variable, don't wrap it in a tuple.
                    if (variadicTypeResults.length === 1 && isVariadicTypeVar(variadicTypeResults[0].type)) {
                        validateVariadicTypeVarIsUnpacked(variadicTypeResults[0].type, variadicTypeResults[0].node);
                    } else {
                        variadicTypeResults.forEach((arg, index) => {
                            validateTypeArg(arg, {
                                allowEmptyTuple: index === 0,
                                allowVariadicTypeVar: true,
                                allowUnpackedTuples: true,
                            });
                        });

                        const variadicTypes: TupleTypeArgument[] = [];
                        if (variadicTypeResults.length !== 1 || !variadicTypeResults[0].isEmptyTupleShorthand) {
                            variadicTypeResults.forEach((typeResult) => {
                                if (isUnpackedClass(typeResult.type) && typeResult.type.tupleTypeArguments) {
                                    appendArray(variadicTypes, typeResult.type.tupleTypeArguments);
                                } else {
                                    variadicTypes.push({
                                        type: convertToInstance(typeResult.type),
                                        isUnbounded: false,
                                    });
                                }
                            });
                        }

                        const tupleObject = convertToInstance(
                            specializeTupleClass(
                                tupleClassType,
                                variadicTypes,
                                /* isTypeArgumentExplicit */ true,
                                /* isUnpackedTuple */ true
                            )
                        );

                        typeArgs = [
                            ...typeArgs.slice(0, variadicIndex),
                            { node: typeArgs[variadicIndex].node, type: tupleObject },
                            ...typeArgs.slice(
                                variadicIndex + 1 + typeArgs.length - typeParameters.length,
                                typeArgs.length
                            ),
                        ];
                    }
                } else if (!variadicTypeVar.details.defaultType) {
                    // Add an empty tuple that maps to the TypeVarTuple type parameter.
                    typeArgs.push({
                        node: errorNode,
                        type: convertToInstance(
                            specializeTupleClass(
                                tupleClassType,
                                [],
                                /* isTypeArgumentExplicit */ true,
                                /* isUnpackedTuple */ true
                            )
                        ),
                    });
                }
            }
        }

        return typeArgs;
    }

    // If the variadic type variable is not unpacked, report an error.
    function validateVariadicTypeVarIsUnpacked(type: TypeVarType, node: ParseNode) {
        if (!type.isVariadicUnpacked) {
            addError(
                Localizer.Diagnostic.unpackedTypeVarTupleExpected().format({
                    name1: type.details.name,
                    name2: type.details.name,
                }),
                node
            );
            return false;
        }

        return true;
    }

    // Handles index expressions that are providing type arguments for a
    // generic type alias.
    function createSpecializedTypeAlias(
        node: IndexNode,
        baseType: Type,
        flags: EvaluatorFlags
    ): TypeResultWithNode | undefined {
        if (
            !baseType.typeAliasInfo?.typeParameters ||
            (baseType.typeAliasInfo.typeParameters.length === 0 && baseType.typeAliasInfo.typeArguments)
        ) {
            return undefined;
        }

        // If this is not instantiable, then the index expression isn't a specialization.
        if (!TypeBase.isInstantiable(baseType)) {
            return undefined;
        }

        // If this is already specialized, the index expression isn't a specialization.
        if (baseType.typeAliasInfo.typeArguments) {
            return undefined;
        }

        const typeParameters = baseType.typeAliasInfo.typeParameters;
        let typeArgs = adjustTypeArgumentsForVariadicTypeVar(getTypeArgs(node, flags), typeParameters, node);

        // PEP 612 says that if the class has only one type parameter consisting
        // of a ParamSpec, the list of arguments does not need to be enclosed in
        // a list. We'll handle that case specially here. Presumably this applies to
        // type aliases as well.
        if (typeParameters.length === 1 && typeParameters[0].details.isParamSpec && typeArgs) {
            if (
                typeArgs.every(
                    (typeArg) => !isEllipsisType(typeArg.type) && !typeArg.typeList && !isParamSpec(typeArg.type)
                )
            ) {
                typeArgs = [
                    {
                        type: UnknownType.create(),
                        node: typeArgs[0].node,
                        typeList: typeArgs,
                    },
                ];
            }
        }

        if (!typeParameters.some((typeVar) => typeVar.details.isVariadic && !typeVar.isVariadicInUnion)) {
            let minTypeArgCount = typeParameters.length;
            const firstNonDefaultParam = typeParameters.findIndex((param) => !!param.details.defaultType);
            if (firstNonDefaultParam >= 0) {
                minTypeArgCount = firstNonDefaultParam;
            }

            if (typeArgs.length > typeParameters.length) {
                addError(
                    Localizer.Diagnostic.typeArgsTooMany().format({
                        name: printType(baseType),
                        expected: typeParameters.length,
                        received: typeArgs.length,
                    }),
                    typeArgs[typeParameters.length].node
                );
            } else if (typeArgs.length < minTypeArgCount) {
                addError(
                    Localizer.Diagnostic.typeArgsTooFew().format({
                        name: printType(baseType),
                        expected: typeParameters.length,
                        received: typeArgs.length,
                    }),
                    node.items[node.items.length - 1]
                );
            }
        }

        // Handle the mypy_extensions.FlexibleAlias type specially.
        if (
            isInstantiableClass(baseType) &&
            baseType.details.fullName === 'mypy_extensions.FlexibleAlias' &&
            typeArgs.length >= 1
        ) {
            return { node, type: typeArgs[0].type };
        }

        const typeVarContext = new TypeVarContext(baseType.typeAliasInfo.typeVarScopeId);
        const diag = new DiagnosticAddendum();
        typeParameters.forEach((param, index) => {
            if (param.details.isParamSpec && index < typeArgs.length) {
                const typeArgType = typeArgs[index].type;

                if (typeArgs[index].typeList) {
                    const functionType = FunctionType.createSynthesizedInstance('', FunctionTypeFlags.ParamSpecValue);
                    TypeBase.setSpecialForm(functionType);
                    typeArgs[index].typeList!.forEach((paramType, paramIndex) => {
                        FunctionType.addParameter(functionType, {
                            category: ParameterCategory.Simple,
                            name: `__p${paramIndex}`,
                            isNameSynthesized: true,
                            type: convertToInstance(paramType.type),
                            hasDeclaredType: true,
                        });
                    });

                    assignTypeToTypeVar(
                        evaluatorInterface,
                        param,
                        functionType,
                        diag,
                        typeVarContext,
                        AssignTypeFlags.RetainLiteralsForTypeVar
                    );
                } else if (isParamSpec(typeArgType)) {
                    assignTypeToTypeVar(
                        evaluatorInterface,
                        param,
                        convertToInstance(typeArgType),
                        diag,
                        typeVarContext,
                        AssignTypeFlags.RetainLiteralsForTypeVar
                    );
                } else if (isInstantiableClass(typeArgType) && ClassType.isBuiltIn(typeArgType, 'Concatenate')) {
                    const concatTypeArgs = typeArgType.typeArguments;
                    const functionType = FunctionType.createInstance('', '', '', FunctionTypeFlags.None);

                    if (concatTypeArgs && concatTypeArgs.length > 0) {
                        concatTypeArgs.forEach((typeArg, index) => {
                            if (index === concatTypeArgs.length - 1) {
                                // Add a position-only separator
                                FunctionType.addParameter(functionType, {
                                    category: ParameterCategory.Simple,
                                    isNameSynthesized: false,
                                    type: UnknownType.create(),
                                });

                                if (isParamSpec(typeArg)) {
                                    functionType.details.paramSpec = typeArg;
                                }
                            } else {
                                FunctionType.addParameter(functionType, {
                                    category: ParameterCategory.Simple,
                                    name: `__p${index}`,
                                    isNameSynthesized: true,
                                    hasDeclaredType: true,
                                    type: typeArg,
                                });
                            }
                        });
                    }

                    assignTypeToTypeVar(
                        evaluatorInterface,
                        param,
                        functionType,
                        diag,
                        typeVarContext,
                        AssignTypeFlags.RetainLiteralsForTypeVar
                    );
                } else if (isEllipsisType(typeArgType)) {
                    const functionType = FunctionType.createSynthesizedInstance(
                        '',
                        FunctionTypeFlags.ParamSpecValue | FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck
                    );
                    TypeBase.setSpecialForm(functionType);
                    FunctionType.addDefaultParameters(functionType);
                    assignTypeToTypeVar(evaluatorInterface, param, functionType, diag, typeVarContext);
                } else {
                    addError(Localizer.Diagnostic.typeArgListExpected(), typeArgs[index].node);
                }
            } else {
                if (index < typeArgs.length && typeArgs[index].typeList) {
                    addError(Localizer.Diagnostic.typeArgListNotAllowed(), typeArgs[index].node);
                }

                let typeArgType: Type;
                if (index < typeArgs.length) {
                    typeArgType = convertToInstance(typeArgs[index].type);
                } else if (param.details.defaultType) {
                    typeArgType = applySolvedTypeVars(param, typeVarContext, { unknownIfNotFound: true });
                } else {
                    typeArgType = UnknownType.create();
                }

                if ((flags & EvaluatorFlags.EnforceTypeVarVarianceConsistency) !== 0) {
                    const usageVariances = inferTypeParameterVarianceForTypeAlias(baseType);
                    if (usageVariances && index < usageVariances.length) {
                        const usageVariance = usageVariances[index];

                        if (!isVarianceOfTypeArgumentCompatible(typeArgType, usageVariance)) {
                            const messageDiag = diag.createAddendum();
                            messageDiag.addMessage(
                                Localizer.DiagnosticAddendum.varianceMismatchForTypeAlias().format({
                                    typeVarName: printType(typeArgType),
                                    typeAliasParam: printType(typeParameters[index]),
                                })
                            );
                            messageDiag.addTextRange(typeArgs[index].node);
                        }
                    }
                }

                assignTypeToTypeVar(
                    evaluatorInterface,
                    param,
                    typeArgType,
                    diag,
                    typeVarContext,
                    AssignTypeFlags.RetainLiteralsForTypeVar
                );
            }
        });

        if (!diag.isEmpty()) {
            addError(
                Localizer.Diagnostic.typeNotSpecializable().format({ type: printType(baseType) }) + diag.getString(),
                node,
                diag.getEffectiveTextRange() ?? node
            );
        }

        const primarySignatureContext = typeVarContext.getPrimarySignature();
        const aliasTypeArgs: Type[] = [];

        baseType.typeAliasInfo.typeParameters?.forEach((typeParam) => {
            let typeVarType: Type | undefined;

            if (isParamSpec(typeParam)) {
                const paramSpecType = primarySignatureContext.getParamSpecType(typeParam);
                typeVarType = paramSpecType ? convertParamSpecValueToType(paramSpecType) : UnknownType.create();
            } else {
                typeVarType = primarySignatureContext.getTypeVarType(typeParam);
            }

            aliasTypeArgs.push(typeVarType || UnknownType.create());
        });

        const type = TypeBase.cloneForTypeAlias(
            applySolvedTypeVars(baseType, typeVarContext),
            baseType.typeAliasInfo.name,
            baseType.typeAliasInfo.fullName,
            baseType.typeAliasInfo.typeVarScopeId,
            baseType.typeAliasInfo.typeParameters,
            aliasTypeArgs
        );

        return { type, node };
    }

    function getTypeOfIndexWithBaseType(
        node: IndexNode,
        baseTypeResult: TypeResult,
        usage: EvaluatorUsage,
        flags: EvaluatorFlags
    ): TypeResult {
        // Handle the case where we're specializing a generic type alias.
        const typeAliasResult = createSpecializedTypeAlias(node, baseTypeResult.type, flags);
        if (typeAliasResult) {
            return typeAliasResult;
        }

        if (isTypeVar(baseTypeResult.type) && isTypeAliasPlaceholder(baseTypeResult.type)) {
            const typeArgTypes = getTypeArgs(node, flags).map((t) => convertToInstance(t.type));
            const type = TypeBase.cloneForTypeAlias(
                baseTypeResult.type,
                baseTypeResult.type.details.recursiveTypeAliasName!,
                '',
                baseTypeResult.type.details.recursiveTypeAliasScopeId!,
                baseTypeResult.type.details.recursiveTypeParameters,
                typeArgTypes
            );
            return { type };
        }

        let isIncomplete = baseTypeResult.isIncomplete;
        let isRequired = false;
        let isNotRequired = false;

        const type = mapSubtypesExpandTypeVars(
            baseTypeResult.type,
            /* conditionFilter */ undefined,
            (concreteSubtype, unexpandedSubtype) => {
                if (isAnyOrUnknown(concreteSubtype)) {
                    return concreteSubtype;
                }

                if (flags & EvaluatorFlags.ExpectingType) {
                    if (isTypeVar(unexpandedSubtype)) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeVarNotSubscriptable().format({
                                type: printType(unexpandedSubtype),
                            }),
                            node.baseExpression
                        );

                        // Evaluate the index expressions as though they are type arguments for error-reporting.
                        getTypeArgs(node, flags);

                        return UnknownType.create();
                    }
                }

                if (isInstantiableClass(concreteSubtype)) {
                    // See if the class has a custom metaclass that supports __getitem__, etc.
                    if (
                        concreteSubtype.details.effectiveMetaclass &&
                        isInstantiableClass(concreteSubtype.details.effectiveMetaclass) &&
                        !ClassType.isBuiltIn(concreteSubtype.details.effectiveMetaclass, ['type', '_InitVarMeta']) &&
                        (flags & EvaluatorFlags.ExpectingType) === 0
                    ) {
                        const itemMethodType = getTypeOfClassMember(
                            node,
                            concreteSubtype,
                            getIndexAccessMagicMethodName(usage),
                            /* usage */ undefined,
                            /* diag */ undefined,
                            MemberAccessFlags.SkipAttributeAccessOverride | MemberAccessFlags.ConsiderMetaclassOnly
                        );

                        if (flags & EvaluatorFlags.ExpectingTypeAnnotation) {
                            // If the class doesn't derive from Generic, a type argument should not be allowed.
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.typeArgsExpectingNone().format({
                                    name: printType(ClassType.cloneAsInstance(concreteSubtype)),
                                }),
                                node
                            );
                        }

                        if (itemMethodType) {
                            return getTypeOfIndexedObjectOrClass(node, concreteSubtype, usage).type;
                        }
                    }

                    // Setting the value of an indexed class will always result
                    // in an exception.
                    if (usage.method === 'set') {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.genericClassAssigned(),
                            node.baseExpression
                        );
                    } else if (usage.method === 'del') {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.genericClassDeleted(),
                            node.baseExpression
                        );
                    }

                    if (ClassType.isSpecialBuiltIn(concreteSubtype, 'Literal')) {
                        // Special-case Literal types.
                        return createLiteralType(node, flags);
                    }

                    if (ClassType.isBuiltIn(concreteSubtype, 'InitVar')) {
                        // Special-case InitVar, used in data classes.
                        const typeArgs = getTypeArgs(node, flags);
                        if (typeArgs.length === 1) {
                            return typeArgs[0].type;
                        } else {
                            addError(
                                Localizer.Diagnostic.typeArgsMismatchOne().format({ received: typeArgs.length }),
                                node.baseExpression
                            );
                            return UnknownType.create();
                        }
                    }

                    if (ClassType.isEnumClass(concreteSubtype)) {
                        // Special-case Enum types.
                        // TODO - validate that there's only one index entry
                        // that is a str type.
                        // TODO - validate that literal strings are referencing
                        // a known enum member.
                        return ClassType.cloneAsInstance(concreteSubtype);
                    }

                    const isAnnotatedClass =
                        isInstantiableClass(concreteSubtype) && ClassType.isBuiltIn(concreteSubtype, 'Annotated');
                    const hasCustomClassGetItem =
                        isInstantiableClass(concreteSubtype) && ClassType.hasCustomClassGetItem(concreteSubtype);
                    const isGenericClass =
                        concreteSubtype.details.typeParameters?.length > 0 ||
                        ClassType.isSpecialBuiltIn(concreteSubtype) ||
                        ClassType.isBuiltIn(concreteSubtype, 'type') ||
                        ClassType.isPartiallyEvaluated(concreteSubtype);
                    const isFinalAnnotation =
                        isInstantiableClass(concreteSubtype) && ClassType.isBuiltIn(concreteSubtype, 'Final');
                    const isClassVarAnnotation =
                        isInstantiableClass(concreteSubtype) && ClassType.isBuiltIn(concreteSubtype, 'ClassVar');

                    // Inlined TypedDicts are supported only for 'dict' (and not for 'Dict').
                    const supportsTypedDictTypeArg =
                        isInstantiableClass(concreteSubtype) &&
                        ClassType.isBuiltIn(concreteSubtype, 'dict') &&
                        !concreteSubtype.aliasName;

                    let typeArgs = getTypeArgs(node, flags, {
                        isAnnotatedClass,
                        hasCustomClassGetItem: hasCustomClassGetItem || !isGenericClass,
                        isFinalAnnotation,
                        isClassVarAnnotation,
                        supportsTypedDictTypeArg,
                    });

                    if (!isAnnotatedClass) {
                        typeArgs = adjustTypeArgumentsForVariadicTypeVar(
                            typeArgs,
                            concreteSubtype.details.typeParameters,
                            node
                        );
                    }

                    // If this is a custom __class_getitem__, there's no need to specialize the class.
                    // Just return it as is.
                    if (hasCustomClassGetItem) {
                        return concreteSubtype;
                    }

                    if (concreteSubtype.typeArguments) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.classAlreadySpecialized().format({
                                type: printType(convertToInstance(concreteSubtype), { expandTypeAlias: true }),
                            }),
                            node.baseExpression
                        );
                        return concreteSubtype;
                    }

                    const result = createSpecializedClassType(concreteSubtype, typeArgs, flags, node);
                    if (result.isRequired) {
                        isRequired = true;
                    } else if (result.isNotRequired) {
                        isNotRequired = true;
                    }

                    return result.type;
                }

                if (isClassInstance(concreteSubtype)) {
                    const typeResult = getTypeOfIndexedObjectOrClass(node, concreteSubtype, usage);
                    if (typeResult.isIncomplete) {
                        isIncomplete = true;
                    }
                    return typeResult.type;
                }

                if (isNever(concreteSubtype) || isUnbound(concreteSubtype)) {
                    return NeverType.createNever();
                }

                if (isNoneInstance(concreteSubtype) && !isIncomplete) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportOptionalSubscript,
                        DiagnosticRule.reportOptionalSubscript,
                        Localizer.Diagnostic.noneNotSubscriptable(),
                        node.baseExpression
                    );

                    return UnknownType.create();
                }

                if (!isIncomplete) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeNotSubscriptable().format({ type: printType(concreteSubtype) }),
                        node.baseExpression
                    );
                }

                return UnknownType.create();
            }
        );

        // In case we didn't walk the list items above, do so now.
        // If we have, this information will be cached.
        if (!baseTypeResult.isIncomplete) {
            node.items.forEach((item) => {
                if (!isTypeCached(item.valueExpression)) {
                    getTypeOfExpression(item.valueExpression, flags & EvaluatorFlags.AllowForwardReferences);
                }
            });
        }

        return { type, isIncomplete, isRequired, isNotRequired };
    }

    // Determines the effective variance of the type parameters for a generic
    // type alias. Normally, variance is not important for type aliases, but
    // it can be important in cases where the type alias is used to specify
    // a base class in a class definition.
    function inferTypeParameterVarianceForTypeAlias(type: Type): Variance[] | undefined {
        // If this isn't a generic type alias, there's nothing to do.
        if (!type.typeAliasInfo || !type.typeAliasInfo.typeParameters) {
            return undefined;
        }

        // Is the usage variance info already cached?
        if (type.typeAliasInfo.usageVariance) {
            return type.typeAliasInfo.usageVariance;
        }

        const typeParams = type.typeAliasInfo.typeParameters;

        // Start with all of the usage variances unknown.
        const usageVariances: Variance[] = typeParams.map(() => Variance.Unknown);

        // Prepopulate the cached value for the type alias to handle
        // recursive type aliases.
        type.typeAliasInfo.usageVariance = usageVariances;

        // Traverse the type alias type definition and adjust the usage
        // variances accordingly.
        updateUsageVariancesRecursive(type, typeParams, usageVariances);

        return usageVariances;
    }

    // Looks at uses of the type parameters within the type and adjusts the
    // variances accordingly. For example, if the type is `Mapping[T1, T2]`,
    // then T1 will be set to invariant and T2 will be set to covariant.
    function updateUsageVariancesRecursive(
        type: Type,
        typeAliasTypeParams: TypeVarType[],
        usageVariances: Variance[],
        recursionCount = 0
    ) {
        if (recursionCount > maxTypeRecursionCount) {
            return;
        }

        recursionCount++;

        // Define a helper function that performs the actual usage variant update.
        function updateUsageVarianceForType(type: Type, variance: Variance) {
            doForEachSubtype(type, (subtype) => {
                const typeParamIndex = typeAliasTypeParams.findIndex((param) => isTypeSame(param, subtype));
                if (typeParamIndex >= 0) {
                    usageVariances[typeParamIndex] = combineVariances(usageVariances[typeParamIndex], variance);
                } else {
                    updateUsageVariancesRecursive(subtype, typeAliasTypeParams, usageVariances, recursionCount);
                }
            });
        }

        doForEachSubtype(type, (subtype) => {
            if (subtype.category === TypeCategory.Function) {
                if (subtype.specializedTypes) {
                    subtype.specializedTypes.parameterTypes.forEach((paramType) => {
                        updateUsageVarianceForType(paramType, Variance.Contravariant);
                    });

                    const returnType = subtype.specializedTypes.returnType;
                    if (returnType) {
                        updateUsageVarianceForType(returnType, Variance.Covariant);
                    }
                }
            } else if (subtype.category === TypeCategory.Class) {
                if (subtype.typeArguments) {
                    // If the class includes type parameters that uses auto variance,
                    // compute the calculated variance.
                    inferTypeParameterVarianceForClass(subtype);

                    // Is the class specialized using any type arguments that correspond to
                    // the type alias' type parameters?
                    subtype.typeArguments.forEach((typeArg, classParamIndex) => {
                        if (isTupleClass(subtype)) {
                            updateUsageVarianceForType(typeArg, Variance.Covariant);
                        } else if (classParamIndex < subtype.details.typeParameters.length) {
                            const classTypeParam = subtype.details.typeParameters[classParamIndex];
                            if (isUnpackedClass(typeArg) && typeArg.tupleTypeArguments) {
                                typeArg.tupleTypeArguments.forEach((tupleTypeArg) => {
                                    updateUsageVarianceForType(tupleTypeArg.type, Variance.Invariant);
                                });
                            } else {
                                updateUsageVarianceForType(
                                    typeArg,
                                    classTypeParam.computedVariance ?? classTypeParam.details.declaredVariance
                                );
                            }
                        }
                    });
                }
            }
        });
    }

    function makeTupleObject(entryTypes: Type[], isUnspecifiedLength = false) {
        if (tupleClassType && isInstantiableClass(tupleClassType)) {
            return convertToInstance(
                specializeTupleClass(
                    tupleClassType,
                    entryTypes.map((t) => {
                        return { type: t, isUnbounded: isUnspecifiedLength };
                    })
                )
            );
        }

        return UnknownType.create();
    }

    function getIndexAccessMagicMethodName(usage: EvaluatorUsage): string {
        if (usage.method === 'get') {
            return '__getitem__';
        } else if (usage.method === 'set') {
            return '__setitem__';
        } else {
            assert(usage.method === 'del');
            return '__delitem__';
        }
    }

    function getTypeOfIndexedObjectOrClass(node: IndexNode, baseType: ClassType, usage: EvaluatorUsage): TypeResult {
        // Handle index operations for TypedDict classes specially.
        if (isClassInstance(baseType) && ClassType.isTypedDictClass(baseType)) {
            const typeFromTypedDict = getTypeOfIndexedTypedDict(evaluatorInterface, node, baseType, usage);
            if (typeFromTypedDict) {
                return typeFromTypedDict;
            }
        }

        const magicMethodName = getIndexAccessMagicMethodName(usage);
        const itemMethodType = isClassInstance(baseType)
            ? getTypeOfObjectMember(
                  node,
                  baseType,
                  magicMethodName,
                  /* usage */ undefined,
                  /* diag */ undefined,
                  MemberAccessFlags.SkipAttributeAccessOverride
              )?.type
            : getTypeOfClassMember(
                  node,
                  baseType,
                  magicMethodName,
                  /* usage */ undefined,
                  /* diag */ undefined,
                  MemberAccessFlags.SkipAttributeAccessOverride | MemberAccessFlags.ConsiderMetaclassOnly
              )?.type;

        if (!itemMethodType) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.methodNotDefinedOnType().format({
                    name: magicMethodName,
                    type: printType(baseType),
                }),
                node.baseExpression
            );
            return { type: UnknownType.create() };
        }

        // Handle the special case where the object is a Tuple and
        // the index is a constant number (integer) or a slice with integer
        // start and end values. In these cases, we can determine
        // the exact type by indexing into the tuple type array.
        if (
            node.items.length === 1 &&
            !node.trailingComma &&
            !node.items[0].name &&
            node.items[0].argumentCategory === ArgumentCategory.Simple &&
            isClassInstance(baseType)
        ) {
            const index0Expr = node.items[0].valueExpression;
            const valueType = getTypeOfExpression(index0Expr).type;

            if (
                isClassInstance(valueType) &&
                ClassType.isBuiltIn(valueType, 'int') &&
                isLiteralType(valueType) &&
                typeof valueType.literalValue === 'number'
            ) {
                const indexValue = valueType.literalValue;
                const tupleType = getSpecializedTupleType(baseType);

                if (tupleType && tupleType.tupleTypeArguments && !isUnboundedTupleClass(tupleType)) {
                    if (indexValue >= 0 && indexValue < tupleType.tupleTypeArguments.length) {
                        return { type: tupleType.tupleTypeArguments[indexValue].type };
                    } else if (indexValue < 0 && tupleType.tupleTypeArguments.length + indexValue >= 0) {
                        return {
                            type: tupleType.tupleTypeArguments[tupleType.tupleTypeArguments.length + indexValue].type,
                        };
                    }
                }
            } else if (isClassInstance(valueType) && ClassType.isBuiltIn(valueType, 'slice')) {
                const tupleType = getSpecializedTupleType(baseType);
                if (tupleType && tupleType.tupleTypeArguments && !isUnboundedTupleClass(tupleType)) {
                    if (index0Expr.nodeType === ParseNodeType.Slice && !index0Expr.stepValue) {
                        // Create a local helper function to evaluate the slice parameters.
                        const getSliceParameter = (expression: ExpressionNode | undefined, defaultValue: number) => {
                            let value = defaultValue;
                            if (expression) {
                                const valType = getTypeOfExpression(expression).type;
                                if (
                                    isClassInstance(valType) &&
                                    ClassType.isBuiltIn(valType, 'int') &&
                                    isLiteralType(valType) &&
                                    typeof valType.literalValue === 'number'
                                ) {
                                    value = valType.literalValue;
                                    if (value < 0) {
                                        value = tupleType.tupleTypeArguments!.length + value;
                                    }
                                } else {
                                    value = -1;
                                }
                            }
                            return value;
                        };

                        const startValue = getSliceParameter(index0Expr.startValue, 0);
                        const endValue = getSliceParameter(index0Expr.endValue, tupleType.tupleTypeArguments.length);

                        if (
                            startValue >= 0 &&
                            endValue > 0 &&
                            endValue <= tupleType.tupleTypeArguments.length &&
                            tupleClassType &&
                            isInstantiableClass(tupleClassType)
                        ) {
                            return {
                                type: ClassType.cloneAsInstance(
                                    specializeTupleClass(
                                        tupleClassType,
                                        tupleType.tupleTypeArguments.slice(startValue, endValue)
                                    )
                                ),
                            };
                        }
                    }
                }
            }
        }

        // Follow PEP 637 rules for positional and keyword arguments.
        const positionalArgs = node.items.filter(
            (item) => item.argumentCategory === ArgumentCategory.Simple && !item.name
        );
        const unpackedListArgs = node.items.filter((item) => item.argumentCategory === ArgumentCategory.UnpackedList);

        const keywordArgs = node.items.filter(
            (item) => item.argumentCategory === ArgumentCategory.Simple && !!item.name
        );
        const unpackedDictArgs = node.items.filter(
            (item) => item.argumentCategory === ArgumentCategory.UnpackedDictionary
        );

        let positionalIndexType: Type;
        let isPositionalIndexTypeIncomplete = false;

        if (positionalArgs.length === 1 && unpackedListArgs.length === 0 && !node.trailingComma) {
            // Handle the common case where there is a single positional argument.
            const typeResult = getTypeOfExpression(positionalArgs[0].valueExpression);
            positionalIndexType = typeResult.type;
            if (typeResult.isIncomplete) {
                isPositionalIndexTypeIncomplete = true;
            }
        } else if (positionalArgs.length === 0 && unpackedListArgs.length === 0) {
            // Handle the case where there are no positionals provided but there are keywords.
            positionalIndexType =
                tupleClassType && isInstantiableClass(tupleClassType)
                    ? convertToInstance(specializeTupleClass(tupleClassType, []))
                    : UnknownType.create();
        } else {
            // Package up all of the positionals into a tuple.
            const tupleEntries: Type[] = [];
            positionalArgs.forEach((arg) => {
                const typeResult = getTypeOfExpression(arg.valueExpression);
                tupleEntries.push(typeResult.type);
                if (typeResult.isIncomplete) {
                    isPositionalIndexTypeIncomplete = true;
                }
            });

            unpackedListArgs.forEach((arg) => {
                const typeResult = getTypeOfExpression(arg.valueExpression);
                if (typeResult.isIncomplete) {
                    isPositionalIndexTypeIncomplete = true;
                }
                const iterableType =
                    getTypeOfIterator(typeResult, /* isAsync */ false, arg.valueExpression)?.type ??
                    UnknownType.create();
                tupleEntries.push(iterableType);
            });

            positionalIndexType = makeTupleObject(tupleEntries, unpackedListArgs.length > 0);
        }

        let argList: FunctionArgument[] = [
            {
                argumentCategory: ArgumentCategory.Simple,
                typeResult: { type: positionalIndexType, isIncomplete: isPositionalIndexTypeIncomplete },
            },
        ];

        if (usage.method === 'set') {
            let setType = usage.setType?.type ?? AnyType.create();

            // Expand constrained type variables.
            if (isTypeVar(setType) && setType.details.constraints.length > 0) {
                const conditionFilter = isClassInstance(baseType) ? baseType.condition : undefined;
                setType = makeTopLevelTypeVarsConcrete(
                    setType,
                    /* makeParamSpecsConcrete */ undefined,
                    conditionFilter
                );
            }

            argList.push({
                argumentCategory: ArgumentCategory.Simple,
                typeResult: {
                    type: setType,
                    isIncomplete: !!usage.setType?.isIncomplete,
                },
            });
        }

        keywordArgs.forEach((arg) => {
            argList.push({
                argumentCategory: ArgumentCategory.Simple,
                valueExpression: arg.valueExpression,
                node: arg,
                name: arg.name,
            });
        });

        unpackedDictArgs.forEach((arg) => {
            argList.push({
                argumentCategory: ArgumentCategory.UnpackedDictionary,
                valueExpression: arg.valueExpression,
                node: arg,
            });
        });

        let callResult: CallResult | undefined;

        // Speculatively attempt the call. We may need to replace the index
        // type with 'int', and we don't want to emit errors before we know
        // which type to use.
        if (keywordArgs.length === 0 && unpackedDictArgs.length === 0 && positionalArgs.length === 1) {
            useSpeculativeMode(node, () => {
                callResult = validateCallArguments(node, argList, { type: itemMethodType });

                if (callResult.argumentErrors) {
                    // If the object supports "__index__" magic method, convert
                    // the index to an int and try again.
                    if (isClassInstance(positionalIndexType)) {
                        const altArgList = [...argList];
                        altArgList[0] = { ...altArgList[0] };
                        const indexMethod = getTypeOfObjectMember(node, positionalIndexType, '__index__');

                        if (indexMethod) {
                            const intType = getBuiltInObject(node, 'int');
                            if (isClassInstance(intType)) {
                                altArgList[0].typeResult = { type: intType };
                            }
                        }

                        callResult = validateCallArguments(node, altArgList, { type: itemMethodType });

                        // We were successful, so replace the arg list.
                        if (!callResult.argumentErrors) {
                            argList = altArgList;
                        }
                    }
                }
            });
        }

        callResult = validateCallArguments(node, argList, { type: itemMethodType });

        return {
            type: callResult.returnType ?? UnknownType.create(),
            isIncomplete: !!callResult.isTypeIncomplete,
        };
    }

    function getTypeArgs(node: IndexNode, flags: EvaluatorFlags, options?: GetTypeArgsOptions): TypeResultWithNode[] {
        const typeArgs: TypeResultWithNode[] = [];
        let adjFlags = flags;

        if (options?.isFinalAnnotation || options?.isClassVarAnnotation) {
            adjFlags |= EvaluatorFlags.DisallowClassVar | EvaluatorFlags.DisallowFinal;
        } else {
            adjFlags &= ~(
                EvaluatorFlags.DoNotSpecialize |
                EvaluatorFlags.DisallowParamSpec |
                EvaluatorFlags.DisallowTypeVarTuple |
                EvaluatorFlags.AllowRequired |
                EvaluatorFlags.EnforceTypeVarVarianceConsistency
            );

            if (!options?.isAnnotatedClass) {
                adjFlags |= EvaluatorFlags.DisallowClassVar | EvaluatorFlags.DisallowFinal;
            }

            adjFlags |= EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple;
        }

        // Create a local function that validates a single type argument.
        const getTypeArgTypeResult = (expr: ExpressionNode, argIndex: number) => {
            let typeResult: TypeResultWithNode;

            // If it's a custom __class_getitem__, none of the arguments should be
            // treated as types. If it's an Annotated[a, b, c], only the first index
            // should be treated as a type. The others can be regular (non-type) objects.
            if (options?.hasCustomClassGetItem || (options?.isAnnotatedClass && argIndex > 0)) {
                typeResult = {
                    ...getTypeOfExpression(
                        expr,
                        EvaluatorFlags.DisallowParamSpec |
                            EvaluatorFlags.DisallowTypeVarTuple |
                            EvaluatorFlags.DoNotSpecialize |
                            EvaluatorFlags.DisallowClassVar
                    ),
                    node: expr,
                };
            } else {
                typeResult = getTypeArg(expr, adjFlags, !!options?.supportsTypedDictTypeArg && argIndex === 0);
            }

            return typeResult;
        };

        // A single (non-empty) tuple is treated the same as a list of items in the index.
        if (
            node.items.length === 1 &&
            !node.trailingComma &&
            !node.items[0].name &&
            node.items[0].valueExpression.nodeType === ParseNodeType.Tuple &&
            node.items[0].valueExpression.expressions.length > 0
        ) {
            node.items[0].valueExpression.expressions.forEach((item, index) => {
                typeArgs.push(getTypeArgTypeResult(item, index));
            });

            // Set the node's type so it isn't reevaluated later.
            setTypeForNode(node.items[0].valueExpression, UnknownType.create());
        } else {
            node.items.forEach((arg, index) => {
                const typeResult = getTypeArgTypeResult(arg.valueExpression, index);

                if (arg.argumentCategory !== ArgumentCategory.Simple) {
                    if (arg.argumentCategory === ArgumentCategory.UnpackedList) {
                        if (isVariadicTypeVar(typeResult.type) && !typeResult.type.isVariadicUnpacked) {
                            typeResult.type = TypeVarType.cloneForUnpacked(typeResult.type);
                        } else if (
                            isInstantiableClass(typeResult.type) &&
                            !typeResult.type.includeSubclasses &&
                            isTupleClass(typeResult.type)
                        ) {
                            typeResult.type = ClassType.cloneForUnpacked(typeResult.type);
                        }
                    }
                }

                if (arg.name) {
                    addError(Localizer.Diagnostic.keywordArgInTypeArgument(), arg.valueExpression);
                }

                typeArgs.push(typeResult);
            });
        }

        return typeArgs;
    }

    function getTypeArg(
        node: ExpressionNode,
        flags: EvaluatorFlags,
        supportsDictExpression: boolean
    ): TypeResultWithNode {
        let typeResult: TypeResultWithNode;

        let adjustedFlags =
            flags |
            EvaluatorFlags.ExpectingType |
            EvaluatorFlags.ConvertEllipsisToAny |
            EvaluatorFlags.EvaluateStringLiteralAsType;

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        if (fileInfo.isStubFile) {
            adjustedFlags |= EvaluatorFlags.AllowForwardReferences;
        }

        if (node.nodeType === ParseNodeType.List) {
            typeResult = {
                type: UnknownType.create(),
                typeList: node.entries.map((entry) => {
                    return { ...getTypeOfExpression(entry, adjustedFlags), node: entry };
                }),
                node,
            };

            // Set the node's type so it isn't reevaluated later.
            setTypeForNode(node, UnknownType.create());
        } else if (node.nodeType === ParseNodeType.Dictionary && supportsDictExpression) {
            const inlinedTypeDict =
                typedDictClassType && isInstantiableClass(typedDictClassType)
                    ? createTypedDictTypeInlined(evaluatorInterface, node, typedDictClassType)
                    : undefined;
            const keyTypeFallback =
                strClassType && isInstantiableClass(strClassType) ? strClassType : UnknownType.create();

            typeResult = {
                type: keyTypeFallback,
                inlinedTypeDict,
                node,
            };
        } else {
            typeResult = { ...getTypeOfExpression(node, adjustedFlags), node };

            if (node.nodeType === ParseNodeType.Dictionary) {
                addError(Localizer.Diagnostic.dictInAnnotation(), node);
            }

            // "Protocol" is not allowed as a type argument.
            if (isClass(typeResult.type) && ClassType.isBuiltIn(typeResult.type, 'Protocol')) {
                addError(Localizer.Diagnostic.protocolNotAllowedInTypeArgument(), node);
            }

            if ((flags & EvaluatorFlags.DisallowClassVar) !== 0) {
                // "ClassVar" is not allowed as a type argument.
                if (isClass(typeResult.type) && ClassType.isBuiltIn(typeResult.type, 'ClassVar')) {
                    addError(Localizer.Diagnostic.classVarNotAllowed(), node);
                }
            }
        }

        return typeResult;
    }

    function getTypeOfTuple(
        node: TupleNode,
        inferenceContext: InferenceContext | undefined,
        flags: EvaluatorFlags
    ): TypeResult {
        if ((flags & EvaluatorFlags.ExpectingType) !== 0 && node.expressions.length === 0 && !inferenceContext) {
            return { type: makeTupleObject([]), isEmptyTupleShorthand: true };
        }

        // If the expected type is a union, recursively call for each of the subtypes
        // to find one that matches.
        let effectiveExpectedType = inferenceContext?.expectedType;
        let expectedTypeContainsAny = inferenceContext && isAny(inferenceContext.expectedType);

        if (inferenceContext && isUnion(inferenceContext.expectedType)) {
            let matchingSubtype: Type | undefined;

            doForEachSubtype(inferenceContext.expectedType, (subtype) => {
                if (isAny(subtype)) {
                    expectedTypeContainsAny = true;
                }

                if (!matchingSubtype) {
                    const subtypeResult = useSpeculativeMode(node, () => {
                        return getTypeOfTupleWithContext(
                            node,
                            makeInferenceContext(subtype, inferenceContext?.typeVarContext)
                        );
                    });

                    if (subtypeResult && assignType(subtype, subtypeResult.type)) {
                        matchingSubtype = subtype;
                    }
                }
            });

            effectiveExpectedType = matchingSubtype;
        }

        let expectedTypeDiagAddendum: DiagnosticAddendum | undefined;
        if (effectiveExpectedType) {
            const result = getTypeOfTupleWithContext(
                node,
                makeInferenceContext(effectiveExpectedType, inferenceContext?.typeVarContext)
            );
            if (result && !result.typeErrors) {
                return result;
            }

            expectedTypeDiagAddendum = result?.expectedTypeDiagAddendum;
        }

        const typeResult = getTypeOfTupleInferred(node);

        // If there was an expected type of Any, replace the resulting type
        // with Any rather than return a type with unknowns.
        if (expectedTypeContainsAny) {
            typeResult.type = AnyType.create();
        }

        return { ...typeResult, expectedTypeDiagAddendum };
    }

    function getTypeOfTupleWithContext(node: TupleNode, inferenceContext: InferenceContext): TypeResult | undefined {
        inferenceContext.expectedType = transformPossibleRecursiveTypeAlias(inferenceContext.expectedType);
        if (!isClassInstance(inferenceContext.expectedType)) {
            return undefined;
        }

        if (!tupleClassType || !isInstantiableClass(tupleClassType)) {
            return undefined;
        }

        // Build an array of expected types.
        let expectedTypes: Type[] = [];

        if (isTupleClass(inferenceContext.expectedType) && inferenceContext.expectedType.tupleTypeArguments) {
            expectedTypes = inferenceContext.expectedType.tupleTypeArguments.map((t) =>
                transformPossibleRecursiveTypeAlias(t.type)
            );
            const unboundedIndex = inferenceContext.expectedType.tupleTypeArguments.findIndex((t) => t.isUnbounded);
            if (unboundedIndex >= 0) {
                if (expectedTypes.length > node.expressions.length) {
                    expectedTypes.splice(unboundedIndex, 1);
                } else {
                    while (expectedTypes.length < node.expressions.length) {
                        expectedTypes.splice(unboundedIndex, 0, expectedTypes[unboundedIndex]);
                    }
                }
            }
        } else {
            const tupleTypeVarContext = new TypeVarContext(getTypeVarScopeId(tupleClassType));
            if (
                !populateTypeVarContextBasedOnExpectedType(
                    evaluatorInterface,
                    ClassType.cloneAsInstance(tupleClassType),
                    inferenceContext.expectedType,
                    tupleTypeVarContext,
                    getTypeVarScopesForNode(node)
                )
            ) {
                return undefined;
            }

            const specializedTuple = applySolvedTypeVars(tupleClassType, tupleTypeVarContext) as ClassType;
            if (!specializedTuple.typeArguments || specializedTuple.typeArguments.length !== 1) {
                return undefined;
            }

            const homogenousType = transformPossibleRecursiveTypeAlias(specializedTuple.typeArguments[0]);
            for (let i = 0; i < node.expressions.length; i++) {
                expectedTypes.push(homogenousType);
            }
        }

        const entryTypeResults = node.expressions.map((expr, index) =>
            getTypeOfExpression(
                expr,
                /* flags */ undefined,
                makeInferenceContext(index < expectedTypes.length ? expectedTypes[index] : undefined)
            )
        );
        const isIncomplete = entryTypeResults.some((result) => result.isIncomplete);

        const type = convertToInstance(
            specializeTupleClass(
                tupleClassType,
                buildTupleTypesList(entryTypeResults),
                /* isTypeArgumentExplicit */ true
            )
        );

        // Copy any expected type diag addenda for precision error reporting.
        let expectedTypeDiagAddendum: DiagnosticAddendum | undefined;
        if (entryTypeResults.some((result) => result.expectedTypeDiagAddendum)) {
            expectedTypeDiagAddendum = new DiagnosticAddendum();
            entryTypeResults.forEach((result) => {
                if (result.expectedTypeDiagAddendum) {
                    expectedTypeDiagAddendum!.addAddendum(result.expectedTypeDiagAddendum);
                }
            });
        }

        return { type, expectedTypeDiagAddendum, isIncomplete };
    }

    function getTypeOfTupleInferred(node: TupleNode): TypeResult {
        const entryTypeResults = node.expressions.map((expr) => getTypeOfExpression(expr));
        const isIncomplete = entryTypeResults.some((result) => result.isIncomplete);

        if (!tupleClassType || !isInstantiableClass(tupleClassType)) {
            return { type: UnknownType.create() };
        }

        const type = convertToInstance(specializeTupleClass(tupleClassType, buildTupleTypesList(entryTypeResults)));

        if (isIncomplete) {
            if (getContainerDepth(type) > maxInferredContainerDepth) {
                return { type: UnknownType.create() };
            }
        }

        return { type, isIncomplete };
    }

    function buildTupleTypesList(entryTypeResults: TypeResult[]): TupleTypeArgument[] {
        const entryTypes: TupleTypeArgument[] = [];

        for (const typeResult of entryTypeResults) {
            let possibleUnpackedTuple: Type | undefined;
            if (typeResult.unpackedType) {
                possibleUnpackedTuple = typeResult.unpackedType;
            } else if (isUnpacked(typeResult.type)) {
                possibleUnpackedTuple = typeResult.type;
            }

            // Is this an unpacked tuple? If so, we can append the individual
            // unpacked entries onto the new tuple. If it's not an upacked tuple
            // but some other iterator (e.g. a List), we won't know the number of
            // items, so we'll need to leave the Tuple open-ended.
            if (
                possibleUnpackedTuple &&
                isClassInstance(possibleUnpackedTuple) &&
                possibleUnpackedTuple.tupleTypeArguments
            ) {
                const typeArgs = possibleUnpackedTuple.tupleTypeArguments;

                if (!typeArgs) {
                    entryTypes.push({ type: UnknownType.create(), isUnbounded: true });
                } else {
                    appendArray(entryTypes, typeArgs);
                }
            } else if (isNever(typeResult.type) && typeResult.isIncomplete && !typeResult.unpackedType) {
                entryTypes.push({ type: UnknownType.create(/* isIncomplete */ true), isUnbounded: false });
            } else {
                entryTypes.push({ type: typeResult.type, isUnbounded: !!typeResult.unpackedType });
            }
        }

        // If there are multiple unbounded entries, combine all of them into a single
        // unbounded entry to avoid violating the invariant that there can be at most
        // one unbounded entry in a tuple.
        if (entryTypes.filter((t) => t.isUnbounded).length > 1) {
            const firstUnboundedEntryIndex = entryTypes.findIndex((t) => t.isUnbounded);
            const removedEntries = entryTypes.splice(firstUnboundedEntryIndex);
            entryTypes.push({ type: combineTypes(removedEntries.map((t) => t.type)), isUnbounded: true });
        }

        return entryTypes;
    }

    function getTypeOfCall(
        node: CallNode,
        inferenceContext: InferenceContext | undefined,
        flags: EvaluatorFlags
    ): TypeResult {
        const baseTypeResult = getTypeOfExpression(node.leftExpression, EvaluatorFlags.DoNotSpecialize);

        const argList = node.arguments.map((arg) => {
            const functionArg: FunctionArgument = {
                valueExpression: arg.valueExpression,
                argumentCategory: arg.argumentCategory,
                node: arg,
                name: arg.name,
            };
            return functionArg;
        });

        let typeResult: TypeResult = { type: UnknownType.create() };

        if (!isTypeAliasPlaceholder(baseTypeResult.type)) {
            if (node.leftExpression.nodeType === ParseNodeType.Name && node.leftExpression.value === 'super') {
                // Handle the built-in "super" call specially.
                typeResult = getTypeOfSuperCall(node);
            } else if (
                isAnyOrUnknown(baseTypeResult.type) &&
                node.leftExpression.nodeType === ParseNodeType.Name &&
                node.leftExpression.value === 'reveal_type'
            ) {
                // Handle the implicit "reveal_type" call.
                typeResult = getTypeOfRevealType(node, inferenceContext);
            } else if (isFunction(baseTypeResult.type) && baseTypeResult.type.details.builtInName === 'reveal_type') {
                // Handle the "typing.reveal_type" call.
                typeResult = getTypeOfRevealType(node, inferenceContext);
            } else if (isFunction(baseTypeResult.type) && baseTypeResult.type.details.builtInName === 'assert_type') {
                // Handle the "typing.assert_type" call.
                typeResult = getTypeOfAssertType(node, inferenceContext);
            } else if (
                isAnyOrUnknown(baseTypeResult.type) &&
                node.leftExpression.nodeType === ParseNodeType.Name &&
                node.leftExpression.value === 'reveal_locals'
            ) {
                if (node.arguments.length === 0) {
                    // Handle the special-case "reveal_locals" call.
                    typeResult.type = getTypeOfRevealLocals(node);
                } else {
                    addError(Localizer.Diagnostic.revealLocalsArgs(), node);
                }
            } else {
                const callResult = validateCallArguments(
                    node,
                    argList,
                    baseTypeResult,
                    /* typeVarContext */ undefined,
                    /* skipUnknownArgCheck */ false,
                    inferenceContext
                );

                typeResult.type = callResult.returnType ?? UnknownType.create();

                if (callResult.argumentErrors) {
                    typeResult.typeErrors = true;
                } else {
                    typeResult.overloadsUsedForCall = callResult.overloadsUsedForCall;
                }

                if (callResult.isTypeIncomplete) {
                    typeResult.isIncomplete = true;
                }
            }

            if (baseTypeResult.isIncomplete) {
                typeResult.isIncomplete = true;
            }
        } else {
            typeResult.isIncomplete = true;
        }

        // Don't bother evaluating the arguments if we're speculatively evaluating the call
        // or the base type is incomplete.
        if (!speculativeTypeTracker.isSpeculative(node) && !baseTypeResult.isIncomplete) {
            // Touch all of the args so they're marked accessed even if there were errors.
            // We skip this if it's a TypeVar() call in the typing.pyi module because
            // this results in a cyclical type resolution problem whereby we try to
            // retrieve the str class, which inherits from Sequence, which inherits from
            // Iterable, which uses a TypeVar. Without this, Iterable and Sequence classes
            // have invalid type parameters.
            const isCyclicalTypeVarCall =
                isInstantiableClass(baseTypeResult.type) &&
                ClassType.isBuiltIn(baseTypeResult.type, 'TypeVar') &&
                AnalyzerNodeInfo.getFileInfo(node).isTypingStubFile;

            if (!isCyclicalTypeVarCall) {
                argList.forEach((arg) => {
                    if (
                        arg.valueExpression &&
                        arg.valueExpression.nodeType !== ParseNodeType.StringList &&
                        !isTypeCached(arg.valueExpression)
                    ) {
                        getTypeOfExpression(arg.valueExpression);
                    }
                });
            }
        }

        if ((flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeAnnotationCall(),
                node
            );

            typeResult = { type: UnknownType.create() };
        }

        return typeResult;
    }

    function getTypeOfAssertType(node: CallNode, inferenceContext: InferenceContext | undefined): TypeResult {
        if (
            node.arguments.length !== 2 ||
            node.arguments[0].argumentCategory !== ArgumentCategory.Simple ||
            node.arguments[0].name !== undefined ||
            node.arguments[0].argumentCategory !== ArgumentCategory.Simple ||
            node.arguments[1].name !== undefined
        ) {
            addError(Localizer.Diagnostic.assertTypeArgs(), node);
            return { type: UnknownType.create() };
        }

        const arg0TypeResult = getTypeOfExpression(
            node.arguments[0].valueExpression,
            /* flags */ undefined,
            inferenceContext
        );
        if (arg0TypeResult.isIncomplete) {
            return { type: UnknownType.create(/* isIncomplete */ true), isIncomplete: true };
        }

        const assertedType = convertToInstance(getTypeOfArgumentExpectingType(node.arguments[1]).type);

        if (
            !isTypeSame(assertedType, arg0TypeResult.type, { treatAnySameAsUnknown: true, ignorePseudoGeneric: true })
        ) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.assertTypeTypeMismatch().format({
                    expected: printType(assertedType),
                    received: printType(arg0TypeResult.type),
                }),
                node.arguments[0].valueExpression
            );
        }

        return { type: arg0TypeResult.type };
    }

    function getTypeOfRevealType(node: CallNode, inferenceContext: InferenceContext | undefined): TypeResult {
        let arg0Value: ExpressionNode | undefined;
        let expectedRevealTypeNode: ExpressionNode | undefined;
        let expectedRevealType: Type | undefined;
        let expectedTextNode: ExpressionNode | undefined;
        let expectedText: string | undefined;

        // Make sure there is only one positional argument passed as arg 0.
        node.arguments.forEach((arg, index) => {
            if (index === 0) {
                if (arg.argumentCategory === ArgumentCategory.Simple && !arg.name) {
                    arg0Value = arg.valueExpression;
                }
            } else if (arg.argumentCategory !== ArgumentCategory.Simple || !arg.name) {
                arg0Value = undefined;
            } else if (arg.name.value === 'expected_text') {
                expectedTextNode = arg.valueExpression;
                const expectedTextType = getTypeOfExpression(arg.valueExpression).type;

                if (
                    !isClassInstance(expectedTextType) ||
                    !ClassType.isBuiltIn(expectedTextType, 'str') ||
                    typeof expectedTextType.literalValue !== 'string'
                ) {
                    addError(Localizer.Diagnostic.revealTypeExpectedTextArg(), arg.valueExpression);
                } else {
                    expectedText = expectedTextType.literalValue;
                }
            } else if (arg.name.value === 'expected_type') {
                expectedRevealTypeNode = arg.valueExpression;
                expectedRevealType = convertToInstance(getTypeOfArgumentExpectingType(arg).type);
            }
        });

        if (!arg0Value) {
            addError(Localizer.Diagnostic.revealTypeArgs(), node);
            return { type: UnknownType.create() };
        }

        const typeResult = getTypeOfExpression(arg0Value, /* flags */ undefined, inferenceContext);
        const type = typeResult.type;

        const exprString = ParseTreeUtils.printExpression(arg0Value);
        const typeString = printType(type, { expandTypeAlias: true });

        if (expectedText !== undefined) {
            if (expectedText !== typeString) {
                addError(
                    Localizer.Diagnostic.revealTypeExpectedTextMismatch().format({
                        expected: expectedText,
                        received: typeString,
                    }),
                    expectedTextNode ?? arg0Value
                );
            }
        }

        if (expectedRevealType) {
            if (!isTypeSame(expectedRevealType, type, { ignorePseudoGeneric: true })) {
                const expectedRevealTypeText = printType(expectedRevealType);
                addError(
                    Localizer.Diagnostic.revealTypeExpectedTypeMismatch().format({
                        expected: expectedRevealTypeText,
                        received: typeString,
                    }),
                    expectedRevealTypeNode ?? arg0Value
                );
            }
        }

        addInformation(
            Localizer.DiagnosticAddendum.typeOfSymbol().format({ name: exprString, type: typeString }),
            node.arguments[0]
        );

        return { type, isIncomplete: typeResult.isIncomplete };
    }

    function getTypeOfRevealLocals(node: CallNode) {
        let curNode: ParseNode | undefined = node;
        let scope: Scope | undefined;

        while (curNode) {
            scope = ScopeUtils.getScopeForNode(curNode);

            // Stop when we get a valid scope that's not a list comprehension
            // scope. That includes lambdas, functions, classes, and modules.
            if (scope && scope.type !== ScopeType.ListComprehension) {
                break;
            }

            curNode = curNode.parent;
        }

        const infoMessages: string[] = [];

        if (scope) {
            scope.symbolTable.forEach((symbol, name) => {
                if (!symbol.isIgnoredForProtocolMatch()) {
                    const typeOfSymbol = getEffectiveTypeOfSymbol(symbol);
                    infoMessages.push(
                        Localizer.DiagnosticAddendum.typeOfSymbol().format({
                            name,
                            type: printType(typeOfSymbol, { expandTypeAlias: true }),
                        })
                    );
                }
            });
        }

        if (infoMessages.length > 0) {
            addInformation(infoMessages.join('\n'), node);
        } else {
            addInformation(Localizer.Diagnostic.revealLocalsNone(), node);
        }

        return NoneType.createInstance();
    }

    function getTypeOfSuperCall(node: CallNode): TypeResult {
        if (node.arguments.length > 2) {
            addError(Localizer.Diagnostic.superCallArgCount(), node.arguments[2]);
        }

        // Determine which class the "super" call is applied to. If
        // there is no first argument, then the class is implicit.
        let targetClassType: Type;
        if (node.arguments.length > 0) {
            targetClassType = getTypeOfExpression(node.arguments[0].valueExpression).type;
            const concreteTargetClassType = makeTopLevelTypeVarsConcrete(targetClassType);

            if (!isAnyOrUnknown(concreteTargetClassType) && !isInstantiableClass(concreteTargetClassType)) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.superCallFirstArg().format({ type: printType(targetClassType) }),
                    node.arguments[0].valueExpression
                );
            }
        } else {
            const enclosingClass = ParseTreeUtils.getEnclosingClass(node);
            if (enclosingClass) {
                const classTypeInfo = getTypeOfClass(enclosingClass);
                targetClassType = classTypeInfo ? classTypeInfo.classType : UnknownType.create();
            } else {
                addError(Localizer.Diagnostic.superCallZeroArgForm(), node.leftExpression);
                targetClassType = UnknownType.create();
            }
        }

        // Determine whether to further narrow the type.
        let bindToType: ClassType | undefined;
        if (node.arguments.length > 1) {
            const secondArgType = makeTopLevelTypeVarsConcrete(
                getTypeOfExpression(node.arguments[1].valueExpression).type
            );

            let reportError = false;

            if (isAnyOrUnknown(secondArgType)) {
                // Ignore unknown or any types.
            } else if (isClassInstance(secondArgType)) {
                if (isInstantiableClass(targetClassType)) {
                    if (
                        !derivesFromClassRecursive(
                            ClassType.cloneAsInstantiable(secondArgType),
                            targetClassType,
                            /* ignoreUnknown */ true
                        )
                    ) {
                        reportError = true;
                    }
                }
                bindToType = secondArgType;
            } else if (isInstantiableClass(secondArgType)) {
                if (isInstantiableClass(targetClassType)) {
                    if (!derivesFromClassRecursive(secondArgType, targetClassType, /* ignoreUnknown */ true)) {
                        reportError = true;
                    }
                }
                bindToType = secondArgType;
            } else {
                reportError = true;
            }

            if (reportError) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.superCallSecondArg().format({ type: printType(targetClassType) }),
                    node.arguments[1].valueExpression
                );
            }
        } else {
            const enclosingMethod = ParseTreeUtils.getEnclosingFunction(node);
            let implicitBindToType: Type | undefined;

            // Get the type from the self or cls parameter if it is explicitly annotated.
            if (enclosingMethod) {
                const methodTypeInfo = getTypeOfFunction(enclosingMethod);
                if (methodTypeInfo) {
                    const methodType = methodTypeInfo.functionType;
                    if (FunctionType.isClassMethod(methodType)) {
                        if (
                            methodType.details.parameters.length > 0 &&
                            methodType.details.parameters[0].hasDeclaredType
                        ) {
                            implicitBindToType = makeTopLevelTypeVarsConcrete(methodType.details.parameters[0].type);
                        }
                    } else if (FunctionType.isInstanceMethod(methodType)) {
                        if (
                            methodType.details.parameters.length > 0 &&
                            methodType.details.parameters[0].hasDeclaredType
                        ) {
                            implicitBindToType = makeTopLevelTypeVarsConcrete(
                                convertToInstantiable(methodType.details.parameters[0].type)
                            );
                        }
                    }
                }
            }

            if (implicitBindToType && isInstantiableClass(implicitBindToType)) {
                bindToType = implicitBindToType;
            } else if (isInstantiableClass(targetClassType)) {
                bindToType = targetClassType;
            }
        }

        // Determine whether super() should return an instance of the class or
        // the class itself. It depends on whether the super() call is located
        // within an instance method or not.
        let resultIsInstance = true;
        if (node.arguments.length <= 1) {
            const enclosingMethod = ParseTreeUtils.getEnclosingFunction(node);
            if (enclosingMethod) {
                const methodType = getTypeOfFunction(enclosingMethod);
                if (methodType) {
                    if (
                        FunctionType.isStaticMethod(methodType.functionType) ||
                        FunctionType.isConstructorMethod(methodType.functionType) ||
                        FunctionType.isClassMethod(methodType.functionType)
                    ) {
                        resultIsInstance = false;
                    }
                }
            }
        }

        // Python docs indicate that super() isn't valid for
        // operations other than member accesses or attribute lookups.
        const parentNode = node.parent!;
        if (parentNode.nodeType === ParseNodeType.MemberAccess) {
            const memberName = parentNode.memberName.value;
            const lookupResults = lookUpClassMember(
                targetClassType,
                memberName,
                ClassMemberLookupFlags.SkipOriginalClass
            );
            if (lookupResults && isInstantiableClass(lookupResults.classType)) {
                return {
                    type: resultIsInstance
                        ? ClassType.cloneAsInstance(lookupResults.classType)
                        : lookupResults.classType,
                    bindToType: bindToType
                        ? TypeBase.cloneForCondition(
                              synthesizeTypeVarForSelfCls(bindToType, !resultIsInstance),
                              bindToType.condition
                          )
                        : undefined,
                };
            }
        }

        // If the lookup failed, try to return the first base class. An error
        // will be reported by the member lookup logic at a later time.
        if (isInstantiableClass(targetClassType)) {
            // If the class derives from one or more unknown classes,
            // return unknown here to prevent spurious errors.
            if (targetClassType.details.mro.some((mroBase) => isAnyOrUnknown(mroBase))) {
                return { type: UnknownType.create() };
            }

            const baseClasses = targetClassType.details.baseClasses;
            if (baseClasses.length > 0) {
                const baseClassType = baseClasses[0];
                if (isInstantiableClass(baseClassType)) {
                    return {
                        type: resultIsInstance ? ClassType.cloneAsInstance(baseClassType) : baseClassType,
                    };
                }
            }
        }

        return { type: UnknownType.create() };
    }

    // Attempts to find an overloaded function for each set of argument
    // types in the expandedArgTypes list. If an argument type is undefined,
    // its type is evaluated from the argument's expression using the
    // corresponding parameter's expected type. The first time this is called,
    // there will be only one argument list in expandedArgTypes, and all entries
    // (one for each argument) will be undefined. On subsequent calls, this
    // list will grow to include union expansions.
    function validateOverloadsWithExpandedTypes(
        errorNode: ExpressionNode,
        expandedArgTypes: (Type | undefined)[][],
        argParamMatches: MatchArgsToParamsResult[],
        typeVarContext: TypeVarContext | undefined,
        skipUnknownArgCheck: boolean,
        inferenceContext: InferenceContext | undefined
    ): CallResult {
        const returnTypes: Type[] = [];
        const matchedOverloads: MatchedOverloadInfo[] = [];
        let isTypeIncomplete = false;
        let overloadsUsedForCall: FunctionType[] = [];
        let isDefinitiveMatchFound = false;

        for (let expandedTypesIndex = 0; expandedTypesIndex < expandedArgTypes.length; expandedTypesIndex++) {
            let matchedOverload: FunctionType | undefined;
            const argTypeOverride = expandedArgTypes[expandedTypesIndex];
            const hasArgTypeOverride = argTypeOverride.some((a) => a !== undefined);
            let possibleMatchResults: MatchedOverloadInfo[] = [];
            isDefinitiveMatchFound = false;

            for (let overloadIndex = 0; overloadIndex < argParamMatches.length; overloadIndex++) {
                const overload = argParamMatches[overloadIndex].overload;

                let matchResults = argParamMatches[overloadIndex];
                if (hasArgTypeOverride) {
                    matchResults = { ...argParamMatches[overloadIndex] };
                    matchResults.argParams = matchResults.argParams.map((argParam, argIndex) => {
                        if (!argTypeOverride[argIndex]) {
                            return argParam;
                        }
                        const argParamCopy = { ...argParam };
                        argParamCopy.argType = argTypeOverride[argIndex];
                        return argParamCopy;
                    });
                }

                // Clone the typeVarContext so we don't modify the original. If this is
                // not the first time through the loop, clone the type var context
                // from the previous successful match.
                const typeVarContextToClone =
                    matchedOverloads.length > 0
                        ? matchedOverloads[matchedOverloads.length - 1].typeVarContext.clone()
                        : typeVarContext;
                const effectiveTypeVarContext =
                    typeVarContextToClone?.clone() ?? new TypeVarContext(getTypeVarScopeId(overload));
                effectiveTypeVarContext.addSolveForScope(getTypeVarScopeId(overload));
                if (overload.details.constructorTypeVarScopeId) {
                    effectiveTypeVarContext.addSolveForScope(overload.details.constructorTypeVarScopeId);
                }
                effectiveTypeVarContext.unlock();

                // Use speculative mode so we don't output any diagnostics or
                // record any final types in the type cache.
                const callResult = useSpeculativeMode(errorNode, () => {
                    return validateFunctionArgumentTypesWithExpectedType(
                        errorNode,
                        matchResults,
                        effectiveTypeVarContext,
                        /* skipUnknownArgCheck */ true,
                        inferenceContext
                    );
                });

                if (callResult.isTypeIncomplete) {
                    isTypeIncomplete = true;
                }

                if (!callResult.argumentErrors && callResult.returnType) {
                    overloadsUsedForCall.push(overload);

                    matchedOverload = overload;
                    const matchedOverloadInfo: MatchedOverloadInfo = {
                        overload: matchedOverload,
                        matchResults,
                        typeVarContext: effectiveTypeVarContext,
                        returnType: callResult.returnType,
                        argResults: callResult.argResults ?? [],
                    };
                    matchedOverloads.push(matchedOverloadInfo);

                    if (callResult.isArgumentAnyOrUnknown) {
                        possibleMatchResults.push(matchedOverloadInfo);
                    } else {
                        returnTypes.push(callResult.returnType);
                        isDefinitiveMatchFound = true;
                        break;
                    }
                }
            }

            // If we didn't find a definitive match that doesn't depend on
            // an Any or Unknown argument, fall back on the possible match.
            // If there were multiple possible matches, evaluate the type as
            // Unknown, but include the "possible types" to allow for completion
            // suggestions.
            if (!isDefinitiveMatchFound) {
                possibleMatchResults = filterOverloadMatchesForAnyArgs(possibleMatchResults);

                // Did the filtering produce a single result? If so, we're done.
                if (possibleMatchResults.length === 1) {
                    overloadsUsedForCall = [possibleMatchResults[0].overload];
                    returnTypes.push(possibleMatchResults[0].returnType);
                } else {
                    // Eliminate any return types that are subsumed by other return types.
                    let dedupedMatchResults: Type[] = [];

                    possibleMatchResults.forEach((result) => {
                        let isSubtypeSubsumed = false;

                        for (let dedupedIndex = 0; dedupedIndex < dedupedMatchResults.length; dedupedIndex++) {
                            if (assignType(dedupedMatchResults[dedupedIndex], result.returnType)) {
                                isSubtypeSubsumed = true;
                                break;
                            } else if (assignType(result.returnType, dedupedMatchResults[dedupedIndex])) {
                                dedupedMatchResults[dedupedIndex] = NeverType.createNever();
                                break;
                            }
                        }

                        if (!isSubtypeSubsumed) {
                            dedupedMatchResults.push(result.returnType);
                        }
                    });

                    dedupedMatchResults = dedupedMatchResults.filter((t) => !isNever(t));
                    const combinedTypes = combineTypes(dedupedMatchResults);

                    returnTypes.push(
                        dedupedMatchResults.length > 1 ? UnknownType.createPossibleType(combinedTypes) : combinedTypes
                    );
                }
            }

            if (!matchedOverload) {
                return { argumentErrors: true, isTypeIncomplete, overloadsUsedForCall };
            }
        }

        // We found a match for all of the expanded argument lists. Copy the
        // resulting type var context back into the caller's type var context.
        // Use the type var context from the last matched overload because it
        // includes the type var solutions for all earlier matched overloads.
        if (typeVarContext && isDefinitiveMatchFound) {
            typeVarContext.copyFromClone(matchedOverloads[matchedOverloads.length - 1].typeVarContext);
        }

        // And run through the first expanded argument list one more time to
        // populate the type cache.
        const finalTypeVarContext = typeVarContext ?? matchedOverloads[0].typeVarContext;
        finalTypeVarContext.unlock();
        finalTypeVarContext.addSolveForScope(getTypeVarScopeId(matchedOverloads[0].overload));
        const finalCallResult = validateFunctionArgumentTypesWithExpectedType(
            errorNode,
            matchedOverloads[0].matchResults,
            finalTypeVarContext,
            skipUnknownArgCheck,
            inferenceContext
        );

        if (finalCallResult.isTypeIncomplete) {
            isTypeIncomplete = true;
        }

        return {
            argumentErrors: finalCallResult.argumentErrors,
            isArgumentAnyOrUnknown: finalCallResult.isArgumentAnyOrUnknown,
            returnType: combineTypes(returnTypes),
            isTypeIncomplete,
            specializedInitSelfType: finalCallResult.specializedInitSelfType,
            overloadsUsedForCall,
        };
    }

    // This function determines whether multiple incompatible overloads match
    // due to an Any or Unknown argument type.
    function filterOverloadMatchesForAnyArgs(matches: MatchedOverloadInfo[]): MatchedOverloadInfo[] {
        if (matches.length < 2) {
            return matches;
        }

        // If all of the return types match, select the first one.
        if (
            areTypesSame(
                matches.map((match) => match.returnType),
                { treatAnySameAsUnknown: true }
            )
        ) {
            return [matches[0]];
        }

        const firstArgResults = matches[0].argResults;
        if (!firstArgResults) {
            return matches;
        }

        for (let i = 0; i < firstArgResults.length; i++) {
            // If the arg is Any or Unknown, see if the corresponding
            // parameter types differ in any way.
            if (isAnyOrUnknown(firstArgResults[i].argType)) {
                const paramTypes = matches.map((match) =>
                    i < match.matchResults.argParams.length
                        ? match.matchResults.argParams[i].paramType
                        : UnknownType.create()
                );
                if (!areTypesSame(paramTypes, { treatAnySameAsUnknown: true })) {
                    return matches;
                }
            }
        }

        return [matches[0]];
    }

    function getBestOverloadForArguments(
        errorNode: ExpressionNode,
        typeResult: TypeResult<OverloadedFunctionType>,
        argList: FunctionArgument[]
    ): FunctionType | undefined {
        let overloadIndex = 0;
        let matches: MatchArgsToParamsResult[] = [];

        // Create a list of potential overload matches based on arguments.
        OverloadedFunctionType.getOverloads(typeResult.type).forEach((overload) => {
            useSpeculativeMode(errorNode, () => {
                const matchResults = matchFunctionArgumentsToParameters(
                    errorNode,
                    argList,
                    { type: overload, isIncomplete: typeResult.isIncomplete },
                    overloadIndex
                );

                if (!matchResults.argumentErrors) {
                    matches.push(matchResults);
                }

                overloadIndex++;
            });
        });

        matches = sortOverloadsByBestMatch(matches);

        let winningOverloadIndex: number | undefined;

        matches.forEach((match, matchIndex) => {
            if (winningOverloadIndex === undefined) {
                useSpeculativeMode(errorNode, () => {
                    const callResult = validateFunctionArgumentTypes(
                        errorNode,
                        match,
                        new TypeVarContext(getTypeVarScopeId(match.overload)),
                        /* skipUnknownArgCheck */ true
                    );

                    if (callResult && !callResult.argumentErrors) {
                        winningOverloadIndex = matchIndex;
                    }
                });
            }
        });

        return winningOverloadIndex === undefined ? undefined : matches[winningOverloadIndex].overload;
    }

    // Sorts the list of overloads based first on "relevance" and second on order.
    function sortOverloadsByBestMatch(matches: MatchArgsToParamsResult[]) {
        return matches.sort((a, b) => {
            if (a.relevance !== b.relevance) {
                return b.relevance - a.relevance;
            }

            return a.overloadIndex - b.overloadIndex;
        });
    }

    function validateOverloadedFunctionArguments(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        typeResult: TypeResult<OverloadedFunctionType>,
        typeVarContext: TypeVarContext | undefined,
        skipUnknownArgCheck: boolean,
        inferenceContext: InferenceContext | undefined
    ): CallResult {
        let filteredMatchResults: MatchArgsToParamsResult[] = [];
        let contextFreeArgTypes: Type[] | undefined;

        // Start by evaluating the types of the arguments without any expected
        // type. Also, filter the list of overloads based on the number of
        // positional and named arguments that are present. We do all of this
        // speculatively because we don't want to record any types in the type
        // cache or record any diagnostics at this stage.
        useSpeculativeMode(errorNode, () => {
            let overloadIndex = 0;
            OverloadedFunctionType.getOverloads(typeResult.type).forEach((overload) => {
                // Consider only the functions that have the @overload decorator,
                // not the final function that omits the overload. This is the
                // intended behavior according to PEP 484.
                const matchResults = matchFunctionArgumentsToParameters(
                    errorNode,
                    argList,
                    { type: overload, isIncomplete: typeResult.isIncomplete },
                    overloadIndex
                );
                if (!matchResults.argumentErrors) {
                    filteredMatchResults.push(matchResults);
                }

                overloadIndex++;
            });
        });

        filteredMatchResults = sortOverloadsByBestMatch(filteredMatchResults);

        // If there are no possible arg/param matches among the overloads,
        // emit an error that includes the argument types.
        if (filteredMatchResults.length === 0) {
            // Skip the error message if we're in speculative mode because it's very
            // expensive, and we're going to suppress the diagnostic anyway.
            if (!isDiagnosticSuppressedForNode(errorNode)) {
                const functionName = typeResult.type.overloads[0].details.name || '<anonymous function>';
                const diagAddendum = new DiagnosticAddendum();
                const argTypes = argList.map((t) => printType(getTypeOfArgument(t).type));

                diagAddendum.addMessage(
                    Localizer.DiagnosticAddendum.argumentTypes().format({ types: argTypes.join(', ') })
                );
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.noOverload().format({ name: functionName }) + diagAddendum.getString(),
                    errorNode
                );
            }

            return { argumentErrors: true, isTypeIncomplete: false, overloadsUsedForCall: [] };
        }

        // Create a helper lambda that evaluates the overload that matches
        // the arg/param lists.
        const evaluateUsingLastMatchingOverload = (skipUnknownArgCheck: boolean) => {
            // Find the match with the largest overload index (i.e. the last overload
            // that was in the overload list).
            const lastMatch = filteredMatchResults.reduce((previous, current) => {
                return current.overloadIndex > previous.overloadIndex ? current : previous;
            });

            const effectiveTypeVarContext = typeVarContext ?? new TypeVarContext();
            effectiveTypeVarContext.addSolveForScope(getTypeVarScopeId(lastMatch.overload));
            if (lastMatch.overload.details.constructorTypeVarScopeId) {
                effectiveTypeVarContext.addSolveForScope(lastMatch.overload.details.constructorTypeVarScopeId);
            }
            effectiveTypeVarContext.unlock();

            return validateFunctionArgumentTypesWithExpectedType(
                errorNode,
                lastMatch,
                effectiveTypeVarContext,
                skipUnknownArgCheck,
                inferenceContext
            );
        };

        // If there is only one possible arg/param match among the overloads,
        // use the normal type matching mechanism because it is faster and
        // will provide a clearer error message.
        if (filteredMatchResults.length === 1) {
            return evaluateUsingLastMatchingOverload(/* skipUnknownArgCheck */ false);
        }

        let expandedArgTypes: (Type | undefined)[][] | undefined = [argList.map((arg) => undefined)];
        let isTypeIncomplete = !!typeResult.isIncomplete;

        while (true) {
            const callResult = validateOverloadsWithExpandedTypes(
                errorNode,
                expandedArgTypes,
                filteredMatchResults,
                typeVarContext,
                skipUnknownArgCheck,
                inferenceContext
            );

            if (callResult.isTypeIncomplete) {
                isTypeIncomplete = true;
            }

            if (!callResult.argumentErrors) {
                return callResult;
            }

            // We didn't find an overload match. Try to expand the next union
            // argument type into individual types and retry with the expanded types.
            if (!contextFreeArgTypes) {
                useSpeculativeMode(errorNode, () => {
                    // Evaluate the types of each argument expression without regard to
                    // the context. We'll use this to determine whether we need to do
                    // union expansion.
                    contextFreeArgTypes = argList.map((arg) => {
                        if (arg.typeResult) {
                            return arg.typeResult.type;
                        }

                        if (arg.valueExpression) {
                            const valueExpressionNode = arg.valueExpression;
                            return useSpeculativeMode(valueExpressionNode, () => {
                                return getTypeOfExpression(valueExpressionNode).type;
                            });
                        }

                        return AnyType.create();
                    });
                });
            }

            expandedArgTypes = expandArgumentUnionTypes(contextFreeArgTypes!, expandedArgTypes);

            // Check for combinatoric explosion and break out of loop.
            if (!expandedArgTypes || expandedArgTypes.length > maxOverloadUnionExpansionCount) {
                break;
            }
        }

        // We couldn't find any valid overloads. Skip the error message if we're
        // in speculative mode because it's very expensive, and we're going to
        // suppress the diagnostic anyway.
        if (!isDiagnosticSuppressedForNode(errorNode) && !isTypeIncomplete) {
            const result = evaluateUsingLastMatchingOverload(/* skipUnknownArgCheck */ true);

            // Replace the result with an unknown type since we don't know
            // what overload should have been used.
            result.returnType = UnknownType.create();
            return { ...result, argumentErrors: true };
        }

        return { argumentErrors: true, isTypeIncomplete: false, overloadsUsedForCall: [] };
    }

    // Replaces each item in the expandedArgTypes with n items where n is
    // the number of subtypes in a union. The contextFreeArgTypes parameter
    // represents the types of the arguments evaluated with no bidirectional
    // type inference (i.e. without the help of the corresponding parameter's
    // expected type). If the function returns undefined, that indicates that
    // all unions have been expanded, and no more expansion is possible.
    function expandArgumentUnionTypes(
        contextFreeArgTypes: Type[],
        expandedArgTypes: (Type | undefined)[][]
    ): (Type | undefined)[][] | undefined {
        // Find the rightmost already-expanded argument.
        let indexToExpand = contextFreeArgTypes.length - 1;
        while (indexToExpand >= 0 && !expandedArgTypes[0][indexToExpand]) {
            indexToExpand--;
        }

        // Move to the next candidate for expansion.
        indexToExpand++;

        if (indexToExpand >= contextFreeArgTypes.length) {
            return undefined;
        }

        let unionToExpand: Type | undefined;
        while (indexToExpand < contextFreeArgTypes.length) {
            // Is this a union type? If so, we can expand it.
            const argType = contextFreeArgTypes[indexToExpand];
            if (isUnion(argType)) {
                unionToExpand = makeTopLevelTypeVarsConcrete(argType);
                break;
            } else if (isTypeVar(argType) && argType.details.constraints.length > 1) {
                unionToExpand = makeTopLevelTypeVarsConcrete(argType);
                break;
            }
            indexToExpand++;
        }

        // We have nothing left to expand.
        if (!unionToExpand) {
            return undefined;
        }

        // Expand entry indexToExpand.
        const newExpandedArgTypes: (Type | undefined)[][] = [];

        expandedArgTypes.forEach((preExpandedTypes) => {
            doForEachSubtype(unionToExpand!, (subtype) => {
                const expandedTypes = [...preExpandedTypes];
                expandedTypes[indexToExpand] = subtype;
                newExpandedArgTypes.push(expandedTypes);
            });
        });

        return newExpandedArgTypes;
    }

    // Tries to match the arguments of a call to the constructor for a class.
    // If successful, it returns the resulting (specialized) object type that
    // is allocated by the constructor. If unsuccessful, it records diagnostic
    // information and returns undefined.
    function validateConstructorArguments(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        type: ClassType,
        skipUnknownArgCheck: boolean,
        inferenceContext: InferenceContext | undefined
    ): CallResult {
        let validatedTypes = false;
        let returnType: Type | undefined;
        let reportedErrors = false;
        let isTypeIncomplete = false;
        let usedMetaclassCallMethod = false;
        const overloadsUsedForCall: FunctionType[] = [];

        // Create a helper function that determines whether we should skip argument
        // validation for either __init__ or __new__. This is required for certain
        // synthesized constructor types, namely NamedTuples.
        const skipConstructorCheck = (type: Type) => {
            return isFunction(type) && FunctionType.isSkipConstructorCheck(type);
        };

        // Validate __init__
        // We validate __init__ before __new__ because the former typically has
        // more specific type annotations, and we want to evaluate the arguments
        // in the context of these types. The __new__ method often uses generic
        // vargs and kwargs.
        const initMethodType = getTypeOfObjectMember(
            errorNode,
            ClassType.cloneAsInstance(type),
            '__init__',
            { method: 'get' },
            /* diag */ undefined,
            MemberAccessFlags.SkipObjectBaseClass | MemberAccessFlags.SkipAttributeAccessOverride
        )?.type;

        if (initMethodType && !skipConstructorCheck(initMethodType)) {
            // If there is an expected type, analyze the constructor call
            // for each of the subtypes that comprise the expected type. If
            // one or more analyzes with no errors, use those results.
            if (inferenceContext) {
                const expectedCallResult = validateConstructorMethodWithExpectedType(
                    errorNode,
                    argList,
                    type,
                    skipUnknownArgCheck,
                    inferenceContext,
                    initMethodType
                );

                if (expectedCallResult && !expectedCallResult.argumentErrors) {
                    returnType = expectedCallResult.returnType;

                    if (expectedCallResult.isTypeIncomplete) {
                        isTypeIncomplete = true;
                    }

                    overloadsUsedForCall.push(...expectedCallResult.overloadsUsedForCall);
                }
            }

            if (!returnType) {
                const typeVarContext = type.typeArguments
                    ? buildTypeVarContextFromSpecializedClass(type, /* makeConcrete */ false)
                    : new TypeVarContext(getTypeVarScopeId(type));

                typeVarContext.addSolveForScope(getTypeVarScopeId(initMethodType));
                const callResult = validateCallArguments(
                    errorNode,
                    argList,
                    { type: initMethodType },
                    typeVarContext,
                    skipUnknownArgCheck
                );

                if (!callResult.argumentErrors) {
                    let adjustedClassType = type;
                    if (
                        callResult.specializedInitSelfType &&
                        isClassInstance(callResult.specializedInitSelfType) &&
                        ClassType.isSameGenericClass(callResult.specializedInitSelfType, type)
                    ) {
                        adjustedClassType = ClassType.cloneAsInstantiable(callResult.specializedInitSelfType);
                    }

                    returnType = applyExpectedTypeForConstructor(
                        adjustedClassType,
                        /* inferenceContext */ undefined,
                        typeVarContext
                    );

                    if (callResult.isTypeIncomplete) {
                        isTypeIncomplete = true;
                    }

                    overloadsUsedForCall.push(...callResult.overloadsUsedForCall);
                } else {
                    reportedErrors = true;
                }
            }

            validatedTypes = true;
            skipUnknownArgCheck = true;
        }

        // Validate __new__
        // Don't report errors for __new__ if __init__ already generated errors. They're
        // probably going to be entirely redundant anyway.
        if (!reportedErrors) {
            const metaclass = type.details.effectiveMetaclass;
            let constructorMethodInfo: ClassMemberLookup | undefined;

            // See if there's a custom `__call__` method on the metaclass. If so, we'll
            // use that rather than the `__new__` method on the class.
            if (metaclass && isInstantiableClass(metaclass) && !ClassType.isSameGenericClass(metaclass, type)) {
                constructorMethodInfo = getTypeOfClassMemberName(
                    errorNode,
                    metaclass,
                    /* isAccessedThroughObject */ true,
                    '__call__',
                    { method: 'get' },
                    /* diag */ undefined,
                    MemberAccessFlags.ConsiderMetaclassOnly |
                        MemberAccessFlags.SkipTypeBaseClass |
                        MemberAccessFlags.SkipAttributeAccessOverride,
                    type
                );

                if (constructorMethodInfo) {
                    usedMetaclassCallMethod = true;
                }
            }

            if (!constructorMethodInfo) {
                constructorMethodInfo = getTypeOfClassMemberName(
                    errorNode,
                    type,
                    /* isAccessedThroughObject */ false,
                    '__new__',
                    { method: 'get' },
                    /* diag */ undefined,
                    MemberAccessFlags.AccessClassMembersOnly |
                        MemberAccessFlags.SkipObjectBaseClass |
                        MemberAccessFlags.TreatConstructorAsClassMethod,
                    type
                );
            }

            if (constructorMethodInfo && !skipConstructorCheck(constructorMethodInfo.type)) {
                const constructorMethodType = constructorMethodInfo.type;
                let newReturnType: Type | undefined;

                // If there is an expected type that was not applied above when
                // handling the __init__ method, try to apply it with the __new__ method.
                if (inferenceContext && !returnType) {
                    const expectedCallResult = validateConstructorMethodWithExpectedType(
                        errorNode,
                        argList,
                        type,
                        skipUnknownArgCheck,
                        inferenceContext,
                        constructorMethodType
                    );

                    if (expectedCallResult && !expectedCallResult.argumentErrors) {
                        newReturnType = expectedCallResult.returnType;
                        returnType = newReturnType;

                        if (expectedCallResult.isTypeIncomplete) {
                            isTypeIncomplete = true;
                        }
                    }
                }

                const typeVarContext = new TypeVarContext(getTypeVarScopeId(type));

                if (type.typeAliasInfo) {
                    typeVarContext.addSolveForScope(type.typeAliasInfo.typeVarScopeId);
                }

                typeVarContext.addSolveForScope(getTypeVarScopeId(constructorMethodType));

                // Skip the unknown argument check if we've already checked for __init__.
                let callResult: CallResult;
                if (hasConstructorTransform(type)) {
                    // Use speculative mode if we're going to later apply
                    // a constructor transform. This allows us to use bidirectional
                    // type inference for arguments in the transform.
                    callResult = useSpeculativeMode(errorNode, () => {
                        return validateCallArguments(
                            errorNode,
                            argList,
                            constructorMethodInfo!,
                            typeVarContext,
                            skipUnknownArgCheck
                        );
                    });
                } else {
                    callResult = validateCallArguments(
                        errorNode,
                        argList,
                        constructorMethodInfo,
                        typeVarContext,
                        skipUnknownArgCheck
                    );
                }

                if (callResult.isTypeIncomplete) {
                    isTypeIncomplete = true;
                }

                if (callResult.argumentErrors) {
                    reportedErrors = true;
                } else if (!newReturnType) {
                    newReturnType = callResult.returnType;

                    // If the constructor returned an object whose type matches the class of
                    // the original type being constructed, use the return type in case it was
                    // specialized. If it doesn't match, we'll fall back on the assumption that
                    // the constructed type is an instance of the class type. We need to do this
                    // in cases where we're inferring the return type based on a call to
                    // super().__new__().
                    if (newReturnType) {
                        if (isClassInstance(newReturnType) && ClassType.isSameGenericClass(newReturnType, type)) {
                            // If the specialized return type derived from the __init__
                            // method is "better" than the return type provided by the
                            // __new__ method (where "better" means that the type arguments
                            // are all known), stick with the __init__ result.
                            if (
                                (!isPartlyUnknown(newReturnType) && !requiresSpecialization(newReturnType)) ||
                                returnType === undefined
                            ) {
                                // Special-case the 'tuple' type specialization to use
                                // the homogenous arbitrary-length form.
                                if (
                                    isClassInstance(newReturnType) &&
                                    ClassType.isTupleClass(newReturnType) &&
                                    !newReturnType.tupleTypeArguments &&
                                    newReturnType.typeArguments &&
                                    newReturnType.typeArguments.length === 1
                                ) {
                                    newReturnType = specializeTupleClass(newReturnType, [
                                        { type: newReturnType.typeArguments[0], isUnbounded: true },
                                    ]);
                                }

                                returnType = newReturnType;
                            }
                        } else if (!returnType && !isUnknown(newReturnType)) {
                            returnType = newReturnType;
                        }
                    }
                }

                if (!returnType) {
                    returnType = applyExpectedTypeForConstructor(type, inferenceContext, typeVarContext);
                } else if (isClassInstance(returnType) && isTupleClass(returnType) && !returnType.tupleTypeArguments) {
                    returnType = applyExpectedTypeForTupleConstructor(returnType, inferenceContext);
                }
                validatedTypes = true;
            }
        }

        // If we weren't able to validate the args, analyze the expressions
        // here to mark symbols as referenced and report expression-level errors.
        if (!validatedTypes) {
            argList.forEach((arg) => {
                if (arg.valueExpression && !speculativeTypeTracker.isSpeculative(arg.valueExpression)) {
                    getTypeOfExpression(arg.valueExpression);
                }
            });
        }

        if (!validatedTypes && argList.some((arg) => arg.argumentCategory === ArgumentCategory.Simple)) {
            // Suppress this error if the class was instantiated from a custom
            // metaclass because it's likely that it's a false positive. Also
            // suppress the error if the class's metaclass has a __call__ method.
            const isCustomMetaclass =
                !!type.details.effectiveMetaclass &&
                isInstantiableClass(type.details.effectiveMetaclass) &&
                !ClassType.isBuiltIn(type.details.effectiveMetaclass);

            if (!isCustomMetaclass && !usedMetaclassCallMethod) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.constructorNoArgs().format({ type: type.aliasName || type.details.name }),
                    errorNode
                );
            }
        }

        if (!returnType) {
            // There was no __init__ or __new__ method or we couldn't match the provided
            // arguments to them.
            if (!inferenceContext && type.typeArguments) {
                // If there was no expected type but the type was already specialized,
                // assume that we're constructing an instance of the specialized type.
                returnType = convertToInstance(type);
            } else {
                // Do our best to specialize the instantiated class based on the expected
                // type if provided.
                const typeVarContext = new TypeVarContext(getTypeVarScopeId(type));

                if (inferenceContext) {
                    populateTypeVarContextBasedOnExpectedType(
                        evaluatorInterface,
                        ClassType.cloneAsInstance(type),
                        inferenceContext.expectedType,
                        typeVarContext,
                        getTypeVarScopesForNode(errorNode)
                    );
                }

                returnType = applyExpectedTypeForConstructor(type, inferenceContext, typeVarContext);
            }
        }

        if (!reportedErrors) {
            const transformed = applyConstructorTransform(evaluatorInterface, errorNode, argList, type, {
                argumentErrors: reportedErrors,
                returnType,
                isTypeIncomplete,
            });

            returnType = transformed.returnType;

            if (transformed.isTypeIncomplete) {
                isTypeIncomplete = true;
            }

            if (transformed.argumentErrors) {
                reportedErrors = true;
            }
        }

        const result: CallResult = {
            argumentErrors: reportedErrors,
            returnType,
            isTypeIncomplete,
            overloadsUsedForCall,
        };

        return result;
    }

    // For a constructor call that targets a generic class and an "expected type"
    // (i.e. bidirectional inference), this function attempts to infer the correct
    // specialized return type for the constructor.
    function validateConstructorMethodWithExpectedType(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        type: ClassType,
        skipUnknownArgCheck: boolean,
        inferenceContext: InferenceContext,
        constructorMethodType: Type
    ): CallResult | undefined {
        let isTypeIncomplete = false;
        let argumentErrors = false;
        const overloadsUsedForCall: FunctionType[] = [];

        const returnType = mapSubtypes(inferenceContext.expectedType, (expectedSubType) => {
            expectedSubType = transformPossibleRecursiveTypeAlias(expectedSubType);
            const typeVarContext = new TypeVarContext(getTypeVarScopeId(type));
            if (
                populateTypeVarContextBasedOnExpectedType(
                    evaluatorInterface,
                    ClassType.cloneAsInstance(type),
                    expectedSubType,
                    typeVarContext,
                    getTypeVarScopesForNode(errorNode)
                )
            ) {
                let callResult: CallResult | undefined;
                useSpeculativeMode(errorNode, () => {
                    callResult = validateCallArguments(
                        errorNode,
                        argList,
                        { type: constructorMethodType },
                        typeVarContext.clone(),
                        skipUnknownArgCheck
                    );
                });

                if (!callResult!.argumentErrors) {
                    // Call validateCallArguments again, this time without speculative
                    // mode, so any errors are reported.
                    callResult = validateCallArguments(
                        errorNode,
                        argList,
                        { type: constructorMethodType },
                        typeVarContext,
                        skipUnknownArgCheck
                    );

                    if (callResult.isTypeIncomplete) {
                        isTypeIncomplete = true;
                    }

                    if (callResult.argumentErrors) {
                        argumentErrors = true;
                    }

                    overloadsUsedForCall.push(...callResult.overloadsUsedForCall);

                    return applyExpectedSubtypeForConstructor(type, expectedSubType, typeVarContext);
                }
            }

            return undefined;
        });

        if (isNever(returnType)) {
            return undefined;
        }

        return { returnType, isTypeIncomplete, argumentErrors, overloadsUsedForCall };
    }

    function applyExpectedSubtypeForConstructor(
        type: ClassType,
        expectedSubtype: Type,
        typeVarContext: TypeVarContext
    ): Type | undefined {
        const specializedType = applySolvedTypeVars(ClassType.cloneAsInstance(type), typeVarContext);

        if (!assignType(expectedSubtype, specializedType)) {
            return undefined;
        }

        // If the expected type is "Any", transform it to an Any.
        if (isAny(expectedSubtype)) {
            return expectedSubtype;
        }

        return specializedType;
    }

    // Handles the case where a constructor is a generic type and the type
    // arguments are not specified but can be provided by the expected type.
    function applyExpectedTypeForConstructor(
        type: ClassType,
        inferenceContext: InferenceContext | undefined,
        typeVarContext: TypeVarContext
    ): Type {
        let unsolvedTypeVarsAreUnknown = true;

        if (inferenceContext) {
            const specializedExpectedType = mapSubtypes(inferenceContext.expectedType, (expectedSubtype) => {
                return applyExpectedSubtypeForConstructor(type, expectedSubtype, typeVarContext);
            });

            if (!isNever(specializedExpectedType)) {
                return specializedExpectedType;
            }

            // If the expected type didn't provide TypeVar values, remaining
            // unsolved TypeVars should be considered Unknown unless they were
            // provided explicitly in the constructor call.
            if (type.typeArguments) {
                unsolvedTypeVarsAreUnknown = false;
            }
        }

        const specializedType = applySolvedTypeVars(type, typeVarContext, {
            unknownIfNotFound: unsolvedTypeVarsAreUnknown,
        }) as ClassType;
        return ClassType.cloneAsInstance(specializedType);
    }

    // Similar to applyExpectedTypeForConstructor, this function handles the
    // special case of the tuple class.
    function applyExpectedTypeForTupleConstructor(type: ClassType, inferenceContext: InferenceContext | undefined) {
        let specializedType = type;

        if (
            inferenceContext &&
            isClassInstance(inferenceContext.expectedType) &&
            isTupleClass(inferenceContext.expectedType) &&
            inferenceContext.expectedType.tupleTypeArguments
        ) {
            specializedType = specializeTupleClass(type, inferenceContext.expectedType.tupleTypeArguments);
        }

        return specializedType;
    }

    // Validates that the arguments can be assigned to the call's parameter
    // list, specializes the call based on arg types, and returns the
    // specialized type of the return value. If it detects an error along
    // the way, it emits a diagnostic and sets argumentErrors to true.
    function validateCallArguments(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        callTypeResult: TypeResult,
        typeVarContext?: TypeVarContext,
        skipUnknownArgCheck = false,
        inferenceContext?: InferenceContext,
        recursionCount = 0
    ): CallResult {
        let argumentErrors = false;
        let isTypeIncomplete = false;
        let specializedInitSelfType: Type | undefined;
        const overloadsUsedForCall: FunctionType[] = [];

        if (recursionCount > maxTypeRecursionCount) {
            return { returnType: UnknownType.create(), argumentErrors: true, overloadsUsedForCall };
        }
        recursionCount++;

        if (TypeBase.isSpecialForm(callTypeResult.type)) {
            const exprNode = errorNode.nodeType === ParseNodeType.Call ? errorNode.leftExpression : errorNode;
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeNotCallable().format({
                    expression: ParseTreeUtils.printExpression(exprNode),
                    type: printType(callTypeResult.type, { expandTypeAlias: true }),
                }),
                exprNode
            );
            return { returnType: UnknownType.create(), argumentErrors: true, overloadsUsedForCall };
        }

        let returnType = mapSubtypesExpandTypeVars(
            callTypeResult.type,
            /* conditionFilter */ undefined,
            (expandedSubtype, unexpandedSubtype) => {
                switch (expandedSubtype.category) {
                    case TypeCategory.Unknown:
                    case TypeCategory.Any: {
                        // Touch all of the args so they're marked accessed. Don't bother
                        // doing this if the call type is incomplete because this will need
                        // to be done again once it is complete.
                        if (!callTypeResult.isIncomplete) {
                            argList.forEach((arg) => {
                                if (arg.valueExpression && !speculativeTypeTracker.isSpeculative(arg.valueExpression)) {
                                    getTypeOfArgument(arg);
                                }
                            });
                        }

                        return expandedSubtype;
                    }

                    case TypeCategory.Function: {
                        if (TypeBase.isInstantiable(expandedSubtype)) {
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.callableNotInstantiable().format({
                                    type: printType(expandedSubtype),
                                }),
                                errorNode
                            );
                            argumentErrors = true;
                            return undefined;
                        }

                        // The stdlib collections/__init__.pyi stub file defines namedtuple
                        // as a function rather than a class, so we need to check for it here.
                        if (expandedSubtype.details.builtInName === 'namedtuple') {
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportUntypedNamedTuple,
                                DiagnosticRule.reportUntypedNamedTuple,
                                Localizer.Diagnostic.namedTupleNoTypes(),
                                errorNode
                            );
                            return createNamedTupleType(
                                evaluatorInterface,
                                errorNode,
                                argList,
                                /* includesTypes */ false
                            );
                        }

                        // Handle the NewType specially, replacing the normal return type.
                        if (expandedSubtype.details.builtInName === 'NewType') {
                            return createNewType(errorNode, argList);
                        }

                        let effectiveTypeVarContext = typeVarContext;
                        if (!effectiveTypeVarContext) {
                            // If a typeVarContext wasn't provided by the caller, allocate one here.
                            effectiveTypeVarContext = new TypeVarContext(getTypeVarScopeId(expandedSubtype));

                            // There are certain cases, such as with super().__new__(cls) calls where
                            // the call is a constructor but the proper TypeVar scope has been lost.
                            // We'll add a wildcard TypeVar scope here. This is a bit of a hack and
                            // we may need to revisit this in the future.
                            if (FunctionType.isConstructorMethod(expandedSubtype)) {
                                effectiveTypeVarContext.addSolveForScope(WildcardTypeVarScopeId);
                            }
                        }

                        const functionResult = validateFunctionArguments(
                            errorNode,
                            argList,
                            { type: expandedSubtype, isIncomplete: callTypeResult.isIncomplete },
                            effectiveTypeVarContext,
                            skipUnknownArgCheck,
                            inferenceContext
                        );

                        if (functionResult.isTypeIncomplete) {
                            isTypeIncomplete = true;
                        }

                        overloadsUsedForCall.push(...functionResult.overloadsUsedForCall);

                        if (functionResult.argumentErrors) {
                            argumentErrors = true;
                        } else {
                            specializedInitSelfType = functionResult.specializedInitSelfType;

                            // Call the function transform logic to handle special-cased functions.
                            const transformed = applyFunctionTransform(
                                evaluatorInterface,
                                errorNode,
                                argList,
                                expandedSubtype,
                                {
                                    argumentErrors: functionResult.argumentErrors,
                                    returnType: functionResult.returnType ?? UnknownType.create(isTypeIncomplete),
                                    isTypeIncomplete,
                                }
                            );

                            functionResult.returnType = transformed.returnType;
                            if (transformed.isTypeIncomplete) {
                                isTypeIncomplete = true;
                            }
                            if (transformed.argumentErrors) {
                                argumentErrors = true;
                            }
                        }

                        if (expandedSubtype.details.builtInName === '__import__') {
                            // For the special __import__ type, we'll override the return type to be "Any".
                            // This is required because we don't know what module was imported, and we don't
                            // want to fail type checks when accessing members of the resulting module type.
                            return AnyType.create();
                        }

                        return functionResult.returnType;
                    }

                    case TypeCategory.OverloadedFunction: {
                        // Handle the 'cast' call as a special case.
                        if (expandedSubtype.overloads[0].details.builtInName === 'cast' && argList.length === 2) {
                            // Verify that the cast is necessary.
                            const castToType = getTypeOfArgumentExpectingType(argList[0]).type;
                            const castFromType = getTypeOfArgument(argList[1]).type;
                            if (isInstantiableClass(castToType) && isClassInstance(castFromType)) {
                                if (
                                    isTypeSame(castToType, ClassType.cloneAsInstantiable(castFromType), {
                                        ignorePseudoGeneric: true,
                                    })
                                ) {
                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportUnnecessaryCast,
                                        DiagnosticRule.reportUnnecessaryCast,
                                        Localizer.Diagnostic.unnecessaryCast().format({
                                            type: printType(castFromType),
                                        }),
                                        errorNode
                                    );
                                }
                            }

                            return convertToInstance(castToType);
                        }

                        const functionResult = validateOverloadedFunctionArguments(
                            errorNode,
                            argList,
                            { type: expandedSubtype, isIncomplete: callTypeResult.isIncomplete },
                            typeVarContext,
                            skipUnknownArgCheck,
                            inferenceContext
                        );

                        overloadsUsedForCall.push(...functionResult.overloadsUsedForCall);

                        if (functionResult.isTypeIncomplete) {
                            isTypeIncomplete = true;
                        }

                        if (functionResult.argumentErrors) {
                            argumentErrors = true;
                        } else {
                            specializedInitSelfType = functionResult.specializedInitSelfType;

                            // Call the function transform logic to handle special-cased functions.
                            const transformed = applyFunctionTransform(
                                evaluatorInterface,
                                errorNode,
                                argList,
                                expandedSubtype,
                                {
                                    argumentErrors: functionResult.argumentErrors,
                                    returnType: functionResult.returnType ?? UnknownType.create(isTypeIncomplete),
                                    isTypeIncomplete,
                                }
                            );

                            functionResult.returnType = transformed.returnType;
                            if (transformed.isTypeIncomplete) {
                                isTypeIncomplete = true;
                            }
                            if (transformed.argumentErrors) {
                                argumentErrors = true;
                            }
                        }

                        return functionResult.returnType ?? UnknownType.create();
                    }

                    case TypeCategory.Class: {
                        if (TypeBase.isInstantiable(expandedSubtype)) {
                            if (expandedSubtype.literalValue !== undefined) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.literalNotCallable(),
                                    errorNode
                                );
                                argumentErrors = true;
                                return UnknownType.create();
                            }

                            if (ClassType.isBuiltIn(expandedSubtype)) {
                                const className = expandedSubtype.aliasName || expandedSubtype.details.name;

                                if (className === 'type') {
                                    // Validate the constructor arguments.
                                    validateConstructorArguments(
                                        errorNode,
                                        argList,
                                        expandedSubtype,
                                        skipUnknownArgCheck,
                                        inferenceContext
                                    );

                                    // Handle the 'type' call specially.
                                    if (argList.length === 1) {
                                        // The one-parameter form of "type" returns the class
                                        // for the specified object.
                                        const argType = getTypeOfArgument(argList[0]).type;
                                        return mapSubtypes(argType, (subtype) => {
                                            if (
                                                isClassInstance(subtype) ||
                                                (isTypeVar(subtype) && TypeBase.isInstance(subtype)) ||
                                                isNoneInstance(subtype)
                                            ) {
                                                return convertToInstantiable(stripLiteralValue(subtype));
                                            } else if (isFunction(subtype) && TypeBase.isInstance(subtype)) {
                                                return FunctionType.cloneAsInstantiable(subtype);
                                            }

                                            return AnyType.create();
                                        });
                                    } else if (argList.length >= 2) {
                                        // The two-parameter form of "type" returns a new class type
                                        // built from the specified base types.
                                        return createType(errorNode, argList) || AnyType.create();
                                    }

                                    // If the parameter to type() is not statically known,
                                    // fall back to Any.
                                    return AnyType.create();
                                }

                                if (className === 'TypeVar') {
                                    return createTypeVarType(errorNode, argList);
                                }

                                if (className === 'TypeVarTuple') {
                                    return createTypeVarTupleType(errorNode, argList);
                                }

                                if (className === 'ParamSpec') {
                                    return createParamSpecType(errorNode, argList);
                                }

                                if (className === 'NamedTuple') {
                                    return createNamedTupleType(
                                        evaluatorInterface,
                                        errorNode,
                                        argList,
                                        /* includesTypes */ true
                                    );
                                }

                                if (className === 'NewType') {
                                    return createNewType(errorNode, argList);
                                }

                                if (
                                    className === 'Protocol' ||
                                    className === 'Generic' ||
                                    className === 'Callable' ||
                                    className === 'Concatenate' ||
                                    className === 'Type'
                                ) {
                                    const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                                    addDiagnostic(
                                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.typeNotIntantiable().format({ type: className }),
                                        errorNode
                                    );
                                    return AnyType.create();
                                }

                                if (isClass(unexpandedSubtype) && isKnownEnumType(className)) {
                                    return createEnumType(errorNode, expandedSubtype, argList) ?? UnknownType.create();
                                }

                                if (className === 'TypedDict') {
                                    return createTypedDictType(evaluatorInterface, errorNode, expandedSubtype, argList);
                                }

                                if (className === 'auto' && argList.length === 0) {
                                    return getEnumAutoValueType(evaluatorInterface, errorNode);
                                }
                            }

                            if (ClassType.supportsAbstractMethods(expandedSubtype)) {
                                const abstractMethods = getAbstractMethods(expandedSubtype);
                                if (
                                    abstractMethods.length > 0 &&
                                    !expandedSubtype.includeSubclasses &&
                                    !isTypeVar(unexpandedSubtype)
                                ) {
                                    // If the class is abstract, it can't be instantiated.
                                    const diagAddendum = new DiagnosticAddendum();
                                    const errorsToDisplay = 2;

                                    abstractMethods.forEach((abstractMethod, index) => {
                                        if (index === errorsToDisplay) {
                                            diagAddendum.addMessage(
                                                Localizer.DiagnosticAddendum.memberIsAbstractMore().format({
                                                    count: abstractMethods.length - errorsToDisplay,
                                                })
                                            );
                                        } else if (index < errorsToDisplay) {
                                            if (isInstantiableClass(abstractMethod.classType)) {
                                                const className = abstractMethod.classType.details.name;
                                                diagAddendum.addMessage(
                                                    Localizer.DiagnosticAddendum.memberIsAbstract().format({
                                                        type: className,
                                                        name: abstractMethod.symbolName,
                                                    })
                                                );
                                            }
                                        }
                                    });

                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet
                                            .reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.instantiateAbstract().format({
                                            type: expandedSubtype.details.name,
                                        }) + diagAddendum.getString(),
                                        errorNode
                                    );
                                }
                            }

                            if (ClassType.isProtocolClass(expandedSubtype) && !expandedSubtype.includeSubclasses) {
                                // If the class is a protocol, it can't be instantiated.
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.instantiateProtocol().format({
                                        type: expandedSubtype.details.name,
                                    }),
                                    errorNode
                                );
                            }

                            // Assume this is a call to the constructor.
                            const constructorResult = validateConstructorArguments(
                                errorNode,
                                argList,
                                expandedSubtype,
                                skipUnknownArgCheck,
                                inferenceContext
                            );

                            overloadsUsedForCall.push(...constructorResult.overloadsUsedForCall);

                            if (constructorResult.argumentErrors) {
                                argumentErrors = true;
                            }

                            if (constructorResult.isTypeIncomplete) {
                                isTypeIncomplete = true;
                            }

                            let returnType = constructorResult.returnType;

                            // If the expandedSubtype originated from a TypeVar, convert
                            // the constructed type back to the TypeVar. For example, if
                            // we have `cls: Type[_T]` followed by `_T()`.
                            if (isTypeVar(unexpandedSubtype)) {
                                returnType = convertToInstance(unexpandedSubtype);
                            }

                            // If we instantiated a type, transform it into a class.
                            // This can happen if someone directly instantiates a metaclass
                            // deriving from type.
                            if (
                                returnType &&
                                isClassInstance(returnType) &&
                                returnType.details.mro.some(
                                    (baseClass) =>
                                        isInstantiableClass(baseClass) && ClassType.isBuiltIn(baseClass, 'type')
                                )
                            ) {
                                let newClassName = '__class_' + returnType.details.name;
                                if (argList.length === 3) {
                                    const firstArgType = getTypeOfArgument(argList[0]).type;
                                    if (
                                        isClassInstance(firstArgType) &&
                                        ClassType.isBuiltIn(firstArgType, 'str') &&
                                        typeof firstArgType.literalValue === 'string'
                                    ) {
                                        newClassName = firstArgType.literalValue;
                                    }
                                }

                                const newClassType = ClassType.createInstantiable(
                                    newClassName,
                                    '',
                                    '',
                                    AnalyzerNodeInfo.getFileInfo(errorNode).filePath,
                                    ClassTypeFlags.None,
                                    ParseTreeUtils.getTypeSourceId(errorNode),
                                    ClassType.cloneAsInstantiable(returnType),
                                    ClassType.cloneAsInstantiable(returnType)
                                );
                                newClassType.details.baseClasses.push(getBuiltInType(errorNode, 'object'));
                                newClassType.details.effectiveMetaclass = expandedSubtype;
                                computeMroLinearization(newClassType);
                                return newClassType;
                            }

                            return returnType;
                        } else {
                            const memberType = getTypeOfObjectMember(
                                errorNode,
                                expandedSubtype,
                                '__call__',
                                /* usage */ undefined,
                                /* diag */ undefined,
                                MemberAccessFlags.SkipAttributeAccessOverride
                            )?.type;

                            if (memberType) {
                                const functionResult = validateCallArguments(
                                    errorNode,
                                    argList,
                                    { type: memberType },
                                    typeVarContext,
                                    skipUnknownArgCheck,
                                    inferenceContext,
                                    recursionCount
                                );

                                overloadsUsedForCall.push(...functionResult.overloadsUsedForCall);

                                if (functionResult.argumentErrors) {
                                    argumentErrors = true;
                                }

                                if (
                                    isTypeVar(unexpandedSubtype) &&
                                    TypeBase.isInstantiable(unexpandedSubtype) &&
                                    isClass(expandedSubtype) &&
                                    ClassType.isBuiltIn(expandedSubtype, 'type')
                                ) {
                                    // Handle the case where a Type[T] is being called. We presume this
                                    // will instantiate an object of type T.
                                    return convertToInstance(unexpandedSubtype);
                                }

                                return functionResult.returnType ?? UnknownType.create();
                            }

                            if (!memberType || !isAnyOrUnknown(memberType)) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.objectNotCallable().format({
                                        type: printType(expandedSubtype),
                                    }),
                                    errorNode
                                );
                            }
                            return UnknownType.create();
                        }
                    }

                    case TypeCategory.None: {
                        if (TypeBase.isInstantiable(expandedSubtype)) {
                            if (noneType && isInstantiableClass(noneType)) {
                                const functionResult = validateCallArguments(
                                    errorNode,
                                    argList,
                                    { type: noneType },
                                    typeVarContext,
                                    skipUnknownArgCheck,
                                    inferenceContext,
                                    recursionCount
                                );

                                if (functionResult.isTypeIncomplete) {
                                    isTypeIncomplete = true;
                                }

                                if (functionResult.argumentErrors) {
                                    argumentErrors = true;
                                }
                            }

                            return NoneType.createInstance();
                        }

                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportOptionalCall,
                            DiagnosticRule.reportOptionalCall,
                            Localizer.Diagnostic.noneNotCallable(),
                            errorNode
                        );
                        return undefined;
                    }

                    // TypeVars should have been expanded in most cases,
                    // but we still need to handle the case of Type[T] where
                    // T is a constrained type that contains a union. We also
                    // need to handle recursive type aliases.
                    case TypeCategory.TypeVar: {
                        expandedSubtype = transformPossibleRecursiveTypeAlias(expandedSubtype);

                        const callResult = validateCallArguments(
                            errorNode,
                            argList,
                            { type: expandedSubtype },
                            typeVarContext,
                            skipUnknownArgCheck,
                            inferenceContext,
                            recursionCount
                        );

                        overloadsUsedForCall.push(...callResult.overloadsUsedForCall);

                        if (callResult.argumentErrors) {
                            argumentErrors = true;
                        }

                        return callResult.returnType ?? UnknownType.create();
                    }

                    case TypeCategory.Module: {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.moduleNotCallable(),
                            errorNode
                        );
                        return undefined;
                    }
                }

                return undefined;
            }
        );

        // If we ended up with a "Never" type because all code paths returned
        // undefined due to argument errors, transform the result into an Unknown
        // to avoid subsequent false positives.
        if (argumentErrors && isNever(returnType) && !returnType.isNoReturn) {
            returnType = UnknownType.create();
        }

        return {
            argumentErrors,
            returnType,
            isTypeIncomplete,
            specializedInitSelfType,
            overloadsUsedForCall,
        };
    }

    // Expands any unpacked tuples within an argument list.
    function expandArgList(argList: FunctionArgument[]): FunctionArgument[] {
        const expandedArgList: FunctionArgument[] = [];

        for (const arg of argList) {
            if (arg.argumentCategory === ArgumentCategory.UnpackedList) {
                const argType = getTypeOfArgument(arg).type;

                // If this is a tuple with specified element types, use those
                // specified types rather than using the more generic iterator
                // type which will be a union of all element types.
                const combinedArgType = combineSameSizedTuples(makeTopLevelTypeVarsConcrete(argType), tupleClassType);

                if (isClassInstance(combinedArgType) && isTupleClass(combinedArgType)) {
                    const tupleTypeArgs = combinedArgType.tupleTypeArguments ?? [];

                    if (tupleTypeArgs.length !== 1) {
                        for (const tupleTypeArg of tupleTypeArgs) {
                            if (tupleTypeArg.isUnbounded) {
                                expandedArgList.push({
                                    ...arg,
                                    argumentCategory: ArgumentCategory.UnpackedList,
                                    valueExpression: undefined,
                                    typeResult: {
                                        type: specializeTupleClass(combinedArgType, [tupleTypeArg]),
                                    },
                                });
                            } else {
                                expandedArgList.push({
                                    ...arg,
                                    argumentCategory: ArgumentCategory.Simple,
                                    valueExpression: undefined,
                                    typeResult: {
                                        type: tupleTypeArg.type,
                                    },
                                });
                            }
                        }
                        continue;
                    }
                }
            }

            expandedArgList.push(arg);
        }

        return expandedArgList;
    }

    // Matches the arguments passed to a function to the corresponding parameters in that
    // function. This matching is done based on positions and keywords. Type evaluation and
    // validation is left to the caller.
    // This logic is based on PEP 3102: https://www.python.org/dev/peps/pep-3102/
    function matchFunctionArgumentsToParameters(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        typeResult: TypeResult<FunctionType>,
        overloadIndex: number
    ): MatchArgsToParamsResult {
        const paramDetails = getParameterListDetails(typeResult.type);
        let argIndex = 0;
        let matchedUnpackedListOfUnknownLength = false;
        let reportedArgError = false;
        let isTypeIncomplete = !!typeResult.isIncomplete;
        let isVariadicTypeVarFullyMatched = false;

        // Expand any unpacked tuples in the arg list.
        argList = expandArgList(argList);

        // Build a map of parameters by name.
        const paramMap = new Map<string, ParamAssignmentInfo>();
        paramDetails.params.forEach((paramInfo) => {
            const param = paramInfo.param;
            if (param.name && param.category === ParameterCategory.Simple) {
                paramMap.set(param.name, {
                    argsNeeded: param.category === ParameterCategory.Simple && !param.hasDefault ? 1 : 0,
                    argsReceived: 0,
                    isPositionalOnly: paramInfo.source === ParameterSource.PositionOnly,
                });
            }
        });

        let positionalOnlyLimitIndex = paramDetails.positionOnlyParamCount;
        let positionParamLimitIndex = paramDetails.firstKeywordOnlyIndex ?? paramDetails.params.length;

        const varArgListParamIndex = paramDetails.argsIndex;
        const varArgDictParamIndex = paramDetails.kwargsIndex;

        // Is this an function that uses the *args and **kwargs
        // from a param spec? If so, we need to treat all positional parameters
        // prior to the *args as positional-only according to PEP 612.
        let paramSpecArgList: FunctionArgument[] | undefined;
        let paramSpecTarget: TypeVarType | undefined;
        let hasParamSpecArgsKwargs = false;

        if (varArgListParamIndex !== undefined && varArgDictParamIndex !== undefined) {
            assert(paramDetails.params[varArgListParamIndex], 'varArgListParamIndex params entry is undefined');
            const varArgListParam = paramDetails.params[varArgListParamIndex].param;
            assert(paramDetails.params[varArgDictParamIndex], 'varArgDictParamIndex params entry is undefined');
            const varArgDictParam = paramDetails.params[varArgDictParamIndex].param;

            if (
                isParamSpec(varArgListParam.type) &&
                varArgListParam.type.paramSpecAccess === 'args' &&
                isParamSpec(varArgDictParam.type) &&
                varArgDictParam.type.paramSpecAccess === 'kwargs' &&
                varArgListParam.type.details.name === varArgDictParam.type.details.name
            ) {
                hasParamSpecArgsKwargs = true;

                // Does this function define the param spec, or is it an inner
                // function nested within another function that defines the param
                // spec? We need to handle these two cases differently.
                if (
                    varArgListParam.type.scopeId === typeResult.type.details.typeVarScopeId ||
                    varArgListParam.type.scopeId === typeResult.type.details.constructorTypeVarScopeId
                ) {
                    paramSpecArgList = [];
                    paramSpecTarget = TypeVarType.cloneForParamSpecAccess(varArgListParam.type, /* access */ undefined);
                } else {
                    positionalOnlyLimitIndex = varArgListParamIndex;
                }
            }
        }

        // If there are keyword arguments present after a *args argument,
        // the keyword arguments may target one or more parameters that are positional.
        // In this case, we will limit the number of positional parameters so the
        // *args doesn't consume them all.
        if (argList.some((arg) => arg.argumentCategory === ArgumentCategory.UnpackedList)) {
            argList.forEach((arg) => {
                if (arg.name) {
                    const keywordParamIndex = paramDetails.params.findIndex(
                        (paramInfo) =>
                            paramInfo.param.name === arg.name!.value &&
                            paramInfo.param.category === ParameterCategory.Simple
                    );

                    // Is this a parameter that can be interpreted as either a keyword or a positional?
                    // If so, we'll treat it as a keyword parameter in this case because it's being
                    // targeted by a keyword argument.
                    if (keywordParamIndex >= 0 && keywordParamIndex >= positionalOnlyLimitIndex) {
                        if (positionParamLimitIndex < 0 || keywordParamIndex < positionParamLimitIndex) {
                            positionParamLimitIndex = keywordParamIndex;
                        }
                    }
                }
            });
        }

        // If we didn't see any special cases, then all parameters are positional.
        if (positionParamLimitIndex < 0) {
            positionParamLimitIndex = paramDetails.params.length;
        }

        // Determine how many positional args are being passed before
        // we see a keyword arg.
        let positionalArgCount = argList.findIndex(
            (arg) => arg.argumentCategory === ArgumentCategory.UnpackedDictionary || arg.name !== undefined
        );
        if (positionalArgCount < 0) {
            positionalArgCount = argList.length;
        }

        let validateArgTypeParams: ValidateArgTypeParams[] = [];

        let activeParam: FunctionParameter | undefined;
        function trySetActive(arg: FunctionArgument, param: FunctionParameter) {
            if (arg.active) {
                activeParam = param;
            }
        }

        const foundUnpackedListArg =
            argList.find((arg) => arg.argumentCategory === ArgumentCategory.UnpackedList) !== undefined;

        // Map the positional args to parameters.
        let paramIndex = 0;

        while (argIndex < positionalArgCount) {
            if (argIndex < positionalOnlyLimitIndex && argList[argIndex].name) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(argList[argIndex].name!);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.argPositional(),
                    argList[argIndex].name!
                );
                reportedArgError = true;
            }

            const remainingArgCount = positionalArgCount - argIndex;
            const remainingParamCount = positionParamLimitIndex - paramIndex - 1;

            if (paramIndex >= positionParamLimitIndex) {
                if (!typeResult.type.details.paramSpec) {
                    let tooManyPositionals = false;

                    if (foundUnpackedListArg && argList[argIndex].argumentCategory === ArgumentCategory.UnpackedList) {
                        // If this is an unpacked iterable, we will conservatively assume that it
                        // might have zero iterations unless we can tell from its type that it
                        // definitely has at least one iterable value.
                        const argType = getTypeOfArgument(argList[argIndex]).type;

                        if (
                            isClassInstance(argType) &&
                            isTupleClass(argType) &&
                            !isUnboundedTupleClass(argType) &&
                            argType.tupleTypeArguments !== undefined &&
                            argType.tupleTypeArguments.length > 0
                        ) {
                            tooManyPositionals = true;
                        }
                    } else {
                        tooManyPositionals = true;
                    }

                    if (tooManyPositionals) {
                        if (!isDiagnosticSuppressedForNode(errorNode)) {
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                positionParamLimitIndex === 1
                                    ? Localizer.Diagnostic.argPositionalExpectedOne()
                                    : Localizer.Diagnostic.argPositionalExpectedCount().format({
                                          expected: positionParamLimitIndex,
                                      }),
                                argList[argIndex].valueExpression ?? errorNode
                            );
                        }
                        reportedArgError = true;
                    }
                }
                break;
            }

            if (paramIndex >= paramDetails.params.length) {
                break;
            }

            assert(paramDetails.params[paramIndex], 'paramIndex params entry is undefined');
            const paramType = paramDetails.params[paramIndex].type;
            const paramName = paramDetails.params[paramIndex].param.name;

            const isParamVariadic =
                paramDetails.params[paramIndex].param.category === ParameterCategory.VarArgList &&
                isVariadicTypeVar(paramType);

            if (argList[argIndex].argumentCategory === ArgumentCategory.UnpackedList) {
                let isArgCompatibleWithVariadic = false;
                const argTypeResult = getTypeOfArgument(argList[argIndex]);
                let listElementType: Type | undefined;
                let advanceToNextArg = false;

                // Handle the case where *args is being passed to a function defined
                // with a ParamSpec and a Concatenate operator. PEP 612 indicates that
                // all positional parameters specified in the Concatenate must be
                // filled explicitly.
                if (typeResult.type.details.paramSpec && paramIndex < positionParamLimitIndex) {
                    if (!isDiagnosticSuppressedForNode(errorNode)) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            positionParamLimitIndex === 1
                                ? Localizer.Diagnostic.argPositionalExpectedOne()
                                : Localizer.Diagnostic.argPositionalExpectedCount().format({
                                      expected: positionParamLimitIndex,
                                  }),
                            argList[argIndex].valueExpression ?? errorNode
                        );
                    }
                    reportedArgError = true;
                }

                const argType = argTypeResult.type;

                if (isParamVariadic && isUnpackedVariadicTypeVar(argType)) {
                    // Allow an unpacked variadic type variable to satisfy an
                    // unpacked variadic type variable.
                    listElementType = argType;
                    isArgCompatibleWithVariadic = true;
                    advanceToNextArg = true;
                    isVariadicTypeVarFullyMatched = true;
                } else if (
                    isClassInstance(argType) &&
                    isTupleClass(argType) &&
                    argType.tupleTypeArguments &&
                    argType.tupleTypeArguments.length === 1 &&
                    isUnpackedVariadicTypeVar(argType.tupleTypeArguments[0].type)
                ) {
                    // Handle the case where an unpacked variadic type var has
                    // been packaged into a tuple.
                    listElementType = argType.tupleTypeArguments[0].type;
                    isArgCompatibleWithVariadic = true;
                    advanceToNextArg = true;
                    isVariadicTypeVarFullyMatched = true;
                } else if (isParamVariadic && isClassInstance(argType) && isTupleClass(argType)) {
                    // Handle the case where an unpacked tuple argument is
                    // matched to a TypeVarTuple parameter.
                    isArgCompatibleWithVariadic = true;
                    advanceToNextArg = true;

                    // Determine whether we should treat the variadic type as fully matched.
                    // This depends on how many args and unmatched parameters exist.
                    if (remainingArgCount < remainingParamCount) {
                        isVariadicTypeVarFullyMatched = true;
                    }

                    listElementType = ClassType.cloneForUnpacked(argType);
                } else if (isParamSpec(argType) && argType.paramSpecAccess === 'args') {
                    listElementType = undefined;
                } else {
                    listElementType =
                        getTypeOfIterator(
                            { type: argType, isIncomplete: argTypeResult.isIncomplete },
                            /* isAsync */ false,
                            argList[argIndex].valueExpression!
                        )?.type ?? UnknownType.create();

                    if (paramDetails.params[paramIndex].param.category !== ParameterCategory.VarArgList) {
                        matchedUnpackedListOfUnknownLength = true;
                    }
                }

                const funcArg: FunctionArgument | undefined = listElementType
                    ? {
                          argumentCategory: ArgumentCategory.Simple,
                          typeResult: { type: listElementType, isIncomplete: argTypeResult.isIncomplete },
                      }
                    : undefined;
                if (funcArg && argTypeResult.isIncomplete) {
                    isTypeIncomplete = true;
                }

                // It's not allowed to use unpacked arguments with a variadic *args
                // parameter unless the argument is a variadic arg as well.
                if (isParamVariadic && !isArgCompatibleWithVariadic) {
                    if (!isDiagnosticSuppressedForNode(errorNode)) {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.unpackedArgWithVariadicParam(),
                            argList[argIndex].valueExpression || errorNode
                        );
                    }
                    reportedArgError = true;
                } else {
                    if (paramSpecArgList) {
                        paramSpecArgList.push(argList[argIndex]);
                    }

                    if (funcArg) {
                        validateArgTypeParams.push({
                            paramCategory: paramDetails.params[paramIndex].param.category,
                            paramType,
                            requiresTypeVarMatching: requiresSpecialization(paramType),
                            argument: funcArg,
                            errorNode: argList[argIndex].valueExpression ?? errorNode,
                            paramName,
                            isParamNameSynthesized: paramDetails.params[paramIndex].param.isNameSynthesized,
                            mapsToVarArgList: isParamVariadic && remainingArgCount > remainingParamCount,
                        });
                    }
                }

                trySetActive(argList[argIndex], paramDetails.params[paramIndex].param);

                // Note that the parameter has received an argument.
                if (
                    paramName &&
                    paramDetails.params[paramIndex].param.category === ParameterCategory.Simple &&
                    paramMap.has(paramName)
                ) {
                    paramMap.get(paramName)!.argsReceived++;
                }

                if (
                    advanceToNextArg ||
                    paramDetails.params[paramIndex].param.category === ParameterCategory.VarArgList
                ) {
                    argIndex++;
                }

                if (
                    isVariadicTypeVarFullyMatched ||
                    paramDetails.params[paramIndex].param.category !== ParameterCategory.VarArgList
                ) {
                    paramIndex++;
                }
            } else if (paramDetails.params[paramIndex].param.category === ParameterCategory.VarArgList) {
                trySetActive(argList[argIndex], paramDetails.params[paramIndex].param);

                if (paramSpecArgList) {
                    paramSpecArgList.push(argList[argIndex]);
                    argIndex++;
                } else {
                    let paramCategory = paramDetails.params[paramIndex].param.category;
                    let effectiveParamType = paramType;
                    const paramName = paramDetails.params[paramIndex].param.name;

                    if (
                        isUnpackedClass(paramType) &&
                        paramType.tupleTypeArguments &&
                        paramType.tupleTypeArguments.length > 0
                    ) {
                        effectiveParamType = paramType.tupleTypeArguments[0].type;
                    }

                    paramCategory = isVariadicTypeVar(effectiveParamType)
                        ? ParameterCategory.VarArgList
                        : ParameterCategory.Simple;

                    if (remainingArgCount <= remainingParamCount) {
                        if (remainingArgCount < remainingParamCount) {
                            if (!isDiagnosticSuppressedForNode(errorNode)) {
                                // Have we run out of arguments and still have parameters left to fill?
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    remainingArgCount === 1
                                        ? Localizer.Diagnostic.argMorePositionalExpectedOne()
                                        : Localizer.Diagnostic.argMorePositionalExpectedCount().format({
                                              expected: remainingArgCount,
                                          }),
                                    argList[argIndex].valueExpression || errorNode
                                );
                            }
                            reportedArgError = true;
                        }

                        paramIndex++;
                    } else {
                        validateArgTypeParams.push({
                            paramCategory,
                            paramType: effectiveParamType,
                            requiresTypeVarMatching: requiresSpecialization(paramType),
                            argument: argList[argIndex],
                            errorNode: argList[argIndex].valueExpression || errorNode,
                            paramName,
                            isParamNameSynthesized: paramDetails.params[paramIndex].param.isNameSynthesized,
                            mapsToVarArgList: true,
                        });

                        argIndex++;
                    }
                }
            } else {
                const paramName = paramDetails.params[paramIndex].param.name;
                validateArgTypeParams.push({
                    paramCategory: paramDetails.params[paramIndex].param.category,
                    paramType,
                    requiresTypeVarMatching: requiresSpecialization(paramType),
                    argument: argList[argIndex],
                    errorNode: argList[argIndex].valueExpression || errorNode,
                    paramName,
                    isParamNameSynthesized: paramDetails.params[paramIndex].param.isNameSynthesized,
                });
                trySetActive(argList[argIndex], paramDetails.params[paramIndex].param);

                // Note that the parameter has received an argument.
                if (paramName && paramMap.has(paramName)) {
                    paramMap.get(paramName)!.argsReceived++;
                }

                argIndex++;
                paramIndex++;
            }
        }

        // If there weren't enough positional arguments to populate all of the
        // positional-only parameters and the next positional-only parameter is
        // an unbounded tuple, skip past it.
        let skippedArgsParam = false;
        if (
            positionalOnlyLimitIndex >= 0 &&
            paramIndex < positionalOnlyLimitIndex &&
            paramDetails.params[paramIndex].param.category === ParameterCategory.VarArgList &&
            !isParamSpec(paramDetails.params[paramIndex].param.type)
        ) {
            paramIndex++;
            skippedArgsParam = true;
        }

        // Check if there weren't enough positional arguments to populate all of
        // the positional-only parameters.
        if (
            positionalOnlyLimitIndex >= 0 &&
            paramIndex < positionalOnlyLimitIndex &&
            (!foundUnpackedListArg || hasParamSpecArgsKwargs)
        ) {
            const firstParamWithDefault = paramDetails.params.findIndex((paramInfo) => paramInfo.param.hasDefault);
            const positionOnlyWithoutDefaultsCount =
                firstParamWithDefault >= 0 && firstParamWithDefault < positionalOnlyLimitIndex
                    ? firstParamWithDefault
                    : positionalOnlyLimitIndex;

            // Calculate the number of remaining positional parameters to report.
            let argsRemainingCount = positionOnlyWithoutDefaultsCount - positionalArgCount;
            if (skippedArgsParam) {
                // If we skipped an args parameter above, reduce the count by one
                // because it's permitted to pass zero arguments to *args.
                argsRemainingCount--;
            }

            const firstArgsParam = paramDetails.params.findIndex(
                (paramInfo) =>
                    paramInfo.param.category === ParameterCategory.VarArgList && !isParamSpec(paramInfo.param.type)
            );
            if (firstArgsParam >= paramIndex && firstArgsParam < positionalOnlyLimitIndex) {
                // If there is another args parameter beyond the current param index,
                // reduce the count by one because it's permitted to pass zero arguments
                // to *args.
                argsRemainingCount--;
            }

            if (argsRemainingCount > 0) {
                if (!isDiagnosticSuppressedForNode(errorNode)) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        argsRemainingCount === 1
                            ? Localizer.Diagnostic.argMorePositionalExpectedOne()
                            : Localizer.Diagnostic.argMorePositionalExpectedCount().format({
                                  expected: argsRemainingCount,
                              }),
                        argList.length > positionalArgCount
                            ? argList[positionalArgCount].valueExpression || errorNode
                            : errorNode
                    );
                }
                reportedArgError = true;
            }
        }

        if (!reportedArgError) {
            let unpackedDictionaryArgType: Type | undefined;

            // Now consume any keyword arguments.
            while (argIndex < argList.length) {
                if (argList[argIndex].argumentCategory === ArgumentCategory.UnpackedDictionary) {
                    // Verify that the type used in this expression is a SupportsKeysAndGetItem[str, T].
                    const argType = getTypeOfArgument(argList[argIndex]).type;
                    if (isAnyOrUnknown(argType)) {
                        unpackedDictionaryArgType = argType;
                    } else if (isClassInstance(argType) && ClassType.isTypedDictClass(argType)) {
                        // Handle the special case where it is a TypedDict and we know which
                        // keys are present.
                        const typedDictEntries = getTypedDictMembersForClass(evaluatorInterface, argType);
                        const diag = new DiagnosticAddendum();

                        typedDictEntries.forEach((entry, name) => {
                            const paramEntry = paramMap.get(name);
                            if (paramEntry && !paramEntry.isPositionalOnly) {
                                if (paramEntry.argsReceived > 0) {
                                    diag.addMessage(Localizer.Diagnostic.paramAlreadyAssigned().format({ name }));
                                } else {
                                    paramEntry.argsReceived++;

                                    const paramInfoIndex = paramDetails.params.findIndex(
                                        (paramInfo) => paramInfo.param.name === name
                                    );
                                    assert(paramInfoIndex >= 0);
                                    const paramType = paramDetails.params[paramInfoIndex].type;

                                    validateArgTypeParams.push({
                                        paramCategory: ParameterCategory.Simple,
                                        paramType,
                                        requiresTypeVarMatching: requiresSpecialization(paramType),
                                        argument: {
                                            argumentCategory: ArgumentCategory.Simple,
                                            typeResult: { type: entry.valueType },
                                        },
                                        errorNode: argList[argIndex].valueExpression || errorNode,
                                        paramName: name,
                                    });
                                }
                            } else if (paramDetails.kwargsIndex !== undefined) {
                                const paramType = paramDetails.params[paramDetails.kwargsIndex].type;
                                validateArgTypeParams.push({
                                    paramCategory: ParameterCategory.VarArgDictionary,
                                    paramType,
                                    requiresTypeVarMatching: requiresSpecialization(paramType),
                                    argument: {
                                        argumentCategory: ArgumentCategory.Simple,
                                        typeResult: { type: entry.valueType },
                                    },
                                    errorNode: argList[argIndex].valueExpression || errorNode,
                                    paramName: name,
                                });

                                // Remember that this parameter has already received a value.
                                paramMap.set(name, {
                                    argsNeeded: 1,
                                    argsReceived: 1,
                                    isPositionalOnly: false,
                                });
                            } else {
                                // If the function doesn't have a **kwargs parameter, we need to emit an error.
                                // However, it's possible that there was a **kwargs but it was eliminated by
                                // getParameterListDetails because it was associated with an unpacked TypedDict.
                                // In this case, we can skip the error.
                                if (!paramDetails.hasUnpackedTypedDict) {
                                    diag.addMessage(Localizer.Diagnostic.paramNameMissing().format({ name }));
                                }
                            }
                        });

                        if (!diag.isEmpty()) {
                            if (!isDiagnosticSuppressedForNode(errorNode)) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.unpackedTypedDictArgument() + diag.getString(),
                                    argList[argIndex].valueExpression || errorNode
                                );
                            }
                            reportedArgError = true;
                        }
                    } else if (isParamSpec(argType) && argType.paramSpecAccess === 'kwargs') {
                        unpackedDictionaryArgType = AnyType.create();
                        if (typeResult.type.details.paramSpec) {
                            validateArgTypeParams.push({
                                paramCategory: ParameterCategory.VarArgDictionary,
                                paramType: typeResult.type.details.paramSpec,
                                requiresTypeVarMatching: false,
                                argument: argList[argIndex],
                                errorNode: argList[argIndex].valueExpression || errorNode,
                            });
                        }
                    } else {
                        let mappingType = getTypeshedType(errorNode, 'SupportsKeysAndGetItem');
                        if (!mappingType) {
                            mappingType = getTypingType(errorNode, 'Mapping');
                        }
                        const strObjType = getBuiltInObject(errorNode, 'str');

                        if (
                            mappingType &&
                            isInstantiableClass(mappingType) &&
                            strObjType &&
                            isClassInstance(strObjType)
                        ) {
                            const mappingTypeVarContext = new TypeVarContext(getTypeVarScopeId(mappingType));
                            let isValidMappingType = false;

                            // If this was a TypeVar (e.g. for pseudo-generic classes),
                            // don't emit this error.
                            if (isTypeVar(argType)) {
                                isValidMappingType = true;
                            } else if (
                                assignType(
                                    ClassType.cloneAsInstance(mappingType),
                                    argType,
                                    /* diag */ undefined,
                                    mappingTypeVarContext
                                )
                            ) {
                                const specializedMapping = applySolvedTypeVars(
                                    mappingType,
                                    mappingTypeVarContext
                                ) as ClassType;
                                const typeArgs = specializedMapping.typeArguments;
                                if (typeArgs && typeArgs.length >= 2) {
                                    if (assignType(strObjType, typeArgs[0])) {
                                        isValidMappingType = true;
                                    }
                                    unpackedDictionaryArgType = typeArgs[1];
                                } else {
                                    isValidMappingType = true;
                                    unpackedDictionaryArgType = UnknownType.create();
                                }
                            }

                            if (!isValidMappingType) {
                                if (!isDiagnosticSuppressedForNode(errorNode)) {
                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet
                                            .reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.unpackedDictArgumentNotMapping(),
                                        argList[argIndex].valueExpression || errorNode
                                    );
                                }
                                reportedArgError = true;
                            }
                        }
                    }

                    if (paramSpecArgList) {
                        paramSpecArgList.push(argList[argIndex]);
                    }
                } else {
                    // Protect against the case where a non-keyword argument appears after
                    // a keyword argument. This will have already been reported as a parse
                    // error, but we need to protect against it here.
                    const paramName = argList[argIndex].name;
                    if (paramName) {
                        const paramNameValue = paramName.value;
                        const paramEntry = paramMap.get(paramNameValue);
                        if (paramEntry && !paramEntry.isPositionalOnly) {
                            if (paramEntry.argsReceived > 0) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(paramName).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.paramAlreadyAssigned().format({ name: paramNameValue }),
                                    paramName
                                );
                                reportedArgError = true;
                            } else {
                                paramEntry.argsReceived++;

                                const paramInfoIndex = paramDetails.params.findIndex(
                                    (paramInfo) => paramInfo.param.name === paramNameValue
                                );
                                assert(paramInfoIndex >= 0);
                                const paramType = paramDetails.params[paramInfoIndex].type;

                                validateArgTypeParams.push({
                                    paramCategory: ParameterCategory.Simple,
                                    paramType,
                                    requiresTypeVarMatching: requiresSpecialization(paramType),
                                    argument: argList[argIndex],
                                    errorNode: argList[argIndex].valueExpression ?? errorNode,
                                    paramName: paramNameValue,
                                });
                                trySetActive(argList[argIndex], paramDetails.params[paramInfoIndex].param);
                            }
                        } else if (paramDetails.kwargsIndex !== undefined) {
                            if (paramSpecArgList) {
                                paramSpecArgList.push(argList[argIndex]);
                            } else {
                                const paramType = paramDetails.params[paramDetails.kwargsIndex].type;
                                validateArgTypeParams.push({
                                    paramCategory: ParameterCategory.VarArgDictionary,
                                    paramType,
                                    requiresTypeVarMatching: requiresSpecialization(paramType),
                                    argument: argList[argIndex],
                                    errorNode: argList[argIndex].valueExpression ?? errorNode,
                                    paramName: paramNameValue,
                                });

                                // Remember that this parameter has already received a value.
                                paramMap.set(paramNameValue, {
                                    argsNeeded: 1,
                                    argsReceived: 1,
                                    isPositionalOnly: false,
                                });
                            }
                            assert(
                                paramDetails.params[paramDetails.kwargsIndex],
                                'paramDetails.kwargsIndex params entry is undefined'
                            );
                            trySetActive(argList[argIndex], paramDetails.params[paramDetails.kwargsIndex].param);
                        } else {
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(paramName).diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.paramNameMissing().format({ name: paramName.value }),
                                paramName
                            );
                            reportedArgError = true;
                        }
                    } else if (argList[argIndex].argumentCategory === ArgumentCategory.Simple) {
                        if (!isDiagnosticSuppressedForNode(errorNode)) {
                            const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                positionParamLimitIndex === 1
                                    ? Localizer.Diagnostic.argPositionalExpectedOne()
                                    : Localizer.Diagnostic.argPositionalExpectedCount().format({
                                          expected: positionParamLimitIndex,
                                      }),
                                argList[argIndex].valueExpression || errorNode
                            );
                        }
                        reportedArgError = true;
                    } else if (argList[argIndex].argumentCategory === ArgumentCategory.UnpackedList) {
                        // Handle the case where a *args: P.args is passed as an argument to
                        // a function that accepts a ParamSpec.
                        if (typeResult.type.details.paramSpec) {
                            const argTypeResult = getTypeOfArgument(argList[argIndex]);
                            const argType = argTypeResult.type;

                            if (argTypeResult.isIncomplete) {
                                isTypeIncomplete = true;
                            }

                            if (isParamSpec(argType) && argType.paramSpecAccess === 'args') {
                                validateArgTypeParams.push({
                                    paramCategory: ParameterCategory.VarArgList,
                                    paramType: typeResult.type.details.paramSpec,
                                    requiresTypeVarMatching: false,
                                    argument: argList[argIndex],
                                    errorNode: argList[argIndex].valueExpression ?? errorNode,
                                });
                            }
                        }
                    }
                }

                argIndex++;
            }

            // If there are keyword-only parameters that haven't been matched but we
            // have an unpacked dictionary arg, assume that it applies to them.
            if (unpackedDictionaryArgType && (!foundUnpackedListArg || paramDetails.argsIndex !== undefined)) {
                // Don't consider any position-only parameters, since they cannot be matched to
                // **kwargs arguments. Consider parameters that are either positional or keyword
                // if there is no *args argument.
                paramDetails.params.forEach((paramInfo, paramIndex) => {
                    const param = paramInfo.param;
                    if (
                        paramIndex >= paramDetails.firstPositionOrKeywordIndex &&
                        param.category === ParameterCategory.Simple &&
                        param.name &&
                        !param.hasDefault &&
                        paramMap.has(param.name) &&
                        paramMap.get(param.name)!.argsReceived === 0
                    ) {
                        const paramType = paramDetails.params[paramIndex].type;
                        validateArgTypeParams.push({
                            paramCategory: ParameterCategory.Simple,
                            paramType,
                            requiresTypeVarMatching: requiresSpecialization(paramType),
                            argument: {
                                argumentCategory: ArgumentCategory.Simple,
                                typeResult: { type: unpackedDictionaryArgType! },
                            },
                            errorNode:
                                argList.find((arg) => arg.argumentCategory === ArgumentCategory.UnpackedDictionary)
                                    ?.valueExpression ?? errorNode,
                            paramName: param.name,
                            isParamNameSynthesized: param.isNameSynthesized,
                        });

                        paramMap.get(param.name)!.argsReceived = 1;
                    }
                });
            }

            // Determine whether there are any parameters that require arguments
            // but have not yet received them. If we received a dictionary argument
            // (i.e. an arg starting with a "**"), we will assume that all parameters
            // are matched.
            if (!unpackedDictionaryArgType && !FunctionType.isDefaultParameterCheckDisabled(typeResult.type)) {
                const unassignedParams = [...paramMap.keys()].filter((name) => {
                    const entry = paramMap.get(name)!;
                    return !entry || entry.argsReceived < entry.argsNeeded;
                });

                if (unassignedParams.length > 0) {
                    if (!isDiagnosticSuppressedForNode(errorNode)) {
                        const missingParamNames = unassignedParams.map((p) => `"${p}"`).join(', ');
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            unassignedParams.length === 1
                                ? Localizer.Diagnostic.argMissingForParam().format({ name: missingParamNames })
                                : Localizer.Diagnostic.argMissingForParams().format({ names: missingParamNames }),
                            errorNode
                        );
                    }
                    reportedArgError = true;
                }

                // Add any implicit (default) arguments that are needed for resolving
                // generic types. For example, if the function is defined as
                // def foo(v1: _T = 'default')
                // and _T is a TypeVar, we need to match the TypeVar to the default
                // value's type if it's not provided by the caller.
                paramDetails.params.forEach((paramInfo) => {
                    const param = paramInfo.param;
                    if (param.category === ParameterCategory.Simple && param.name) {
                        const entry = paramMap.get(param.name)!;
                        if (entry.argsNeeded === 0 && entry.argsReceived === 0) {
                            const defaultArgType = paramInfo.defaultArgType ?? param.defaultType;

                            if (
                                defaultArgType &&
                                !isEllipsisType(defaultArgType) &&
                                requiresSpecialization(param.type)
                            ) {
                                validateArgTypeParams.push({
                                    paramCategory: param.category,
                                    paramType: paramInfo.type,
                                    requiresTypeVarMatching: true,
                                    argument: {
                                        argumentCategory: ArgumentCategory.Simple,
                                        typeResult: { type: defaultArgType },
                                    },
                                    errorNode,
                                    paramName: param.name,
                                    isParamNameSynthesized: param.isNameSynthesized,
                                });
                            }
                        }
                    }
                });
            }
        }

        // If we're in speculative mode and an arg/param mismatch has already been reported, don't
        // bother doing the extra work here. This occurs frequently when attempting to find the
        // correct overload.
        if (!reportedArgError || !speculativeTypeTracker.isSpeculative(undefined)) {
            // If there are arguments that map to a variadic *args parameter that hasn't
            // already been matched, see if the type of that *args parameter is a variadic
            // type variable. If so, we'll preprocess those arguments and combine them
            // into a tuple.
            if (
                paramDetails.argsIndex !== undefined &&
                paramDetails.argsIndex >= 0 &&
                paramDetails.params[paramDetails.argsIndex].param.hasDeclaredType &&
                !isVariadicTypeVarFullyMatched
            ) {
                const paramType = paramDetails.params[paramDetails.argsIndex].type;
                const variadicArgs = validateArgTypeParams.filter((argParam) => argParam.mapsToVarArgList);

                if (isVariadicTypeVar(paramType) && !paramType.isVariadicInUnion) {
                    if (tupleClassType && isInstantiableClass(tupleClassType)) {
                        const tupleTypeArgs: TupleTypeArgument[] = variadicArgs.map((argParam) => {
                            const argType = getTypeOfArgument(argParam.argument).type;
                            const containsVariadicTypeVar =
                                isUnpackedVariadicTypeVar(argType) ||
                                (isClassInstance(argType) &&
                                    isTupleClass(argType) &&
                                    argType.tupleTypeArguments &&
                                    argType.tupleTypeArguments.length === 1 &&
                                    isUnpackedVariadicTypeVar(argType.tupleTypeArguments[0].type));

                            if (
                                containsVariadicTypeVar &&
                                argParam.argument.argumentCategory !== ArgumentCategory.UnpackedList &&
                                !argParam.mapsToVarArgList
                            ) {
                                if (!isDiagnosticSuppressedForNode(errorNode)) {
                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet
                                            .reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.typeVarTupleMustBeUnpacked(),
                                        argParam.argument.valueExpression ?? errorNode
                                    );
                                }
                                reportedArgError = true;
                            }

                            return {
                                type: stripLiteralValue(argType),
                                isUnbounded: argParam.argument.argumentCategory === ArgumentCategory.UnpackedList,
                            };
                        });

                        let specializedTuple: Type;
                        if (
                            tupleTypeArgs.length === 1 &&
                            !tupleTypeArgs[0].isUnbounded &&
                            (isUnpackedClass(tupleTypeArgs[0].type) || isVariadicTypeVar(tupleTypeArgs[0].type))
                        ) {
                            // If there is a single unpacked tuple or unpacked variadic type variable
                            // (including an unpacked TypeVarTuple union) within this tuple,
                            // simplify the type.
                            specializedTuple = tupleTypeArgs[0].type;
                        } else {
                            specializedTuple = ClassType.cloneAsInstance(
                                specializeTupleClass(
                                    tupleClassType,
                                    tupleTypeArgs,
                                    /* isTypeArgumentExplicit */ true,
                                    /* isUnpackedTuple */ true
                                )
                            );
                        }

                        const combinedArg: ValidateArgTypeParams = {
                            paramCategory: ParameterCategory.VarArgList,
                            paramType,
                            requiresTypeVarMatching: true,
                            argument: {
                                argumentCategory: ArgumentCategory.Simple,
                                typeResult: { type: specializedTuple },
                            },
                            errorNode,
                            paramName: paramDetails.params[paramDetails.argsIndex].param.name,
                            isParamNameSynthesized: paramDetails.params[paramDetails.argsIndex].param.isNameSynthesized,
                            mapsToVarArgList: true,
                        };

                        validateArgTypeParams = [
                            ...validateArgTypeParams.filter((argParam) => !argParam.mapsToVarArgList),
                            combinedArg,
                        ];
                    }
                }
            }
        }

        let relevance = 0;
        if (matchedUnpackedListOfUnknownLength) {
            // Lower the relevance if we made assumptions about the length
            // of an unpacked argument. This will favor overloads that
            // associate this case with a *args parameter.
            relevance--;
        }

        // Special-case the builtin isinstance and issubclass functions.
        if (
            ['isinstance', 'issubclass'].some((name) => name === typeResult.type.details.builtInName) &&
            validateArgTypeParams.length === 2
        ) {
            validateArgTypeParams[1].expectingType = true;
        }

        return {
            overload: typeResult.type,
            overloadIndex,
            argumentErrors: reportedArgError,
            isTypeIncomplete,
            argParams: validateArgTypeParams,
            paramSpecTarget,
            paramSpecArgList,
            activeParam,
            relevance,
        };
    }

    // After having matched arguments with parameters, this function evaluates the
    // types of each argument expression and validates that the resulting type is
    // compatible with the declared type of the corresponding parameter.
    function validateFunctionArgumentTypesWithExpectedType(
        errorNode: ExpressionNode,
        matchResults: MatchArgsToParamsResult,
        typeVarContext: TypeVarContext,
        skipUnknownArgCheck = false,
        inferenceContext: InferenceContext | undefined
    ): CallResult {
        const type = matchResults.overload;

        if (
            !inferenceContext ||
            isAnyOrUnknown(inferenceContext.expectedType) ||
            isNever(inferenceContext.expectedType) ||
            requiresSpecialization(inferenceContext.expectedType) ||
            !type.details.declaredReturnType ||
            !requiresSpecialization(FunctionType.getSpecializedReturnType(type) ?? UnknownType.create())
        ) {
            return validateFunctionArgumentTypes(errorNode, matchResults, typeVarContext, skipUnknownArgCheck);
        }

        const effectiveReturnType = getFunctionEffectiveReturnType(type);
        let effectiveExpectedType: Type | undefined = inferenceContext.expectedType;
        let effectiveFlags = AssignTypeFlags.Default;
        if (containsLiteralType(effectiveExpectedType, /* includeTypeArgs */ true)) {
            effectiveFlags |= AssignTypeFlags.RetainLiteralsForTypeVar;
        }

        // If the expected type is a union, we don't know which type is expected.
        // We may or may not be able to make use of the expected type. We'll evaluate
        // speculatively to see if using the expected type works.
        if (isUnion(inferenceContext.expectedType)) {
            let speculativeResults: CallResult | undefined;

            useSpeculativeMode(errorNode, () => {
                const typeVarContextCopy = typeVarContext.clone();
                assignType(
                    effectiveReturnType,
                    effectiveExpectedType!,
                    /* diag */ undefined,
                    typeVarContextCopy,
                    /* srcTypeVarContext */ undefined,
                    effectiveFlags | AssignTypeFlags.PopulatingExpectedType
                );
                speculativeResults = validateFunctionArgumentTypes(
                    errorNode,
                    matchResults,
                    typeVarContextCopy,
                    skipUnknownArgCheck
                );
            });

            if (speculativeResults && speculativeResults.argumentErrors) {
                effectiveExpectedType = undefined;
            }
        }

        if (effectiveExpectedType) {
            // Prepopulate the typeVarContext based on the specialized expected type if the
            // callee has a declared return type. This will allow us to more closely match
            // the expected type if possible.

            // If the return type is not the same as the expected type but is
            // assignable to the expected type, determine which type arguments
            // are needed to match the expected type.
            if (
                isClassInstance(effectiveReturnType) &&
                isClassInstance(effectiveExpectedType) &&
                !ClassType.isSameGenericClass(effectiveReturnType, effectiveExpectedType)
            ) {
                const tempTypeVarContext = new TypeVarContext(getTypeVarScopeId(effectiveReturnType));
                populateTypeVarContextBasedOnExpectedType(
                    evaluatorInterface,
                    effectiveReturnType,
                    effectiveExpectedType,
                    tempTypeVarContext,
                    getTypeVarScopesForNode(errorNode)
                );

                const genericReturnType = ClassType.cloneForSpecialization(
                    effectiveReturnType,
                    /* typeArguments */ undefined,
                    /* isTypeArgumentExplicit */ false
                );

                effectiveExpectedType = applySolvedTypeVars(genericReturnType, tempTypeVarContext);
            }

            assignType(
                effectiveReturnType,
                effectiveExpectedType,
                /* diag */ undefined,
                typeVarContext,
                /* srcTypeVarContext */ undefined,
                effectiveFlags | AssignTypeFlags.PopulatingExpectedType
            );
        }

        return validateFunctionArgumentTypes(errorNode, matchResults, typeVarContext, skipUnknownArgCheck);
    }

    function validateFunctionArgumentTypes(
        errorNode: ExpressionNode,
        matchResults: MatchArgsToParamsResult,
        typeVarContext: TypeVarContext,
        skipUnknownArgCheck = false
    ): CallResult {
        const type = matchResults.overload;
        let isTypeIncomplete = matchResults.isTypeIncomplete;
        let argumentErrors = false;
        let specializedInitSelfType: Type | undefined;
        let isArgumentAnyOrUnknown = false;
        const typeCondition = getTypeCondition(type);

        if (type.boundTypeVarScopeId) {
            // If the function was bound to a class or object and was a constructor, a
            // static method or a class method, it's possible that some of that class's
            // type variables have not yet been solved. Add that class's TypeVar scope ID.
            if (type.preBoundFlags !== undefined && type.boundToType && requiresSpecialization(type.boundToType)) {
                if (
                    type.preBoundFlags &
                    (FunctionTypeFlags.StaticMethod | FunctionTypeFlags.ClassMethod | FunctionTypeFlags.StaticMethod)
                ) {
                    typeVarContext.addSolveForScope(type.boundTypeVarScopeId);
                }
            }

            // Some typeshed stubs use specialized type annotations in the "self" parameter
            // of an overloaded __init__ method to specify which specialized type should
            // be constructed. Although this isn't part of the official Python spec, other
            // type checkers appear to honor it.
            if (
                type.details.name === '__init__' &&
                FunctionType.isOverloaded(type) &&
                type.strippedFirstParamType &&
                type.boundToType &&
                isClassInstance(type.strippedFirstParamType) &&
                isClassInstance(type.boundToType) &&
                ClassType.isSameGenericClass(type.strippedFirstParamType, type.boundToType) &&
                type.strippedFirstParamType.typeArguments
            ) {
                const typeParams = type.strippedFirstParamType.details.typeParameters;
                specializedInitSelfType = type.strippedFirstParamType;
                type.strippedFirstParamType.typeArguments.forEach((typeArg, index) => {
                    if (index < typeParams.length) {
                        const typeParam = typeParams[index];
                        if (!isTypeSame(typeParam, typeArg, { ignorePseudoGeneric: true })) {
                            typeVarContext.setTypeVarType(typeParams[index], typeArg);
                        }
                    }
                });
            }
        }

        // Special-case a few built-in calls that are often used for
        // casting or checking for unknown types.
        if (['cast', 'isinstance', 'issubclass'].some((name) => name === type.details.builtInName)) {
            skipUnknownArgCheck = true;
        }

        // Run through all args and validate them against their matched parameter.
        // We'll do two passes. The first one will match any type arguments. The second
        // will perform the actual validation. We can skip the first pass if there
        // are no type vars to match.
        const typeVarMatchingCount = matchResults.argParams.filter((arg) => arg.requiresTypeVarMatching).length;
        if (typeVarMatchingCount > 0) {
            // In theory, we may need to do up to n passes where n is the number of
            // arguments that need type var matching. That's because later matches
            // can provide bidirectional type hints for earlier matches. The best
            // example of this is the built-in "map" method whose first parameter is
            // a lambda and second parameter indicates what type the lambda should accept.
            // In practice, we will limit the number of passes to 2 because it can get
            // very expensive to go beyond this, and we don't see generally see cases
            // where more than two passes are needed.
            let passCount = Math.min(typeVarMatchingCount, 2);
            for (let i = 0; i < passCount; i++) {
                const signatureTracker = new UniqueSignatureTracker();

                useSpeculativeMode(errorNode, () => {
                    matchResults.argParams.forEach((argParam) => {
                        if (!argParam.requiresTypeVarMatching) {
                            return;
                        }

                        // Populate the typeVarContext for the argument. If the argument
                        // is an overload function, skip it during the first pass
                        // because the selection of the proper overload may depend
                        // on type arguments supplied by other function arguments.
                        // Set useNarrowBoundOnly to true the first time through
                        // the loop if we're going to go through the loop multiple
                        // times.
                        const argResult = validateArgType(
                            argParam,
                            typeVarContext,
                            signatureTracker,
                            { type, isIncomplete: matchResults.isTypeIncomplete },
                            skipUnknownArgCheck,
                            /* skipOverloadArg */ i === 0,
                            /* useNarrowBoundOnly */ passCount > 1 && i === 0,
                            typeCondition
                        );

                        if (argResult.isTypeIncomplete) {
                            isTypeIncomplete = true;
                        }

                        // If we skipped a overload arg during the first pass,
                        // add another pass to ensure that we handle all of the
                        // type variables.
                        if (i === 0 && argResult.skippedOverloadArg) {
                            passCount++;
                        }
                    });
                });
            }

            // Lock the type var map so it cannot be modified and revalidate the
            // arguments in a second pass.
            typeVarContext.lock();
        }

        let sawParamSpecArgs = false;
        let sawParamSpecKwargs = false;

        let condition: TypeCondition[] = [];
        const argResults: ArgResult[] = [];

        const signatureTracker = new UniqueSignatureTracker();
        matchResults.argParams.forEach((argParam) => {
            const argResult = validateArgType(
                argParam,
                typeVarContext,
                signatureTracker,
                { type, isIncomplete: matchResults.isTypeIncomplete },
                skipUnknownArgCheck,
                /* skipOverloadArg */ false,
                /* useNarrowBoundOnly */ false,
                typeCondition
            );

            argResults.push(argResult);

            if (!argResult.isCompatible) {
                argumentErrors = true;
            }

            if (argResult.isTypeIncomplete) {
                isTypeIncomplete = true;
            }

            if (argResult.condition) {
                condition = TypeCondition.combine(condition, argResult.condition) ?? [];
            }

            if (isAnyOrUnknown(argResult.argType)) {
                isArgumentAnyOrUnknown = true;
            }

            if (type.details.paramSpec) {
                if (argParam.argument.argumentCategory === ArgumentCategory.UnpackedList) {
                    if (isParamSpec(argResult.argType) && argResult.argType.paramSpecAccess === 'args') {
                        sawParamSpecArgs = true;
                    }
                }

                if (argParam.argument.argumentCategory === ArgumentCategory.UnpackedDictionary) {
                    if (isParamSpec(argResult.argType) && argResult.argType.paramSpecAccess === 'kwargs') {
                        sawParamSpecKwargs = true;
                    }
                }
            }
        });

        // Handle the assignment of additional arguments that map to a param spec.
        if (matchResults.paramSpecArgList && matchResults.paramSpecTarget) {
            if (
                !validateFunctionArgumentsForParamSpec(
                    errorNode,
                    matchResults.paramSpecArgList,
                    matchResults.paramSpecTarget,
                    typeVarContext,
                    typeCondition
                )
            ) {
                argumentErrors = true;
            }
        } else if (type.details.paramSpec) {
            if (!sawParamSpecArgs || !sawParamSpecKwargs) {
                if (!isTypeIncomplete) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.paramSpecArgsMissing().format({ type: printType(type.details.paramSpec) }),
                        errorNode
                    );
                }
                argumentErrors = true;
            }
        }

        // Calculate the return type.
        let returnType = getFunctionEffectiveReturnType(type, matchResults.argParams);

        if (condition.length > 0) {
            returnType = TypeBase.cloneForCondition(returnType, condition);
        }

        // Determine whether the expression being evaluated is within the current TypeVar
        // scope. If not, then the expression is invoking a function in another scope,
        // and we should eliminate unsolved type variables from union types that appear
        // in the return type. If we're within the same scope, we should retain these
        // extra type variables because they are still potentially relevant within this
        // scope.
        let eliminateUnsolvedInUnions = true;
        let curNode: ParseNode | undefined = errorNode;
        while (curNode) {
            const typeVarScopeNode = ParseTreeUtils.getTypeVarScopeNode(curNode);
            if (!typeVarScopeNode) {
                break;
            }

            const typeVarScopeId = getScopeIdForNode(typeVarScopeNode);
            if (typeVarContext.hasSolveForScope(typeVarScopeId)) {
                eliminateUnsolvedInUnions = false;
            }

            curNode = typeVarScopeNode.parent;
        }

        // If the function is returning a callable, don't eliminate unsolved
        // type vars within a union. There are legit uses for unsolved type vars
        // within a callable.
        if (
            isFunction(returnType) ||
            isOverloadedFunction(returnType) ||
            type.details.typeVarScopeId === WildcardTypeVarScopeId
        ) {
            eliminateUnsolvedInUnions = false;
        }

        // In general, we want to replace in-scope TypeVars with Unknown
        // if they were not solved. However, if the return type is a
        // Callable, we'll leave the TypeVars unsolved because
        // the call below to adjustCallableReturnType will "detach" these
        // TypeVars from the scope of this function and "attach" them to
        // the scope of the callable.
        let unknownIfNotFound = !isFunction(returnType);

        // We'll also leave TypeVars unsolved if the call is a recursive
        // call to a generic function.
        const typeVarScopes = getTypeVarScopesForNode(errorNode);
        if (typeVarScopes.some((typeVarScope) => typeVarContext.hasSolveForScope(typeVarScope))) {
            unknownIfNotFound = false;
        }

        let specializedReturnType = addConditionToType(
            applySolvedTypeVars(returnType, typeVarContext, {
                unknownIfNotFound,
                useUnknownOverDefault: skipUnknownArgCheck,
                eliminateUnsolvedInUnions,
            }),
            typeCondition
        );

        // If the final return type is an unpacked tuple, turn it into a normal (unpacked) tuple.
        if (isUnpackedClass(specializedReturnType)) {
            specializedReturnType = ClassType.cloneForUnpacked(specializedReturnType, /* isUnpackedTuple */ false);
        }

        // Handle 'TypeGuard' and 'StrictTypeGuard' specially. We'll transform the
        // return type into a 'bool' object with a type argument that reflects the
        // narrowed type.
        if (
            isClassInstance(specializedReturnType) &&
            ClassType.isBuiltIn(specializedReturnType, ['TypeGuard', 'StrictTypeGuard']) &&
            specializedReturnType.typeArguments &&
            specializedReturnType.typeArguments.length > 0
        ) {
            if (boolClassType && isInstantiableClass(boolClassType)) {
                let typeGuardType = specializedReturnType.typeArguments[0];

                // If the first argument is a simple (non-constrained) TypeVar,
                // associate that TypeVar with the resulting TypeGuard type.
                if (argResults.length > 0) {
                    const arg0Type = argResults[0].argType;
                    if (
                        isTypeVar(arg0Type) &&
                        !arg0Type.details.isParamSpec &&
                        arg0Type.details.constraints.length === 0
                    ) {
                        typeGuardType = addConditionToType(typeGuardType, [
                            {
                                typeVarName: TypeVarType.getNameWithScope(arg0Type),
                                constraintIndex: 0,
                                isConstrainedTypeVar: false,
                            },
                        ]) as ClassType;
                    }
                }

                specializedReturnType = ClassType.cloneAsInstance(
                    ClassType.cloneForTypeGuard(
                        boolClassType,
                        typeGuardType,
                        ClassType.isBuiltIn(specializedReturnType, 'StrictTypeGuard')
                    )
                );
            }
        }

        specializedReturnType = adjustCallableReturnType(specializedReturnType);

        if (specializedInitSelfType) {
            specializedInitSelfType = applySolvedTypeVars(specializedInitSelfType, typeVarContext);
        }

        return {
            argumentErrors,
            argResults,
            isArgumentAnyOrUnknown,
            returnType: specializedReturnType,
            isTypeIncomplete,
            activeParam: matchResults.activeParam,
            specializedInitSelfType,
            overloadsUsedForCall: argumentErrors ? [] : [type],
        };
    }

    function adjustCallableReturnType(type: Type): Type {
        // If the return type includes a generic Callable type, set the type var
        // scope to a wildcard to allow these type vars to be solved. This won't
        // work with overloads or unions of callables. It's intended for a
        // specific use case. We may need to make this more sophisticated in
        // the future.
        if (isFunction(type) && !type.details.name) {
            type.details = {
                ...type.details,
                typeVarScopeId: WildcardTypeVarScopeId,
            };
        }

        return type;
    }

    // Tries to assign the call arguments to the function parameter
    // list and reports any mismatches in types or counts. Returns the
    // specialized return type of the call.
    function validateFunctionArguments(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        typeResult: TypeResult<FunctionType>,
        typeVarContext: TypeVarContext,
        skipUnknownArgCheck = false,
        inferenceContext?: InferenceContext
    ): CallResult {
        const matchResults = matchFunctionArgumentsToParameters(errorNode, argList, typeResult, 0);

        if (matchResults.argumentErrors) {
            // Evaluate types of all args. This will ensure that referenced symbols are
            // not reported as unaccessed.
            argList.forEach((arg) => {
                if (arg.valueExpression && !speculativeTypeTracker.isSpeculative(arg.valueExpression)) {
                    getTypeOfExpression(arg.valueExpression);
                }
            });

            return {
                argumentErrors: true,
                activeParam: matchResults.activeParam,
                overloadsUsedForCall: [],
            };
        }

        return validateFunctionArgumentTypesWithExpectedType(
            errorNode,
            matchResults,
            typeVarContext,
            skipUnknownArgCheck,
            inferenceContext
        );
    }

    // Determines whether the specified argument list satisfies the function
    // signature bound to the specified ParamSpec. Return value indicates success.
    function validateFunctionArgumentsForParamSpec(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        paramSpec: TypeVarType,
        destTypeVarContext: TypeVarContext,
        conditionFilter: TypeCondition[] | undefined
    ): boolean {
        const signatureContexts = destTypeVarContext.getSignatureContexts();

        // Handle the common case where there is only one signature context.
        if (signatureContexts.length === 1) {
            return validateFunctionArgumentsForParamSpecSignature(
                errorNode,
                argList,
                paramSpec,
                signatureContexts[0],
                conditionFilter
            );
        }

        const filteredSignatureContexts: TypeVarSignatureContext[] = [];
        signatureContexts.forEach((context) => {
            // Use speculative mode to avoid emitting errors or caching types.
            useSpeculativeMode(errorNode, () => {
                if (
                    validateFunctionArgumentsForParamSpecSignature(
                        errorNode,
                        argList,
                        paramSpec,
                        context,
                        conditionFilter
                    )
                ) {
                    filteredSignatureContexts.push(context);
                }
            });
        });

        // Copy back any compatible signature contexts if any were compatible.
        if (filteredSignatureContexts.length > 0) {
            destTypeVarContext.copySignatureContexts(filteredSignatureContexts);
        }

        // Evaluate non-speculatively to produce a final result and cache types.
        return validateFunctionArgumentsForParamSpecSignature(
            errorNode,
            argList,
            paramSpec,
            filteredSignatureContexts.length > 0 ? filteredSignatureContexts[0] : signatureContexts[0],
            conditionFilter
        );
    }

    function validateFunctionArgumentsForParamSpecSignature(
        errorNode: ExpressionNode,
        argList: FunctionArgument[],
        paramSpec: TypeVarType,
        typeVarContext: TypeVarSignatureContext,
        conditionFilter: TypeCondition[] | undefined
    ): boolean {
        const paramSpecType = typeVarContext.getParamSpecType(paramSpec);
        if (!paramSpecType) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.paramSpecNotBound().format({ type: printType(paramSpec) }),
                argList[0]?.valueExpression || errorNode
            );
            return false;
        }

        const srcTypeVarContext = new TypeVarContext(paramSpecType.details.typeVarScopeId);
        let reportedArgError = false;

        // Build a map of all named parameters.
        const paramMap = new Map<string, FunctionParameter>();
        const paramSpecParams = paramSpecType.details.parameters;
        paramSpecParams.forEach((param) => {
            if (param.name) {
                paramMap.set(param.name, param);
            }
        });

        let positionalIndex = 0;
        let positionalIndexLimit = paramSpecParams.findIndex(
            (paramInfo) => paramInfo.category !== ParameterCategory.Simple
        );
        if (positionalIndexLimit < 0) {
            positionalIndexLimit = paramSpecParams.length;
        }
        const argsParam = paramSpecParams.find((paramInfo) => paramInfo.category === ParameterCategory.VarArgList);
        const kwargsParam = paramSpecParams.find(
            (paramInfo) => paramInfo.category === ParameterCategory.VarArgDictionary
        );

        const signatureTracker = new UniqueSignatureTracker();

        argList.forEach((arg) => {
            if (arg.argumentCategory === ArgumentCategory.Simple) {
                let paramType: Type | undefined;

                if (arg.name) {
                    const paramInfo = paramMap.get(arg.name.value);
                    if (paramInfo) {
                        paramType = paramInfo.type;
                        paramMap.delete(arg.name.value);
                    } else if (kwargsParam) {
                        paramType = kwargsParam.type;
                    } else {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.paramNameMissing().format({ name: arg.name.value }),
                            arg.name || errorNode
                        );
                        reportedArgError = true;
                    }
                } else {
                    if (positionalIndex < positionalIndexLimit) {
                        const paramInfo = paramSpecParams[positionalIndex];
                        paramType = paramInfo.type;
                        if (paramInfo.name) {
                            paramMap.delete(paramInfo.name);
                        }
                    } else if (argsParam) {
                        paramType = argsParam.type;
                    } else {
                        addDiagnostic(
                            AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            paramSpecParams.length === 1
                                ? Localizer.Diagnostic.argPositionalExpectedOne()
                                : Localizer.Diagnostic.argPositionalExpectedCount().format({
                                      expected: paramSpecParams.length,
                                  }),
                            arg.valueExpression ?? errorNode
                        );
                        reportedArgError = true;
                    }

                    positionalIndex++;
                }

                if (paramType) {
                    const argResult = validateArgType(
                        {
                            paramCategory: ParameterCategory.Simple,
                            paramType,
                            requiresTypeVarMatching: false,
                            argument: arg,
                            errorNode: arg.valueExpression || errorNode,
                        },
                        srcTypeVarContext,
                        signatureTracker,
                        /* functionType */ undefined,
                        /* skipUnknownArgCheck */ false,
                        /* skipOverloadArg */ false,
                        /* useNarrowBoundOnly */ false,
                        conditionFilter
                    );

                    if (!argResult.isCompatible) {
                        reportedArgError = true;
                    }
                }
            } else {
                // TODO - handle *args and **kwargs
                paramMap.clear();
            }
        });

        // Report any missing parameters.
        if (!reportedArgError) {
            let unassignedParams = [...paramMap.keys()];

            // Parameters that have defaults can be left unspecified.
            unassignedParams = unassignedParams.filter((name) => {
                const paramInfo = paramMap.get(name)!;
                return paramInfo.category === ParameterCategory.Simple && !paramInfo.hasDefault;
            });

            if (unassignedParams.length > 0 && !paramSpecType.details.paramSpec) {
                const missingParamNames = unassignedParams.map((p) => `"${p}"`).join(', ');
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    unassignedParams.length === 1
                        ? Localizer.Diagnostic.argMissingForParam().format({ name: missingParamNames })
                        : Localizer.Diagnostic.argMissingForParams().format({ names: missingParamNames }),
                    errorNode
                );
                reportedArgError = true;
            }
        }

        if (!reportedArgError) {
            typeVarContext.applySourceContextTypeVars(srcTypeVarContext);
        }

        return !reportedArgError;
    }

    function validateArgType(
        argParam: ValidateArgTypeParams,
        typeVarContext: TypeVarContext,
        signatureTracker: UniqueSignatureTracker,
        typeResult: TypeResult<FunctionType> | undefined,
        skipUnknownCheck: boolean,
        skipOverloadArg: boolean,
        useNarrowBoundOnly: boolean,
        conditionFilter: TypeCondition[] | undefined
    ): ArgResult {
        let argType: Type | undefined;
        let expectedTypeDiag: DiagnosticAddendum | undefined;
        let isTypeIncomplete = !!typeResult?.isIncomplete;
        let isCompatible = true;
        const functionName = typeResult?.type.details.name;

        if (argParam.argument.valueExpression) {
            // If the param type is a "bare" TypeVar, don't use it as an expected
            // type. This causes problems for cases where the the call expression
            // result can influence the type of the TypeVar, such as in
            // the expression "min(1, max(2, 0.5))". We set useNarrowBoundOnly
            // to true if this is the first pass through the parameter list because
            // a wide bound on a TypeVar (if a narrow bound has not yet been established)
            // will unnecessarily constrain the expected type.
            let expectedType: Type | undefined;
            if (
                !isTypeVar(argParam.paramType) ||
                argParam.paramType.scopeId !== typeResult?.type.details.typeVarScopeId
            ) {
                expectedType = applySolvedTypeVars(argParam.paramType, typeVarContext, { useNarrowBoundOnly });
            }

            // If the expected type is unknown, don't use an expected type. Instead,
            // use default rules for evaluating the expression type.
            if (expectedType && isUnknown(expectedType)) {
                expectedType = undefined;
            }

            // Was the argument's type precomputed by the caller?
            if (argParam.argType) {
                argType = argParam.argType;
            } else {
                const flags = argParam.expectingType
                    ? EvaluatorFlags.EvaluateStringLiteralAsType |
                      EvaluatorFlags.DisallowParamSpec |
                      EvaluatorFlags.DisallowTypeVarTuple
                    : EvaluatorFlags.None;
                const exprTypeResult = getTypeOfExpression(
                    argParam.argument.valueExpression,
                    flags,
                    makeInferenceContext(expectedType, typeVarContext, !!typeResult?.isIncomplete)
                );

                argType = exprTypeResult.type;

                if (exprTypeResult.isIncomplete) {
                    isTypeIncomplete = true;
                }

                if (exprTypeResult.typeErrors) {
                    isCompatible = false;
                }

                expectedTypeDiag = exprTypeResult.expectedTypeDiagAddendum;
            }

            if (
                argParam.argument &&
                argParam.argument.name &&
                !speculativeTypeTracker.isSpeculative(argParam.errorNode)
            ) {
                writeTypeCache(
                    argParam.argument.name,
                    { type: expectedType ?? argType, isIncomplete: isTypeIncomplete },
                    EvaluatorFlags.None
                );
            }
        } else {
            // Was the argument's type precomputed by the caller?
            if (argParam.argType) {
                argType = argParam.argType;
            } else if (argParam.expectingType && !argParam.argument.typeResult && argParam.argument.valueExpression) {
                const argTypeResult = getTypeOfExpression(
                    argParam.argument.valueExpression,
                    EvaluatorFlags.EvaluateStringLiteralAsType |
                        EvaluatorFlags.DisallowParamSpec |
                        EvaluatorFlags.DisallowTypeVarTuple
                );

                argType = argTypeResult.type;

                if (argTypeResult.isIncomplete) {
                    isTypeIncomplete = true;
                }
            } else {
                const argTypeResult = getTypeOfArgument(argParam.argument);
                argType = argTypeResult.type;
                if (argTypeResult.isIncomplete) {
                    isTypeIncomplete = true;
                }
            }
        }

        // If the type includes multiple instances of a generic function
        // signature, force the type arguments for the duplicates to have
        // unique names.
        argType = ensureFunctionSignaturesAreUnique(argType, signatureTracker);

        // If we're assigning to a var arg dictionary with a TypeVar type,
        // strip literals before performing the assignment. This is used in
        // places like a dict constructor.
        if (argParam.paramCategory === ParameterCategory.VarArgDictionary && isTypeVar(argParam.paramType)) {
            argType = stripLiteralValue(argType);
        }

        // If there's a constraint filter, apply it to top-level type variables
        // if appropriate. This doesn't properly handle non-top-level constrained
        // type variables.
        if (conditionFilter) {
            argType = mapSubtypesExpandTypeVars(argType, conditionFilter, (expandedSubtype) => {
                return expandedSubtype;
            });
        }

        const condition = argType.condition;

        let diag = new DiagnosticAddendum();

        // Handle the case where we're assigning a *args or **kwargs argument
        // to a *P.args or **P.kwargs parameter.
        if (isParamSpec(argParam.paramType) && argParam.paramType.paramSpecAccess !== undefined) {
            return { isCompatible, argType, isTypeIncomplete, condition };
        }

        // Handle the case where we're assigning a *P.args or **P.kwargs argument
        // to a *P.args or **P.kwargs parameter.
        if (isParamSpec(argType) && argType.paramSpecAccess !== undefined) {
            return { isCompatible, argType, isTypeIncomplete, condition };
        }

        // If we are asked to skip overload arguments, determine whether the argument
        // is an explicit overload type, an overloaded class constructor, or a
        // an overloaded callback protocol.
        if (skipOverloadArg) {
            if (isOverloadedFunction(argType)) {
                return { isCompatible, argType, isTypeIncomplete, skippedOverloadArg: true, condition };
            }

            const concreteParamType = makeTopLevelTypeVarsConcrete(argParam.paramType);
            if (isFunction(concreteParamType) || isOverloadedFunction(concreteParamType)) {
                if (isInstantiableClass(argType)) {
                    const constructor = createFunctionFromConstructor(argType);
                    if (constructor && isOverloadedFunction(constructor)) {
                        return { isCompatible, argType, isTypeIncomplete, skippedOverloadArg: true, condition };
                    }
                }

                if (isClassInstance(argType)) {
                    const callMember = lookUpObjectMember(argType, '__call__');
                    if (callMember) {
                        const memberType = getTypeOfMember(callMember);
                        if (isOverloadedFunction(memberType)) {
                            return { isCompatible, argType, isTypeIncomplete, skippedOverloadArg: true, condition };
                        }
                    }
                }
            }
        }

        if (!assignType(argParam.paramType, argType, diag.createAddendum(), typeVarContext)) {
            // Mismatching parameter types are common in untyped code; don't bother spending time
            // printing types if the diagnostic is disabled.
            const fileInfo = AnalyzerNodeInfo.getFileInfo(argParam.errorNode);
            if (
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues !== 'none' &&
                !isDiagnosticSuppressedForNode(argParam.errorNode) &&
                !isTypeIncomplete
            ) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(argParam.errorNode);
                const argTypeText = printType(argType);
                const paramTypeText = printType(argParam.paramType);

                let message: string;
                if (argParam.paramName && !argParam.isParamNameSynthesized) {
                    if (functionName) {
                        message = Localizer.Diagnostic.argAssignmentParamFunction().format({
                            argType: argTypeText,
                            paramType: paramTypeText,
                            functionName,
                            paramName: argParam.paramName,
                        });
                    } else {
                        message = Localizer.Diagnostic.argAssignmentParam().format({
                            argType: argTypeText,
                            paramType: paramTypeText,
                            paramName: argParam.paramName,
                        });
                    }
                } else {
                    if (functionName) {
                        message = Localizer.Diagnostic.argAssignmentFunction().format({
                            argType: argTypeText,
                            paramType: paramTypeText,
                            functionName,
                        });
                    } else {
                        message = Localizer.Diagnostic.argAssignment().format({
                            argType: argTypeText,
                            paramType: paramTypeText,
                        });
                    }
                }

                // If we have an expected type diagnostic addendum, use that
                // instead of the local diagnostic addendum because it will
                // be more informative.
                if (expectedTypeDiag) {
                    diag = expectedTypeDiag;
                }

                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    message + diag.getString(),
                    argParam.errorNode,
                    diag.getEffectiveTextRange() ?? argParam.errorNode
                );
            }

            return { isCompatible: false, argType, isTypeIncomplete, condition };
        }

        if (!skipUnknownCheck) {
            const simplifiedType = removeUnbound(argType);
            const fileInfo = AnalyzerNodeInfo.getFileInfo(argParam.errorNode);

            const getDiagAddendum = () => {
                const diagAddendum = new DiagnosticAddendum();
                if (argParam.paramName) {
                    diagAddendum.addMessage(
                        (functionName
                            ? Localizer.DiagnosticAddendum.argParamFunction().format({
                                  paramName: argParam.paramName,
                                  functionName,
                              })
                            : Localizer.DiagnosticAddendum.argParam().format({ paramName: argParam.paramName })) +
                            diagAddendum.getString()
                    );
                }
                return diagAddendum;
            };

            // Do not check for unknown types if the expected type is "Any".
            // Don't print types if reportUnknownArgumentType is disabled for performance.
            if (
                fileInfo.diagnosticRuleSet.reportUnknownArgumentType !== 'none' &&
                !isAny(argParam.paramType) &&
                !isTypeIncomplete
            ) {
                if (isUnknown(simplifiedType)) {
                    const diagAddendum = getDiagAddendum();
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportUnknownArgumentType,
                        DiagnosticRule.reportUnknownArgumentType,
                        Localizer.Diagnostic.argTypeUnknown() + diagAddendum.getString(),
                        argParam.errorNode
                    );
                } else if (isPartlyUnknown(simplifiedType, /* allowUnknownTypeArgsForClasses */ true)) {
                    let suppressPartialUnknown = false;

                    // Don't report an error if the type is a partially-specialized
                    // class. This comes up frequently in cases where a type is passed
                    // as an argument (e.g. "defaultdict(list)").
                    if (isInstantiableClass(simplifiedType)) {
                        suppressPartialUnknown = true;
                    }

                    // If the parameter type is also partially unknown, don't report
                    // the error because it's likely that the partially-unknown type
                    // arose due to bidirectional type matching.
                    if (isPartlyUnknown(argParam.paramType)) {
                        suppressPartialUnknown = true;
                    }

                    // If the argument type comes from a `[]` or `{}` expression,
                    // don't bother reporting it.
                    if (isClassInstance(simplifiedType) && simplifiedType.isEmptyContainer) {
                        suppressPartialUnknown = true;
                    }

                    if (!suppressPartialUnknown) {
                        const diagAddendum = getDiagAddendum();
                        diagAddendum.addMessage(
                            Localizer.DiagnosticAddendum.argumentType().format({
                                type: printType(simplifiedType, { expandTypeAlias: true }),
                            })
                        );
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportUnknownArgumentType,
                            DiagnosticRule.reportUnknownArgumentType,
                            Localizer.Diagnostic.argTypePartiallyUnknown() + diagAddendum.getString(),
                            argParam.errorNode
                        );
                    }
                }
            }
        }

        return { isCompatible, argType, isTypeIncomplete, condition };
    }

    function createTypeVarType(errorNode: ExpressionNode, argList: FunctionArgument[]): Type | undefined {
        let typeVarName = '';
        let firstConstraintArg: FunctionArgument | undefined;
        let defaultValueNode: ExpressionNode | undefined;

        if (argList.length === 0) {
            addError(Localizer.Diagnostic.typeVarFirstArg(), errorNode);
            return undefined;
        }

        const firstArg = argList[0];
        if (firstArg.valueExpression && firstArg.valueExpression.nodeType === ParseNodeType.StringList) {
            typeVarName = firstArg.valueExpression.strings.map((s) => s.value).join('');
        } else {
            addError(Localizer.Diagnostic.typeVarFirstArg(), firstArg.valueExpression || errorNode);
        }

        const typeVar = TypeVarType.createInstantiable(typeVarName, /* isParamSpec */ false);

        // Parse the remaining parameters.
        const paramNameMap = new Map<string, string>();
        for (let i = 1; i < argList.length; i++) {
            const paramNameNode = argList[i].name;
            const paramName = paramNameNode ? paramNameNode.value : undefined;

            if (paramName) {
                if (paramNameMap.get(paramName)) {
                    addError(
                        Localizer.Diagnostic.duplicateParam().format({ name: paramName }),
                        argList[i].valueExpression || errorNode
                    );
                }

                if (paramName === 'bound') {
                    if (typeVar.details.constraints.length > 0) {
                        addError(
                            Localizer.Diagnostic.typeVarBoundAndConstrained(),
                            argList[i].valueExpression || errorNode
                        );
                    } else {
                        const argType =
                            argList[i].typeResult?.type ??
                            getTypeOfExpressionExpectingType(argList[i].valueExpression!).type;
                        if (requiresSpecialization(argType, /* ignorePseudoGeneric */ true)) {
                            addError(
                                Localizer.Diagnostic.typeVarBoundGeneric(),
                                argList[i].valueExpression || errorNode
                            );
                        }
                        typeVar.details.boundType = convertToInstance(argType);
                    }
                } else if (paramName === 'covariant') {
                    if (argList[i].valueExpression && getBooleanValue(argList[i].valueExpression!)) {
                        if (
                            typeVar.details.declaredVariance === Variance.Contravariant ||
                            typeVar.details.declaredVariance === Variance.Auto
                        ) {
                            addError(Localizer.Diagnostic.typeVarVariance(), argList[i].valueExpression!);
                        } else {
                            typeVar.details.declaredVariance = Variance.Covariant;
                        }
                    }
                } else if (paramName === 'contravariant') {
                    if (argList[i].valueExpression && getBooleanValue(argList[i].valueExpression!)) {
                        if (
                            typeVar.details.declaredVariance === Variance.Covariant ||
                            typeVar.details.declaredVariance === Variance.Auto
                        ) {
                            addError(Localizer.Diagnostic.typeVarVariance(), argList[i].valueExpression!);
                        } else {
                            typeVar.details.declaredVariance = Variance.Contravariant;
                        }
                    }
                } else if (paramName === 'infer_variance') {
                    if (argList[i].valueExpression && getBooleanValue(argList[i].valueExpression!)) {
                        if (
                            typeVar.details.declaredVariance === Variance.Covariant ||
                            typeVar.details.declaredVariance === Variance.Contravariant
                        ) {
                            addError(Localizer.Diagnostic.typeVarVariance(), argList[i].valueExpression!);
                        } else {
                            typeVar.details.declaredVariance = Variance.Auto;
                        }
                    }
                } else if (paramName === 'default') {
                    defaultValueNode = argList[i].valueExpression;
                    const argType =
                        argList[i].typeResult?.type ??
                        getTypeOfExpressionExpectingType(argList[i].valueExpression!).type;
                    typeVar.details.defaultType = convertToInstance(argType);
                } else {
                    addError(
                        Localizer.Diagnostic.typeVarUnknownParam().format({ name: paramName }),
                        argList[i].node?.name || argList[i].valueExpression || errorNode
                    );
                }

                paramNameMap.set(paramName, paramName);
            } else {
                if (typeVar.details.boundType) {
                    addError(
                        Localizer.Diagnostic.typeVarBoundAndConstrained(),
                        argList[i].valueExpression || errorNode
                    );
                } else {
                    const argType =
                        argList[i].typeResult?.type ??
                        getTypeOfExpressionExpectingType(argList[i].valueExpression!).type;

                    if (requiresSpecialization(argType, /* ignorePseudoGeneric */ true)) {
                        addError(
                            Localizer.Diagnostic.typeVarConstraintGeneric(),
                            argList[i].valueExpression || errorNode
                        );
                    }
                    TypeVarType.addConstraint(typeVar, convertToInstance(argType));
                    if (firstConstraintArg === undefined) {
                        firstConstraintArg = argList[i];
                    }
                }
            }
        }

        if (typeVar.details.constraints.length === 1 && firstConstraintArg) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeVarSingleConstraint(),
                firstConstraintArg.valueExpression || errorNode
            );
        }

        // If a default is provided, make sure it is compatible with the bound
        // or constraint.
        if (typeVar.details.defaultType && defaultValueNode) {
            const typeVarContext = new TypeVarContext(WildcardTypeVarScopeId);
            const concreteDefaultType = makeTopLevelTypeVarsConcrete(
                applySolvedTypeVars(typeVar.details.defaultType, typeVarContext, {
                    unknownIfNotFound: true,
                })
            );

            if (typeVar.details.boundType) {
                if (!assignType(typeVar.details.boundType, concreteDefaultType)) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarDefaultBoundMismatch(),
                        defaultValueNode
                    );
                }
            } else if (typeVar.details.constraints.length > 0) {
                if (!typeVar.details.constraints.some((constraint) => isTypeSame(constraint, concreteDefaultType))) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarDefaultConstraintMismatch(),
                        defaultValueNode
                    );
                }
            }
        }

        return typeVar;
    }

    function createTypeVarTupleType(errorNode: ExpressionNode, argList: FunctionArgument[]): Type | undefined {
        let typeVarName = '';

        if (argList.length === 0) {
            addError(Localizer.Diagnostic.typeVarFirstArg(), errorNode);
            return undefined;
        }

        const firstArg = argList[0];
        if (firstArg.valueExpression && firstArg.valueExpression.nodeType === ParseNodeType.StringList) {
            typeVarName = firstArg.valueExpression.strings.map((s) => s.value).join('');
        } else {
            addError(Localizer.Diagnostic.typeVarFirstArg(), firstArg.valueExpression || errorNode);
        }

        const typeVar = TypeVarType.createInstantiable(typeVarName, /* isParamSpec */ false);
        typeVar.details.isVariadic = true;

        // Parse the remaining parameters.
        for (let i = 1; i < argList.length; i++) {
            const paramNameNode = argList[i].name;
            const paramName = paramNameNode ? paramNameNode.value : undefined;

            if (paramName) {
                if (paramName === 'default') {
                    const expr = argList[i].valueExpression;
                    if (expr) {
                        typeVar.details.defaultType = getTypeVarTupleDefaultType(expr);
                    }
                } else {
                    addError(
                        Localizer.Diagnostic.typeVarTupleUnknownParam().format({ name: argList[i].name?.value || '?' }),
                        argList[i].node?.name || argList[i].valueExpression || errorNode
                    );
                }
            }
        }

        return typeVar;
    }

    function getTypeVarTupleDefaultType(node: ExpressionNode): Type | undefined {
        const argType = getTypeOfExpressionExpectingType(node, { allowUnpackedTuple: true }).type;
        const isUnpackedTuple = isClass(argType) && isTupleClass(argType) && argType.isUnpacked;
        const isUnpackedTypeVarTuple = isUnpackedVariadicTypeVar(argType);

        if (!isUnpackedTuple && !isUnpackedTypeVarTuple) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeVarTupleDefaultNotUnpacked(),
                node
            );
            return undefined;
        }

        return convertToInstance(argType);
    }

    function createParamSpecType(errorNode: ExpressionNode, argList: FunctionArgument[]): Type | undefined {
        if (argList.length === 0) {
            addError(Localizer.Diagnostic.paramSpecFirstArg(), errorNode);
            return undefined;
        }

        const firstArg = argList[0];
        let paramSpecName = '';
        if (firstArg.valueExpression && firstArg.valueExpression.nodeType === ParseNodeType.StringList) {
            paramSpecName = firstArg.valueExpression.strings.map((s) => s.value).join('');
        } else {
            addError(Localizer.Diagnostic.paramSpecFirstArg(), firstArg.valueExpression || errorNode);
        }

        const paramSpec = TypeVarType.createInstantiable(paramSpecName, /* isParamSpec */ true);

        // Parse the remaining parameters.
        for (let i = 1; i < argList.length; i++) {
            const paramNameNode = argList[i].name;
            const paramName = paramNameNode ? paramNameNode.value : undefined;

            if (paramName) {
                if (paramName === 'default') {
                    const expr = argList[i].valueExpression;
                    if (expr) {
                        paramSpec.details.defaultType = getParamSpecDefaultType(expr);
                    }
                } else {
                    addError(
                        Localizer.Diagnostic.paramSpecUnknownParam().format({ name: paramName }),
                        paramNameNode || argList[i].valueExpression || errorNode
                    );
                }
            } else {
                addError(Localizer.Diagnostic.paramSpecUnknownArg(), argList[i].valueExpression || errorNode);
                break;
            }
        }

        return paramSpec;
    }

    function getParamSpecDefaultType(node: ExpressionNode): Type | undefined {
        const functionType = FunctionType.createSynthesizedInstance(
            '',
            FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck | FunctionTypeFlags.ParamSpecValue
        );
        TypeBase.setSpecialForm(functionType);

        if (node.nodeType === ParseNodeType.Ellipsis) {
            FunctionType.addDefaultParameters(functionType);
            return functionType;
        }

        if (node.nodeType === ParseNodeType.List) {
            node.entries.forEach((paramExpr, index) => {
                const typeResult = getTypeOfExpressionExpectingType(paramExpr);

                FunctionType.addParameter(functionType, {
                    category: ParameterCategory.Simple,
                    name: `__p${index}`,
                    isNameSynthesized: true,
                    hasDeclaredType: true,
                    type: convertToInstance(typeResult.type),
                });
            });

            // Update the type cache so we don't attempt to re-evaluate this node.
            // The type doesn't matter, so use Any.
            writeTypeCache(node, { type: AnyType.create() }, /* flags */ undefined);
            return functionType;
        } else {
            const typeResult = getTypeOfExpressionExpectingType(node, { allowParamSpec: true });
            if (typeResult.typeErrors) {
                return undefined;
            }

            if (isParamSpec(typeResult.type)) {
                functionType.details.paramSpec = typeResult.type;
                return functionType;
            }
        }

        addDiagnostic(
            AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
            DiagnosticRule.reportGeneralTypeIssues,
            Localizer.Diagnostic.paramSpecDefaultNotTuple(),
            node
        );

        return undefined;
    }

    function getBooleanValue(node: ExpressionNode): boolean {
        if (node.nodeType === ParseNodeType.Constant) {
            if (node.constType === KeywordType.False) {
                return false;
            } else if (node.constType === KeywordType.True) {
                return true;
            }
        }

        addError(Localizer.Diagnostic.expectedBoolLiteral(), node);
        return false;
    }

    function getFunctionFullName(functionNode: ParseNode, moduleName: string, functionName: string): string {
        const nameParts: string[] = [functionName];

        let curNode: ParseNode | undefined = functionNode;

        // Walk the parse tree looking for classes or functions.
        while (curNode) {
            curNode = ParseTreeUtils.getEnclosingClassOrFunction(curNode);
            if (curNode) {
                nameParts.push(curNode.name.value);
            }
        }

        nameParts.push(moduleName);

        return nameParts.reverse().join('.');
    }

    // Implements the semantics of the NewType call as documented
    // in the Python specification: The static type checker will treat
    // the new type as if it were a subclass of the original type.
    function createNewType(errorNode: ExpressionNode, argList: FunctionArgument[]): ClassType | undefined {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
        let className = '';

        if (argList.length !== 2) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.newTypeParamCount(),
                errorNode
            );
            return undefined;
        }

        const nameArg = argList[0];
        if (
            nameArg.argumentCategory === ArgumentCategory.Simple &&
            nameArg.valueExpression &&
            nameArg.valueExpression.nodeType === ParseNodeType.StringList
        ) {
            className = nameArg.valueExpression.strings.map((s) => s.value).join('');
        }

        if (!className) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.newTypeBadName(),
                argList[0].node ?? errorNode
            );
            return undefined;
        }

        const baseClass = getTypeOfArgumentExpectingType(argList[1]).type;

        if (isInstantiableClass(baseClass)) {
            if (ClassType.isProtocolClass(baseClass)) {
                addError(Localizer.Diagnostic.newTypeProtocolClass(), argList[1].node || errorNode);
            } else if (baseClass.literalValue !== undefined) {
                addError(Localizer.Diagnostic.newTypeLiteral(), argList[1].node || errorNode);
            }

            const classFlags = baseClass.details.flags & ~(ClassTypeFlags.BuiltInClass | ClassTypeFlags.SpecialBuiltIn);
            const classType = ClassType.createInstantiable(
                className,
                ParseTreeUtils.getClassFullName(errorNode, fileInfo.moduleName, className),
                fileInfo.moduleName,
                fileInfo.filePath,
                classFlags,
                ParseTreeUtils.getTypeSourceId(errorNode),
                /* declaredMetaclass */ undefined,
                baseClass.details.effectiveMetaclass
            );
            classType.details.baseClasses.push(baseClass);
            computeMroLinearization(classType);

            // Synthesize an __init__ method that accepts only the specified type.
            const initType = FunctionType.createSynthesizedInstance('__init__');
            FunctionType.addParameter(initType, {
                category: ParameterCategory.Simple,
                name: 'self',
                type: ClassType.cloneAsInstance(classType),
                hasDeclaredType: true,
            });
            FunctionType.addParameter(initType, {
                category: ParameterCategory.Simple,
                name: '_x',
                type: ClassType.cloneAsInstance(baseClass),
                hasDeclaredType: true,
            });
            initType.details.declaredReturnType = NoneType.createInstance();
            classType.details.fields.set('__init__', Symbol.createWithType(SymbolFlags.ClassMember, initType));

            // Synthesize a trivial __new__ method.
            const newType = FunctionType.createSynthesizedInstance('__new__', FunctionTypeFlags.ConstructorMethod);
            FunctionType.addParameter(newType, {
                category: ParameterCategory.Simple,
                name: 'cls',
                type: classType,
                hasDeclaredType: true,
            });
            FunctionType.addDefaultParameters(newType);
            newType.details.declaredReturnType = ClassType.cloneAsInstance(classType);
            classType.details.fields.set('__new__', Symbol.createWithType(SymbolFlags.ClassMember, newType));
            return classType;
        }

        if (!isAnyOrUnknown(baseClass)) {
            addError(Localizer.Diagnostic.newTypeNotAClass(), argList[1].node || errorNode);
        }

        return undefined;
    }

    // Implements the semantics of the multi-parameter variant of the "type" call.
    function createType(errorNode: ExpressionNode, argList: FunctionArgument[]): ClassType | undefined {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
        const arg0Type = getTypeOfArgument(argList[0]).type;
        if (!isClassInstance(arg0Type) || !ClassType.isBuiltIn(arg0Type, 'str')) {
            return undefined;
        }
        const className = (arg0Type.literalValue as string) || '_';

        const arg1Type = getTypeOfArgument(argList[1]).type;
        if (!isClassInstance(arg1Type) || !isTupleClass(arg1Type) || arg1Type.tupleTypeArguments === undefined) {
            return undefined;
        }

        const classType = ClassType.createInstantiable(
            className,
            ParseTreeUtils.getClassFullName(errorNode, fileInfo.moduleName, className),
            fileInfo.moduleName,
            fileInfo.filePath,
            ClassTypeFlags.None,
            ParseTreeUtils.getTypeSourceId(errorNode),
            /* declaredMetaclass */ undefined,
            arg1Type.details.effectiveMetaclass
        );
        arg1Type.tupleTypeArguments.forEach((typeArg) => {
            const specializedType = makeTopLevelTypeVarsConcrete(typeArg.type);

            if (isEffectivelyInstantiable(specializedType)) {
                classType.details.baseClasses.push(specializedType);
            } else {
                addExpectedClassDiagnostic(typeArg.type, argList[1].valueExpression || errorNode);
            }
        });

        if (!computeMroLinearization(classType)) {
            addError(Localizer.Diagnostic.methodOrdering(), errorNode);
        }

        return classType;
    }

    function getTypeOfConstant(node: ConstantNode, flags: EvaluatorFlags): TypeResult | undefined {
        let type: Type | undefined;

        if (node.constType === KeywordType.None) {
            type = (flags & EvaluatorFlags.ExpectingType) !== 0 ? NoneType.createType() : NoneType.createInstance();
        } else if (
            node.constType === KeywordType.True ||
            node.constType === KeywordType.False ||
            node.constType === KeywordType.Debug
        ) {
            type = getBuiltInObject(node, 'bool');

            // For True and False, we can create truthy and falsy
            // versions of 'bool'.
            if (type && isClassInstance(type)) {
                if (node.constType === KeywordType.True) {
                    type = ClassType.cloneWithLiteral(type, /* value */ true);
                } else if (node.constType === KeywordType.False) {
                    type = ClassType.cloneWithLiteral(type, /* value */ false);
                }
            }
        }

        if (!type) {
            return undefined;
        }

        return { type };
    }

    function getTypeOfUnaryOperation(
        node: UnaryOperationNode,
        inferenceContext: InferenceContext | undefined
    ): TypeResult {
        const exprTypeResult = getTypeOfExpression(node.expression);
        let exprType = makeTopLevelTypeVarsConcrete(exprTypeResult.type);
        const isIncomplete = exprTypeResult.isIncomplete;

        if (isNever(exprType)) {
            return { type: NeverType.createNever(), isIncomplete };
        }

        // Map unary operators to magic functions. Note that the bitwise
        // invert has two magic functions that are aliases of each other.
        const unaryOperatorMap: { [operator: number]: string } = {
            [OperatorType.Add]: '__pos__',
            [OperatorType.Subtract]: '__neg__',
            [OperatorType.BitwiseInvert]: '__invert__',
        };

        let type: Type | undefined;

        if (node.operator !== OperatorType.Not) {
            if (isOptionalType(exprType)) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportOptionalOperand,
                    DiagnosticRule.reportOptionalOperand,
                    Localizer.Diagnostic.noneOperator().format({
                        operator: ParseTreeUtils.printOperator(node.operator),
                    }),
                    node.expression
                );
                exprType = removeNoneFromUnion(exprType);
            }
        }

        // Handle certain operations on certain literal types
        // using special-case math. Do not apply this if the input type
        // is incomplete because we may be evaluating an expression within
        // a loop, so the literal value may change each time.
        if (!exprTypeResult.isIncomplete) {
            const literalClassName = getLiteralTypeClassName(exprType);
            if (literalClassName === 'int') {
                if (node.operator === OperatorType.Add) {
                    type = exprType;
                } else if (node.operator === OperatorType.Subtract) {
                    type = mapSubtypes(exprType, (subtype) => {
                        const classSubtype = subtype as ClassType;
                        return ClassType.cloneWithLiteral(
                            classSubtype,
                            -(classSubtype.literalValue as number | bigint)
                        );
                    });
                }
            } else if (literalClassName === 'bool') {
                if (node.operator === OperatorType.Not) {
                    type = mapSubtypes(exprType, (subtype) => {
                        const classSubtype = subtype as ClassType;
                        return ClassType.cloneWithLiteral(classSubtype, !(classSubtype.literalValue as boolean));
                    });
                }
            }
        }

        if (!type) {
            // __not__ always returns a boolean.
            if (node.operator === OperatorType.Not) {
                type = getBuiltInObject(node, 'bool');
                if (!type) {
                    type = UnknownType.create();
                }
            } else {
                if (isAnyOrUnknown(exprType)) {
                    type = exprType;
                } else {
                    const magicMethodName = unaryOperatorMap[node.operator];
                    type = getTypeOfMagicMethodReturn(exprType, [], magicMethodName, node, inferenceContext);
                }

                if (!type) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

                    if (inferenceContext) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeNotSupportUnaryOperatorBidirectional().format({
                                operator: ParseTreeUtils.printOperator(node.operator),
                                type: printType(exprType),
                                expectedType: printType(inferenceContext.expectedType),
                            }),
                            node
                        );
                    } else {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeNotSupportUnaryOperator().format({
                                operator: ParseTreeUtils.printOperator(node.operator),
                                type: printType(exprType),
                            }),
                            node
                        );
                    }
                    type = UnknownType.create();
                }
            }
        }

        return { type, isIncomplete };
    }

    function getTypeOfBinaryOperation(
        node: BinaryOperationNode,
        inferenceContext: InferenceContext | undefined,
        flags: EvaluatorFlags
    ): TypeResult {
        const leftExpression = node.leftExpression;
        let rightExpression = node.rightExpression;
        let isIncomplete = false;
        let typeErrors = false;

        // If this is a comparison and the left expression is also a comparison,
        // we need to change the behavior to accommodate python's "chained
        // comparisons" feature.
        if (ParseTreeUtils.operatorSupportsChaining(node.operator)) {
            if (
                rightExpression.nodeType === ParseNodeType.BinaryOperation &&
                !rightExpression.parenthesized &&
                ParseTreeUtils.operatorSupportsChaining(rightExpression.operator)
            ) {
                // Evaluate the right expression so it is type checked.
                getTypeOfBinaryOperation(rightExpression, inferenceContext, flags);

                // Use the left side of the right expression for comparison purposes.
                rightExpression = rightExpression.leftExpression;
            }
        }

        // For most binary operations, the "expected type" is applied to the output
        // of the magic method for that operation. However, the "or" and "and" operators
        // have no magic method, so we apply the expected type directly to both operands.
        let expectedOperandType =
            node.operator === OperatorType.Or || node.operator === OperatorType.And
                ? inferenceContext?.expectedType
                : undefined;

        // Handle the very special case where the expected type is a list
        // and the operator is a multiply. This comes up in the common case
        // of "x: List[Optional[X]] = [None] * y" where y is an integer literal.
        let expectedLeftOperandType: Type | undefined;
        if (
            node.operator === OperatorType.Multiply &&
            inferenceContext &&
            isClassInstance(inferenceContext.expectedType) &&
            ClassType.isBuiltIn(inferenceContext.expectedType, 'list') &&
            inferenceContext.expectedType.typeArguments &&
            inferenceContext.expectedType.typeArguments.length >= 1 &&
            node.leftExpression.nodeType === ParseNodeType.List
        ) {
            expectedLeftOperandType = inferenceContext.expectedType;
        }

        const effectiveExpectedType = expectedOperandType ?? expectedLeftOperandType;
        const leftTypeResult = getTypeOfExpression(leftExpression, flags, makeInferenceContext(effectiveExpectedType));
        let leftType = leftTypeResult.type;

        if (!expectedOperandType) {
            if (node.operator === OperatorType.Or || node.operator === OperatorType.And) {
                // For "or" and "and", use the type of the left operand. This allows us to
                // infer a better type for expressions like `x or []`.
                expectedOperandType = leftType;
            } else if (node.operator === OperatorType.Add && node.rightExpression.nodeType === ParseNodeType.List) {
                // For the "+" operator , use this technique only if the right operand is
                // a list expression. This heuristic handles the common case of `my_list + [0]`.
                expectedOperandType = leftType;
            } else if (node.operator === OperatorType.BitwiseOr) {
                // If this is a bitwise or ("|"), use the type of the left operand. This allows
                // us to support the case where a TypedDict is being updated with a dict expression.
                if (isClassInstance(leftType) && ClassType.isTypedDictClass(leftType)) {
                    expectedOperandType = leftType;
                }
            }
        }

        const rightTypeResult = getTypeOfExpression(rightExpression, flags, makeInferenceContext(expectedOperandType));
        let rightType = rightTypeResult.type;

        if (leftTypeResult.isIncomplete || rightTypeResult.isIncomplete) {
            isIncomplete = true;
        }

        // Is this a "|" operator used in a context where it is supposed to be
        // interpreted as a union operator?
        if (
            node.operator === OperatorType.BitwiseOr &&
            !customMetaclassSupportsMethod(leftType, '__or__') &&
            !customMetaclassSupportsMethod(rightType, '__ror__')
        ) {
            let adjustedRightType = rightType;
            let adjustedLeftType = leftType;
            if (!isNoneInstance(leftType) && isNoneInstance(rightType)) {
                // Handle the special case where "None" is being added to the union
                // with something else. Even though "None" will normally be interpreted
                // as the None singleton object in contexts where a type annotation isn't
                // assumed, we'll allow it here.
                adjustedRightType = NoneType.createType();
            } else if (!isNoneInstance(rightType) && isNoneInstance(leftType)) {
                adjustedLeftType = NoneType.createType();
            }

            if (isUnionableType([adjustedLeftType, adjustedRightType])) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                const unionNotationSupported =
                    fileInfo.isStubFile ||
                    (flags & EvaluatorFlags.AllowForwardReferences) !== 0 ||
                    fileInfo.executionEnvironment.pythonVersion >= PythonVersion.V3_10;
                if (!unionNotationSupported) {
                    // If the left type is Any, we can't say for sure whether this
                    // is an illegal syntax or a valid application of the "|" operator.
                    if (!isAnyOrUnknown(adjustedLeftType)) {
                        addError(Localizer.Diagnostic.unionSyntaxIllegal(), node, node.operatorToken);
                    }
                }

                if (
                    !validateTypeArg(
                        { ...leftTypeResult, node: leftExpression },
                        { allowVariadicTypeVar: true, allowUnpackedTuples: true }
                    ) ||
                    !validateTypeArg(
                        { ...rightTypeResult, node: rightExpression },
                        { allowVariadicTypeVar: true, allowUnpackedTuples: true }
                    )
                ) {
                    return { type: UnknownType.create() };
                }

                const newUnion = combineTypes([adjustedLeftType, adjustedRightType]);
                if (isUnion(newUnion)) {
                    TypeBase.setSpecialForm(newUnion);
                }

                // Check for "stringified" forward reference type expressions. The "|" operator
                // doesn't support these except in certain circumstances. Notably, it can't be used
                // with other strings or with types that are not specialized using an index form.
                if (!fileInfo.isStubFile) {
                    let stringNode: ExpressionNode | undefined;
                    let otherNode: ExpressionNode | undefined;
                    let otherType: Type | undefined;

                    if (leftExpression.nodeType === ParseNodeType.StringList) {
                        stringNode = leftExpression;
                        otherNode = rightExpression;
                        otherType = rightType;
                    } else if (rightExpression.nodeType === ParseNodeType.StringList) {
                        stringNode = rightExpression;
                        otherNode = leftExpression;
                        otherType = leftType;
                    }

                    if (stringNode && otherNode && otherType) {
                        let isAllowed = true;
                        if (isClass(otherType)) {
                            if (!otherType.isTypeArgumentExplicit || isClassInstance(otherType)) {
                                isAllowed = false;
                            }
                        }

                        if (!isAllowed) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.unionForwardReferenceNotAllowed(),
                                stringNode
                            );
                        }
                    }
                }

                return { type: newUnion };
            }
        }

        // Optional checks apply to all operations except for boolean operations.
        let isLeftOptionalType = false;
        if (booleanOperatorMap[node.operator] === undefined) {
            // None is a valid operand for == and != even if the type stub says otherwise.
            if (node.operator === OperatorType.Equals || node.operator === OperatorType.NotEquals) {
                leftType = removeNoneFromUnion(leftType);
            } else {
                isLeftOptionalType = isOptionalType(leftType);
            }

            // None is a valid operand for == and != even if the type stub says otherwise.
            if (node.operator === OperatorType.Equals || node.operator === OperatorType.NotEquals) {
                rightType = removeNoneFromUnion(rightType);
            }
        }

        const diag = new DiagnosticAddendum();

        // Don't use literal math if either of the operation is within a loop
        // because the literal values may change each time.
        const isLiteralMathAllowed = !ParseTreeUtils.isWithinLoop(node);

        // Don't special-case tuple __add__ if the left type is a union. This
        // can result in an infinite loop if we keep creating new tuple types
        // within a loop construct using __add__.
        const isTupleAddAllowed = !isUnion(leftType);

        let type = validateBinaryOperation(
            node.operator,
            { type: leftType, isIncomplete: leftTypeResult.isIncomplete },
            { type: rightType, isIncomplete: rightTypeResult.isIncomplete },
            node,
            inferenceContext,
            diag,
            { isLiteralMathAllowed, isTupleAddAllowed }
        );

        if (!diag.isEmpty() || !type) {
            typeErrors = true;

            if (!isIncomplete) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

                if (isLeftOptionalType && diag.getMessages().length === 1) {
                    // If the left was an optional type and there is just one diagnostic,
                    // assume that it was due to a "None" not being supported. Report
                    // this as a reportOptionalOperand diagnostic rather than a
                    // reportGeneralTypeIssues diagnostic.
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportOptionalOperand,
                        DiagnosticRule.reportOptionalOperand,
                        Localizer.Diagnostic.noneOperator().format({
                            operator: ParseTreeUtils.printOperator(node.operator),
                        }),
                        node.leftExpression
                    );
                } else {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeNotSupportBinaryOperator().format({
                            operator: ParseTreeUtils.printOperator(node.operator),
                            leftType: printType(leftType),
                            rightType: printType(rightType),
                        }) + diag.getString(),
                        node
                    );
                }
            }

            type = UnknownType.create();
        }

        return { type, isIncomplete, typeErrors };
    }

    function customMetaclassSupportsMethod(type: Type, methodName: string): boolean {
        if (!isInstantiableClass(type)) {
            return false;
        }

        const metaclass = type.details.effectiveMetaclass;
        if (!metaclass || !isInstantiableClass(metaclass)) {
            return false;
        }

        if (ClassType.isBuiltIn(metaclass, 'type')) {
            return false;
        }

        const memberInfo = lookUpClassMember(metaclass, methodName);
        if (!memberInfo) {
            return false;
        }

        if (isInstantiableClass(memberInfo.classType) && ClassType.isBuiltIn(memberInfo.classType, 'type')) {
            return false;
        }

        return true;
    }

    function getTypeOfAugmentedAssignment(
        node: AugmentedAssignmentNode,
        inferenceContext: InferenceContext | undefined
    ): TypeResult {
        const operatorMap: { [operator: number]: [string, OperatorType] } = {
            [OperatorType.AddEqual]: ['__iadd__', OperatorType.Add],
            [OperatorType.SubtractEqual]: ['__isub__', OperatorType.Subtract],
            [OperatorType.MultiplyEqual]: ['__imul__', OperatorType.Multiply],
            [OperatorType.FloorDivideEqual]: ['__ifloordiv__', OperatorType.FloorDivide],
            [OperatorType.DivideEqual]: ['__itruediv__', OperatorType.Divide],
            [OperatorType.ModEqual]: ['__imod__', OperatorType.Mod],
            [OperatorType.PowerEqual]: ['__ipow__', OperatorType.Power],
            [OperatorType.MatrixMultiplyEqual]: ['__imatmul__', OperatorType.MatrixMultiply],
            [OperatorType.BitwiseAndEqual]: ['__iand__', OperatorType.BitwiseAnd],
            [OperatorType.BitwiseOrEqual]: ['__ior__', OperatorType.BitwiseOr],
            [OperatorType.BitwiseXorEqual]: ['__ixor__', OperatorType.BitwiseXor],
            [OperatorType.LeftShiftEqual]: ['__ilshift__', OperatorType.LeftShift],
            [OperatorType.RightShiftEqual]: ['__irshift__', OperatorType.RightShift],
        };

        let type: Type | undefined;
        let typeResult: TypeResult | undefined;
        const diag = new DiagnosticAddendum();

        const leftTypeResult = getTypeOfExpression(node.leftExpression);
        const leftType = leftTypeResult.type;

        let expectedOperandType: Type | undefined;
        if (node.operator === OperatorType.BitwiseOrEqual) {
            // If this is a bitwise or ("|="), use the type of the left operand. This allows
            // us to support the case where a TypedDict is being updated with a dict expression.
            expectedOperandType = leftType;
        }

        const rightTypeResult = getTypeOfExpression(
            node.rightExpression,
            /* flags */ undefined,
            makeInferenceContext(expectedOperandType)
        );
        const rightType = rightTypeResult.type;
        const isIncomplete = !!rightTypeResult.isIncomplete || !!leftTypeResult.isIncomplete;

        if (isNever(leftType) || isNever(rightType)) {
            typeResult = { type: NeverType.createNever(), isIncomplete };
        } else {
            type = mapSubtypesExpandTypeVars(
                leftType,
                /* conditionFilter */ undefined,
                (leftSubtypeExpanded, leftSubtypeUnexpanded) => {
                    return mapSubtypesExpandTypeVars(
                        rightType,
                        getTypeCondition(leftSubtypeExpanded),
                        (rightSubtypeExpanded, rightSubtypeUnexpanded) => {
                            if (isAnyOrUnknown(leftSubtypeUnexpanded) || isAnyOrUnknown(rightSubtypeUnexpanded)) {
                                return preserveUnknown(leftSubtypeUnexpanded, rightSubtypeUnexpanded);
                            }

                            const magicMethodName = operatorMap[node.operator][0];
                            let returnType = getTypeOfMagicMethodReturn(
                                leftSubtypeUnexpanded,
                                [{ type: rightSubtypeUnexpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                magicMethodName,
                                node,
                                inferenceContext
                            );

                            if (!returnType && leftSubtypeUnexpanded !== leftSubtypeExpanded) {
                                // Try with the expanded left type.
                                returnType = getTypeOfMagicMethodReturn(
                                    leftSubtypeExpanded,
                                    [{ type: rightSubtypeUnexpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                    magicMethodName,
                                    node,
                                    inferenceContext
                                );
                            }

                            if (!returnType && rightSubtypeUnexpanded !== rightSubtypeExpanded) {
                                // Try with the expanded left and right type.
                                returnType = getTypeOfMagicMethodReturn(
                                    leftSubtypeExpanded,
                                    [{ type: rightSubtypeExpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                    magicMethodName,
                                    node,
                                    inferenceContext
                                );
                            }

                            if (!returnType) {
                                // If the LHS class didn't support the magic method for augmented
                                // assignment, fall back on the normal binary expression evaluator.
                                const binaryOperator = operatorMap[node.operator][1];

                                // Don't use literal math if either of the operation is within a loop
                                // because the literal values may change each time.
                                const isLiteralMathAllowed =
                                    !ParseTreeUtils.isWithinLoop(node) &&
                                    getUnionSubtypeCount(leftType) * getUnionSubtypeCount(rightType) <
                                        maxLiteralMathSubtypeCount;

                                // Don't special-case tuple __add__ if the left type is a union. This
                                // can result in an infinite loop if we keep creating new tuple types
                                // within a loop construct using __add__.
                                const isTupleAddAllowed = !isUnion(leftType);

                                returnType = validateBinaryOperation(
                                    binaryOperator,
                                    { type: leftSubtypeUnexpanded, isIncomplete: leftTypeResult.isIncomplete },
                                    { type: rightSubtypeUnexpanded, isIncomplete: rightTypeResult.isIncomplete },
                                    node,
                                    inferenceContext,
                                    diag,
                                    { isLiteralMathAllowed, isTupleAddAllowed }
                                );
                            }

                            return returnType;
                        }
                    );
                }
            );

            // If the LHS class didn't support the magic method for augmented
            // assignment, fall back on the normal binary expression evaluator.
            if (!diag.isEmpty() || !type || isNever(type)) {
                if (!isIncomplete) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeNotSupportBinaryOperator().format({
                            operator: ParseTreeUtils.printOperator(node.operator),
                            leftType: printType(leftType),
                            rightType: printType(rightType),
                        }) + diag.getString(),
                        node
                    );
                }

                type = UnknownType.create();
            }

            typeResult = { type, isIncomplete };
        }

        assignTypeToExpression(node.destExpression, typeResult.type, !!typeResult.isIncomplete, node.rightExpression);

        return typeResult;
    }

    function validateBinaryOperation(
        operator: OperatorType,
        leftTypeResult: TypeResult,
        rightTypeResult: TypeResult,
        errorNode: ExpressionNode,
        inferenceContext: InferenceContext | undefined,
        diag: DiagnosticAddendum,
        options: BinaryOperationOptions
    ): Type | undefined {
        const leftType = leftTypeResult.type;
        const rightType = rightTypeResult.type;
        let type: Type | undefined;
        let concreteLeftType = makeTopLevelTypeVarsConcrete(leftType);

        if (booleanOperatorMap[operator] !== undefined) {
            // If it's an AND or OR, we need to handle short-circuiting by
            // eliminating any known-truthy or known-falsy types.
            if (operator === OperatorType.And) {
                // If the LHS evaluates to falsy, the And expression will
                // always return the type of the left-hand side.
                if (!canBeTruthy(concreteLeftType)) {
                    return leftType;
                }

                // If the LHS evaluates to truthy, the And expression will
                // always return the type of the right-hand side.
                if (!canBeFalsy(concreteLeftType)) {
                    return rightType;
                }

                concreteLeftType = removeTruthinessFromType(concreteLeftType);

                if (isNever(rightType)) {
                    return concreteLeftType;
                }
            } else if (operator === OperatorType.Or) {
                // If the LHS evaluates to truthy, the Or expression will
                // always return the type of the left-hand side.
                if (!canBeFalsy(concreteLeftType)) {
                    return leftType;
                }

                // If the LHS evaluates to falsy, the Or expression will
                // always return the type of the right-hand side.
                if (!canBeTruthy(concreteLeftType)) {
                    return rightType;
                }

                concreteLeftType = removeFalsinessFromType(concreteLeftType);

                if (isNever(rightType)) {
                    return concreteLeftType;
                }
            }

            if (isNever(leftType) || isNever(rightType)) {
                return NeverType.createNever();
            }

            // The "in" and "not in" operators make use of the __contains__
            // magic method.
            if (operator === OperatorType.In || operator === OperatorType.NotIn) {
                type = mapSubtypesExpandTypeVars(
                    rightType,
                    /* conditionFilter */ undefined,
                    (rightSubtypeExpanded, rightSubtypeUnexpanded) => {
                        return mapSubtypesExpandTypeVars(
                            concreteLeftType,
                            getTypeCondition(rightSubtypeExpanded),
                            (leftSubtype) => {
                                if (isAnyOrUnknown(leftSubtype) || isAnyOrUnknown(rightSubtypeUnexpanded)) {
                                    return preserveUnknown(leftSubtype, rightSubtypeExpanded);
                                }

                                let returnType = getTypeOfMagicMethodReturn(
                                    rightSubtypeExpanded,
                                    [{ type: leftSubtype, isIncomplete: leftTypeResult.isIncomplete }],
                                    '__contains__',
                                    errorNode,
                                    /* inferenceContext */ undefined
                                );

                                if (!returnType) {
                                    // If __contains__ was not supported, fall back
                                    // on an iterable.
                                    const iteratorType = getTypeOfIterator(
                                        { type: rightSubtypeExpanded, isIncomplete: rightTypeResult.isIncomplete },
                                        /* isAsync */ false,
                                        /* errorNode */ undefined
                                    )?.type;

                                    if (iteratorType && assignType(iteratorType, leftSubtype)) {
                                        returnType = getBuiltInObject(errorNode, 'bool');
                                    }
                                }

                                if (!returnType) {
                                    diag.addMessage(
                                        Localizer.Diagnostic.typeNotSupportBinaryOperator().format({
                                            operator: ParseTreeUtils.printOperator(operator),
                                            leftType: printType(leftSubtype),
                                            rightType: printType(rightSubtypeExpanded),
                                        })
                                    );
                                }

                                return returnType;
                            }
                        );
                    }
                );

                // Assume that a bool is returned even if the type is unknown
                if (type && !isNever(type)) {
                    type = getBuiltInObject(errorNode, 'bool');
                }
            } else {
                type = mapSubtypesExpandTypeVars(
                    concreteLeftType,
                    /* conditionFilter */ undefined,
                    (leftSubtypeExpanded, leftSubtypeUnexpanded) => {
                        return mapSubtypesExpandTypeVars(
                            rightType,
                            getTypeCondition(leftSubtypeExpanded),
                            (rightSubtypeExpanded, rightSubtypeUnexpanded) => {
                                // If the operator is an AND or OR, we need to combine the two types.
                                if (operator === OperatorType.And || operator === OperatorType.Or) {
                                    return combineTypes([leftSubtypeUnexpanded, rightSubtypeUnexpanded]);
                                }
                                // The other boolean operators always return a bool value.
                                return getBuiltInObject(errorNode, 'bool');
                            }
                        );
                    }
                );
            }
        } else if (binaryOperatorMap[operator]) {
            if (isNever(leftType) || isNever(rightType)) {
                return NeverType.createNever();
            }

            // Handle certain operations on certain homogenous literal types
            // using special-case math. For example, Literal[1, 2] + Literal[3, 4]
            // should result in Literal[4, 5, 6].
            if (options.isLiteralMathAllowed) {
                const leftLiteralClassName = getLiteralTypeClassName(leftType);
                if (leftLiteralClassName && !getTypeCondition(leftType)) {
                    const rightLiteralClassName = getLiteralTypeClassName(rightType);

                    if (
                        leftLiteralClassName === rightLiteralClassName &&
                        !getTypeCondition(rightType) &&
                        getUnionSubtypeCount(leftType) * getUnionSubtypeCount(rightType) < maxLiteralMathSubtypeCount
                    ) {
                        if (leftLiteralClassName === 'str' || leftLiteralClassName === 'bytes') {
                            if (operator === OperatorType.Add) {
                                type = mapSubtypes(leftType, (leftSubtype) => {
                                    return mapSubtypes(rightType, (rightSubtype) => {
                                        const leftClassSubtype = leftSubtype as ClassType;
                                        const rightClassSubtype = rightSubtype as ClassType;

                                        return ClassType.cloneWithLiteral(
                                            leftClassSubtype,
                                            ((leftClassSubtype.literalValue as string) +
                                                rightClassSubtype.literalValue) as string
                                        );
                                    });
                                });
                            }
                        } else if (leftLiteralClassName === 'int') {
                            if (
                                operator === OperatorType.Add ||
                                operator === OperatorType.Subtract ||
                                operator === OperatorType.Multiply ||
                                operator === OperatorType.FloorDivide ||
                                operator === OperatorType.Mod
                            ) {
                                let isValidResult = true;

                                type = mapSubtypes(leftType, (leftSubtype) => {
                                    return mapSubtypes(rightType, (rightSubtype) => {
                                        try {
                                            const leftClassSubtype = leftSubtype as ClassType;
                                            const rightClassSubtype = rightSubtype as ClassType;
                                            const leftLiteralValue = BigInt(
                                                leftClassSubtype.literalValue as number | bigint
                                            );
                                            const rightLiteralValue = BigInt(
                                                rightClassSubtype.literalValue as number | bigint
                                            );

                                            let newValue: number | bigint | undefined;
                                            if (operator === OperatorType.Add) {
                                                newValue = leftLiteralValue + rightLiteralValue;
                                            } else if (operator === OperatorType.Subtract) {
                                                newValue = leftLiteralValue - rightLiteralValue;
                                            } else if (operator === OperatorType.Multiply) {
                                                newValue = leftLiteralValue * rightLiteralValue;
                                            } else if (operator === OperatorType.FloorDivide) {
                                                if (rightLiteralValue !== BigInt(0)) {
                                                    newValue = leftLiteralValue / rightLiteralValue;
                                                }
                                            } else if (operator === OperatorType.Mod) {
                                                if (rightLiteralValue !== BigInt(0)) {
                                                    newValue = leftLiteralValue % rightLiteralValue;
                                                }
                                            }

                                            if (newValue === undefined) {
                                                isValidResult = false;
                                                return undefined;
                                            } else if (typeof newValue === 'number' && isNaN(newValue)) {
                                                isValidResult = false;
                                                return undefined;
                                            } else {
                                                // Convert back to a simple number if it fits. Leave as a bigint
                                                // if it doesn't.
                                                if (
                                                    newValue >= Number.MIN_SAFE_INTEGER &&
                                                    newValue <= Number.MAX_SAFE_INTEGER
                                                ) {
                                                    newValue = Number(newValue);
                                                }

                                                return ClassType.cloneWithLiteral(leftClassSubtype, newValue);
                                            }
                                        } catch {
                                            isValidResult = false;
                                            return undefined;
                                        }
                                    });
                                });

                                if (!isValidResult) {
                                    type = undefined;
                                }
                            }
                        }
                    }
                }
            }

            if (!type) {
                type = mapSubtypesExpandTypeVars(
                    leftType,
                    /* conditionFilter */ undefined,
                    (leftSubtypeExpanded, leftSubtypeUnexpanded) => {
                        return mapSubtypesExpandTypeVars(
                            rightType,
                            getTypeCondition(leftSubtypeExpanded),
                            (rightSubtypeExpanded, rightSubtypeUnexpanded) => {
                                if (isAnyOrUnknown(leftSubtypeUnexpanded) || isAnyOrUnknown(rightSubtypeUnexpanded)) {
                                    return preserveUnknown(leftSubtypeUnexpanded, rightSubtypeUnexpanded);
                                }

                                // Special-case __add__ for tuples when the types for both tuples are known.
                                if (
                                    options.isTupleAddAllowed &&
                                    operator === OperatorType.Add &&
                                    isClassInstance(leftSubtypeExpanded) &&
                                    isTupleClass(leftSubtypeExpanded) &&
                                    leftSubtypeExpanded.tupleTypeArguments &&
                                    !isUnboundedTupleClass(leftSubtypeExpanded) &&
                                    isClassInstance(rightSubtypeExpanded) &&
                                    isTupleClass(rightSubtypeExpanded) &&
                                    rightSubtypeExpanded.tupleTypeArguments &&
                                    !isUnboundedTupleClass(rightSubtypeExpanded) &&
                                    tupleClassType &&
                                    isInstantiableClass(tupleClassType)
                                ) {
                                    return ClassType.cloneAsInstance(
                                        specializeTupleClass(tupleClassType, [
                                            ...leftSubtypeExpanded.tupleTypeArguments,
                                            ...rightSubtypeExpanded.tupleTypeArguments,
                                        ])
                                    );
                                }

                                const magicMethodName = binaryOperatorMap[operator][0];
                                let resultType = getTypeOfMagicMethodReturn(
                                    convertFunctionToObject(leftSubtypeUnexpanded),
                                    [{ type: rightSubtypeUnexpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                    magicMethodName,
                                    errorNode,
                                    inferenceContext
                                );

                                if (!resultType && leftSubtypeUnexpanded !== leftSubtypeExpanded) {
                                    // Try the expanded left type.
                                    resultType = getTypeOfMagicMethodReturn(
                                        convertFunctionToObject(leftSubtypeExpanded),
                                        [{ type: rightSubtypeUnexpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                        magicMethodName,
                                        errorNode,
                                        inferenceContext
                                    );
                                }

                                if (!resultType && rightSubtypeUnexpanded !== rightSubtypeExpanded) {
                                    // Try the expanded left and right type.
                                    resultType = getTypeOfMagicMethodReturn(
                                        convertFunctionToObject(leftSubtypeExpanded),
                                        [{ type: rightSubtypeExpanded, isIncomplete: rightTypeResult.isIncomplete }],
                                        magicMethodName,
                                        errorNode,
                                        inferenceContext
                                    );
                                }

                                if (!resultType) {
                                    // Try the alternate form (swapping right and left).
                                    const altMagicMethodName = binaryOperatorMap[operator][1];
                                    resultType = getTypeOfMagicMethodReturn(
                                        convertFunctionToObject(rightSubtypeUnexpanded),
                                        [{ type: leftSubtypeUnexpanded, isIncomplete: leftTypeResult.isIncomplete }],
                                        altMagicMethodName,
                                        errorNode,
                                        inferenceContext
                                    );

                                    if (!resultType && rightSubtypeUnexpanded !== rightSubtypeExpanded) {
                                        // Try the expanded right type.
                                        resultType = getTypeOfMagicMethodReturn(
                                            convertFunctionToObject(rightSubtypeExpanded),
                                            [
                                                {
                                                    type: leftSubtypeUnexpanded,
                                                    isIncomplete: leftTypeResult.isIncomplete,
                                                },
                                            ],
                                            altMagicMethodName,
                                            errorNode,
                                            inferenceContext
                                        );
                                    }

                                    if (!resultType && leftSubtypeUnexpanded !== leftSubtypeExpanded) {
                                        // Try the expanded right and left type.
                                        resultType = getTypeOfMagicMethodReturn(
                                            convertFunctionToObject(rightSubtypeExpanded),
                                            [{ type: leftSubtypeExpanded, isIncomplete: leftTypeResult.isIncomplete }],
                                            altMagicMethodName,
                                            errorNode,
                                            inferenceContext
                                        );
                                    }
                                }

                                if (!resultType) {
                                    if (inferenceContext) {
                                        diag.addMessage(
                                            Localizer.Diagnostic.typeNotSupportBinaryOperatorBidirectional().format({
                                                operator: ParseTreeUtils.printOperator(operator),
                                                leftType: printType(leftSubtypeExpanded),
                                                rightType: printType(rightSubtypeExpanded),
                                                expectedType: printType(inferenceContext.expectedType),
                                            })
                                        );
                                    } else {
                                        diag.addMessage(
                                            Localizer.Diagnostic.typeNotSupportBinaryOperator().format({
                                                operator: ParseTreeUtils.printOperator(operator),
                                                leftType: printType(leftSubtypeExpanded),
                                                rightType: printType(rightSubtypeExpanded),
                                            })
                                        );
                                    }
                                }
                                return resultType;
                            }
                        );
                    }
                );
            }
        }

        return type && isNever(type) ? undefined : type;
    }

    function getTypeOfMagicMethodReturn(
        objType: Type,
        args: TypeResult[],
        magicMethodName: string,
        errorNode: ExpressionNode,
        inferenceContext: InferenceContext | undefined
    ): Type | undefined {
        let magicMethodSupported = true;

        // Create a helper lambda for object subtypes.
        const handleSubtype = (subtype: ClassType | TypeVarType) => {
            let magicMethodType: Type | undefined;
            const concreteSubtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isClassInstance(concreteSubtype)) {
                magicMethodType = getTypeOfObjectMember(
                    errorNode,
                    concreteSubtype,
                    magicMethodName,
                    /* usage */ undefined,
                    /* diag */ undefined,
                    MemberAccessFlags.SkipAttributeAccessOverride | MemberAccessFlags.AccessClassMembersOnly,
                    subtype
                )?.type;
            } else if (isInstantiableClass(concreteSubtype)) {
                magicMethodType = getTypeOfClassMember(
                    errorNode,
                    concreteSubtype,
                    magicMethodName,
                    /* usage */ undefined,
                    /* diag */ undefined,
                    MemberAccessFlags.SkipAttributeAccessOverride | MemberAccessFlags.ConsiderMetaclassOnly
                )?.type;
            }

            if (magicMethodType) {
                const functionArgs: FunctionArgument[] = args.map((arg) => {
                    return {
                        argumentCategory: ArgumentCategory.Simple,
                        typeResult: arg,
                    };
                });

                let callResult: CallResult | undefined;

                useSpeculativeMode(errorNode, () => {
                    callResult = validateCallArguments(
                        errorNode,
                        functionArgs,
                        { type: magicMethodType! },
                        /* typeVarContext */ undefined,
                        /* skipUnknownArgCheck */ true,
                        inferenceContext
                    );
                });

                // If there were errors with the expected type, try
                // to evaluate without the expected type.
                if (callResult!.argumentErrors && inferenceContext) {
                    useSpeculativeMode(errorNode, () => {
                        callResult = validateCallArguments(
                            errorNode,
                            functionArgs,
                            { type: magicMethodType! },
                            /* typeVarContext */ undefined,
                            /* skipUnknownArgCheck */ true
                        );
                    });
                }

                if (callResult!.argumentErrors) {
                    magicMethodSupported = false;
                }

                return callResult!.returnType;
            }

            magicMethodSupported = false;
            return undefined;
        };

        const returnType = mapSubtypes(objType, (subtype) => {
            if (isAnyOrUnknown(subtype)) {
                return subtype;
            }

            if (isClassInstance(subtype) || isInstantiableClass(subtype) || isTypeVar(subtype)) {
                return handleSubtype(subtype);
            }

            if (isNoneInstance(subtype)) {
                if (objectType && isClassInstance(objectType)) {
                    // Use 'object' for 'None'.
                    return handleSubtype(objectType);
                }
            }

            if (isNoneTypeClass(subtype)) {
                if (typeClassType && isInstantiableClass(typeClassType)) {
                    // Use 'type' for 'type[None]'.
                    return handleSubtype(ClassType.cloneAsInstance(typeClassType));
                }
            }

            magicMethodSupported = false;
            return undefined;
        });

        if (!magicMethodSupported) {
            return undefined;
        }

        return returnType;
    }

    // All functions in Python derive from object, so they inherit all
    // of the capabilities of an object. This function converts a function
    // to an object instance.
    function convertFunctionToObject(type: Type) {
        if (isFunction(type) || isOverloadedFunction(type)) {
            if (objectType) {
                return objectType;
            }
        }

        return type;
    }

    function getTypeOfDictionary(node: DictionaryNode, inferenceContext: InferenceContext | undefined): TypeResult {
        // If the expected type is a union, analyze for each of the subtypes
        // to find one that matches.
        let effectiveExpectedType = inferenceContext?.expectedType;

        if (inferenceContext && isUnion(inferenceContext.expectedType)) {
            let matchingSubtype: Type | undefined;
            let matchingSubtypeResult: TypeResult | undefined;

            doForEachSubtype(inferenceContext.expectedType, (subtype) => {
                // Use shortcut if we've already found a match.
                if (matchingSubtypeResult && !matchingSubtypeResult.typeErrors) {
                    return;
                }

                const subtypeResult = useSpeculativeMode(node, () => {
                    return getTypeOfDictionaryWithContext(
                        node,
                        makeInferenceContext(subtype, inferenceContext?.typeVarContext)
                    );
                });

                if (subtypeResult && assignType(subtype, subtypeResult.type)) {
                    // If this is the first result we're seeing or it's the first result
                    // without errors, select it as the match.
                    if (!matchingSubtypeResult || (matchingSubtypeResult.typeErrors && !subtypeResult.typeErrors)) {
                        matchingSubtype = subtype;
                        matchingSubtypeResult = subtypeResult;
                    }
                }
            });

            effectiveExpectedType = matchingSubtype;
        }

        let expectedTypeDiagAddendum = undefined;
        if (effectiveExpectedType) {
            expectedTypeDiagAddendum = new DiagnosticAddendum();
            const result = getTypeOfDictionaryWithContext(
                node,
                makeInferenceContext(effectiveExpectedType, inferenceContext?.typeVarContext),
                expectedTypeDiagAddendum
            );
            if (result) {
                return result;
            }
        }

        const result = getTypeOfDictionaryInferred(node, /* hasExpectedType */ !!inferenceContext);
        return { ...result, expectedTypeDiagAddendum };
    }

    function getTypeOfDictionaryWithContext(
        node: DictionaryNode,
        inferenceContext: InferenceContext,
        expectedDiagAddendum?: DiagnosticAddendum
    ): TypeResult | undefined {
        inferenceContext.expectedType = transformPossibleRecursiveTypeAlias(inferenceContext.expectedType);
        const concreteExpectedType = makeTopLevelTypeVarsConcrete(inferenceContext.expectedType);

        if (!isClassInstance(concreteExpectedType)) {
            return undefined;
        }

        const keyTypes: TypeResultWithNode[] = [];
        const valueTypes: TypeResultWithNode[] = [];
        let isIncomplete = false;

        // Handle TypedDict's as a special case.
        if (ClassType.isTypedDictClass(concreteExpectedType)) {
            const expectedTypedDictEntries = getTypedDictMembersForClass(evaluatorInterface, concreteExpectedType);

            // Infer the key and value types if possible.
            if (
                getKeyAndValueTypesFromDictionary(
                    node,
                    keyTypes,
                    valueTypes,
                    /* forceStrictInference */ true,
                    /* expectedKeyType */ undefined,
                    /* expectedValueType */ undefined,
                    expectedTypedDictEntries,
                    expectedDiagAddendum
                )
            ) {
                isIncomplete = true;
            }

            if (ClassType.isTypedDictClass(concreteExpectedType)) {
                const resultTypedDict = assignToTypedDict(
                    evaluatorInterface,
                    concreteExpectedType,
                    keyTypes,
                    valueTypes,
                    // Don't overwrite existing expectedDiagAddendum messages if they were
                    // already provided by getKeyValueTypesFromDictionary.
                    expectedDiagAddendum?.isEmpty() ? expectedDiagAddendum : undefined
                );
                if (resultTypedDict) {
                    return {
                        type: resultTypedDict,
                        isIncomplete,
                    };
                }
            }

            return undefined;
        }

        const builtInDict = getBuiltInObject(node, 'dict');
        if (!isClassInstance(builtInDict)) {
            return undefined;
        }

        const dictTypeVarContext = new TypeVarContext(getTypeVarScopeId(builtInDict));
        if (
            !populateTypeVarContextBasedOnExpectedType(
                evaluatorInterface,
                builtInDict,
                inferenceContext.expectedType,
                dictTypeVarContext,
                getTypeVarScopesForNode(node)
            )
        ) {
            return undefined;
        }

        const specializedDict = applySolvedTypeVars(
            ClassType.cloneAsInstantiable(builtInDict),
            dictTypeVarContext
        ) as ClassType;
        if (!specializedDict.typeArguments || specializedDict.typeArguments.length !== 2) {
            return undefined;
        }

        const expectedKeyType = specializedDict.typeArguments[0];
        const expectedValueType = specializedDict.typeArguments[1];

        // Infer the key and value types if possible.
        if (
            getKeyAndValueTypesFromDictionary(
                node,
                keyTypes,
                valueTypes,
                /* forceStrictInference */ true,
                expectedKeyType,
                expectedValueType,
                undefined,
                expectedDiagAddendum
            )
        ) {
            isIncomplete = true;
        }

        // Dict and MutableMapping types have invariant value types, so they
        // cannot be narrowed further. Other super-types like Mapping, Collection,
        // and Iterable use covariant value types, so they can be narrowed.
        const isValueTypeInvariant =
            isClassInstance(inferenceContext.expectedType) &&
            (ClassType.isBuiltIn(inferenceContext.expectedType, 'dict') ||
                ClassType.isBuiltIn(inferenceContext.expectedType, 'MutableMapping'));

        const specializedKeyType = inferTypeArgFromExpectedType(
            expectedKeyType,
            keyTypes.map((result) => result.type),
            /* isNarrowable */ false
        );
        const specializedValueType = inferTypeArgFromExpectedType(
            expectedValueType,
            valueTypes.map((result) => result.type),
            /* isNarrowable */ !isValueTypeInvariant
        );
        if (!specializedKeyType || !specializedValueType) {
            return undefined;
        }

        const type = getBuiltInObject(node, 'dict', [specializedKeyType, specializedValueType]);
        return { type, isIncomplete };
    }

    // Attempts to infer the type of a dictionary statement. If hasExpectedType
    // is true, strict inference is used for the subexpressions.
    function getTypeOfDictionaryInferred(node: DictionaryNode, hasExpectedType: boolean): TypeResult {
        const fallbackType = hasExpectedType ? AnyType.create() : UnknownType.create();
        let keyType: Type = fallbackType;
        let valueType: Type = fallbackType;

        const keyTypeResults: TypeResultWithNode[] = [];
        const valueTypeResults: TypeResultWithNode[] = [];

        let isEmptyContainer = false;
        let isIncomplete = false;

        // Infer the key and value types if possible.
        if (
            getKeyAndValueTypesFromDictionary(
                node,
                keyTypeResults,
                valueTypeResults,
                /* forceStrictInference */ hasExpectedType
            )
        ) {
            isIncomplete = true;
        }

        // Strip any literal values.
        const keyTypes = keyTypeResults.map((t) => stripLiteralValue(t.type));
        const valueTypes = valueTypeResults.map((t) => stripLiteralValue(t.type));

        keyType = keyTypes.length > 0 ? combineTypes(keyTypes) : fallbackType;

        // If the value type differs and we're not using "strict inference mode",
        // we need to back off because we can't properly represent the mappings
        // between different keys and associated value types. If all the values
        // are the same type, we'll assume that all values in this dictionary should
        // be the same.
        if (valueTypes.length > 0) {
            if (AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.strictDictionaryInference || hasExpectedType) {
                valueType = combineTypes(valueTypes);
            } else {
                valueType = areTypesSame(valueTypes, { ignorePseudoGeneric: true }) ? valueTypes[0] : fallbackType;
            }
        } else {
            valueType = fallbackType;
            isEmptyContainer = true;
        }

        const dictClass = getBuiltInType(node, 'dict');
        const type = isInstantiableClass(dictClass)
            ? ClassType.cloneAsInstance(
                  ClassType.cloneForSpecialization(
                      dictClass,
                      [keyType, valueType],
                      /* isTypeArgumentExplicit */ true,
                      /* includeSubclasses */ undefined,
                      /* TupleTypeArguments */ undefined,
                      isEmptyContainer
                  )
              )
            : UnknownType.create();

        if (isIncomplete) {
            if (getContainerDepth(type) > maxInferredContainerDepth) {
                return { type: UnknownType.create() };
            }
        }

        return { type, isIncomplete };
    }

    function getKeyAndValueTypesFromDictionary(
        node: DictionaryNode,
        keyTypes: TypeResultWithNode[],
        valueTypes: TypeResultWithNode[],
        forceStrictInference: boolean,
        expectedKeyType?: Type,
        expectedValueType?: Type,
        expectedTypedDictEntries?: Map<string, TypedDictEntry>,
        expectedDiagAddendum?: DiagnosticAddendum
    ): boolean {
        let isIncomplete = false;

        // Infer the key and value types if possible.
        node.entries.forEach((entryNode, index) => {
            let addUnknown = true;

            if (entryNode.nodeType === ParseNodeType.DictionaryKeyEntry) {
                const keyTypeResult = getTypeOfExpression(
                    entryNode.keyExpression,
                    /* flags */ undefined,
                    makeInferenceContext(
                        expectedKeyType ?? (forceStrictInference ? NeverType.createNever() : undefined)
                    )
                );

                if (keyTypeResult.isIncomplete) {
                    isIncomplete = true;
                }

                const keyType = keyTypeResult.type;

                if (!keyTypeResult.isIncomplete && !keyTypeResult.typeErrors) {
                    verifySetEntryOrDictKeyIsHashable(entryNode.keyExpression, keyType, /* isDictKey */ true);
                }

                if (expectedDiagAddendum && keyTypeResult.expectedTypeDiagAddendum) {
                    expectedDiagAddendum.addAddendum(keyTypeResult.expectedTypeDiagAddendum);
                }

                let valueTypeResult: TypeResult;

                if (
                    expectedTypedDictEntries &&
                    isClassInstance(keyType) &&
                    ClassType.isBuiltIn(keyType, 'str') &&
                    isLiteralType(keyType) &&
                    expectedTypedDictEntries.has(keyType.literalValue as string)
                ) {
                    const effectiveValueType = expectedTypedDictEntries.get(keyType.literalValue as string)!.valueType;
                    valueTypeResult = getTypeOfExpression(
                        entryNode.valueExpression,
                        /* flags */ undefined,
                        makeInferenceContext(effectiveValueType)
                    );
                } else {
                    const effectiveValueType =
                        expectedValueType ?? (forceStrictInference ? NeverType.createNever() : undefined);
                    valueTypeResult = getTypeOfExpression(
                        entryNode.valueExpression,
                        /* flags */ undefined,
                        makeInferenceContext(effectiveValueType)
                    );
                }

                if (expectedDiagAddendum && valueTypeResult.expectedTypeDiagAddendum) {
                    expectedDiagAddendum.addAddendum(valueTypeResult.expectedTypeDiagAddendum);
                }

                const valueType = valueTypeResult.type;
                if (valueTypeResult.isIncomplete) {
                    isIncomplete = true;
                }

                if (forceStrictInference || index < maxEntriesToUseForInference) {
                    keyTypes.push({ node: entryNode.keyExpression, type: keyType });
                    valueTypes.push({ node: entryNode.valueExpression, type: valueType });
                }

                addUnknown = false;
            } else if (entryNode.nodeType === ParseNodeType.DictionaryExpandEntry) {
                // Verify that the type supports the `keys` and `__getitem__` methods.
                // This protocol is defined in the _typeshed stub. If we can't find
                // it there, fall back on typing.Mapping.
                let mappingType = getTypeshedType(node, 'SupportsKeysAndGetItem');
                if (!mappingType) {
                    mappingType = getTypingType(node, 'Mapping');
                }

                let expectedType: Type | undefined;
                if (expectedKeyType && expectedValueType) {
                    if (mappingType && isInstantiableClass(mappingType)) {
                        expectedType = ClassType.cloneAsInstance(
                            ClassType.cloneForSpecialization(
                                mappingType,
                                [expectedKeyType, expectedValueType],
                                /* isTypeArgumentExplicit */ true
                            )
                        );
                    }
                }

                const unexpandedTypeResult = getTypeOfExpression(
                    entryNode.expandExpression,
                    /* flags */ undefined,
                    makeInferenceContext(expectedType)
                );

                if (unexpandedTypeResult.isIncomplete) {
                    isIncomplete = true;
                }

                const unexpandedType = unexpandedTypeResult.type;
                if (isAnyOrUnknown(unexpandedType)) {
                    addUnknown = false;
                } else if (isClassInstance(unexpandedType) && ClassType.isTypedDictClass(unexpandedType)) {
                    // Handle dictionary expansion for a TypedDict.
                    if (strClassType && isInstantiableClass(strClassType)) {
                        const strObject = ClassType.cloneAsInstance(strClassType);
                        const tdEntries = getTypedDictMembersForClass(
                            evaluatorInterface,
                            unexpandedType,
                            /* allowNarrowed */ true
                        );

                        tdEntries.forEach((entry, name) => {
                            if (entry.isRequired || entry.isProvided) {
                                keyTypes.push({ node: entryNode, type: ClassType.cloneWithLiteral(strObject, name) });
                                valueTypes.push({ node: entryNode, type: entry.valueType });
                            }
                        });

                        addUnknown = false;
                    }
                } else {
                    if (mappingType && isInstantiableClass(mappingType)) {
                        const mappingTypeVarContext = new TypeVarContext(getTypeVarScopeId(mappingType));

                        // Self-specialize the class.
                        mappingType = ClassType.cloneForSpecialization(
                            mappingType,
                            mappingType.details.typeParameters,
                            /* isTypeArgumentExplicit */ true
                        );

                        if (
                            assignType(
                                ClassType.cloneAsInstance(mappingType),
                                unexpandedType,
                                /* diag */ undefined,
                                mappingTypeVarContext,
                                /* srcTypeVarContext */ undefined,
                                AssignTypeFlags.RetainLiteralsForTypeVar
                            )
                        ) {
                            const specializedMapping = applySolvedTypeVars(
                                mappingType,
                                mappingTypeVarContext
                            ) as ClassType;
                            const typeArgs = specializedMapping.typeArguments;
                            if (typeArgs && typeArgs.length >= 2) {
                                if (forceStrictInference || index < maxEntriesToUseForInference) {
                                    keyTypes.push({ node: entryNode, type: typeArgs[0] });
                                    valueTypes.push({ node: entryNode, type: typeArgs[1] });
                                }
                                addUnknown = false;
                            }
                        } else {
                            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.dictUnpackIsNotMapping(),
                                entryNode
                            );
                        }
                    }
                }
            } else if (entryNode.nodeType === ParseNodeType.ListComprehension) {
                const dictEntryTypeResult = getElementTypeFromListComprehension(
                    entryNode,
                    expectedValueType,
                    expectedKeyType
                );
                const dictEntryType = dictEntryTypeResult.type;
                if (dictEntryTypeResult.isIncomplete) {
                    isIncomplete = true;
                }

                // The result should be a tuple.
                if (isClassInstance(dictEntryType) && isTupleClass(dictEntryType)) {
                    const typeArgs = dictEntryType.tupleTypeArguments?.map((t) => t.type);
                    if (typeArgs && typeArgs.length === 2) {
                        if (forceStrictInference || index < maxEntriesToUseForInference) {
                            keyTypes.push({ node: entryNode, type: typeArgs[0] });
                            valueTypes.push({ node: entryNode, type: typeArgs[1] });
                        }
                        addUnknown = false;
                    }
                }
            }

            if (addUnknown) {
                if (forceStrictInference || index < maxEntriesToUseForInference) {
                    keyTypes.push({ node: entryNode, type: UnknownType.create() });
                    valueTypes.push({ node: entryNode, type: UnknownType.create() });
                }
            }
        });

        return isIncomplete;
    }

    function getTypeOfListOrSet(node: ListNode | SetNode, inferenceContext: InferenceContext | undefined): TypeResult {
        // If the expected type is a union, recursively call for each of the subtypes
        // to find one that matches.
        let effectiveExpectedType = inferenceContext?.expectedType;

        if (inferenceContext && isUnion(inferenceContext.expectedType)) {
            let matchingSubtype: Type | undefined;
            let matchingSubtypeResult: TypeResult | undefined;

            doForEachSubtype(inferenceContext.expectedType, (subtype) => {
                // Use shortcut if we've already found a match.
                if (matchingSubtypeResult && !matchingSubtypeResult.typeErrors) {
                    return;
                }

                const subtypeResult = useSpeculativeMode(node, () => {
                    return getTypeOfListOrSetWithContext(
                        node,
                        makeInferenceContext(subtype, inferenceContext?.typeVarContext)
                    );
                });

                if (subtypeResult && assignType(subtype, subtypeResult.type)) {
                    // If this is the first result we're seeing or it's the first result
                    // without errors, select it as the match.
                    if (!matchingSubtypeResult || (matchingSubtypeResult.typeErrors && !subtypeResult.typeErrors)) {
                        matchingSubtype = subtype;
                        matchingSubtypeResult = subtypeResult;
                    }
                }
            });

            effectiveExpectedType = matchingSubtype;
        }

        let expectedTypeDiagAddendum: DiagnosticAddendum | undefined;
        if (effectiveExpectedType) {
            const result = getTypeOfListOrSetWithContext(
                node,
                makeInferenceContext(effectiveExpectedType, inferenceContext?.typeVarContext)
            );
            if (result && !result.typeErrors) {
                return result;
            }

            expectedTypeDiagAddendum = result?.expectedTypeDiagAddendum;
        }

        const typeResult = getTypeOfListOrSetInferred(node, /* hasExpectedType */ inferenceContext !== undefined);
        return { ...typeResult, expectedTypeDiagAddendum };
    }

    // Attempts to determine the type of a list or set statement based on an expected type.
    // Returns undefined if that type cannot be honored.
    function getTypeOfListOrSetWithContext(
        node: ListNode | SetNode,
        inferenceContext: InferenceContext
    ): TypeResult | undefined {
        const builtInClassName = node.nodeType === ParseNodeType.List ? 'list' : 'set';
        inferenceContext.expectedType = transformPossibleRecursiveTypeAlias(inferenceContext.expectedType);

        let isIncomplete = false;
        let typeErrors = false;
        const verifyHashable = node.nodeType === ParseNodeType.Set;

        if (!isClassInstance(inferenceContext.expectedType)) {
            return undefined;
        }

        const builtInListOrSet = getBuiltInObject(node, builtInClassName);
        if (!isClassInstance(builtInListOrSet)) {
            return undefined;
        }

        const typeVarContext = new TypeVarContext(getTypeVarScopeId(builtInListOrSet));
        if (
            !populateTypeVarContextBasedOnExpectedType(
                evaluatorInterface,
                builtInListOrSet,
                inferenceContext.expectedType,
                typeVarContext,
                getTypeVarScopesForNode(node)
            )
        ) {
            return undefined;
        }

        const specializedListOrSet = applySolvedTypeVars(
            ClassType.cloneAsInstantiable(builtInListOrSet),
            typeVarContext
        ) as ClassType;
        if (!specializedListOrSet.typeArguments || specializedListOrSet.typeArguments.length !== 1) {
            return undefined;
        }

        const expectedEntryType = specializedListOrSet.typeArguments[0];

        const entryTypes: Type[] = [];
        const expectedTypeDiagAddendum = new DiagnosticAddendum();
        node.entries.forEach((entry) => {
            let entryTypeResult: TypeResult;

            if (entry.nodeType === ParseNodeType.ListComprehension) {
                entryTypeResult = getElementTypeFromListComprehension(entry, expectedEntryType);
            } else {
                entryTypeResult = getTypeOfExpression(
                    entry,
                    /* flags */ undefined,
                    makeInferenceContext(expectedEntryType, inferenceContext?.typeVarContext)
                );
            }

            entryTypes.push(entryTypeResult.type);

            if (entryTypeResult.isIncomplete) {
                isIncomplete = true;
            }

            if (entryTypeResult.typeErrors) {
                typeErrors = true;
            }

            if (entryTypeResult.expectedTypeDiagAddendum) {
                expectedTypeDiagAddendum.addAddendum(entryTypeResult.expectedTypeDiagAddendum);
            }

            if (verifyHashable && !entryTypeResult.isIncomplete && !entryTypeResult.typeErrors) {
                verifySetEntryOrDictKeyIsHashable(entry, entryTypeResult.type, /* isDictKey */ false);
            }
        });

        const isExpectedTypeListOrSet =
            isClassInstance(inferenceContext.expectedType) &&
            ClassType.isBuiltIn(inferenceContext.expectedType, builtInClassName);
        const specializedEntryType = inferTypeArgFromExpectedType(
            expectedEntryType,
            entryTypes,
            /* isNarrowable */ !isExpectedTypeListOrSet
        );
        if (!specializedEntryType) {
            return { type: UnknownType.create(), isIncomplete, typeErrors: true, expectedTypeDiagAddendum };
        }

        const type = getBuiltInObject(node, builtInClassName, [specializedEntryType]);
        return { type, isIncomplete, typeErrors, expectedTypeDiagAddendum };
    }

    // Attempts to infer the type of a list or set statement with no "expected type".
    function getTypeOfListOrSetInferred(node: ListNode | SetNode, hasExpectedType: boolean): TypeResult {
        const builtInClassName = node.nodeType === ParseNodeType.List ? 'list' : 'set';
        const verifyHashable = node.nodeType === ParseNodeType.Set;
        let isEmptyContainer = false;
        let isIncomplete = false;
        let typeErrors = false;

        let entryTypes: Type[] = [];
        node.entries.forEach((entry, index) => {
            let entryTypeResult: TypeResult;

            if (entry.nodeType === ParseNodeType.ListComprehension) {
                entryTypeResult = getElementTypeFromListComprehension(entry);
            } else {
                entryTypeResult = getTypeOfExpression(entry);
            }

            if (entryTypeResult.isIncomplete) {
                isIncomplete = true;
            }

            if (entryTypeResult.typeErrors) {
                typeErrors = true;
            }

            if (index < maxEntriesToUseForInference) {
                entryTypes.push(entryTypeResult.type);
            }

            if (verifyHashable && !entryTypeResult.isIncomplete && !entryTypeResult.typeErrors) {
                verifySetEntryOrDictKeyIsHashable(entry, entryTypeResult.type, /* isDictKey */ false);
            }
        });

        entryTypes = entryTypes.map((t) => stripLiteralValue(t));

        let inferredEntryType: Type = hasExpectedType ? AnyType.create() : UnknownType.create();
        if (entryTypes.length > 0) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            // If there was an expected type or we're using strict list inference,
            // combine the types into a union.
            if (
                (builtInClassName === 'list' && fileInfo.diagnosticRuleSet.strictListInference) ||
                (builtInClassName === 'set' && fileInfo.diagnosticRuleSet.strictSetInference) ||
                hasExpectedType
            ) {
                inferredEntryType = combineTypes(entryTypes, maxSubtypesForInferredType);
            } else {
                // Is the list or set homogeneous? If so, use stricter rules. Otherwise relax the rules.
                inferredEntryType = areTypesSame(entryTypes, { ignorePseudoGeneric: true })
                    ? entryTypes[0]
                    : inferredEntryType;
            }
        } else {
            isEmptyContainer = true;
        }

        const listOrSetClass = getBuiltInType(node, builtInClassName);
        const type = isInstantiableClass(listOrSetClass)
            ? ClassType.cloneAsInstance(
                  ClassType.cloneForSpecialization(
                      listOrSetClass,
                      [inferredEntryType],
                      /* isTypeArgumentExplicit */ true,
                      /* includeSubclasses */ undefined,
                      /* TupleTypeArguments */ undefined,
                      isEmptyContainer
                  )
              )
            : UnknownType.create();

        if (isIncomplete) {
            if (getContainerDepth(type) > maxInferredContainerDepth) {
                return { type: UnknownType.create() };
            }
        }

        return { type, isIncomplete, typeErrors };
    }

    function verifySetEntryOrDictKeyIsHashable(entry: ExpressionNode, type: Type, isDictKey: boolean) {
        // Verify that the type is hashable.
        if (!isTypeHashable(type)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(entry);
            const diag = new DiagnosticAddendum();
            diag.addMessage(Localizer.DiagnosticAddendum.unhashableType().format({ type: printType(type) }));

            const message = isDictKey
                ? Localizer.Diagnostic.unhashableDictKey()
                : Localizer.Diagnostic.unhashableSetEntry();

            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                message + diag.getString(),
                entry
            );
        }
    }

    function inferTypeArgFromExpectedType(
        expectedType: Type,
        entryTypes: Type[],
        isNarrowable: boolean
    ): Type | undefined {
        let targetTypeVar: TypeVarType;
        let useSynthesizedTypeVar = false;

        // If the expected type is a TypeVar, use it as a target to find
        // a common (narrowest) type among the entry types.
        if (isTypeVar(expectedType)) {
            if (expectedType.details.isParamSpec || expectedType.details.isVariadic) {
                return undefined;
            }

            targetTypeVar = expectedType;
        } else {
            // Synthesize a temporary bound type var. We will attempt to assign all list
            // entries to this type var, possibly narrowing the type in the process.
            targetTypeVar = TypeVarType.createInstance('__typeArg');
            targetTypeVar.details.isSynthesized = true;
            targetTypeVar.details.boundType = makeTopLevelTypeVarsConcrete(expectedType);

            // Use a dummy scope ID. It needs to be a non-empty string.
            targetTypeVar.scopeId = '__typeArgScopeId';
            useSynthesizedTypeVar = true;
        }

        // First, try to assign entries with their literal values stripped.
        // The only time we don't want to strip them is if the expected
        // type explicitly includes literals.
        let typeVarContext = new TypeVarContext(targetTypeVar.scopeId);

        if (useSynthesizedTypeVar) {
            typeVarContext.setTypeVarType(
                targetTypeVar,
                isNarrowable ? undefined : expectedType,
                /* narrowBoundNoLiterals */ undefined,
                expectedType
            );
        }

        if (
            entryTypes.every((entryType) =>
                assignType(targetTypeVar, stripLiteralValue(entryType), /* diag */ undefined, typeVarContext)
            )
        ) {
            return applySolvedTypeVars(targetTypeVar, typeVarContext);
        }

        // Allocate a fresh typeVarContext before we try again with literals not stripped.
        typeVarContext = new TypeVarContext(targetTypeVar.scopeId);

        if (useSynthesizedTypeVar) {
            typeVarContext.setTypeVarType(
                targetTypeVar,
                isNarrowable ? undefined : expectedType,
                /* narrowBoundNoLiterals */ undefined,
                expectedType
            );
        }

        if (
            entryTypes.every((entryType) => assignType(targetTypeVar!, entryType, /* diag */ undefined, typeVarContext))
        ) {
            return applySolvedTypeVars(targetTypeVar, typeVarContext);
        }

        return undefined;
    }

    function getTypeOfTernary(
        node: TernaryNode,
        flags: EvaluatorFlags,
        inferenceContext: InferenceContext | undefined
    ): TypeResult {
        getTypeOfExpression(node.testExpression);

        const typesToCombine: Type[] = [];
        let isIncomplete = false;
        let typeErrors = false;

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const constExprValue = evaluateStaticBoolExpression(
            node.testExpression,
            fileInfo.executionEnvironment,
            fileInfo.definedConstants
        );

        if (constExprValue !== false && isNodeReachable(node.ifExpression)) {
            const ifType = getTypeOfExpression(node.ifExpression, flags, inferenceContext);
            typesToCombine.push(ifType.type);
            if (ifType.isIncomplete) {
                isIncomplete = true;
            }
            if (ifType.typeErrors) {
                typeErrors = true;
            }
        }

        if (constExprValue !== true && isNodeReachable(node.elseExpression)) {
            const elseType = getTypeOfExpression(node.elseExpression, flags, inferenceContext);
            typesToCombine.push(elseType.type);
            if (elseType.isIncomplete) {
                isIncomplete = true;
            }
            if (elseType.typeErrors) {
                typeErrors = true;
            }
        }

        return { type: combineTypes(typesToCombine), isIncomplete, typeErrors };
    }

    function getTypeOfYield(node: YieldNode): TypeResult {
        let expectedYieldType: Type | undefined;
        let sentType: Type | undefined;
        let isIncomplete = false;

        const enclosingFunction = ParseTreeUtils.getEnclosingFunction(node);
        if (enclosingFunction) {
            const functionTypeInfo = getTypeOfFunction(enclosingFunction);
            if (functionTypeInfo) {
                const returnType = FunctionType.getSpecializedReturnType(functionTypeInfo.functionType);
                if (returnType) {
                    expectedYieldType = getGeneratorYieldType(returnType, !!enclosingFunction.isAsync);

                    const generatorTypeArgs = getGeneratorTypeArgs(returnType);
                    if (generatorTypeArgs && generatorTypeArgs.length >= 2) {
                        sentType = generatorTypeArgs[1];
                    }
                }
            }
        }

        if (node.expression) {
            const exprResult = getTypeOfExpression(
                node.expression,
                /* flags */ undefined,
                makeInferenceContext(expectedYieldType)
            );
            if (exprResult.isIncomplete) {
                isIncomplete = true;
            }
        }

        return { type: sentType || UnknownType.create(), isIncomplete };
    }

    function getTypeOfYieldFrom(node: YieldFromNode): TypeResult {
        const yieldFromTypeResult = getTypeOfExpression(node.expression);
        const yieldFromType = yieldFromTypeResult.type;
        let generatorTypeArgs = getGeneratorTypeArgs(yieldFromType);

        let returnedType: Type | undefined;

        // Is the expression a Generator type?
        if (generatorTypeArgs) {
            returnedType = generatorTypeArgs.length >= 2 ? generatorTypeArgs[2] : UnknownType.create();
        } else if (isClassInstance(yieldFromType) && ClassType.isBuiltIn(yieldFromType, 'Coroutine')) {
            // Handle old-style (pre-await) Coroutines as a special case.
            returnedType = UnknownType.create();
        } else {
            const iterableType =
                getTypeOfIterable(yieldFromTypeResult, /* isAsync */ false, node)?.type ?? UnknownType.create();

            // Does the iterable return a Generator?
            generatorTypeArgs = getGeneratorTypeArgs(iterableType);
            if (generatorTypeArgs) {
                returnedType = generatorTypeArgs.length >= 2 ? generatorTypeArgs[2] : UnknownType.create();
            }
        }

        return { type: returnedType || UnknownType.create() };
    }

    function getTypeOfLambda(node: LambdaNode, inferenceContext: InferenceContext | undefined): TypeResult {
        let isIncomplete = !!inferenceContext?.isTypeIncomplete;
        const functionType = FunctionType.createInstance('', '', '', FunctionTypeFlags.PartiallyEvaluated);
        functionType.details.typeVarScopeId = getScopeIdForNode(node);

        // Pre-cache the incomplete function type in case the evaluation of the
        // lambda depends on itself.
        writeTypeCache(node, { type: functionType, isIncomplete: true }, EvaluatorFlags.None);

        let expectedFunctionTypes: FunctionType[] = [];
        if (inferenceContext) {
            mapSubtypes(inferenceContext.expectedType, (subtype) => {
                if (isFunction(subtype)) {
                    expectedFunctionTypes.push(subtype);
                }

                if (isClassInstance(subtype)) {
                    const boundMethod = getBoundMethod(subtype, '__call__');
                    if (boundMethod && isFunction(boundMethod)) {
                        expectedFunctionTypes.push(boundMethod as FunctionType);
                    }
                }

                return undefined;
            });

            // Determine the minimum number of parameters that are required to
            // satisfy the lambda.
            const minLambdaParamCount = node.parameters.filter(
                (param) => param.category === ParameterCategory.Simple && param.defaultValue === undefined
            ).length;
            const maxLambdaParamCount = node.parameters.filter(
                (param) => param.category === ParameterCategory.Simple
            ).length;

            // Remove any expected subtypes that don't satisfy the minimum
            // parameter count requirement.
            expectedFunctionTypes = expectedFunctionTypes.filter((functionType) => {
                const functionParamCount = functionType.details.parameters.filter(
                    (param) => !!param.name && !param.hasDefault
                ).length;
                const hasVarArgs = functionType.details.parameters.some(
                    (param) => !!param.name && param.category !== ParameterCategory.Simple
                );
                return (
                    hasVarArgs ||
                    (functionParamCount >= minLambdaParamCount && functionParamCount <= maxLambdaParamCount)
                );
            });
        }

        // For now, use only the first expected type.
        const expectedFunctionType = expectedFunctionTypes.length > 0 ? expectedFunctionTypes[0] : undefined;
        let paramsArePositionOnly = true;

        node.parameters.forEach((param, index) => {
            let paramType: Type = UnknownType.create();
            if (expectedFunctionType && index < expectedFunctionType.details.parameters.length) {
                paramType = FunctionType.getEffectiveParameterType(expectedFunctionType, index);
            }

            if (param.name) {
                writeTypeCache(
                    param.name,
                    { type: transformVariadicParamType(node, param.category, paramType) },
                    EvaluatorFlags.None
                );
            }

            if (param.defaultValue) {
                // Evaluate the default value if it's present.
                getTypeOfExpression(param.defaultValue, EvaluatorFlags.ConvertEllipsisToAny);
            }

            // Determine whether we need to insert an implied position-only parameter.
            // This is needed when a function's parameters are named using the old-style
            // way of specifying position-only parameters.
            if (index >= 0) {
                let isImplicitPositionOnlyParam = false;

                if (param.category === ParameterCategory.Simple && param.name) {
                    if (isPrivateName(param.name.value)) {
                        isImplicitPositionOnlyParam = true;
                    }
                } else {
                    paramsArePositionOnly = false;
                }

                if (
                    paramsArePositionOnly &&
                    !isImplicitPositionOnlyParam &&
                    functionType.details.parameters.length > 0
                ) {
                    // Insert an implicit "position-only parameter" separator.
                    FunctionType.addParameter(functionType, {
                        category: ParameterCategory.Simple,
                        type: UnknownType.create(),
                    });
                }

                if (!isImplicitPositionOnlyParam) {
                    paramsArePositionOnly = false;
                }
            }

            const functionParam: FunctionParameter = {
                category: param.category,
                name: param.name ? param.name.value : undefined,
                hasDefault: !!param.defaultValue,
                defaultValueExpression: param.defaultValue,
                hasDeclaredType: true,
                type: paramType,
            };

            FunctionType.addParameter(functionType, functionParam);
        });

        if (paramsArePositionOnly && functionType.details.parameters.length > 0) {
            // Insert an implicit "position-only parameter" separator.
            FunctionType.addParameter(functionType, {
                category: ParameterCategory.Simple,
                type: UnknownType.create(),
            });
        }

        const expectedReturnType = expectedFunctionType
            ? getFunctionEffectiveReturnType(expectedFunctionType)
            : undefined;

        // If we're speculatively evaluating the lambda, create another speculative
        // evaluation scope for the return expression and do not allow retention
        // of the cached types.
        const inferLambdaReturnType = () => {
            const returnTypeResult = getTypeOfExpression(
                node.expression,
                /* flags */ undefined,
                makeInferenceContext(expectedReturnType)
            );

            functionType.inferredReturnType = returnTypeResult.type;
            if (returnTypeResult.isIncomplete) {
                isIncomplete = true;
            }
        };

        if (speculativeTypeTracker.isSpeculative(node) || inferenceContext?.isTypeIncomplete) {
            // We need to set allowCacheRetention to false because we don't want to
            // cache the type of the lambda return expression because it depends on
            // the parameter types that we set above, and the speculative type cache
            // doesn't know about that context.
            useSpeculativeMode(
                node.expression,
                () => {
                    inferLambdaReturnType();
                },
                /* allowCacheRetention */ false
            );
        } else {
            inferLambdaReturnType();
        }

        // Mark the function type as no longer being evaluated.
        functionType.details.flags &= ~FunctionTypeFlags.PartiallyEvaluated;

        return { type: functionType, isIncomplete };
    }

    function getTypeOfListComprehension(node: ListComprehensionNode, inferenceContext?: InferenceContext): TypeResult {
        let isIncomplete = false;
        let typeErrors = false;

        let isAsync = node.forIfNodes.some((comp) => {
            return (
                (comp.nodeType === ParseNodeType.ListComprehensionFor && comp.isAsync) ||
                (comp.nodeType === ParseNodeType.ListComprehensionIf &&
                    comp.testExpression.nodeType === ParseNodeType.Await)
            );
        });
        let type: Type = UnknownType.create();

        if (node.expression.nodeType === ParseNodeType.Await) {
            isAsync = true;
        }

        let expectedElementType: Type | undefined;
        if (inferenceContext) {
            expectedElementType = getTypeOfIterator(
                { type: inferenceContext.expectedType },
                isAsync,
                /* errorNode */ undefined
            )?.type;
        }

        const elementTypeResult = getElementTypeFromListComprehension(node, expectedElementType);
        if (elementTypeResult.isIncomplete) {
            isIncomplete = true;
        }
        if (elementTypeResult.typeErrors) {
            typeErrors = true;
        }
        const elementType = elementTypeResult.type;

        // Handle the special case where a generator function (e.g. `(await x for x in y)`)
        // is expected to be an AsyncGenerator.
        if (
            !isAsync &&
            inferenceContext &&
            isClassInstance(inferenceContext.expectedType) &&
            ClassType.isBuiltIn(inferenceContext.expectedType, 'AsyncGenerator')
        ) {
            isAsync = true;
        }
        const builtInIteratorType = getTypingType(node, isAsync ? 'AsyncGenerator' : 'Generator');

        if (builtInIteratorType && isInstantiableClass(builtInIteratorType)) {
            type = ClassType.cloneAsInstance(
                ClassType.cloneForSpecialization(
                    builtInIteratorType,
                    isAsync
                        ? [elementType, NoneType.createInstance()]
                        : [elementType, NoneType.createInstance(), NoneType.createInstance()],
                    /* isTypeArgumentExplicit */ true
                )
            );
        }

        return { type, isIncomplete, typeErrors };
    }

    function reportPossibleUnknownAssignment(
        diagLevel: DiagnosticLevel,
        rule: string,
        target: NameNode,
        type: Type,
        errorNode: ExpressionNode,
        ignoreEmptyContainers: boolean
    ) {
        // Don't bother if the feature is disabled.
        if (diagLevel === 'none') {
            return;
        }

        const nameValue = target.value;

        // Sometimes variables contain an "unbound" type if they're
        // assigned only within conditional statements. Remove this
        // to avoid confusion.
        const simplifiedType = removeUnbound(type);

        if (isUnknown(simplifiedType)) {
            addDiagnostic(diagLevel, rule, Localizer.Diagnostic.typeUnknown().format({ name: nameValue }), errorNode);
        } else if (isPartlyUnknown(simplifiedType)) {
            // If ignoreEmptyContainers is true, don't report the problem for
            // empty containers (lists or dictionaries). We'll report the problem
            // only if the assigned value is used later.
            if (!ignoreEmptyContainers || !isClassInstance(type) || !type.isEmptyContainer) {
                const diagAddendum = new DiagnosticAddendum();
                diagAddendum.addMessage(
                    Localizer.DiagnosticAddendum.typeOfSymbol().format({
                        name: nameValue,
                        type: printType(simplifiedType, { expandTypeAlias: true }),
                    })
                );
                addDiagnostic(
                    diagLevel,
                    rule,
                    Localizer.Diagnostic.typePartiallyUnknown().format({ name: nameValue }) + diagAddendum.getString(),
                    errorNode
                );
            }
        }
    }

    function evaluateListComprehensionForIf(node: ListComprehensionForIfNode) {
        let isIncomplete = false;

        if (node.nodeType === ParseNodeType.ListComprehensionFor) {
            const iterableTypeResult = getTypeOfExpression(node.iterableExpression);
            if (iterableTypeResult.isIncomplete) {
                isIncomplete = true;
            }
            const iterableType = stripLiteralValue(iterableTypeResult.type);
            const itemTypeResult = getTypeOfIterator(
                { type: iterableType, isIncomplete: iterableTypeResult.isIncomplete },
                !!node.isAsync,
                node.iterableExpression
            ) ?? { type: UnknownType.create(), isIncomplete: iterableTypeResult.isIncomplete };

            const targetExpr = node.targetExpression;
            assignTypeToExpression(
                targetExpr,
                itemTypeResult.type,
                !!itemTypeResult.isIncomplete,
                node.iterableExpression
            );
        } else {
            assert(node.nodeType === ParseNodeType.ListComprehensionIf);

            // Evaluate the test expression to validate it and mark symbols
            // as referenced. Don't bother doing this if we're in speculative
            // mode because it doesn't affect the element type.
            if (!speculativeTypeTracker.isSpeculative(node.testExpression)) {
                getTypeOfExpression(node.testExpression);
            }
        }

        return isIncomplete;
    }

    // Returns the type of one entry returned by the list comprehension,
    // as opposed to the entire list.
    function getElementTypeFromListComprehension(
        node: ListComprehensionNode,
        expectedValueOrElementType?: Type,
        expectedKeyType?: Type
    ): TypeResult {
        let isIncomplete = false;
        let typeErrors = false;

        // "Execute" the list comprehensions from start to finish.
        for (const forIfNode of node.forIfNodes) {
            if (evaluateListComprehensionForIf(forIfNode)) {
                isIncomplete = true;
            }
        }

        let type: Type = UnknownType.create();
        if (node.expression.nodeType === ParseNodeType.DictionaryKeyEntry) {
            // Create a tuple with the key/value types.
            const keyTypeResult = getTypeOfExpression(
                node.expression.keyExpression,
                /* flags */ undefined,
                makeInferenceContext(expectedKeyType)
            );
            if (keyTypeResult.isIncomplete) {
                isIncomplete = true;
            }
            if (keyTypeResult.typeErrors) {
                typeErrors = true;
            }
            let keyType = keyTypeResult.type;
            if (!expectedKeyType || !containsLiteralType(expectedKeyType)) {
                keyType = stripLiteralValue(keyType);
            }

            const valueTypeResult = getTypeOfExpression(
                node.expression.valueExpression,
                /* flags */ undefined,
                makeInferenceContext(expectedValueOrElementType)
            );
            if (valueTypeResult.isIncomplete) {
                isIncomplete = true;
            }
            if (valueTypeResult.typeErrors) {
                typeErrors = true;
            }
            let valueType = valueTypeResult.type;
            if (!expectedValueOrElementType || !containsLiteralType(expectedValueOrElementType)) {
                valueType = stripLiteralValue(valueType);
            }

            type = makeTupleObject([keyType, valueType]);
        } else if (node.expression.nodeType === ParseNodeType.DictionaryExpandEntry) {
            // The parser should have reported an error in this case because it's not allowed.
            getTypeOfExpression(
                node.expression.expandExpression,
                /* flags */ undefined,
                makeInferenceContext(expectedValueOrElementType)
            );
        } else if (isExpressionNode(node)) {
            const exprTypeResult = getTypeOfExpression(
                node.expression as ExpressionNode,
                /* flags */ undefined,
                makeInferenceContext(expectedValueOrElementType)
            );
            if (exprTypeResult.isIncomplete) {
                isIncomplete = true;
            }
            if (exprTypeResult.typeErrors) {
                typeErrors = true;
            }
            type = exprTypeResult.type;
        }

        return { type, isIncomplete, typeErrors };
    }

    function getTypeOfSlice(node: SliceNode): TypeResult {
        // Evaluate the expressions to report errors and record symbol
        // references. We can skip this if we're executing speculatively.
        if (!speculativeTypeTracker.isSpeculative(node)) {
            if (node.startValue) {
                getTypeOfExpression(node.startValue);
            }

            if (node.endValue) {
                getTypeOfExpression(node.endValue);
            }

            if (node.stepValue) {
                getTypeOfExpression(node.stepValue);
            }
        }

        return { type: getBuiltInObject(node, 'slice') };
    }

    // Verifies that a type argument's type is not disallowed.
    function validateTypeArg(argResult: TypeResultWithNode, options?: ValidateTypeArgsOptions): boolean {
        if (argResult.typeList) {
            if (!options?.allowTypeArgList) {
                addError(Localizer.Diagnostic.typeArgListNotAllowed(), argResult.node);
                return false;
            } else {
                argResult.typeList!.forEach((typeArg) => {
                    validateTypeArg(typeArg);
                });
            }
        }

        if (isEllipsisType(argResult.type)) {
            if (!options?.allowTypeArgList) {
                addError(Localizer.Diagnostic.ellipsisContext(), argResult.node);
                return false;
            }
        }

        if (isModule(argResult.type)) {
            addError(Localizer.Diagnostic.moduleAsType(), argResult.node);
            return false;
        }

        if (isParamSpec(argResult.type)) {
            if (!options?.allowParamSpec) {
                addError(Localizer.Diagnostic.paramSpecContext(), argResult.node);
                return false;
            }
        }

        if (isVariadicTypeVar(argResult.type) && !argResult.type.isVariadicInUnion) {
            if (!options?.allowVariadicTypeVar) {
                addError(Localizer.Diagnostic.typeVarTupleContext(), argResult.node);
                return false;
            } else {
                validateVariadicTypeVarIsUnpacked(argResult.type, argResult.node);
            }
        }

        if (!options?.allowEmptyTuple && argResult.isEmptyTupleShorthand) {
            addError(Localizer.Diagnostic.zeroLengthTupleNotAllowed(), argResult.node);
            return false;
        }

        if (isUnpackedClass(argResult.type)) {
            if (!options?.allowUnpackedTuples) {
                addError(Localizer.Diagnostic.unpackedArgInTypeArgument(), argResult.node);
                return false;
            }
        }

        return true;
    }

    // Converts the type parameters for a Callable type. It should
    // have zero to two parameters. The first parameter, if present, should be
    // either an ellipsis or a list of parameter types. The second parameter, if
    // present, should specify the return type.
    function createCallableType(typeArgs: TypeResultWithNode[] | undefined, errorNode: ParseNode): FunctionType {
        // Create a new function that is marked as "static" so there is later
        // no attempt to bind it as though it's an instance or class method.
        const functionType = FunctionType.createInstantiable(FunctionTypeFlags.None);
        TypeBase.setSpecialForm(functionType);
        functionType.details.declaredReturnType = UnknownType.create();
        functionType.details.typeVarScopeId = getScopeIdForNode(errorNode);

        if (typeArgs && typeArgs.length > 0) {
            if (typeArgs[0].typeList) {
                const typeList = typeArgs[0].typeList;
                let sawUnpacked = false;
                let reportedUnpackedError = false;
                const noteSawUnpacked = (entry: TypeResultWithNode) => {
                    // Make sure we have at most one unpacked variadic type variable.
                    if (sawUnpacked) {
                        if (!reportedUnpackedError) {
                            addError(Localizer.Diagnostic.variadicTypeArgsTooMany(), entry.node);
                            reportedUnpackedError = true;
                        }
                    }
                    sawUnpacked = true;
                };

                typeList.forEach((entry, index) => {
                    let entryType = entry.type;
                    let paramCategory: ParameterCategory = ParameterCategory.Simple;
                    const paramName = `__p${index.toString()}`;

                    if (isVariadicTypeVar(entryType)) {
                        validateVariadicTypeVarIsUnpacked(entryType, entry.node);
                        paramCategory = ParameterCategory.VarArgList;
                        noteSawUnpacked(entry);
                    } else if (validateTypeArg(entry, { allowUnpackedTuples: true })) {
                        if (isUnpackedClass(entryType)) {
                            paramCategory = ParameterCategory.VarArgList;
                            noteSawUnpacked(entry);
                        }
                    } else {
                        entryType = UnknownType.create();
                    }

                    FunctionType.addParameter(functionType, {
                        category: paramCategory,
                        name: paramName,
                        isNameSynthesized: true,
                        type: convertToInstance(entryType),
                        hasDeclaredType: true,
                    });
                });

                FunctionType.addParameter(functionType, {
                    category: ParameterCategory.Simple,
                    isNameSynthesized: false,
                    type: UnknownType.create(),
                });
            } else if (isEllipsisType(typeArgs[0].type)) {
                FunctionType.addDefaultParameters(functionType);
                functionType.details.flags |= FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck;
            } else if (isParamSpec(typeArgs[0].type)) {
                functionType.details.paramSpec = typeArgs[0].type;
            } else {
                if (isInstantiableClass(typeArgs[0].type) && ClassType.isBuiltIn(typeArgs[0].type, 'Concatenate')) {
                    const concatTypeArgs = typeArgs[0].type.typeArguments;
                    if (concatTypeArgs && concatTypeArgs.length > 0) {
                        concatTypeArgs.forEach((typeArg, index) => {
                            if (index === concatTypeArgs.length - 1) {
                                // Add a position-only separator
                                FunctionType.addParameter(functionType, {
                                    category: ParameterCategory.Simple,
                                    isNameSynthesized: false,
                                    type: UnknownType.create(),
                                });

                                if (isParamSpec(typeArg)) {
                                    functionType.details.paramSpec = typeArg;
                                }
                            } else {
                                FunctionType.addParameter(functionType, {
                                    category: ParameterCategory.Simple,
                                    name: `__p${index}`,
                                    isNameSynthesized: true,
                                    hasDeclaredType: true,
                                    type: typeArg,
                                });
                            }
                        });
                    }
                } else {
                    addError(Localizer.Diagnostic.callableFirstArg(), typeArgs[0].node);
                }
            }

            if (typeArgs.length > 1) {
                let typeArg1Type = typeArgs[1].type;
                if (!validateTypeArg(typeArgs[1])) {
                    typeArg1Type = UnknownType.create();
                }
                functionType.details.declaredReturnType = convertToInstance(typeArg1Type);
            } else {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.callableSecondArg(),
                    errorNode
                );

                functionType.details.declaredReturnType = UnknownType.create();
            }

            if (typeArgs.length > 2) {
                addError(Localizer.Diagnostic.callableExtraArgs(), typeArgs[2].node);
            }
        } else {
            FunctionType.addDefaultParameters(functionType, /* useUnknown */ true);
            functionType.details.flags |= FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck;
        }

        return functionType;
    }

    // Creates an Optional[X] type.
    function createOptionalType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        if (!typeArgs) {
            // If no type arguments are provided, the resulting type
            // depends on whether we're evaluating a type annotation or
            // we're in some other context.
            if ((flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0) {
                addError(Localizer.Diagnostic.optionalExtraArgs(), errorNode);
                return UnknownType.create();
            }

            return classType;
        }

        if (typeArgs.length > 1) {
            addError(Localizer.Diagnostic.optionalExtraArgs(), errorNode);
            return UnknownType.create();
        }

        let typeArg0Type = typeArgs[0].type;
        if (!validateTypeArg(typeArgs[0])) {
            typeArg0Type = UnknownType.create();
        } else if (!isEffectivelyInstantiable(typeArg0Type)) {
            addExpectedClassDiagnostic(typeArg0Type, typeArgs[0].node);
            typeArg0Type = UnknownType.create();
        }

        const optionalType = combineTypes([typeArg0Type, NoneType.createType()]);

        if (isUnion(optionalType)) {
            TypeBase.setSpecialForm(optionalType);
        }

        return optionalType;
    }

    function cloneBuiltinObjectWithLiteral(node: ParseNode, builtInName: string, value: LiteralValue): Type {
        const type = getBuiltInObject(node, builtInName);
        if (isClassInstance(type)) {
            return ClassType.cloneWithLiteral(type, value);
        }

        return UnknownType.create();
    }

    function cloneBuiltinClassWithLiteral(node: ParseNode, builtInName: string, value: LiteralValue): Type {
        const type = getBuiltInType(node, builtInName);
        if (isInstantiableClass(type)) {
            return ClassType.cloneWithLiteral(type, value);
        }

        return UnknownType.create();
    }

    // Creates a type that represents a Literal.
    function createLiteralType(node: IndexNode, flags: EvaluatorFlags): Type {
        if (node.items.length === 0) {
            addError(Localizer.Diagnostic.literalEmptyArgs(), node.baseExpression);
            return UnknownType.create();
        }

        // As per the specification, we support None, int, bool, str, bytes literals
        // plus enum values.
        const literalTypes: Type[] = [];

        for (const item of node.items) {
            let type: Type | undefined;
            const itemExpr = item.valueExpression;

            if (item.argumentCategory !== ArgumentCategory.Simple) {
                addError(Localizer.Diagnostic.unpackedArgInTypeArgument(), itemExpr);
                type = UnknownType.create();
            } else if (item.name) {
                addError(Localizer.Diagnostic.keywordArgInTypeArgument(), itemExpr);
                type = UnknownType.create();
            } else if (itemExpr.nodeType === ParseNodeType.StringList) {
                const isBytes = (itemExpr.strings[0].token.flags & StringTokenFlags.Bytes) !== 0;
                const value = itemExpr.strings.map((s) => s.value).join('');
                if (isBytes) {
                    type = cloneBuiltinClassWithLiteral(node, 'bytes', value);
                } else {
                    type = cloneBuiltinClassWithLiteral(node, 'str', value);
                }
            } else if (itemExpr.nodeType === ParseNodeType.Number) {
                if (!itemExpr.isImaginary && itemExpr.isInteger) {
                    type = cloneBuiltinClassWithLiteral(node, 'int', itemExpr.value);
                }
            } else if (itemExpr.nodeType === ParseNodeType.Constant) {
                if (itemExpr.constType === KeywordType.True) {
                    type = cloneBuiltinClassWithLiteral(node, 'bool', true);
                } else if (itemExpr.constType === KeywordType.False) {
                    type = cloneBuiltinClassWithLiteral(node, 'bool', false);
                } else if (itemExpr.constType === KeywordType.None) {
                    type = NoneType.createType();
                }
            } else if (
                itemExpr.nodeType === ParseNodeType.UnaryOperation &&
                itemExpr.operator === OperatorType.Subtract
            ) {
                if (itemExpr.expression.nodeType === ParseNodeType.Number) {
                    if (!itemExpr.expression.isImaginary && itemExpr.expression.isInteger) {
                        type = cloneBuiltinClassWithLiteral(node, 'int', -itemExpr.expression.value);
                    }
                }
            }

            if (!type) {
                const exprType = getTypeOfExpression(itemExpr);

                // Is this an enum type?
                if (
                    isClassInstance(exprType.type) &&
                    ClassType.isEnumClass(exprType.type) &&
                    exprType.type.literalValue !== undefined
                ) {
                    type = ClassType.cloneAsInstantiable(exprType.type);
                } else {
                    // Is this a type alias to an existing literal type?
                    let isLiteralType = true;

                    doForEachSubtype(exprType.type, (subtype) => {
                        if (!isInstantiableClass(subtype) || subtype.literalValue === undefined) {
                            isLiteralType = false;
                        }
                    });

                    if (isLiteralType) {
                        type = exprType.type;
                    }
                }
            }

            if (!type) {
                addError(Localizer.Diagnostic.literalUnsupportedType(), item);
                type = UnknownType.create();
            }

            literalTypes.push(type);
        }

        return combineTypes(literalTypes);
    }

    // Creates a ClassVar type.
    function createClassVarType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        if (flags & EvaluatorFlags.DisallowClassVar) {
            addError(Localizer.Diagnostic.classVarNotAllowed(), errorNode);
            return AnyType.create();
        }

        if (!typeArgs) {
            return classType;
        } else if (typeArgs.length === 0) {
            addError(Localizer.Diagnostic.classVarFirstArgMissing(), errorNode);
            return UnknownType.create();
        } else if (typeArgs.length > 1) {
            addError(Localizer.Diagnostic.classVarTooManyArgs(), typeArgs[1].node);
            return UnknownType.create();
        }

        const type = typeArgs[0].type;

        // A ClassVar should not allow TypeVars or generic types parameterized
        // by TypeVars.
        if (requiresSpecialization(type, /* ignorePseudoGeneric */ true, /* ignoreSelf */ true)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);

            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.classVarWithTypeVar(),
                typeArgs[0].node ?? errorNode
            );
        }

        return type;
    }

    // Creates a "TypeGuard" type. This is an alias for 'bool', which
    // isn't a generic type and therefore doesn't have a typeParameter.
    // We'll abuse our internal types a bit by specializing it with
    // a type argument anyway.
    function createTypeGuardType(
        errorNode: ParseNode,
        classType: ClassType,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        // If no type arguments are provided, the resulting type
        // depends on whether we're evaluating a type annotation or
        // we're in some other context.
        if (!typeArgs) {
            if ((flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0) {
                addError(Localizer.Diagnostic.typeGuardArgCount(), errorNode);
            }

            return classType;
        } else if (typeArgs.length !== 1) {
            addError(Localizer.Diagnostic.typeGuardArgCount(), errorNode);
            return UnknownType.create();
        }

        const convertedTypeArgs = typeArgs.map((typeArg) => {
            return convertToInstance(validateTypeArg(typeArg) ? typeArg.type : UnknownType.create());
        });

        return ClassType.cloneForSpecialization(classType, convertedTypeArgs, /* isTypeArgumentExplicit */ true);
    }

    function createSelfType(classType: ClassType, errorNode: ParseNode, typeArgs: TypeResultWithNode[] | undefined) {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);

        // Self doesn't support any type arguments.
        if (typeArgs) {
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeArgsExpectingNone().format({
                    name: classType.details.name,
                }),
                typeArgs[0].node ?? errorNode
            );
        }

        const enclosingClass = ParseTreeUtils.getEnclosingClass(errorNode);
        const enclosingClassTypeResult = enclosingClass ? getTypeOfClass(enclosingClass) : undefined;
        if (!enclosingClassTypeResult) {
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.selfTypeContext(),
                errorNode
            );

            return UnknownType.create();
        }

        const enclosingFunction = ParseTreeUtils.getEnclosingFunction(errorNode);
        if (enclosingFunction) {
            const functionFlags = getFunctionFlagsFromDecorators(enclosingFunction, /* isInClass */ true);

            const isInnerFunction = !!ParseTreeUtils.getEnclosingFunction(enclosingFunction);
            if (!isInnerFunction) {
                // Check for static methods.
                if (functionFlags & FunctionTypeFlags.StaticMethod) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.selfTypeContext(),
                        errorNode
                    );

                    return UnknownType.create();
                }

                if (enclosingFunction.parameters.length > 0) {
                    const firstParamTypeAnnotation = ParseTreeUtils.getTypeAnnotationForParameter(enclosingFunction, 0);
                    if (
                        firstParamTypeAnnotation &&
                        !ParseTreeUtils.isNodeContainedWithin(errorNode, firstParamTypeAnnotation)
                    ) {
                        const annotationType = getTypeOfAnnotation(firstParamTypeAnnotation, {
                            associateTypeVarsWithScope: true,
                        });
                        if (!isTypeVar(annotationType) || !annotationType.details.isSynthesizedSelf) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.selfTypeWithTypedSelfOrCls(),
                                errorNode
                            );
                        }
                    }
                }
            }
        }

        return synthesizeTypeVarForSelfCls(enclosingClassTypeResult.classType, /* isClsParam */ true);
    }

    function createRequiredType(
        classType: ClassType,
        errorNode: ParseNode,
        isRequired: boolean,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): TypeResult {
        // If no type arguments are provided, the resulting type
        // depends on whether we're evaluating a type annotation or
        // we're in some other context.
        if (!typeArgs && (flags & EvaluatorFlags.ExpectingTypeAnnotation) === 0) {
            return { type: classType };
        }

        if (!typeArgs || typeArgs.length !== 1) {
            addError(
                isRequired ? Localizer.Diagnostic.requiredArgCount() : Localizer.Diagnostic.notRequiredArgCount(),
                errorNode
            );
            return { type: classType };
        }

        const typeArgType = typeArgs[0].type;

        // Make sure this is used only in a dataclass.
        const containingClassNode = ParseTreeUtils.getEnclosingClass(errorNode, /* stopAtFunction */ true);
        const classTypeInfo = containingClassNode ? getTypeOfClass(containingClassNode) : undefined;

        let isUsageLegal = false;

        if (
            classTypeInfo &&
            isInstantiableClass(classTypeInfo.classType) &&
            ClassType.isTypedDictClass(classTypeInfo.classType)
        ) {
            // The only legal usage is when used in a type annotation statement.
            if (ParseTreeUtils.isNodeContainedWithinNodeType(errorNode, ParseNodeType.TypeAnnotation)) {
                isUsageLegal = true;
            }
        }

        if ((flags & EvaluatorFlags.AllowRequired) !== 0) {
            isUsageLegal = true;
        }

        // Nested Required/NotRequired are not allowed.
        if (typeArgs[0].isRequired || typeArgs[0].isNotRequired) {
            isUsageLegal = false;
        }

        if (!isUsageLegal) {
            addError(
                isRequired
                    ? Localizer.Diagnostic.requiredNotInTypedDict()
                    : Localizer.Diagnostic.notRequiredNotInTypedDict(),
                errorNode
            );
            return { type: ClassType.cloneForSpecialization(classType, [convertToInstance(typeArgType)], !!typeArgs) };
        }

        return { type: typeArgType, isRequired, isNotRequired: !isRequired };
    }

    function createUnpackType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        // If no type arguments are provided, the resulting type
        // depends on whether we're evaluating a type annotation or
        // we're in some other context.
        if (!typeArgs && (flags & EvaluatorFlags.ExpectingTypeAnnotation) === 0) {
            return classType;
        }

        if (!typeArgs || typeArgs.length !== 1) {
            addError(Localizer.Diagnostic.unpackArgCount(), errorNode);
            return UnknownType.create();
        }

        let typeArgType = typeArgs[0].type;
        if (isUnion(typeArgType) && typeArgType.subtypes.length === 1) {
            typeArgType = typeArgType.subtypes[0];
        }

        const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);

        if ((flags & EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple) !== 0) {
            if (isInstantiableClass(typeArgType) && !typeArgType.includeSubclasses && isTupleClass(typeArgType)) {
                return ClassType.cloneForUnpacked(typeArgType);
            }

            if (isVariadicTypeVar(typeArgType) && !typeArgType.isVariadicUnpacked) {
                return TypeVarType.cloneForUnpacked(typeArgType);
            }

            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.unpackExpectedTypeVarTuple(),
                errorNode
            );
            return UnknownType.create();
        }

        if ((flags & EvaluatorFlags.AllowUnpackedTypedDict) !== 0) {
            if (isInstantiableClass(typeArgType) && ClassType.isTypedDictClass(typeArgType)) {
                return ClassType.cloneForUnpacked(typeArgType);
            }

            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.unpackExpectedTypedDict(),
                errorNode
            );
            return UnknownType.create();
        }

        addDiagnostic(
            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
            DiagnosticRule.reportGeneralTypeIssues,
            Localizer.Diagnostic.unpackNotAllowed(),
            errorNode
        );
        return UnknownType.create();
    }

    // Creates a "Final" type.
    function createFinalType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        if (flags & EvaluatorFlags.DisallowFinal) {
            addError(Localizer.Diagnostic.finalContext(), errorNode);
            return AnyType.create();
        }

        if (!typeArgs || typeArgs.length === 0) {
            return classType;
        }

        if (typeArgs.length > 1) {
            addError(Localizer.Diagnostic.finalTooManyArgs(), errorNode);
        }

        return typeArgs[0].type;
    }

    function createConcatenateType(
        errorNode: ParseNode,
        classType: ClassType,
        typeArgs: TypeResultWithNode[] | undefined
    ): Type {
        if (!typeArgs || typeArgs.length === 0) {
            addError(Localizer.Diagnostic.concatenateTypeArgsMissing(), errorNode);
        } else {
            typeArgs.forEach((typeArg, index) => {
                if (index === typeArgs.length - 1) {
                    if (!isParamSpec(typeArg.type)) {
                        addError(Localizer.Diagnostic.concatenateParamSpecMissing(), typeArg.node);
                    }
                } else {
                    if (isParamSpec(typeArg.type)) {
                        addError(Localizer.Diagnostic.paramSpecContext(), typeArg.node);
                    } else if (isUnpackedVariadicTypeVar(typeArg.type)) {
                        addError(Localizer.Diagnostic.typeVarTupleContext(), typeArg.node);
                    } else if (isUnpackedClass(typeArg.type)) {
                        addError(Localizer.Diagnostic.unpackedArgInTypeArgument(), typeArg.node);
                    }
                }
            });
        }

        return createSpecialType(classType, typeArgs, /* paramLimit */ undefined, /* allowParamSpec */ true);
    }

    function createAnnotatedType(errorNode: ParseNode, typeArgs: TypeResultWithNode[] | undefined): TypeResult {
        if (typeArgs && typeArgs.length < 2) {
            addError(Localizer.Diagnostic.annotatedTypeArgMissing(), errorNode);
        }

        if (!typeArgs || typeArgs.length === 0) {
            return { type: AnyType.create() };
        }

        return {
            type: TypeBase.cloneForAnnotated(typeArgs[0].type),
            isRequired: typeArgs[0].isRequired,
            isNotRequired: typeArgs[0].isNotRequired,
        };
    }

    // Creates one of several "special" types that are defined in typing.pyi
    // but not declared in their entirety. This includes the likes of "Tuple",
    // "Dict", etc.
    function createSpecialType(
        classType: ClassType,
        typeArgs: TypeResultWithNode[] | undefined,
        paramLimit?: number,
        allowParamSpec = false,
        isCallable = false
    ): Type {
        const isTupleTypeParam = ClassType.isTupleClass(classType);

        if (typeArgs) {
            if (isTupleTypeParam && typeArgs.length === 1 && typeArgs[0].isEmptyTupleShorthand) {
                typeArgs = [];
            } else {
                let sawUnpacked = false;
                const noteSawUnpacked = (typeArg: TypeResultWithNode) => {
                    if (sawUnpacked) {
                        if (!reportedUnpackedError) {
                            addError(Localizer.Diagnostic.variadicTypeArgsTooMany(), typeArg.node);
                            reportedUnpackedError = true;
                        }
                    }
                    sawUnpacked = true;
                };
                let reportedUnpackedError = false;

                // Verify that we didn't receive any inappropriate types.
                typeArgs.forEach((typeArg, index) => {
                    if (isEllipsisType(typeArg.type)) {
                        if (!isTupleTypeParam) {
                            addError(Localizer.Diagnostic.ellipsisContext(), typeArg.node);
                        } else if (typeArgs!.length !== 2 || index !== 1) {
                            addError(Localizer.Diagnostic.ellipsisSecondArg(), typeArg.node);
                        } else {
                            if (
                                isTypeVar(typeArgs![0].type) &&
                                isVariadicTypeVar(typeArgs![0].type) &&
                                !typeArgs![0].type.isVariadicInUnion
                            ) {
                                addError(Localizer.Diagnostic.typeVarTupleContext(), typeArgs![0].node);
                            }
                        }
                    } else if (isParamSpec(typeArg.type) && allowParamSpec) {
                        // Nothing to do - this is allowed.
                    } else if (isVariadicTypeVar(typeArg.type) && paramLimit === undefined) {
                        noteSawUnpacked(typeArg);
                        validateVariadicTypeVarIsUnpacked(typeArg.type, typeArg.node);
                    } else if (paramLimit === undefined && isUnpacked(typeArg.type)) {
                        noteSawUnpacked(typeArg);
                        validateTypeArg(typeArg, { allowUnpackedTuples: true });
                    } else {
                        validateTypeArg(typeArg);
                    }
                });
            }
        }

        let typeArgTypes = typeArgs ? typeArgs.map((t) => convertToInstance(t.type)) : [];

        // Make sure the argument list count is correct.
        if (paramLimit !== undefined) {
            if (typeArgs && typeArgTypes.length > paramLimit) {
                addError(
                    Localizer.Diagnostic.typeArgsTooMany().format({
                        name: classType.aliasName || classType.details.name,
                        expected: paramLimit,
                        received: typeArgTypes.length,
                    }),
                    typeArgs[paramLimit].node
                );
                typeArgTypes = typeArgTypes.slice(0, paramLimit);
            } else if (typeArgTypes.length < paramLimit) {
                // Fill up the remainder of the slots with unknown types.
                while (typeArgTypes.length < paramLimit) {
                    typeArgTypes.push(UnknownType.create());
                }
            }
        }

        // Handle tuple type params as a special case.
        let returnType: Type;
        if (isTupleTypeParam) {
            const tupleTypeArgTypes: TupleTypeArgument[] = [];

            // If no type args are provided and it's a tuple, default to [Unknown, ...].
            if (!typeArgs) {
                tupleTypeArgTypes.push({ type: UnknownType.create(), isUnbounded: true });
            } else {
                typeArgs.forEach((typeArg, index) => {
                    if (index === 1 && isEllipsisType(typeArgTypes[index])) {
                        if (tupleTypeArgTypes.length === 1 && !tupleTypeArgTypes[0].isUnbounded) {
                            tupleTypeArgTypes[0] = { type: tupleTypeArgTypes[0].type, isUnbounded: true };
                        } else {
                            addError(Localizer.Diagnostic.ellipsisSecondArg(), typeArg.node);
                        }
                    } else if (isUnpackedClass(typeArg.type) && typeArg.type.tupleTypeArguments) {
                        appendArray(tupleTypeArgTypes, typeArg.type.tupleTypeArguments);
                    } else {
                        tupleTypeArgTypes.push({ type: typeArgTypes[index], isUnbounded: false });
                    }
                });
            }

            returnType = specializeTupleClass(classType, tupleTypeArgTypes, typeArgs !== undefined);
        } else {
            returnType = ClassType.cloneForSpecialization(classType, typeArgTypes, typeArgs !== undefined);
        }

        if (!isCallable) {
            TypeBase.setSpecialForm(returnType);
        }

        return returnType;
    }

    // Unpacks the index expression for a "Union[X, Y, Z]" type annotation.
    function createUnionType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        const types: Type[] = [];

        if (!typeArgs) {
            // If no type arguments are provided, the resulting type
            // depends on whether we're evaluating a type annotation or
            // we're in some other context.
            if ((flags & EvaluatorFlags.ExpectingTypeAnnotation) !== 0) {
                addError(Localizer.Diagnostic.unionTypeArgCount(), errorNode);
                return NeverType.createNever();
            }

            return classType;
        }

        for (const typeArg of typeArgs) {
            let typeArgType = typeArg.type;

            if (!validateTypeArg(typeArg, { allowVariadicTypeVar: true, allowUnpackedTuples: true })) {
                typeArgType = UnknownType.create();
            } else if (!isEffectivelyInstantiable(typeArgType)) {
                addExpectedClassDiagnostic(typeArgType, typeArg.node);
                typeArgType = UnknownType.create();
            }

            // If this is an unpacked tuple, explode out the individual items.
            if (isUnpackedClass(typeArg.type) && typeArg.type.tupleTypeArguments) {
                typeArg.type.tupleTypeArguments.forEach((tupleTypeArg) => {
                    types.push(convertToInstantiable(tupleTypeArg.type));
                });
            } else {
                // If this is an unpacked TypeVar, note that it is in a union so we can differentiate
                // between Unpack[Vs] and Union[Unpack[Vs]].
                if (isTypeVar(typeArgType) && isUnpackedVariadicTypeVar(typeArgType)) {
                    typeArgType = TypeVarType.cloneForUnpacked(typeArgType, /* isInUnion */ true);
                }

                types.push(typeArgType);
            }
        }

        // Validate that we received at least two type arguments. One type argument
        // is allowed if it's an unpacked variadic type var or tuple. None is also allowed
        // since it is used to define NoReturn in typeshed stubs).
        if (types.length === 1) {
            if (!isVariadicTypeVar(types[0]) && !isUnpacked(types[0]) && !isNoneInstance(types[0])) {
                addError(Localizer.Diagnostic.unionTypeArgCount(), errorNode);
            }
        }

        const unionType = combineTypes(types);
        if (isUnion(unionType)) {
            TypeBase.setSpecialForm(unionType);
        }

        return unionType;
    }

    // Creates a type that represents "Generic[T1, T2, ...]", used in the
    // definition of a generic class.
    function createGenericType(
        classType: ClassType,
        errorNode: ParseNode,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags
    ): Type {
        if (!typeArgs) {
            // If no type arguments are provided, the resulting type
            // depends on whether we're evaluating a type annotation or
            // we're in some other context.
            if ((flags & (EvaluatorFlags.ExpectingTypeAnnotation | EvaluatorFlags.DisallowNakedGeneric)) !== 0) {
                addError(Localizer.Diagnostic.genericTypeArgMissing(), errorNode);
            }

            return classType;
        }

        const uniqueTypeVars: TypeVarType[] = [];
        if (typeArgs) {
            // Make sure there's at least one type arg.
            if (typeArgs.length === 0) {
                addError(Localizer.Diagnostic.genericTypeArgMissing(), errorNode);
            }

            // Make sure that all of the type args are typeVars and are unique.
            typeArgs.forEach((typeArg) => {
                if (!isTypeVar(typeArg.type)) {
                    addError(Localizer.Diagnostic.genericTypeArgTypeVar(), typeArg.node);
                } else {
                    if (uniqueTypeVars.some((t) => isTypeSame(t, typeArg.type))) {
                        addError(Localizer.Diagnostic.genericTypeArgUnique(), typeArg.node);
                    }

                    uniqueTypeVars.push(typeArg.type);
                }
            });
        }

        return createSpecialType(classType, typeArgs, /* paramLimit */ undefined, /* allowParamSpec */ true);
    }

    function transformTypeForTypeAlias(
        type: Type,
        name: NameNode,
        errorNode: ExpressionNode,
        typeParameters?: TypeVarType[],
        typeParamNodes?: TypeParameterNode[]
    ): Type {
        if (!TypeBase.isInstantiable(type)) {
            return type;
        }

        // If this is a recursive type alias that hasn't yet been fully resolved
        // (i.e. there is no boundType associated with it), don't apply the transform.
        if (isTypeAliasPlaceholder(type)) {
            return type;
        }

        if (!typeParameters) {
            // Determine if there are any generic type parameters associated
            // with this type alias.
            typeParameters = [];

            // Skip this for a simple TypeVar (one that's not part of a union).
            if (!isTypeVar(type) || TypeBase.isAnnotated(type)) {
                doForEachSubtype(type, (subtype) => {
                    addTypeVarsToListIfUnique(typeParameters!, getTypeVarArgumentsRecursive(subtype));
                });
            }

            // Don't include any synthesized type variables.
            typeParameters = typeParameters.filter((typeVar) => !typeVar.details.isSynthesized);
        }

        // Convert all type variables to instances.
        typeParameters = typeParameters.map((typeVar) => {
            if (TypeBase.isInstance(typeVar)) {
                return typeVar;
            }
            return convertToInstance(typeVar) as TypeVarType;
        });

        // Validate the default types for all type parameters.
        typeParameters.forEach((typeParam, index) => {
            let bestErrorNode = errorNode;
            if (typeParamNodes && index < typeParamNodes.length) {
                bestErrorNode = typeParamNodes[index].defaultExpression ?? typeParamNodes[index].name;
            }
            validateTypeParameterDefault(bestErrorNode, typeParam, typeParameters!.slice(0, index));
        });

        // Verify that we have at most one variadic type variable.
        const variadics = typeParameters.filter((param) => isVariadicTypeVar(param));
        if (variadics.length > 1) {
            addError(
                Localizer.Diagnostic.variadicTypeParamTooManyAlias().format({
                    names: variadics.map((v) => `"${v.details.name}"`).join(', '),
                }),
                errorNode
            );
        }

        const fileInfo = AnalyzerNodeInfo.getFileInfo(name);
        const typeAliasScopeId = getScopeIdForNode(name);

        const boundTypeVars = typeParameters.filter(
            (typeVar) => typeVar.scopeId !== typeAliasScopeId && typeVar.scopeType === TypeVarScopeType.Class
        );
        if (boundTypeVars.length > 0) {
            addError(
                Localizer.Diagnostic.genericTypeAliasBoundTypeVar().format({
                    names: boundTypeVars.map((t) => `${t.details.name}`).join(', '),
                }),
                errorNode
            );
        }

        return TypeBase.cloneForTypeAlias(
            type,
            name.value,
            `${fileInfo.moduleName}.${name.value}`,
            typeAliasScopeId,
            typeParameters.length > 0 ? typeParameters : undefined
        );
    }

    function createSpecialBuiltInClass(node: ParseNode, assignedName: string, aliasMapEntry: AliasMapEntry): ClassType {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        let specialClassType = ClassType.createInstantiable(
            assignedName,
            ParseTreeUtils.getClassFullName(node, fileInfo.moduleName, assignedName),
            fileInfo.moduleName,
            fileInfo.filePath,
            ClassTypeFlags.BuiltInClass | ClassTypeFlags.SpecialBuiltIn,
            /* typeSourceId */ 0,
            /* declaredMetaclass */ undefined,
            /* effectiveMetaclass */ undefined
        );

        if (fileInfo.isTypingExtensionsStubFile) {
            specialClassType.details.flags |= ClassTypeFlags.TypingExtensionClass;
        }

        const baseClassName = aliasMapEntry.alias || 'object';

        let baseClass: Type | undefined;
        if (aliasMapEntry.module === 'builtins') {
            baseClass = getBuiltInType(node, baseClassName);
        } else if (aliasMapEntry.module === 'collections') {
            // The typing.pyi file imports collections.
            baseClass = getTypeOfModule(node, baseClassName, ['collections']);
        } else if (aliasMapEntry.module === 'self') {
            const symbolWithScope = lookUpSymbolRecursive(node, baseClassName, /* honorCodeFlow */ false);
            if (symbolWithScope) {
                baseClass = getEffectiveTypeOfSymbol(symbolWithScope.symbol);
                // The _TypedDict class is marked as abstract, but the
                // methods that are abstract are overridden and shouldn't
                // cause the TypedDict to be marked as abstract.
                if (isInstantiableClass(baseClass) && ClassType.isBuiltIn(baseClass, '_TypedDict')) {
                    baseClass.details.flags &= ~ClassTypeFlags.SupportsAbstractMethods;
                }
            }
        }

        if (baseClass && isInstantiableClass(baseClass)) {
            if (aliasMapEntry.alias) {
                specialClassType = ClassType.cloneForTypingAlias(baseClass, assignedName);
            } else {
                specialClassType.details.baseClasses.push(baseClass);
                specialClassType.details.effectiveMetaclass = baseClass.details.effectiveMetaclass;
                computeMroLinearization(specialClassType);
            }
        } else {
            specialClassType.details.baseClasses.push(UnknownType.create());
            specialClassType.details.effectiveMetaclass = UnknownType.create();
            computeMroLinearization(specialClassType);
        }

        return specialClassType;
    }

    // Handles some special-case type annotations that are found
    // within the typings.pyi file.
    function handleTypingStubTypeAnnotation(node: ExpressionNode): ClassType | undefined {
        if (!node.parent || node.parent.nodeType !== ParseNodeType.TypeAnnotation) {
            return undefined;
        }

        if (node.parent.valueExpression.nodeType !== ParseNodeType.Name) {
            return undefined;
        }

        const nameNode = node.parent.valueExpression;
        const assignedName = nameNode.value;

        const specialTypes: Map<string, AliasMapEntry> = new Map([
            ['Tuple', { alias: 'tuple', module: 'builtins' }],
            ['Generic', { alias: '', module: 'builtins' }],
            ['Protocol', { alias: '', module: 'builtins' }],
            ['Callable', { alias: '', module: 'builtins' }],
            ['Type', { alias: 'type', module: 'builtins' }],
            ['ClassVar', { alias: '', module: 'builtins' }],
            ['Final', { alias: '', module: 'builtins' }],
            ['Literal', { alias: '', module: 'builtins' }],
            ['TypedDict', { alias: '_TypedDict', module: 'self' }],
            ['Union', { alias: '', module: 'builtins' }],
            ['Optional', { alias: '', module: 'builtins' }],
            ['Annotated', { alias: '', module: 'builtins' }],
            ['TypeAlias', { alias: '', module: 'builtins' }],
            ['Concatenate', { alias: '', module: 'builtins' }],
            ['TypeGuard', { alias: '', module: 'builtins' }],
            ['StrictTypeGuard', { alias: '', module: 'builtins' }],
            ['Unpack', { alias: '', module: 'builtins' }],
            ['Required', { alias: '', module: 'builtins' }],
            ['NotRequired', { alias: '', module: 'builtins' }],
            ['Self', { alias: '', module: 'builtins' }],
            ['NoReturn', { alias: '', module: 'builtins' }],
            ['Never', { alias: '', module: 'builtins' }],
            ['LiteralString', { alias: '', module: 'builtins' }],
        ]);

        const aliasMapEntry = specialTypes.get(assignedName);
        if (aliasMapEntry) {
            const cachedType = readTypeCache(node, EvaluatorFlags.None);
            if (cachedType) {
                assert(isInstantiableClass(cachedType));
                return cachedType as ClassType;
            }

            const specialType = createSpecialBuiltInClass(node, assignedName, aliasMapEntry);

            // Handle 'LiteralString' specially because we want it to act as
            // though it derives from 'str'.
            if (assignedName === 'LiteralString') {
                specialType.details.baseClasses.push(strClassType ?? AnyType.create());
                computeMroLinearization(specialType);
            }

            writeTypeCache(node, { type: specialType }, EvaluatorFlags.None);
            return specialType;
        }

        return undefined;
    }

    // Handles some special-case assignment statements that are found
    // within the typings.pyi file.
    function handleTypingStubAssignment(node: AssignmentNode): Type | undefined {
        if (node.leftExpression.nodeType !== ParseNodeType.Name) {
            return undefined;
        }

        const nameNode = node.leftExpression;
        const assignedName = nameNode.value;

        if (assignedName === 'Any') {
            return AnyType.create();
        }

        const specialTypes: Map<string, AliasMapEntry> = new Map([
            ['overload', { alias: '', module: 'builtins' }],
            ['TypeVar', { alias: '', module: 'builtins' }],
            ['_promote', { alias: '', module: 'builtins' }],
            ['no_type_check', { alias: '', module: 'builtins' }],
            ['NoReturn', { alias: '', module: 'builtins' }],
            ['Never', { alias: '', module: 'builtins' }],
            ['Counter', { alias: 'Counter', module: 'collections' }],
            ['List', { alias: 'list', module: 'builtins' }],
            ['Dict', { alias: 'dict', module: 'builtins' }],
            ['DefaultDict', { alias: 'defaultdict', module: 'collections' }],
            ['Set', { alias: 'set', module: 'builtins' }],
            ['FrozenSet', { alias: 'frozenset', module: 'builtins' }],
            ['Deque', { alias: 'deque', module: 'collections' }],
            ['ChainMap', { alias: 'ChainMap', module: 'collections' }],
            ['OrderedDict', { alias: 'OrderedDict', module: 'collections' }],
        ]);

        const aliasMapEntry = specialTypes.get(assignedName);
        if (aliasMapEntry) {
            // Evaluate the expression so symbols are marked as accessed.
            getTypeOfExpression(node.rightExpression);
            return createSpecialBuiltInClass(node, assignedName, aliasMapEntry);
        }

        return undefined;
    }

    function evaluateTypesForAssignmentStatement(node: AssignmentNode): void {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        // If the entire statement has already been evaluated, don't
        // re-evaluate it.
        if (isTypeCached(node)) {
            return;
        }

        let flags: EvaluatorFlags = EvaluatorFlags.None;
        if (fileInfo.isStubFile) {
            // An assignment of ellipsis means "Any" within a type stub file.
            flags |= EvaluatorFlags.ConvertEllipsisToUnknown;
        }

        if (
            node.rightExpression.nodeType === ParseNodeType.Name ||
            node.rightExpression.nodeType === ParseNodeType.MemberAccess
        ) {
            // Don't specialize a generic class on assignment (e.g. "x = list"
            // or "x = collections.OrderedDict") because we may want to later
            // specialize it (e.g. "x[int]").
            flags |= EvaluatorFlags.DoNotSpecialize;
        }

        if (isDeclaredTypeAlias(node.leftExpression)) {
            flags |=
                EvaluatorFlags.ExpectingType |
                EvaluatorFlags.ExpectingTypeAnnotation |
                EvaluatorFlags.EvaluateStringLiteralAsType |
                EvaluatorFlags.DisallowParamSpec |
                EvaluatorFlags.DisallowTypeVarTuple |
                EvaluatorFlags.DisallowClassVar;
            flags &= ~EvaluatorFlags.DoNotSpecialize;
        }

        // Is this type already cached?
        let rightHandType = readTypeCache(node.rightExpression, flags);
        let isIncomplete = false;
        let expectedTypeDiagAddendum: DiagnosticAddendum | undefined;

        if (!rightHandType) {
            // Special-case the typing.pyi file, which contains some special
            // types that the type analyzer needs to interpret differently.
            if (fileInfo.isTypingStubFile || fileInfo.isTypingExtensionsStubFile) {
                rightHandType = handleTypingStubAssignment(node);
                if (rightHandType) {
                    writeTypeCache(node.rightExpression, { type: rightHandType }, EvaluatorFlags.None);
                }
            }

            if (!rightHandType) {
                // Determine whether there is a declared type.
                const declaredType = getDeclaredTypeForExpression(node.leftExpression, {
                    method: 'set',
                });

                let typeAliasNameNode: NameNode | undefined;
                let isSpeculativeTypeAlias = false;

                if (isDeclaredTypeAlias(node.leftExpression)) {
                    typeAliasNameNode = (node.leftExpression as TypeAnnotationNode).valueExpression as NameNode;

                    if (!isLegalTypeAliasExpressionForm(node.rightExpression)) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeAliasIllegalExpressionForm(),
                            node.rightExpression
                        );
                    }
                } else if (node.leftExpression.nodeType === ParseNodeType.Name) {
                    const symbolWithScope = lookUpSymbolRecursive(
                        node.leftExpression,
                        node.leftExpression.value,
                        /* honorCodeFlow */ false
                    );
                    if (symbolWithScope) {
                        const decls = symbolWithScope.symbol.getDeclarations();
                        if (decls.length === 1 && isPossibleTypeAliasOrTypedDict(decls[0])) {
                            typeAliasNameNode = node.leftExpression;
                            isSpeculativeTypeAlias = true;
                        }
                    }
                }

                // Synthesize a type variable that represents the type alias while we're
                // evaluating it. This allows us to handle recursive definitions.
                let typeAliasTypeVar: TypeVarType | undefined;
                if (typeAliasNameNode) {
                    typeAliasTypeVar = TypeVarType.createInstantiable(`__type_alias_${typeAliasNameNode.value}`);
                    typeAliasTypeVar.details.isSynthesized = true;
                    typeAliasTypeVar.details.recursiveTypeAliasName = typeAliasNameNode.value;
                    const scopeId = getScopeIdForNode(typeAliasNameNode);
                    typeAliasTypeVar.details.recursiveTypeAliasScopeId = scopeId;
                    typeAliasTypeVar.scopeId = scopeId;

                    // Write the type back to the type cache. It will be replaced below.
                    writeTypeCache(node, { type: typeAliasTypeVar }, /* flags */ undefined);
                    writeTypeCache(node.leftExpression, { type: typeAliasTypeVar }, /* flags */ undefined);
                    if (node.leftExpression.nodeType === ParseNodeType.TypeAnnotation) {
                        writeTypeCache(
                            node.leftExpression.valueExpression,
                            { type: typeAliasTypeVar },
                            /* flags */ undefined
                        );
                    }
                }

                const srcTypeResult = getTypeOfExpression(
                    node.rightExpression,
                    flags,
                    makeInferenceContext(declaredType)
                );
                let srcType = srcTypeResult.type;
                expectedTypeDiagAddendum = srcTypeResult.expectedTypeDiagAddendum;
                if (srcTypeResult.isIncomplete) {
                    isIncomplete = true;
                }

                // If the RHS is a constant boolean expression, assign it a literal type.
                const constExprValue = evaluateStaticBoolExpression(
                    node.rightExpression,
                    fileInfo.executionEnvironment,
                    fileInfo.definedConstants
                );

                if (constExprValue !== undefined) {
                    const boolType = getBuiltInObject(node, 'bool');
                    if (isClassInstance(boolType)) {
                        srcType = ClassType.cloneWithLiteral(boolType, constExprValue);
                    }
                }

                // If there was a declared type, make sure the RHS value is compatible.
                if (declaredType) {
                    if (assignType(declaredType, srcType)) {
                        // Narrow the resulting type if possible.
                        if (!isAnyOrUnknown(srcType)) {
                            srcType = narrowTypeBasedOnAssignment(declaredType, srcType);
                        }
                    }
                }

                // If this is an enum, transform the type as required.
                rightHandType = srcType;
                if (node.leftExpression.nodeType === ParseNodeType.Name && !node.typeAnnotationComment) {
                    rightHandType =
                        transformTypeForPossibleEnumClass(
                            evaluatorInterface,
                            node.leftExpression,
                            () => rightHandType!
                        ) || rightHandType;
                }

                if (typeAliasNameNode) {
                    // If this was a speculative type alias, it becomes a real type alias
                    // only if the evaluated type is an instantiable type.
                    if (
                        !isSpeculativeTypeAlias ||
                        (TypeBase.isInstantiable(rightHandType) && !isUnknown(rightHandType))
                    ) {
                        // If this is a type alias, record its name based on the assignment target.
                        rightHandType = transformTypeForTypeAlias(
                            rightHandType,
                            typeAliasNameNode,
                            node.rightExpression
                        );

                        if (isTypeAliasRecursive(typeAliasTypeVar!, rightHandType)) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.typeAliasIsRecursiveDirect().format({
                                    name: typeAliasNameNode.value,
                                }),
                                node.rightExpression
                            );

                            rightHandType = UnknownType.create();
                        }

                        // Set the resulting type to the boundType of the original type alias
                        // to support recursive type aliases.
                        typeAliasTypeVar!.details.boundType = rightHandType;

                        // Record the type parameters within the recursive type alias so it
                        // can be specialized.
                        typeAliasTypeVar!.details.recursiveTypeParameters = rightHandType.typeAliasInfo?.typeParameters;
                    }
                }
            }
        }

        assignTypeToExpression(
            node.leftExpression,
            rightHandType,
            isIncomplete,
            node.rightExpression,
            /* ignoreEmptyContainers */ true,
            /* allowAssignmentToFinalVar */ true,
            expectedTypeDiagAddendum
        );

        writeTypeCache(node, { type: rightHandType, isIncomplete }, EvaluatorFlags.None);
    }

    function isPossibleTypeAliasOrTypedDict(decl: Declaration) {
        if (isPossibleTypeAliasDeclaration(decl)) {
            return true;
        }

        if (
            decl.type === DeclarationType.Variable &&
            decl.node.parent &&
            decl.node.parent.nodeType === ParseNodeType.Assignment &&
            decl.node.parent.rightExpression?.nodeType === ParseNodeType.Call
        ) {
            const callLeftNode = decl.node.parent.rightExpression.leftExpression;

            // Use a simple heuristic to determine whether this is potentially
            // a call to the TypedDict call. This avoids the expensive (and potentially
            // recursive) call to getTypeOfExpression in cases where it's not needed.
            if (
                (callLeftNode.nodeType === ParseNodeType.Name && callLeftNode.value) === 'TypedDict' ||
                (callLeftNode.nodeType === ParseNodeType.MemberAccess &&
                    callLeftNode.memberName.value === 'TypedDict' &&
                    callLeftNode.leftExpression.nodeType === ParseNodeType.Name)
            ) {
                // See if this is a call to TypedDict. We want to support
                // recursive type references in a TypedDict call.
                const callType = getTypeOfExpression(callLeftNode, EvaluatorFlags.DoNotSpecialize).type;

                if (isInstantiableClass(callType) && ClassType.isBuiltIn(callType, 'TypedDict')) {
                    return true;
                }
            }
        }

        return false;
    }

    // Evaluates the type of a type alias (i.e. "type") statement. This code
    // path does not handle traditional type aliases, which are treated as
    // variables since they use normal variable assignment syntax.
    function getTypeOfTypeAlias(node: TypeAliasNode): Type {
        const cachedType = readTypeCache(node.name, EvaluatorFlags.None);
        if (cachedType) {
            return cachedType;
        }

        // Synthesize a type variable that represents the type alias while we're
        // evaluating it. This allows us to handle recursive definitions.
        const typeAliasTypeVar = TypeVarType.createInstantiable(`__type_alias_${node.name.value}`);
        typeAliasTypeVar.details.isSynthesized = true;
        typeAliasTypeVar.details.recursiveTypeAliasName = node.name.value;
        const scopeId = getScopeIdForNode(node.name);
        typeAliasTypeVar.details.recursiveTypeAliasScopeId = scopeId;
        typeAliasTypeVar.scopeId = scopeId;

        // Write the type to the type cache. It will be replaced below.
        writeTypeCache(node.name, { type: typeAliasTypeVar }, /* flags */ undefined);

        // Set a partial type to handle recursive (self-referential) type aliases.
        const scope = ScopeUtils.getScopeForNode(node);
        const typeAliasSymbol = scope?.lookUpSymbolRecursive(node.name.value);
        const typeAliasDecl = AnalyzerNodeInfo.getDeclaration(node);
        if (typeAliasDecl && typeAliasSymbol) {
            setSymbolResolutionPartialType(typeAliasSymbol.symbol, typeAliasDecl, typeAliasTypeVar);
        }

        let typeParameters: TypeVarType[] = [];
        if (node.typeParameters) {
            typeParameters = evaluateTypeParameterList(node.typeParameters);
            typeAliasTypeVar.details.recursiveTypeParameters = typeParameters;
        }

        if (!isLegalTypeAliasExpressionForm(node.expression)) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeAliasIllegalExpressionForm(),
                node.expression
            );
        }

        const aliasTypeResult = getTypeOfExpressionExpectingType(node.expression);
        let isIncomplete = false;
        let aliasType = aliasTypeResult.type;
        if (aliasTypeResult.isIncomplete) {
            isIncomplete = true;
        }

        aliasType = transformTypeForTypeAlias(
            aliasType,
            node.name,
            node.expression,
            typeParameters,
            node.typeParameters?.parameters
        );

        if (isTypeAliasRecursive(typeAliasTypeVar, aliasType)) {
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeAliasIsRecursiveDirect().format({
                    name: node.name.value,
                }),
                node.expression
            );

            aliasType = UnknownType.create();
        }

        // Set the resulting type to the boundType of the original type alias
        // to support recursive type aliases.
        typeAliasTypeVar.details.boundType = aliasType;

        writeTypeCache(node.name, { type: aliasType, isIncomplete }, EvaluatorFlags.None);

        return aliasType;
    }

    function evaluateTypesForAugmentedAssignment(node: AugmentedAssignmentNode): void {
        if (isTypeCached(node)) {
            return;
        }

        const destTypeResult = getTypeOfAugmentedAssignment(node, /* inferenceContext */ undefined);

        writeTypeCache(node, destTypeResult, EvaluatorFlags.None);
    }

    function getPseudoGenericTypeVarName(paramName: string) {
        return `__type_of_${paramName}`;
    }

    function getTypeOfClass(node: ClassNode): ClassTypeResult | undefined {
        // Is this type already cached?
        const cachedClassType = readTypeCache(node.name, EvaluatorFlags.None);

        if (cachedClassType) {
            if (!isInstantiableClass(cachedClassType)) {
                // This can happen in rare circumstances where the class declaration
                // is located in an unreachable code block.
                return undefined;
            }
            return {
                classType: cachedClassType,
                decoratedType: readTypeCache(node, EvaluatorFlags.None) || UnknownType.create(),
            };
        }

        // The type wasn't cached, so we need to create a new one.
        const scope = ScopeUtils.getScopeForNode(node);

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        let classFlags = ClassTypeFlags.None;
        if (
            scope?.type === ScopeType.Builtin ||
            fileInfo.isTypingStubFile ||
            fileInfo.isTypingExtensionsStubFile ||
            fileInfo.isBuiltInStubFile
        ) {
            classFlags |= ClassTypeFlags.BuiltInClass;

            if (fileInfo.isTypingExtensionsStubFile) {
                classFlags |= ClassTypeFlags.TypingExtensionClass;
            }

            if (node.name.value === 'property') {
                classFlags |= ClassTypeFlags.PropertyClass;
            }

            if (node.name.value === 'tuple') {
                classFlags |= ClassTypeFlags.TupleClass;
            }
        }

        if (fileInfo.isStubFile) {
            classFlags |= ClassTypeFlags.DefinedInStub;
        }

        const classType = ClassType.createInstantiable(
            node.name.value,
            ParseTreeUtils.getClassFullName(node, fileInfo.moduleName, node.name.value),
            fileInfo.moduleName,
            fileInfo.filePath,
            classFlags,
            /* typeSourceId */ 0,
            /* declaredMetaclass */ undefined,
            /* effectiveMetaclass */ undefined,
            ParseTreeUtils.getDocString(node.suite.statements)
        );

        classType.details.typeVarScopeId = getScopeIdForNode(node);

        // Some classes refer to themselves within type arguments used within
        // base classes. We'll register the partially-constructed class type
        // to allow these to be resolved.
        const classSymbol = scope?.lookUpSymbol(node.name.value);
        let classDecl: ClassDeclaration | undefined;
        const decl = AnalyzerNodeInfo.getDeclaration(node);
        if (decl) {
            classDecl = decl as ClassDeclaration;
        }
        if (classDecl && classSymbol) {
            setSymbolResolutionPartialType(classSymbol, classDecl, classType);
        }
        classType.details.flags |= ClassTypeFlags.PartiallyEvaluated;
        writeTypeCache(node, { type: classType }, /* flags */ undefined);
        writeTypeCache(node.name, { type: classType }, /* flags */ undefined);

        // Keep a list of unique type parameters that are used in the
        // base class arguments.
        let typeParameters: TypeVarType[] = [];

        if (node.typeParameters) {
            typeParameters = evaluateTypeParameterList(node.typeParameters).map((t) => TypeVarType.cloneAsInstance(t));
        }

        // If the class derives from "Generic" directly, it will provide
        // all of the type parameters in the specified order.
        let genericTypeParameters: TypeVarType[] | undefined;
        let protocolTypeParameters: TypeVarType[] | undefined;

        const initSubclassArgs: FunctionArgument[] = [];
        let metaclassNode: ExpressionNode | undefined;
        let isMetaclassDeferred = false;
        let exprFlags =
            EvaluatorFlags.ExpectingType |
            EvaluatorFlags.AllowGenericClassType |
            EvaluatorFlags.DisallowNakedGeneric |
            EvaluatorFlags.DisallowTypeVarsWithScopeId |
            EvaluatorFlags.AssociateTypeVarsWithCurrentScope |
            EvaluatorFlags.EnforceTypeVarVarianceConsistency;
        if (fileInfo.isStubFile) {
            exprFlags |= EvaluatorFlags.AllowForwardReferences;
        }

        node.arguments.forEach((arg) => {
            // Ignore unpacked arguments.
            if (arg.argumentCategory === ArgumentCategory.UnpackedDictionary) {
                // Evaluate the expression's type so symbols are marked accessed
                // and errors are reported.
                getTypeOfExpression(arg.valueExpression);
                return;
            }

            if (!arg.name) {
                let argType: Type;

                if (arg.argumentCategory === ArgumentCategory.UnpackedList) {
                    getTypeOfExpression(arg.valueExpression);
                    argType = UnknownType.create();
                } else {
                    argType = getTypeOfExpression(arg.valueExpression, exprFlags).type;
                }

                // In some stub files, classes are conditionally defined (e.g. based
                // on platform type). We'll assume that the conditional logic is correct
                // and strip off the "unbound" union.
                if (isUnion(argType)) {
                    argType = removeUnbound(argType);
                }

                if (!isAnyOrUnknown(argType) && !isUnbound(argType)) {
                    if (isMetaclassInstance(argType)) {
                        assert(isClassInstance(argType));
                        argType =
                            argType.typeArguments && argType.typeArguments.length > 0
                                ? argType.typeArguments[0]
                                : UnknownType.create();
                    } else if (!isInstantiableClass(argType)) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.baseClassInvalid(),
                            arg
                        );
                        argType = UnknownType.create();
                    } else {
                        if (
                            ClassType.isPartiallyEvaluated(argType) ||
                            argType.details.mro.some((t) => isClass(t) && ClassType.isPartiallyEvaluated(t))
                        ) {
                            // If the base class is partially evaluated, install a callback
                            // so we can fix up this class (e.g. compute the MRO) when the
                            // dependent class is completed.
                            classTypeHooks.push({
                                dependency: argType,
                                callback: () => completeClassTypeDeferred(classType, node, node.name),
                            });
                            isMetaclassDeferred = true;
                        }

                        if (ClassType.isBuiltIn(argType, 'Protocol')) {
                            if (
                                !fileInfo.isStubFile &&
                                !ClassType.isTypingExtensionClass(argType) &&
                                fileInfo.executionEnvironment.pythonVersion < PythonVersion.V3_7
                            ) {
                                addError(Localizer.Diagnostic.protocolIllegal(), arg.valueExpression);
                            }
                            classType.details.flags |= ClassTypeFlags.ProtocolClass;
                        }

                        if (ClassType.isBuiltIn(argType, 'property')) {
                            classType.details.flags |= ClassTypeFlags.PropertyClass;
                        }

                        // If the class directly derives from NamedTuple (in Python 3.6 or
                        // newer), it's considered a (read-only) dataclass.
                        if (fileInfo.executionEnvironment.pythonVersion >= PythonVersion.V3_6) {
                            if (ClassType.isBuiltIn(argType, 'NamedTuple')) {
                                classType.details.flags |=
                                    ClassTypeFlags.DataClass |
                                    ClassTypeFlags.SkipSynthesizedDataClassEq |
                                    ClassTypeFlags.ReadOnlyInstanceVariables;
                            }
                        }

                        // If the class directly derives from TypedDict or from a class that is
                        // a TypedDict, it is considered a TypedDict.
                        if (ClassType.isBuiltIn(argType, 'TypedDict') || ClassType.isTypedDictClass(argType)) {
                            classType.details.flags |= ClassTypeFlags.TypedDictClass;
                        } else if (ClassType.isTypedDictClass(classType) && !ClassType.isTypedDictClass(argType)) {
                            // Exempt Generic from this test. As of Python 3.11, generic TypedDict
                            // classes are supported.
                            if (!isInstantiableClass(argType) || !ClassType.isBuiltIn(argType, 'Generic')) {
                                // TypedDict classes must derive only from other TypedDict classes.
                                addError(Localizer.Diagnostic.typedDictBaseClass(), arg);
                            }
                        }

                        // Validate that the class isn't deriving from itself, creating a
                        // circular dependency.
                        if (derivesFromClassRecursive(argType, classType, /* ignoreUnknown */ true)) {
                            addError(Localizer.Diagnostic.baseClassCircular(), arg);
                            argType = UnknownType.create();
                        }
                    }
                }

                if (isUnknown(argType)) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportUntypedBaseClass,
                        DiagnosticRule.reportUntypedBaseClass,
                        Localizer.Diagnostic.baseClassUnknown(),
                        arg
                    );
                }

                // Check for a duplicate class.
                if (
                    classType.details.baseClasses.some((prevBaseClass) => {
                        return (
                            isInstantiableClass(prevBaseClass) &&
                            isInstantiableClass(argType) &&
                            ClassType.isSameGenericClass(argType, prevBaseClass)
                        );
                    })
                ) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.duplicateBaseClass(),
                        arg.name || arg
                    );
                }

                classType.details.baseClasses.push(argType);
                if (isInstantiableClass(argType)) {
                    if (ClassType.isEnumClass(argType)) {
                        classType.details.flags |= ClassTypeFlags.EnumClass;
                    }

                    // Determine if the class is abstract. Protocol classes support abstract methods
                    // even though they don't derive from the ABCMeta class. We'll exclude built-in
                    // protocol classes because these are known not to contain any abstract methods
                    // and getAbstractMethods causes problems because of dependencies on some of these
                    // built-in protocol classes.
                    if (
                        ClassType.supportsAbstractMethods(argType) ||
                        (ClassType.isProtocolClass(argType) && !ClassType.isBuiltIn(argType))
                    ) {
                        classType.details.flags |= ClassTypeFlags.SupportsAbstractMethods;
                    }

                    if (ClassType.isPropertyClass(argType)) {
                        classType.details.flags |= ClassTypeFlags.PropertyClass;
                    }

                    if (ClassType.isFinal(argType)) {
                        const className = printObjectTypeForClass(argType);
                        addError(
                            Localizer.Diagnostic.baseClassFinal().format({ type: className }),
                            arg.valueExpression
                        );
                    }
                }

                addTypeVarsToListIfUnique(typeParameters, getTypeVarArgumentsRecursive(argType));
                if (isInstantiableClass(argType)) {
                    if (ClassType.isBuiltIn(argType, 'Generic')) {
                        // 'Generic' is implicitly added if type parameter syntax is used.
                        if (node.typeParameters) {
                            addDiagnostic(
                                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                Localizer.Diagnostic.genericBaseClassNotAllowed(),
                                arg.valueExpression
                            );
                        } else {
                            if (!genericTypeParameters) {
                                if (protocolTypeParameters) {
                                    addDiagnostic(
                                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.duplicateGenericAndProtocolBase(),
                                        arg.valueExpression
                                    );
                                }
                                genericTypeParameters = [];
                                addTypeVarsToListIfUnique(genericTypeParameters, getTypeVarArgumentsRecursive(argType));
                            }
                        }
                    } else if (
                        ClassType.isBuiltIn(argType, 'Protocol') &&
                        argType.typeArguments &&
                        argType.typeArguments.length > 0
                    ) {
                        if (!protocolTypeParameters) {
                            if (genericTypeParameters) {
                                addDiagnostic(
                                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.duplicateGenericAndProtocolBase(),
                                    arg.valueExpression
                                );
                            }
                            protocolTypeParameters = [];
                            addTypeVarsToListIfUnique(protocolTypeParameters, getTypeVarArgumentsRecursive(argType));

                            if (node.typeParameters && protocolTypeParameters.length > 0) {
                                addDiagnostic(
                                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.protocolBaseClassWithTypeArgs(),
                                    arg.valueExpression
                                );
                                protocolTypeParameters = [];
                            }
                        }
                    }
                }
            } else if (arg.name.value === 'metaclass') {
                if (metaclassNode) {
                    addError(Localizer.Diagnostic.metaclassDuplicate(), arg);
                } else {
                    metaclassNode = arg.valueExpression;
                }
            } else if (arg.name.value === 'total' && ClassType.isTypedDictClass(classType)) {
                // The "total" parameter name applies only for TypedDict classes.
                // PEP 589 specifies that the parameter must be either True or False.
                const constArgValue = evaluateStaticBoolExpression(
                    arg.valueExpression,
                    fileInfo.executionEnvironment,
                    fileInfo.definedConstants
                );
                if (constArgValue === undefined) {
                    addError(Localizer.Diagnostic.typedDictTotalParam(), arg.valueExpression);
                } else if (!constArgValue) {
                    classType.details.flags |= ClassTypeFlags.CanOmitDictValues;
                }
            } else {
                // Collect arguments that will be passed to the `__init_subclass__`
                // method described in PEP 487.
                initSubclassArgs.push({
                    argumentCategory: ArgumentCategory.Simple,
                    node: arg,
                    name: arg.name,
                    valueExpression: arg.valueExpression,
                });
            }
        });

        // Check for NamedTuple multiple inheritance.
        if (classType.details.baseClasses.length > 1) {
            let derivesFromNamedTuple = false;
            let foundIllegalBaseClass = false;

            classType.details.baseClasses.forEach((baseClass) => {
                if (isInstantiableClass(baseClass)) {
                    if (ClassType.isBuiltIn(baseClass, 'NamedTuple')) {
                        derivesFromNamedTuple = true;
                    } else if (!ClassType.isBuiltIn(baseClass, 'Generic')) {
                        foundIllegalBaseClass = true;
                    }
                }
            });

            if (derivesFromNamedTuple && foundIllegalBaseClass) {
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.namedTupleMultipleInheritance(),
                    node.name
                );
            }
        }

        // Make sure we don't have 'object' derive from itself. Infinite
        // recursion will result.
        if (
            !ClassType.isBuiltIn(classType, 'object') &&
            classType.details.baseClasses.filter((baseClass) => isClass(baseClass)).length === 0
        ) {
            // If there are no other (known) base classes, the class implicitly derives from object.
            classType.details.baseClasses.push(getBuiltInType(node, 'object'));
        }

        // If genericTypeParameters or protocolTypeParameters are provided,
        // make sure that typeParameters is a proper subset.
        genericTypeParameters = genericTypeParameters ?? protocolTypeParameters;
        if (genericTypeParameters && !node.typeParameters) {
            verifyGenericTypeParameters(node.name, typeParameters, genericTypeParameters);
        }
        classType.details.typeParameters = genericTypeParameters ?? typeParameters;

        // Determine if one or more type parameters is autovariance.
        if (
            classType.details.typeParameters.some(
                (param) => param.details.declaredVariance === Variance.Auto && param.computedVariance === undefined
            )
        ) {
            classType.details.requiresVarianceInference = true;
        }

        // Make sure there's at most one variadic type parameter.
        const variadics = classType.details.typeParameters.filter((param) => isVariadicTypeVar(param));
        if (variadics.length > 1) {
            addError(
                Localizer.Diagnostic.variadicTypeParamTooManyClass().format({
                    names: variadics.map((v) => `"${v.details.name}"`).join(', '),
                }),
                node.name,
                TextRange.combine(node.arguments) || node.name
            );
        }

        // Validate the default types for all type parameters.
        classType.details.typeParameters.forEach((typeParam, index) => {
            let bestErrorNode: ExpressionNode = node.name;
            if (node.typeParameters && index < node.typeParameters.parameters.length) {
                const typeParamNode = node.typeParameters.parameters[index];
                bestErrorNode = typeParamNode.defaultExpression ?? typeParamNode.name;
            }
            validateTypeParameterDefault(bestErrorNode, typeParam, classType.details.typeParameters.slice(0, index));
        });

        if (!computeMroLinearization(classType)) {
            addError(Localizer.Diagnostic.methodOrdering(), node.name);
        }

        // The scope for this class becomes the "fields" for the corresponding type.
        const innerScope = ScopeUtils.getScopeForNode(node.suite);
        classType.details.fields = innerScope?.symbolTable
            ? new Map<string, Symbol>(innerScope.symbolTable)
            : new Map<string, Symbol>();

        // Determine whether the class's instance variables are constrained
        // to those defined by __slots__. We need to do this prior to dataclass
        // processing because dataclasses can implicitly add to the slots
        // list.
        const slotsNames = innerScope?.getSlotsNames();
        if (slotsNames) {
            classType.details.localSlotsNames = slotsNames;
        }

        // Determine if the class should be a "pseudo-generic" class, characterized
        // by having an __init__ method with parameters that lack type annotations.
        // For such classes, we'll treat them as generic, with the type arguments provided
        // by the callers of the constructor.
        if (!fileInfo.isStubFile && classType.details.typeParameters.length === 0) {
            const initMethod = classType.details.fields.get('__init__');
            if (initMethod) {
                const initDecls = initMethod.getTypedDeclarations();
                if (initDecls.length === 1 && initDecls[0].type === DeclarationType.Function) {
                    const initDeclNode = initDecls[0].node;
                    const initParams = initDeclNode.parameters;

                    if (
                        initParams.length > 1 &&
                        !initParams.some(
                            (param, index) => !!ParseTreeUtils.getTypeAnnotationForParameter(initDeclNode, index)
                        )
                    ) {
                        const genericParams = initParams.filter(
                            (param, index) =>
                                index > 0 &&
                                param.name &&
                                param.category === ParameterCategory.Simple &&
                                !param.defaultValue
                        );

                        if (genericParams.length > 0) {
                            classType.details.flags |= ClassTypeFlags.PseudoGenericClass;

                            // Create a type parameter for each simple, named parameter
                            // in the __init__ method.
                            classType.details.typeParameters = genericParams.map((param) => {
                                const typeVar = TypeVarType.createInstance(
                                    getPseudoGenericTypeVarName(param.name!.value)
                                );
                                typeVar.details.isSynthesized = true;
                                typeVar.scopeId = getScopeIdForNode(initDeclNode);
                                typeVar.details.boundType = UnknownType.create();
                                return TypeVarType.cloneForScopeId(
                                    typeVar,
                                    getScopeIdForNode(node),
                                    node.name.value,
                                    TypeVarScopeType.Class
                                );
                            });
                        }
                    }
                }
            }
        }

        // Determine if the class has a custom __class_getitem__ method. This applies
        // only to classes that have no type parameters, since those with type parameters
        // are assumed to follow normal subscripting semantics for generic classes.
        if (classType.details.typeParameters.length === 0 && !ClassType.isBuiltIn(classType, 'type')) {
            if (
                classType.details.baseClasses.some(
                    (baseClass) => isInstantiableClass(baseClass) && ClassType.hasCustomClassGetItem(baseClass)
                ) ||
                classType.details.fields.has('__class_getitem__')
            ) {
                classType.details.flags |= ClassTypeFlags.HasCustomClassGetItem;
            }
        }

        // Determine the effective metaclass and detect metaclass conflicts.
        if (metaclassNode) {
            const metaclassType = getTypeOfExpression(metaclassNode, exprFlags).type;
            if (isInstantiableClass(metaclassType) || isUnknown(metaclassType)) {
                if (requiresSpecialization(metaclassType, /* ignorePseudoGeneric */ true)) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.metaclassIsGeneric(),
                        metaclassNode
                    );
                }

                classType.details.declaredMetaclass = metaclassType;
                if (isInstantiableClass(metaclassType)) {
                    if (ClassType.isBuiltIn(metaclassType, 'EnumMeta')) {
                        classType.details.flags |= ClassTypeFlags.EnumClass;
                    } else if (ClassType.isBuiltIn(metaclassType, 'ABCMeta')) {
                        classType.details.flags |= ClassTypeFlags.SupportsAbstractMethods;
                    }
                }
            }
        }

        const effectiveMetaclass = computeEffectiveMetaclass(classType, node.name);

        // Clear the "partially constructed" flag.
        classType.details.flags &= ~ClassTypeFlags.PartiallyEvaluated;

        // Now determine the decorated type of the class.
        let decoratedType: Type = classType;
        let foundUnknown = false;

        for (let i = node.decorators.length - 1; i >= 0; i--) {
            const decorator = node.decorators[i];

            const newDecoratedType = applyClassDecorator(decoratedType, classType, decorator);
            if (containsUnknown(newDecoratedType)) {
                // Report this error only on the first unknown type.
                if (!foundUnknown) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportUntypedClassDecorator,
                        DiagnosticRule.reportUntypedClassDecorator,
                        Localizer.Diagnostic.classDecoratorTypeUnknown(),
                        node.decorators[i].expression
                    );

                    foundUnknown = true;
                }
            } else {
                // Apply the decorator only if the type is known.
                decoratedType = newDecoratedType;
            }
        }

        // Determine whether this class derives from (or has a metaclass) that imbues
        // it with dataclass-like behaviors. If so, we'll apply those here.
        let dataClassBehaviors: DataClassBehaviors | undefined;
        if (isInstantiableClass(effectiveMetaclass) && effectiveMetaclass.details.classDataClassTransform) {
            dataClassBehaviors = effectiveMetaclass.details.classDataClassTransform;
        } else {
            const baseClassDataTransform = classType.details.mro.find((mroClass) => {
                return (
                    isClass(mroClass) &&
                    mroClass.details.classDataClassTransform !== undefined &&
                    !ClassType.isSameGenericClass(mroClass, classType)
                );
            });

            if (baseClassDataTransform) {
                dataClassBehaviors = (baseClassDataTransform as ClassType).details.classDataClassTransform!;
            }
        }

        if (dataClassBehaviors) {
            applyDataClassDefaultBehaviors(classType, dataClassBehaviors);
            applyDataClassClassBehaviorOverrides(
                evaluatorInterface,
                node.name,
                classType,
                initSubclassArgs,
                dataClassBehaviors
            );
        }

        // Run any class hooks that depend on this class.
        runClassTypeHooks(classType);

        // Synthesize TypedDict methods.
        if (ClassType.isTypedDictClass(classType)) {
            synthesizeTypedDictClassMethods(
                evaluatorInterface,
                node,
                classType,
                isClass(decoratedType) && ClassType.isFinal(decoratedType)
            );
        }

        // Synthesize dataclass methods.
        if (ClassType.isDataClass(classType)) {
            const skipSynthesizedInit = ClassType.isSkipSynthesizedDataClassInit(classType);
            let hasExistingInitMethod = skipSynthesizedInit;

            // See if there's already a non-synthesized __init__ method.
            // We shouldn't override it.
            if (!skipSynthesizedInit) {
                const initSymbol = lookUpClassMember(classType, '__init__', ClassMemberLookupFlags.SkipBaseClasses);
                if (initSymbol) {
                    hasExistingInitMethod = true;
                }
            }

            let skipSynthesizeHash = false;
            const hashSymbol = lookUpClassMember(classType, '__hash__', ClassMemberLookupFlags.SkipBaseClasses);
            if (hashSymbol) {
                skipSynthesizeHash = true;
            }

            synthesizeDataClassMethods(
                evaluatorInterface,
                node,
                classType,
                skipSynthesizedInit,
                hasExistingInitMethod,
                skipSynthesizeHash
            );
        }

        // Build a complete list of all slots names defined by the class hierarchy.
        // This needs to be done after dataclass processing.
        if (classType.details.localSlotsNames) {
            let isLimitedToSlots = true;
            const extendedSlotsNames = [...classType.details.localSlotsNames];

            classType.details.baseClasses.forEach((baseClass) => {
                if (isInstantiableClass(baseClass)) {
                    if (
                        !ClassType.isBuiltIn(baseClass, 'object') &&
                        !ClassType.isBuiltIn(baseClass, 'type') &&
                        !ClassType.isBuiltIn(baseClass, 'Generic')
                    ) {
                        if (baseClass.details.inheritedSlotsNames === undefined) {
                            isLimitedToSlots = false;
                        } else {
                            appendArray(extendedSlotsNames, baseClass.details.inheritedSlotsNames);
                        }
                    }
                } else {
                    isLimitedToSlots = false;
                }
            });

            if (isLimitedToSlots) {
                classType.details.inheritedSlotsNames = extendedSlotsNames;
            }
        }

        // Update the undecorated class type.
        writeTypeCache(node.name, { type: classType }, EvaluatorFlags.None);

        // Update the decorated class type.
        writeTypeCache(node, { type: decoratedType }, EvaluatorFlags.None);

        // Stash away a reference to the UnionType class if we encounter it.
        // There's no easy way to otherwise reference it.
        if (ClassType.isBuiltIn(classType, 'UnionType')) {
            unionType = ClassType.cloneAsInstance(classType);
        }

        // Validate that arguments passed to `__init_subclass__` are of the correct type.
        // Defer this if the metaclass calculation is deferred.
        if (!isMetaclassDeferred) {
            validateInitSubclassArgs(node, classType);
        }

        return { classType, decoratedType };
    }

    // Determines whether the type parameters has a default that refers to another
    // type parameter. If so, validates that it is in the list of "live" type
    // parameters and updates the scope of the type parameter referred to in the
    // default type expression.
    function validateTypeParameterDefault(
        errorNode: ExpressionNode,
        typeParam: TypeVarType,
        otherLiveTypeParams: TypeVarType[]
    ) {
        if (
            !typeParam.details.defaultType &&
            !typeParam.details.isSynthesized &&
            !typeParam.details.isSynthesizedSelf
        ) {
            const typeVarWithDefault = otherLiveTypeParams.find((param) => param.details.defaultType);
            if (typeVarWithDefault) {
                addDiagnostic(
                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeVarWithoutDefault().format({
                        name: typeParam.details.name,
                        other: typeVarWithDefault.details.name,
                    }),
                    errorNode
                );
            }
            return;
        }

        const invalidTypeVars = new Set<string>();
        validateTypeVarDefault(typeParam, otherLiveTypeParams, invalidTypeVars);

        // If we found one or more unapplied type variable, report an error.
        if (invalidTypeVars.size > 0) {
            const diag = new DiagnosticAddendum();
            invalidTypeVars.forEach((name) => {
                diag.addMessage(Localizer.DiagnosticAddendum.typeVarDefaultOutOfScope().format({ name }));
            });

            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeVarDefaultInvalidTypeVar().format({
                    name: typeParam.details.name,
                }) + diag.getString(),
                errorNode
            );
        }
    }

    function inferTypeParameterVarianceForClass(classType: ClassType): void {
        if (!classType.details.requiresVarianceInference) {
            return;
        }

        if (!objectType || !isClassInstance(objectType)) {
            return;
        }

        // Presumptively mark the variance inference as complete. This
        // prevents potential recursion.
        classType.details.requiresVarianceInference = false;

        // Presumptively mark the computed variance to "unknown". We'll
        // replace this below once the variance has been inferred.
        classType.details.typeParameters.forEach((param) => {
            if (param.details.declaredVariance === Variance.Auto) {
                param.computedVariance = Variance.Unknown;
            }
        });

        // Replace all of the type parameters with invariant TypeVars.
        const updatedTypeParams = classType.details.typeParameters.map((typeParam) =>
            TypeVarType.cloneAsInvariant(typeParam)
        );
        const updatedClassType = ClassType.cloneWithNewTypeParameters(classType, updatedTypeParams);

        const dummyTypeObject = ClassType.createInstantiable('__varianceDummy', '', '', '', 0, 0, undefined, undefined);

        updatedTypeParams.forEach((param, paramIndex) => {
            // Skip variadics and ParamSpecs.
            if (param.details.isVariadic || param.details.isParamSpec) {
                return;
            }

            // Skip type variables without auto-variance.
            if (param.details.declaredVariance !== Variance.Auto) {
                return;
            }

            // Replace all type arguments with a dummy type except for the
            // TypeVar of interest, which is replaced with an object instance.
            const srcTypeArgs = updatedTypeParams.map((p, i) => {
                if (p.details.isVariadic) {
                    return p;
                }
                return i === paramIndex ? objectType! : dummyTypeObject;
            });

            // Replace all type arguments with a dummy type except for the
            // TypeVar of interest, which is replaced with itself.
            const destTypeArgs = updatedTypeParams.map((p, i) => {
                return i === paramIndex || p.details.isVariadic ? p : dummyTypeObject;
            });

            const srcType = ClassType.cloneForSpecialization(
                updatedClassType,
                srcTypeArgs,
                /* isTypeArgumentExplicit */ true
            );
            const destType = ClassType.cloneForSpecialization(
                updatedClassType,
                destTypeArgs,
                /* isTypeArgumentExplicit */ true
            );

            const isDestSubtypeOfSrc = assignClassToSelf(srcType, destType, /* ignoreBaseClassVariance */ false);

            let inferredVariance: Variance;
            if (isDestSubtypeOfSrc) {
                inferredVariance = Variance.Covariant;
            } else {
                const isSrcSubtypeOfDest = assignClassToSelf(destType, srcType, /* ignoreBaseClassVariance */ false);
                if (isSrcSubtypeOfDest) {
                    inferredVariance = Variance.Contravariant;
                } else {
                    inferredVariance = Variance.Invariant;
                }
            }

            // We assume here that we don't need to clone the type var object
            // because it was already cloned when it was associated with this
            // class scope.
            classType.details.typeParameters[paramIndex].computedVariance = inferredVariance;
        });
    }

    function evaluateTypeParameterList(node: TypeParameterListNode): TypeVarType[] {
        const paramTypes: TypeVarType[] = [];

        node.parameters.forEach((param) => {
            const paramSymbol = AnalyzerNodeInfo.getTypeParameterSymbol(param.name);
            assert(paramSymbol);

            const typeOfParam = getDeclaredTypeOfSymbol(paramSymbol, param.name)?.type;
            if (!typeOfParam || !isTypeVar(typeOfParam)) {
                return;
            }

            writeTypeCache(param.name, { type: typeOfParam }, EvaluatorFlags.None);
            paramTypes.push(typeOfParam);
        });

        return paramTypes;
    }

    function computeEffectiveMetaclass(classType: ClassType, errorNode: ParseNode) {
        let effectiveMetaclass = classType.details.declaredMetaclass;
        let reportedMetaclassConflict = false;

        if (!effectiveMetaclass || isInstantiableClass(effectiveMetaclass)) {
            for (const baseClass of classType.details.baseClasses) {
                if (isInstantiableClass(baseClass)) {
                    const baseClassMeta = baseClass.details.effectiveMetaclass || typeClassType;
                    if (baseClassMeta && isInstantiableClass(baseClassMeta)) {
                        // Make sure there is no metaclass conflict.
                        if (!effectiveMetaclass) {
                            effectiveMetaclass = baseClassMeta;
                        } else if (
                            derivesFromClassRecursive(baseClassMeta, effectiveMetaclass, /* ignoreUnknown */ false)
                        ) {
                            effectiveMetaclass = baseClassMeta;
                        } else if (
                            !derivesFromClassRecursive(effectiveMetaclass, baseClassMeta, /* ignoreUnknown */ false)
                        ) {
                            if (!reportedMetaclassConflict) {
                                addDiagnostic(
                                    AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                    DiagnosticRule.reportGeneralTypeIssues,
                                    Localizer.Diagnostic.metaclassConflict(),
                                    errorNode
                                );
                                // Don't report more than once.
                                reportedMetaclassConflict = true;
                            }
                        }
                    } else {
                        effectiveMetaclass = baseClassMeta ? UnknownType.create() : undefined;
                        break;
                    }

                    if (ClassType.isEnumClass(baseClass)) {
                        classType.details.flags |= ClassTypeFlags.EnumClass;
                    }
                } else {
                    // If one of the base classes is unknown, then the effective
                    // metaclass is also unknowable.
                    effectiveMetaclass = UnknownType.create();
                    break;
                }
            }
        }

        // If we haven't found an effective metaclass, assume "type", which
        // is the metaclass for "object".
        if (!effectiveMetaclass) {
            const typeMetaclass = getBuiltInType(errorNode, 'type');
            effectiveMetaclass =
                typeMetaclass && isInstantiableClass(typeMetaclass) ? typeMetaclass : UnknownType.create();
        }

        classType.details.effectiveMetaclass = effectiveMetaclass;

        return effectiveMetaclass;
    }

    // Verifies that the type variables provided outside of "Generic"
    // or "Protocol" are also provided within the "Generic". For example:
    //    class Foo(Mapping[K, V], Generic[V])
    // is illegal because K is not included in Generic.
    function verifyGenericTypeParameters(
        errorNode: ExpressionNode,
        typeVars: TypeVarType[],
        genericTypeVars: TypeVarType[]
    ) {
        const missingFromGeneric = typeVars.filter((typeVar) => {
            return !genericTypeVars.some((genericTypeVar) => genericTypeVar.details.name === typeVar.details.name);
        });

        if (missingFromGeneric.length > 0) {
            const diag = new DiagnosticAddendum();
            diag.addMessage(
                Localizer.DiagnosticAddendum.typeVarsMissing().format({
                    names: missingFromGeneric.map((typeVar) => `"${typeVar.details.name}"`).join(', '),
                })
            );
            addDiagnostic(
                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeVarsNotInGenericOrProtocol() + diag.getString(),
                errorNode
            );
        }
    }

    function applyClassDecorator(
        inputClassType: Type,
        originalClassType: ClassType,
        decoratorNode: DecoratorNode
    ): Type {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(decoratorNode);
        let flags = fileInfo.isStubFile ? EvaluatorFlags.AllowForwardReferences : EvaluatorFlags.None;
        if (decoratorNode.expression.nodeType !== ParseNodeType.Call) {
            flags |= EvaluatorFlags.DoNotSpecialize;
        }
        const decoratorType = getTypeOfExpression(decoratorNode.expression, flags).type;

        if (decoratorNode.expression.nodeType === ParseNodeType.Call) {
            const decoratorCallType = getTypeOfExpression(
                decoratorNode.expression.leftExpression,
                flags | EvaluatorFlags.DoNotSpecialize
            ).type;

            if (isFunction(decoratorCallType)) {
                if (
                    decoratorCallType.details.name === '__dataclass_transform__' ||
                    decoratorCallType.details.builtInName === 'dataclass_transform'
                ) {
                    originalClassType.details.classDataClassTransform = validateDataClassTransformDecorator(
                        evaluatorInterface,
                        decoratorNode.expression
                    );
                } else if (decoratorCallType.details.name === 'deprecated') {
                    originalClassType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                    return inputClassType;
                }
            }

            if (isOverloadedFunction(decoratorCallType)) {
                if (
                    decoratorCallType.overloads.length > 0 &&
                    decoratorCallType.overloads[0].details.builtInName === 'deprecated'
                ) {
                    originalClassType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                    return inputClassType;
                }
            }
        }

        if (isOverloadedFunction(decoratorType)) {
            const dataclassBehaviors = getDataclassDecoratorBehaviors(decoratorType);
            if (dataclassBehaviors) {
                applyDataClassDecorator(
                    evaluatorInterface,
                    decoratorNode,
                    originalClassType,
                    dataclassBehaviors,
                    /* callNode */ undefined
                );
                return inputClassType;
            }

            if (decoratorType.overloads.length > 0 && decoratorType.overloads[0].details.builtInName === 'deprecated') {
                originalClassType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                return inputClassType;
            }
        } else if (isFunction(decoratorType)) {
            if (decoratorType.details.builtInName === 'final') {
                originalClassType.details.flags |= ClassTypeFlags.Final;

                // Don't call getTypeOfDecorator for final. We'll hard-code its
                // behavior because its function definition results in a cyclical
                // dependency between builtins, typing and _typeshed stubs.
                return inputClassType;
            }

            if (decoratorType.details.builtInName === 'deprecated') {
                originalClassType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                return inputClassType;
            }

            if (decoratorType.details.builtInName === 'runtime_checkable') {
                originalClassType.details.flags |= ClassTypeFlags.RuntimeCheckable;

                // Don't call getTypeOfDecorator for runtime_checkable. It appears
                // frequently in stubs, and it's a waste of time to validate its
                // parameters.
                return inputClassType;
            }

            // Is this a dataclass decorator?
            let dataclassBehaviors: DataClassBehaviors | undefined;
            let callNode: CallNode | undefined;

            if (decoratorNode.expression.nodeType === ParseNodeType.Call) {
                callNode = decoratorNode.expression;
                const decoratorCallType = getTypeOfExpression(
                    callNode.leftExpression,
                    flags | EvaluatorFlags.DoNotSpecialize
                ).type;
                dataclassBehaviors = getDataclassDecoratorBehaviors(decoratorCallType);
            } else {
                const decoratorType = getTypeOfExpression(decoratorNode.expression, flags).type;
                dataclassBehaviors = getDataclassDecoratorBehaviors(decoratorType);
            }

            if (dataclassBehaviors) {
                applyDataClassDecorator(
                    evaluatorInterface,
                    decoratorNode,
                    originalClassType,
                    dataclassBehaviors,
                    callNode
                );
                return inputClassType;
            }
        }

        return getTypeOfDecorator(decoratorNode, inputClassType);
    }

    // Given a @typing.deprecated decorator node, returns either '' or a custom
    // deprecation message if one is provided.
    function getCustomDeprecationMessage(decorator: DecoratorNode): string {
        if (
            decorator.expression.nodeType === ParseNodeType.Call &&
            decorator.expression.arguments.length > 0 &&
            decorator.expression.arguments[0].argumentCategory === ArgumentCategory.Simple &&
            decorator.expression.arguments[0].valueExpression.nodeType === ParseNodeType.StringList &&
            decorator.expression.arguments[0].valueExpression.strings.length === 1
        ) {
            return decorator.expression.arguments[0].valueExpression.strings[0].value;
        }

        return '';
    }

    // Runs any registered "callback hooks" that depend on the specified class type.
    // This allows us to complete any work that requires dependent classes to be
    // completed.
    function runClassTypeHooks(type: ClassType) {
        classTypeHooks.forEach((hook) => {
            if (ClassType.isSameGenericClass(hook.dependency, type)) {
                hook.callback();
            }
        });

        // Remove any hooks that depend on this type.
        classTypeHooks = classTypeHooks.filter((hook) => !ClassType.isSameGenericClass(hook.dependency, type));
    }

    // Recomputes the MRO and effective metaclass for the class after dependent
    // classes have been fully constructed.
    function completeClassTypeDeferred(type: ClassType, node: ClassNode, errorNode: ParseNode) {
        // Recompute the MRO linearization.
        if (!computeMroLinearization(type)) {
            addError(Localizer.Diagnostic.methodOrdering(), errorNode);
        }

        // Recompute the effective metaclass.
        computeEffectiveMetaclass(type, errorNode);

        validateInitSubclassArgs(node, type);
    }

    function validateInitSubclassArgs(node: ClassNode, classType: ClassType) {
        // Collect arguments that will be passed to the `__init_subclass__`
        // method described in PEP 487 and validate it.
        const argList: FunctionArgument[] = [];

        node.arguments.forEach((arg) => {
            if (arg.name && arg.name.value !== 'metaclass') {
                argList.push({
                    argumentCategory: ArgumentCategory.Simple,
                    node: arg,
                    name: arg.name,
                    valueExpression: arg.valueExpression,
                });
            }
        });

        const errorNode = argList.length > 0 ? argList[0].node!.name! : node.name;
        const initSubclassMethodInfo = getTypeOfClassMemberName(
            errorNode,
            classType,
            /* isAccessedThroughObject */ false,
            '__init_subclass__',
            { method: 'get' },
            /* diag */ undefined,
            MemberAccessFlags.AccessClassMembersOnly |
                MemberAccessFlags.SkipObjectBaseClass |
                MemberAccessFlags.SkipOriginalClass,
            classType
        );

        if (initSubclassMethodInfo) {
            const initSubclassMethodType = initSubclassMethodInfo.type;

            if (initSubclassMethodType) {
                validateCallArguments(
                    errorNode,
                    argList,
                    { type: initSubclassMethodType },
                    /* typeVarContext */ undefined,
                    /* skipUnknownArgCheck */ false,
                    makeInferenceContext(NoneType.createInstance())
                );
            }
        } else if (classType.details.effectiveMetaclass && isClass(classType.details.effectiveMetaclass)) {
            // See if the metaclass has a `__new__` method that accepts keyword parameters.
            const newMethodMember = lookUpClassMember(
                classType.details.effectiveMetaclass,
                '__new__',
                ClassMemberLookupFlags.SkipTypeBaseClass
            );

            if (newMethodMember) {
                const newMethodType = getTypeOfMember(newMethodMember);
                if (isFunction(newMethodType)) {
                    const paramListDetails = getParameterListDetails(newMethodType);

                    if (paramListDetails.firstKeywordOnlyIndex !== undefined) {
                        // Build a map of the keyword-only parameters.
                        const paramMap = new Map<string, number>();
                        for (let i = paramListDetails.firstKeywordOnlyIndex; i < paramListDetails.params.length; i++) {
                            const paramInfo = paramListDetails.params[i];
                            if (paramInfo.param.category === ParameterCategory.Simple && paramInfo.param.name) {
                                paramMap.set(paramInfo.param.name, i);
                            }
                        }

                        argList.forEach((arg) => {
                            const signatureTracker = new UniqueSignatureTracker();

                            if (arg.argumentCategory === ArgumentCategory.Simple && arg.name) {
                                const paramIndex = paramMap.get(arg.name.value) ?? paramListDetails.kwargsIndex;

                                if (paramIndex !== undefined) {
                                    const paramInfo = paramListDetails.params[paramIndex];
                                    const argParam: ValidateArgTypeParams = {
                                        paramCategory: paramInfo.param.category,
                                        paramType: FunctionType.getEffectiveParameterType(
                                            newMethodType,
                                            paramInfo.index
                                        ),
                                        requiresTypeVarMatching: false,
                                        argument: arg,
                                        errorNode: arg.valueExpression ?? errorNode,
                                    };

                                    validateArgType(
                                        argParam,
                                        new TypeVarContext(),
                                        signatureTracker,
                                        { type: newMethodType },
                                        /* skipUnknownCheck */ true,
                                        /* skipOverloadArg */ true,
                                        /* useNarrowBoundOnly */ false,
                                        /* conditionFilter */ undefined
                                    );
                                    paramMap.delete(arg.name.value);
                                } else {
                                    addDiagnostic(
                                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                                        DiagnosticRule.reportGeneralTypeIssues,
                                        Localizer.Diagnostic.paramNameMissing().format({ name: arg.name.value }),
                                        arg.name ?? errorNode
                                    );
                                }
                            }
                        });

                        // See if we have any remaining unmatched parameters without
                        // default values.
                        const unassignedParams: string[] = [];
                        paramMap.forEach((index, paramName) => {
                            const paramInfo = paramListDetails.params[index];
                            if (!paramInfo.param.hasDefault) {
                                unassignedParams.push(paramName);
                            }
                        });

                        if (unassignedParams.length > 0) {
                            const missingParamNames = unassignedParams.map((p) => `"${p}"`).join(', ');
                            addDiagnostic(
                                AnalyzerNodeInfo.getFileInfo(errorNode).diagnosticRuleSet.reportGeneralTypeIssues,
                                DiagnosticRule.reportGeneralTypeIssues,
                                unassignedParams.length === 1
                                    ? Localizer.Diagnostic.argMissingForParam().format({ name: missingParamNames })
                                    : Localizer.Diagnostic.argMissingForParams().format({ names: missingParamNames }),
                                errorNode
                            );
                        }
                    }
                }
            }
        }

        // Evaluate all of the expressions so they are checked and marked referenced.
        argList.forEach((arg) => {
            if (arg.valueExpression) {
                getTypeOfExpression(arg.valueExpression);
            }
        });
    }

    function getTypeOfFunction(node: FunctionNode): FunctionTypeResult | undefined {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        // Is this type already cached?
        const cachedFunctionType = readTypeCache(node.name, EvaluatorFlags.None) as FunctionType;

        if (cachedFunctionType) {
            if (!isFunction(cachedFunctionType)) {
                // This can happen in certain rare circumstances where the
                // function declaration falls within an unreachable code block.
                return undefined;
            }
            return {
                functionType: cachedFunctionType,
                decoratedType: readTypeCache(node, EvaluatorFlags.None) || UnknownType.create(),
            };
        }

        let functionDecl: FunctionDeclaration | undefined;
        const decl = AnalyzerNodeInfo.getDeclaration(node);
        if (decl) {
            functionDecl = decl as FunctionDeclaration;
        }

        // There was no cached type, so create a new one.
        // Retrieve the containing class node if the function is a method.
        const containingClassNode = ParseTreeUtils.getEnclosingClass(node, /* stopAtFunction */ true);
        let containingClassType: ClassType | undefined;
        if (containingClassNode) {
            const classInfo = getTypeOfClass(containingClassNode);
            if (!classInfo) {
                return undefined;
            }
            containingClassType = classInfo.classType;
        }

        let functionFlags = getFunctionFlagsFromDecorators(node, !!containingClassNode);
        if (functionDecl?.isGenerator) {
            functionFlags |= FunctionTypeFlags.Generator;
        }

        if (fileInfo.isStubFile) {
            functionFlags |= FunctionTypeFlags.StubDefinition;
        } else if (fileInfo.isInPyTypedPackage) {
            functionFlags |= FunctionTypeFlags.PyTypedDefinition;
        }

        if (node.isAsync) {
            functionFlags |= FunctionTypeFlags.Async;
        }

        const functionType = FunctionType.createInstance(
            node.name.value,
            getFunctionFullName(node, fileInfo.moduleName, node.name.value),
            fileInfo.moduleName,
            functionFlags | FunctionTypeFlags.PartiallyEvaluated,
            ParseTreeUtils.getDocString(node.suite.statements)
        );

        functionType.details.typeVarScopeId = getScopeIdForNode(node);

        if (node.name.value === '__init__' || node.name.value === '__new__') {
            if (containingClassNode) {
                functionType.details.constructorTypeVarScopeId = getScopeIdForNode(containingClassNode);
            }
        }

        if (fileInfo.isBuiltInStubFile || fileInfo.isTypingStubFile || fileInfo.isTypingExtensionsStubFile) {
            // Stash away the name of the function since we need to handle
            // 'namedtuple', 'abstractmethod', 'dataclass' and 'NewType'
            // specially.
            functionType.details.builtInName = node.name.value;
        }

        functionType.details.declaration = functionDecl;

        // Allow recursion by registering the partially-constructed
        // function type.
        const scope = ScopeUtils.getScopeForNode(node);
        const functionSymbol = scope?.lookUpSymbolRecursive(node.name.value);
        if (functionDecl && functionSymbol) {
            setSymbolResolutionPartialType(functionSymbol.symbol, functionDecl, functionType);
        }
        writeTypeCache(node, { type: functionType }, /* flags */ undefined);
        writeTypeCache(node.name, { type: functionType }, /* flags */ undefined);

        // Is this an "__init__" method within a pseudo-generic class? If so,
        // we'll add generic types to the constructor's parameters.
        const addGenericParamTypes =
            containingClassType &&
            ClassType.isPseudoGenericClass(containingClassType) &&
            node.name.value === '__init__';

        const paramTypes: Type[] = [];

        // Determine if the first parameter should be skipped for comment-based
        // function annotations.
        let firstCommentAnnotationIndex = 0;
        if (containingClassType && (functionType.details.flags & FunctionTypeFlags.StaticMethod) === 0) {
            firstCommentAnnotationIndex = 1;
        }

        // If there is a function annotation comment, validate that it has the correct
        // number of parameter annotations.
        if (node.functionAnnotationComment && !node.functionAnnotationComment.isParamListEllipsis) {
            const expected = node.parameters.length - firstCommentAnnotationIndex;
            const received = node.functionAnnotationComment.paramTypeAnnotations.length;

            // For methods with "self" or "cls" parameters, the annotation list
            // can either include or exclude the annotation for the first parameter.
            if (firstCommentAnnotationIndex > 0 && received === node.parameters.length) {
                firstCommentAnnotationIndex = 0;
            } else if (received !== expected) {
                addError(
                    Localizer.Diagnostic.annotatedParamCountMismatch().format({
                        expected,
                        received,
                    }),
                    node.functionAnnotationComment
                );
            }
        }

        // If this function uses PEP 695 syntax for type parameters,
        // accumulate the list of type parameters upfront.
        const typeParametersSeen: TypeVarType[] = [];
        if (node.typeParameters) {
            functionType.details.typeParameters = evaluateTypeParameterList(node.typeParameters);
        } else {
            functionType.details.typeParameters = typeParametersSeen;
        }

        const markParamAccessed = (param: ParameterNode) => {
            if (param.name) {
                const symbolWithScope = lookUpSymbolRecursive(param.name, param.name.value, /* honorCodeFlow */ false);
                if (symbolWithScope) {
                    setSymbolAccessed(fileInfo, symbolWithScope.symbol, param.name);
                }
            }
        };

        let paramsArePositionOnly = true;
        const isFirstParamClsOrSelf =
            containingClassType &&
            (FunctionType.isClassMethod(functionType) ||
                FunctionType.isInstanceMethod(functionType) ||
                FunctionType.isConstructorMethod(functionType));
        const firstNonClsSelfParamIndex = isFirstParamClsOrSelf ? 1 : 0;

        node.parameters.forEach((param, index) => {
            let paramType: Type | undefined;
            let annotatedType: Type | undefined;
            let isNoneWithoutOptional = false;
            let paramTypeNode: ExpressionNode | undefined;

            if (param.name) {
                if (index === 0 && isFirstParamClsOrSelf) {
                    // Mark "self/cls" as accessed.
                    markParamAccessed(param);
                } else if (FunctionType.isAbstractMethod(functionType)) {
                    // Mark all parameters in abstract methods as accessed.
                    markParamAccessed(param);
                } else if (containingClassType && ClassType.isProtocolClass(containingClassType)) {
                    // Mark all parameters in protocol methods as accessed.
                    markParamAccessed(param);
                }
            }

            if (param.typeAnnotation) {
                paramTypeNode = param.typeAnnotation;
            } else if (param.typeAnnotationComment) {
                paramTypeNode = param.typeAnnotationComment;
            } else if (node.functionAnnotationComment && !node.functionAnnotationComment.isParamListEllipsis) {
                const adjustedIndex = index - firstCommentAnnotationIndex;
                if (adjustedIndex >= 0 && adjustedIndex < node.functionAnnotationComment.paramTypeAnnotations.length) {
                    paramTypeNode = node.functionAnnotationComment.paramTypeAnnotations[adjustedIndex];
                }
            }

            if (paramTypeNode) {
                annotatedType = getTypeOfParameterAnnotation(paramTypeNode, param.category);

                if (isVariadicTypeVar(annotatedType) && !annotatedType.isVariadicUnpacked) {
                    addError(
                        Localizer.Diagnostic.unpackedTypeVarTupleExpected().format({
                            name1: annotatedType.details.name,
                            name2: annotatedType.details.name,
                        }),
                        paramTypeNode
                    );
                    annotatedType = UnknownType.create();
                }
            }

            if (!annotatedType && addGenericParamTypes) {
                if (index > 0 && param.category === ParameterCategory.Simple && param.name && !param.defaultValue) {
                    const typeParamName = getPseudoGenericTypeVarName(param.name.value);
                    annotatedType = containingClassType!.details.typeParameters.find(
                        (param) => param.details.name === typeParamName
                    );
                }
            }

            if (annotatedType) {
                const adjustedAnnotatedType = adjustParameterAnnotatedType(param, annotatedType);
                if (adjustedAnnotatedType !== annotatedType) {
                    annotatedType = adjustedAnnotatedType;
                    isNoneWithoutOptional = true;
                }
            }

            let defaultValueType: Type | undefined;
            if (param.defaultValue) {
                defaultValueType = getTypeOfExpression(
                    param.defaultValue,
                    EvaluatorFlags.ConvertEllipsisToAny,
                    makeInferenceContext(annotatedType)
                ).type;
            }

            if (annotatedType) {
                // If there was both a type annotation and a default value, verify
                // that the default value matches the annotation.
                if (param.defaultValue && defaultValueType) {
                    const diagAddendum = new DiagnosticAddendum();
                    const typeVarContext = new TypeVarContext(functionType.details.typeVarScopeId);
                    if (containingClassType && containingClassType.details.typeVarScopeId !== undefined) {
                        if (node.name.value === '__init__' || node.name.value === '__new__') {
                            typeVarContext.addSolveForScope(containingClassType.details.typeVarScopeId);
                        }
                    }

                    if (!assignType(annotatedType, defaultValueType, diagAddendum, typeVarContext)) {
                        const diag = addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.paramAssignmentMismatch().format({
                                sourceType: printType(defaultValueType),
                                paramType: printType(annotatedType),
                            }) + diagAddendum.getString(),
                            param.defaultValue
                        );

                        if (isNoneWithoutOptional && paramTypeNode) {
                            const addOptionalAction: AddMissingOptionalToParamAction = {
                                action: Commands.addMissingOptionalToParam,
                                offsetOfTypeNode: paramTypeNode.start + 1,
                            };
                            diag?.addAction(addOptionalAction);
                        }
                    }
                }

                paramType = annotatedType;
            }

            // Determine whether we need to insert an implied position-only parameter.
            // This is needed when a function's parameters are named using the old-style
            // way of specifying position-only parameters.
            if (index >= firstNonClsSelfParamIndex) {
                let isImplicitPositionOnlyParam = false;

                if (param.category === ParameterCategory.Simple && param.name) {
                    if (isPrivateName(param.name.value)) {
                        isImplicitPositionOnlyParam = true;
                    }
                } else {
                    paramsArePositionOnly = false;
                }

                if (
                    paramsArePositionOnly &&
                    !isImplicitPositionOnlyParam &&
                    functionType.details.parameters.length > firstNonClsSelfParamIndex
                ) {
                    // Insert an implicit "position-only parameter" separator.
                    FunctionType.addParameter(functionType, {
                        category: ParameterCategory.Simple,
                        type: UnknownType.create(),
                    });
                }

                if (!isImplicitPositionOnlyParam) {
                    paramsArePositionOnly = false;
                }
            }

            // If there was no annotation for the parameter, infer its type if possible.
            let isTypeInferred = false;
            if (!paramTypeNode) {
                isTypeInferred = true;
                const inferredType = inferParameterType(node, functionType.details.flags, index, containingClassType);
                if (inferredType) {
                    paramType = inferredType;
                }
            }

            const functionParam: FunctionParameter = {
                category: param.category,
                name: param.name ? param.name.value : undefined,
                hasDefault: !!param.defaultValue,
                defaultValueExpression: param.defaultValue,
                defaultType: defaultValueType,
                type: paramType ?? UnknownType.create(),
                typeAnnotation: paramTypeNode,
                hasDeclaredType: !!paramTypeNode,
                isTypeInferred,
            };

            FunctionType.addParameter(functionType, functionParam);

            if (functionParam.hasDeclaredType) {
                addTypeVarsToListIfUnique(typeParametersSeen, getTypeVarArgumentsRecursive(functionParam.type));
            }

            if (param.name) {
                const variadicParamType = transformVariadicParamType(node, param.category, functionParam.type);
                paramTypes.push(variadicParamType);
            } else {
                paramTypes.push(functionParam.type);
            }
        });

        if (paramsArePositionOnly && functionType.details.parameters.length > firstNonClsSelfParamIndex) {
            // Insert an implicit "position-only parameter" separator.
            FunctionType.addParameter(functionType, {
                category: ParameterCategory.Simple,
                type: UnknownType.create(),
            });
        }

        // Update the types for the nodes associated with the parameters.
        paramTypes.forEach((paramType, index) => {
            const paramNameNode = node.parameters[index].name;
            if (paramNameNode) {
                if (isUnknown(paramType)) {
                    functionType.details.flags |= FunctionTypeFlags.UnannotatedParams;
                }
                writeTypeCache(paramNameNode, { type: paramType }, EvaluatorFlags.None);
            }
        });

        // If the function ends in P.args and P.kwargs parameters, make it exempt from
        // args/kwargs compatibility checks. This is important for protocol comparisons.
        if (paramTypes.length >= 2) {
            const paramType1 = paramTypes[paramTypes.length - 2];
            const paramType2 = paramTypes[paramTypes.length - 1];
            if (
                isParamSpec(paramType1) &&
                paramType1.paramSpecAccess === 'args' &&
                isParamSpec(paramType2) &&
                paramType2.paramSpecAccess === 'kwargs'
            ) {
                functionType.details.flags |= FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck;
            }
        }

        // If there was a defined return type, analyze that first so when we
        // walk the contents of the function, return statements can be
        // validated against this type.
        const returnTypeAnnotationNode =
            node.returnTypeAnnotation ?? node.functionAnnotationComment?.returnTypeAnnotation;
        if (returnTypeAnnotationNode) {
            // Temporarily set the return type to unknown in case of recursion.
            functionType.details.declaredReturnType = UnknownType.create();

            const returnType = getTypeOfAnnotation(returnTypeAnnotationNode, {
                associateTypeVarsWithScope: true,
            });
            functionType.details.declaredReturnType = returnType;
        } else {
            // If there was no return type annotation and this is a type stub,
            // we have no opportunity to infer the return type, so we'll indicate
            // that it's unknown.
            if (fileInfo.isStubFile) {
                // Special-case the __init__ method, which is commonly left without
                // an annotated return type, but we can assume it returns None.
                if (node.name.value === '__init__') {
                    functionType.details.declaredReturnType = NoneType.createInstance();
                } else {
                    functionType.details.declaredReturnType = UnknownType.create();
                }
            }
        }

        // Accumulate any type parameters used in the return type.
        if (functionType.details.declaredReturnType && returnTypeAnnotationNode) {
            rescopeTypeVarsForCallableReturnType(
                functionType.details.declaredReturnType,
                functionType,
                typeParametersSeen
            );
        }

        // If the return type is explicitly annotated as a generator, mark the
        // function as a generator even though it may not contain a "yield" statement.
        // This is important for generator functions declared in stub files, abstract
        // methods or protocol definitions.
        if (fileInfo.isStubFile || ParseTreeUtils.isSuiteEmpty(node.suite)) {
            if (
                functionType.details.declaredReturnType &&
                isClassInstance(functionType.details.declaredReturnType) &&
                ClassType.isBuiltIn(functionType.details.declaredReturnType, [
                    'Generator',
                    'AsyncGenerator',
                    'AwaitableGenerator',
                ])
            ) {
                functionType.details.flags |= FunctionTypeFlags.Generator;
            }
        }

        // Validate the default types for all type parameters.
        functionType.details.typeParameters.forEach((typeParam, index) => {
            let bestErrorNode: ExpressionNode = node.name;
            if (node.typeParameters && index < node.typeParameters.parameters.length) {
                const typeParamNode = node.typeParameters.parameters[index];
                bestErrorNode = typeParamNode.defaultExpression ?? typeParamNode.name;
            }
            validateTypeParameterDefault(bestErrorNode, typeParam, functionType.details.typeParameters.slice(0, index));
        });

        // Clear the "partially evaluated" flag to indicate that the functionType
        // is fully evaluated.
        functionType.details.flags &= ~FunctionTypeFlags.PartiallyEvaluated;

        // If it's an async function, wrap the return type in an Awaitable or Generator.
        const preDecoratedType = node.isAsync ? createAsyncFunction(node, functionType) : functionType;

        // Apply all of the decorators in reverse order.
        let decoratedType: Type = preDecoratedType;
        let foundUnknown = false;
        for (let i = node.decorators.length - 1; i >= 0; i--) {
            const decorator = node.decorators[i];

            const newDecoratedType = applyFunctionDecorator(decoratedType, functionType, decorator, node);
            if (containsUnknown(newDecoratedType)) {
                // Report this error only on the first unknown type.
                if (!foundUnknown) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportUntypedFunctionDecorator,
                        DiagnosticRule.reportUntypedFunctionDecorator,
                        Localizer.Diagnostic.functionDecoratorTypeUnknown(),
                        node.decorators[i].expression
                    );

                    foundUnknown = true;
                }
            } else {
                // Apply the decorator only if the type is known.
                decoratedType = newDecoratedType;
            }
        }

        // See if there are any overloads provided by previous function declarations.
        if (isFunction(decoratedType)) {
            if (FunctionType.isOverloaded(decoratedType)) {
                // Mark all the parameters as accessed.
                node.parameters.forEach((param) => {
                    markParamAccessed(param);
                });
            }

            decoratedType = addOverloadsToFunctionType(node, decoratedType);
        }

        writeTypeCache(node.name, { type: functionType }, EvaluatorFlags.None);
        writeTypeCache(node, { type: decoratedType }, EvaluatorFlags.None);

        return { functionType, decoratedType };
    }

    // If the declared return type of a function contains type variables that
    // are found nowhere else in the signature and are contained within a
    // Callable, these type variables are "rescoped" from the function to
    // the Callable.
    function rescopeTypeVarsForCallableReturnType(
        returnType: Type,
        functionType: FunctionType,
        typeParametersSeen: TypeVarType[]
    ) {
        const typeVarsInReturnType = getTypeVarArgumentsRecursive(returnType);
        const rescopedTypeVars: TypeVarType[] = [];

        typeVarsInReturnType.forEach((typeVar) => {
            if (TypeBase.isInstantiable(typeVar)) {
                typeVar = TypeVarType.cloneAsInstance(typeVar);
            }

            // If this type variable isn't scoped to this function, it is probably
            // associated with an outer scope.
            if (typeVar.scopeId !== functionType.details.typeVarScopeId) {
                return;
            }

            // If this type variable was already seen in one or more input parameters,
            // don't attempt to rescope it.
            if (typeParametersSeen.some((tp) => isTypeSame(convertToInstance(tp), typeVar))) {
                return;
            }

            // Is this type variable seen outside of a single callable?
            if (isTypeVarLimitedToCallable(returnType, typeVar)) {
                rescopedTypeVars.push(typeVar);
            }
        });

        addTypeVarsToListIfUnique(typeParametersSeen, typeVarsInReturnType);

        // Note that the type parameters have been rescoped so they are not
        // considered valid for the body of this function.
        functionType.details.rescopedTypeParameters = rescopedTypeVars;
    }

    function adjustParameterAnnotatedType(param: ParameterNode, type: Type): Type {
        // PEP 484 indicates that if a parameter has a default value of 'None'
        // the type checker should assume that the type is optional (i.e. a union
        // of the specified type and 'None'). Skip this step if the type is already
        // optional to avoid losing alias names when combining the types.
        if (
            param.defaultValue?.nodeType === ParseNodeType.Constant &&
            param.defaultValue.constType === KeywordType.None &&
            !isOptionalType(type) &&
            !AnalyzerNodeInfo.getFileInfo(param).diagnosticRuleSet.strictParameterNoneValue
        ) {
            type = combineTypes([type, NoneType.createInstance()]);
        } else if (isTypeVar(type) && param.defaultValue && type.scopeType === TypeVarScopeType.Function) {
            // Handle the case where a default argument type is provided when the
            // parameter is annotated with a "raw" function-scoped type variable, as in:
            //   "def foo(value: T = 3)"
            // In this case, we need to include the default value type in a union.
            const defaultArgType = getTypeOfExpression(
                param.defaultValue,
                EvaluatorFlags.ConvertEllipsisToAny,
                makeInferenceContext(type)
            ).type;

            if (!isAny(defaultArgType)) {
                type = combineTypes([type, defaultArgType]);
            }
        }

        return type;
    }

    // Attempts to infer an unannotated parameter type from available context.
    function inferParameterType(
        functionNode: FunctionNode,
        functionFlags: FunctionTypeFlags,
        paramIndex: number,
        containingClassType: ClassType | undefined
    ) {
        // Is the function a method within a class? If so, see if a base class
        // defines the same method and provides annotations.
        if (containingClassType) {
            if (paramIndex === 0) {
                if ((functionFlags & FunctionTypeFlags.StaticMethod) === 0) {
                    const hasClsParam =
                        (functionFlags & (FunctionTypeFlags.ClassMethod | FunctionTypeFlags.ConstructorMethod)) !== 0;
                    return synthesizeTypeVarForSelfCls(containingClassType, hasClsParam);
                }
            }

            const methodName = functionNode.name.value;

            const baseClassMemberInfo = lookUpClassMember(
                containingClassType,
                methodName,
                ClassMemberLookupFlags.SkipOriginalClass
            );

            if (baseClassMemberInfo) {
                const memberDecls = baseClassMemberInfo.symbol.getDeclarations();
                if (memberDecls.length === 1 && memberDecls[0].type === DeclarationType.Function) {
                    const baseClassMethodNode = memberDecls[0].node;

                    // Does the signature match exactly with the exception of annotations?
                    if (
                        baseClassMethodNode.parameters.length === functionNode.parameters.length &&
                        baseClassMethodNode.parameters.every((param, index) => {
                            const overrideParam = functionNode.parameters[index];
                            return (
                                overrideParam.name?.value === param.name?.value &&
                                overrideParam.category === param.category
                            );
                        })
                    ) {
                        const baseClassParam = baseClassMethodNode.parameters[paramIndex];
                        const baseClassParamAnnotation =
                            baseClassParam.typeAnnotation ?? baseClassParam.typeAnnotationComment;
                        if (baseClassParamAnnotation) {
                            let inferredParamType = getTypeOfParameterAnnotation(
                                baseClassParamAnnotation,
                                functionNode.parameters[paramIndex].category
                            );

                            const fileInfo = AnalyzerNodeInfo.getFileInfo(functionNode);
                            if (fileInfo.isInPyTypedPackage && !fileInfo.isStubFile) {
                                inferredParamType = TypeBase.cloneForAmbiguousType(inferredParamType);
                            }

                            return inferredParamType;
                        }
                    }
                }
            }
        }

        // If the parameter has a default argument value, we may be able to infer its
        // type from this information.
        const paramValueExpr = functionNode.parameters[paramIndex].defaultValue;
        if (paramValueExpr) {
            const defaultValueType = getTypeOfExpression(paramValueExpr, EvaluatorFlags.ConvertEllipsisToAny).type;

            let inferredParamType: Type | undefined;

            // Is the default value a "None" or an instance of some private class (one
            // whose name starts with an underscore)? If so, we will assume that the
            // value is a singleton sentinel. The actual supported type is going to be
            // a union of this type and Unknown.
            if (
                isNoneInstance(defaultValueType) ||
                (isClassInstance(defaultValueType) && isPrivateOrProtectedName(defaultValueType.details.name))
            ) {
                inferredParamType = combineTypes([defaultValueType, UnknownType.create()]);
            } else {
                // Do not infer certain types like tuple because it's likely to be
                // more restrictive (narrower) than intended.
                if (
                    !isClassInstance(defaultValueType) ||
                    !ClassType.isBuiltIn(defaultValueType, ['tuple', 'list', 'set', 'dict'])
                ) {
                    inferredParamType = stripLiteralValue(defaultValueType);
                }
            }

            if (inferredParamType) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(functionNode);
                if (fileInfo.isInPyTypedPackage && !fileInfo.isStubFile) {
                    inferredParamType = TypeBase.cloneForAmbiguousType(inferredParamType);
                }
            }

            return inferredParamType;
        }

        return undefined;
    }

    // Transforms the parameter type based on its category. If it's a simple parameter,
    // no transform is applied. If it's a var-arg or keyword-arg parameter, the type
    // is wrapped in a List or Dict.
    function transformVariadicParamType(node: ParseNode, paramCategory: ParameterCategory, type: Type): Type {
        switch (paramCategory) {
            case ParameterCategory.Simple: {
                return type;
            }

            case ParameterCategory.VarArgList: {
                if (isTypeVar(type) && type.paramSpecAccess) {
                    return type;
                }

                if (isUnpackedClass(type)) {
                    return ClassType.cloneForUnpacked(type, /* isUnpackedTuple */ false);
                }

                if (tupleClassType && isInstantiableClass(tupleClassType)) {
                    return ClassType.cloneAsInstance(
                        specializeTupleClass(
                            tupleClassType,
                            [{ type, isUnbounded: !isVariadicTypeVar(type) }],
                            /* isTypeArgumentExplicit */ true
                        )
                    );
                }

                return UnknownType.create();
            }

            case ParameterCategory.VarArgDictionary: {
                // Leave a ParamSpec alone.
                if (isTypeVar(type) && type.paramSpecAccess) {
                    return type;
                }

                // Is this an unpacked TypedDict? If so, return it unmodified.
                if (isClassInstance(type) && ClassType.isTypedDictClass(type) && type.isUnpacked) {
                    return type;
                }

                // Wrap the type in a dict with str keys.
                const dictType = getBuiltInType(node, 'dict');
                const strType = getBuiltInObject(node, 'str');

                if (isInstantiableClass(dictType) && isClassInstance(strType)) {
                    return ClassType.cloneAsInstance(
                        ClassType.cloneForSpecialization(dictType, [strType, type], /* isTypeArgumentExplicit */ true)
                    );
                }

                return UnknownType.create();
            }
        }
    }

    // Scans through the decorators to find a few built-in decorators
    // that affect the function flags.
    function getFunctionFlagsFromDecorators(node: FunctionNode, isInClass: boolean) {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        let flags = FunctionTypeFlags.None;

        if (isInClass) {
            // The "__new__" magic method is not an instance method.
            // It acts as a static method instead.
            if (node.name.value === '__new__') {
                flags |= FunctionTypeFlags.ConstructorMethod;
            }

            // Several magic methods are treated as class methods implicitly
            // by the runtime. Check for these here.
            const implicitClassMethods = ['__init_subclass__', '__class_getitem__'];
            if (implicitClassMethods.some((name) => node.name.value === name)) {
                flags |= FunctionTypeFlags.ClassMethod;
            }
        }

        for (const decoratorNode of node.decorators) {
            // Some stub files (e.g. builtins.pyi) rely on forward declarations of decorators.
            let evaluatorFlags = fileInfo.isStubFile ? EvaluatorFlags.AllowForwardReferences : EvaluatorFlags.None;
            if (decoratorNode.expression.nodeType !== ParseNodeType.Call) {
                evaluatorFlags |= EvaluatorFlags.DoNotSpecialize;
            }

            const decoratorTypeResult = getTypeOfExpression(decoratorNode.expression, evaluatorFlags);
            const decoratorType = decoratorTypeResult.type;

            if (isFunction(decoratorType)) {
                if (decoratorType.details.builtInName === 'abstractmethod') {
                    if (isInClass) {
                        flags |= FunctionTypeFlags.AbstractMethod;
                    }
                } else if (decoratorType.details.builtInName === 'final') {
                    flags |= FunctionTypeFlags.Final;
                } else if (decoratorType.details.builtInName === 'override') {
                    flags |= FunctionTypeFlags.Overridden;
                }
            } else if (isInstantiableClass(decoratorType)) {
                if (ClassType.isBuiltIn(decoratorType, 'staticmethod')) {
                    if (isInClass) {
                        flags |= FunctionTypeFlags.StaticMethod;
                    }
                } else if (ClassType.isBuiltIn(decoratorType, 'classmethod')) {
                    if (isInClass) {
                        flags |= FunctionTypeFlags.ClassMethod;
                    }
                }
            }
        }

        return flags;
    }

    // Transforms the input function type into an output type based on the
    // decorator function described by the decoratorNode.
    function applyFunctionDecorator(
        inputFunctionType: Type,
        undecoratedType: FunctionType,
        decoratorNode: DecoratorNode,
        functionNode: FunctionNode
    ): Type {
        const fileInfo = AnalyzerNodeInfo.getFileInfo(decoratorNode);

        // Some stub files (e.g. builtins.pyi) rely on forward declarations of decorators.
        let evaluatorFlags = fileInfo.isStubFile ? EvaluatorFlags.AllowForwardReferences : EvaluatorFlags.None;
        if (decoratorNode.expression.nodeType !== ParseNodeType.Call) {
            evaluatorFlags |= EvaluatorFlags.DoNotSpecialize;
        }

        const decoratorTypeResult = getTypeOfExpression(decoratorNode.expression, evaluatorFlags);
        const decoratorType = decoratorTypeResult.type;

        // Special-case the "overload" because it has no definition. Older versions of typeshed
        // defined "overload" as an object, but newer versions define it as a function.
        if (
            (isInstantiableClass(decoratorType) && ClassType.isSpecialBuiltIn(decoratorType, 'overload')) ||
            (isFunction(decoratorType) && decoratorType.details.builtInName === 'overload')
        ) {
            if (isFunction(inputFunctionType)) {
                inputFunctionType.details.flags |= FunctionTypeFlags.Overloaded;
                undecoratedType.details.flags |= FunctionTypeFlags.Overloaded;
                return inputFunctionType;
            }
        }

        if (decoratorNode.expression.nodeType === ParseNodeType.Call) {
            const decoratorCallType = getTypeOfExpression(
                decoratorNode.expression.leftExpression,
                evaluatorFlags | EvaluatorFlags.DoNotSpecialize
            ).type;

            if (isFunction(decoratorCallType)) {
                if (
                    decoratorCallType.details.name === '__dataclass_transform__' ||
                    decoratorCallType.details.builtInName === 'dataclass_transform'
                ) {
                    undecoratedType.details.decoratorDataClassBehaviors = validateDataClassTransformDecorator(
                        evaluatorInterface,
                        decoratorNode.expression
                    );
                    return inputFunctionType;
                }

                if (decoratorCallType.details.name === 'deprecated') {
                    undecoratedType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                    return inputFunctionType;
                }
            }

            if (isOverloadedFunction(decoratorCallType)) {
                if (
                    decoratorCallType.overloads.length > 0 &&
                    decoratorCallType.overloads[0].details.builtInName === 'deprecated'
                ) {
                    undecoratedType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                    return inputFunctionType;
                }
            }
        }

        let returnType = getTypeOfDecorator(decoratorNode, inputFunctionType);

        // Check for some built-in decorator types with known semantics.
        if (isFunction(decoratorType)) {
            if (decoratorType.details.builtInName === 'abstractmethod') {
                return inputFunctionType;
            }

            if (decoratorType.details.builtInName === 'deprecated') {
                undecoratedType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                return inputFunctionType;
            }

            // Handle property setters and deleters.
            if (decoratorNode.expression.nodeType === ParseNodeType.MemberAccess) {
                const baseType = getTypeOfExpression(
                    decoratorNode.expression.leftExpression,
                    evaluatorFlags | EvaluatorFlags.DoNotSpecialize
                ).type;

                if (isProperty(baseType)) {
                    const memberName = decoratorNode.expression.memberName.value;
                    if (memberName === 'setter') {
                        if (isFunction(inputFunctionType)) {
                            validatePropertyMethod(evaluatorInterface, inputFunctionType, decoratorNode);
                            return clonePropertyWithSetter(
                                evaluatorInterface,
                                baseType,
                                inputFunctionType,
                                functionNode
                            );
                        } else {
                            return inputFunctionType;
                        }
                    } else if (memberName === 'deleter') {
                        if (isFunction(inputFunctionType)) {
                            validatePropertyMethod(evaluatorInterface, inputFunctionType, decoratorNode);
                            return clonePropertyWithDeleter(
                                evaluatorInterface,
                                baseType,
                                inputFunctionType,
                                functionNode
                            );
                        } else {
                            return inputFunctionType;
                        }
                    }
                }
            }
        } else if (isOverloadedFunction(decoratorType)) {
            if (decoratorType.overloads.length > 0 && decoratorType.overloads[0].details.builtInName === 'deprecated') {
                undecoratedType.details.deprecatedMessage = getCustomDeprecationMessage(decoratorNode);
                return inputFunctionType;
            }
        } else if (isInstantiableClass(decoratorType)) {
            if (ClassType.isBuiltIn(decoratorType)) {
                switch (decoratorType.details.name) {
                    case 'classmethod':
                    case 'staticmethod': {
                        const requiredFlag =
                            decoratorType.details.name === 'classmethod'
                                ? FunctionTypeFlags.ClassMethod
                                : FunctionTypeFlags.StaticMethod;

                        // If the function isn't currently a class method or static method
                        // (which can happen if the function was wrapped in a decorator),
                        // add the appropriate flag.
                        if (isFunction(inputFunctionType) && (inputFunctionType.details.flags & requiredFlag) === 0) {
                            const newFunction = FunctionType.clone(inputFunctionType);
                            newFunction.details.flags &= ~(
                                FunctionTypeFlags.ConstructorMethod |
                                FunctionTypeFlags.StaticMethod |
                                FunctionTypeFlags.ClassMethod
                            );
                            newFunction.details.flags |= requiredFlag;
                            return newFunction;
                        }

                        return inputFunctionType;
                    }
                }
            }

            // Handle properties and subclasses of properties specially.
            if (ClassType.isPropertyClass(decoratorType)) {
                if (isFunction(inputFunctionType)) {
                    validatePropertyMethod(evaluatorInterface, inputFunctionType, decoratorNode);
                    return createProperty(evaluatorInterface, decoratorNode, decoratorType, inputFunctionType);
                } else if (isClassInstance(inputFunctionType)) {
                    const boundMethod = getBoundMethod(inputFunctionType, '__call__');
                    if (boundMethod && isFunction(boundMethod)) {
                        return createProperty(evaluatorInterface, decoratorNode, decoratorType, boundMethod);
                    }

                    return UnknownType.create();
                }
            }
        }

        if (isFunction(inputFunctionType) && isFunction(returnType)) {
            returnType = FunctionType.clone(returnType);

            // Copy the overload flag from the input function type.
            if (FunctionType.isOverloaded(inputFunctionType)) {
                returnType.details.flags |= FunctionTypeFlags.Overloaded;
            }

            // Copy the docstrings from the input function type if the
            // decorator didn't have its own docstring.
            if (!returnType.details.docString) {
                returnType.details.docString = inputFunctionType.details.docString;
            }
        }

        return returnType;
    }

    // Given a function node and the function type associated with it, this
    // method searches for prior function nodes that are marked as @overload
    // and creates an OverloadedFunctionType that includes this function and
    // all previous ones.
    function addOverloadsToFunctionType(node: FunctionNode, type: FunctionType): Type {
        let functionDecl: FunctionDeclaration | undefined;
        const decl = AnalyzerNodeInfo.getDeclaration(node);
        if (decl) {
            functionDecl = decl as FunctionDeclaration;
        }
        const symbolWithScope = lookUpSymbolRecursive(node, node.name.value, /* honorCodeFlow */ false);
        if (symbolWithScope) {
            const decls = symbolWithScope.symbol.getDeclarations();

            // Find this function's declaration.
            const declIndex = decls.findIndex((decl) => decl === functionDecl);
            if (declIndex > 0) {
                // Evaluate all of the previous function declarations. They will
                // be cached. We do it in this order to avoid a stack overflow due
                // to recursion if there is a large number (1000's) of overloads.
                for (let i = 0; i < declIndex; i++) {
                    const decl = decls[i];
                    if (decl.type === DeclarationType.Function) {
                        getTypeOfFunction(decl.node);
                    }
                }

                let overloadedTypes: FunctionType[] = [];

                // Look at the previous declaration's type.
                const prevDecl = decls[declIndex - 1];
                if (prevDecl.type === DeclarationType.Function) {
                    const prevDeclDeclTypeInfo = getTypeOfFunction(prevDecl.node);
                    if (prevDeclDeclTypeInfo) {
                        if (isFunction(prevDeclDeclTypeInfo.decoratedType)) {
                            if (FunctionType.isOverloaded(prevDeclDeclTypeInfo.decoratedType)) {
                                overloadedTypes.push(prevDeclDeclTypeInfo.decoratedType);
                            }
                        } else if (isOverloadedFunction(prevDeclDeclTypeInfo.decoratedType)) {
                            // If the previous declaration was itself an overloaded function,
                            // copy the entries from it.
                            appendArray(overloadedTypes, prevDeclDeclTypeInfo.decoratedType.overloads);
                        }
                    }
                }

                overloadedTypes.push(type);

                if (overloadedTypes.length === 1) {
                    return overloadedTypes[0];
                }

                // Apply the implementation's docstring to any overloads that don't
                // have their own docstrings.
                const implementation = overloadedTypes.find((signature) => !FunctionType.isOverloaded(signature));
                if (implementation?.details.docString) {
                    overloadedTypes = overloadedTypes.map((overload) => {
                        if (FunctionType.isOverloaded(overload) && !overload.details.docString) {
                            return FunctionType.cloneWithDocString(overload, implementation.details.docString);
                        }
                        return overload;
                    });
                }

                // Create a new overloaded type that copies the contents of the previous
                // one and adds a new function.
                const newOverload = OverloadedFunctionType.create(overloadedTypes);

                const prevOverload = overloadedTypes[overloadedTypes.length - 2];
                const isPrevOverloadAbstract = FunctionType.isAbstractMethod(prevOverload);
                const isCurrentOverloadAbstract = FunctionType.isAbstractMethod(type);

                if (isPrevOverloadAbstract !== isCurrentOverloadAbstract) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.overloadAbstractMismatch().format({ name: node.name.value }),
                        node.name
                    );
                }

                return newOverload;
            }
        }

        return type;
    }

    function createAsyncFunction(node: FunctionNode, functionType: FunctionType): FunctionType {
        // Clone the original function and replace its return type with an
        // Awaitable[<returnType>].
        const awaitableFunctionType = FunctionType.clone(functionType);

        if (functionType.details.declaredReturnType) {
            awaitableFunctionType.details.declaredReturnType = createAwaitableReturnType(
                node,
                functionType.details.declaredReturnType,
                FunctionType.isGenerator(functionType)
            );
        }

        // Note that the inferred type, once lazily computed, needs to wrap the
        // resulting type in an awaitable.
        awaitableFunctionType.details.flags |= FunctionTypeFlags.WrapReturnTypeInAwait;

        return awaitableFunctionType;
    }

    function createAwaitableReturnType(node: ParseNode, returnType: Type, isGenerator: boolean): Type {
        let awaitableReturnType: Type | undefined;

        if (isClassInstance(returnType)) {
            if (ClassType.isBuiltIn(returnType)) {
                if (returnType.details.name === 'Generator') {
                    // If the return type is a Generator, change it to an AsyncGenerator.
                    const asyncGeneratorType = getTypingType(node, 'AsyncGenerator');
                    if (asyncGeneratorType && isInstantiableClass(asyncGeneratorType)) {
                        const typeArgs: Type[] = [];
                        const generatorTypeArgs = returnType.typeArguments;
                        if (generatorTypeArgs && generatorTypeArgs.length > 0) {
                            typeArgs.push(generatorTypeArgs[0]);
                        }
                        if (generatorTypeArgs && generatorTypeArgs.length > 1) {
                            typeArgs.push(generatorTypeArgs[1]);
                        }
                        awaitableReturnType = ClassType.cloneAsInstance(
                            ClassType.cloneForSpecialization(
                                asyncGeneratorType,
                                typeArgs,
                                /* isTypeArgumentExplicit */ true
                            )
                        );
                    }
                } else if (
                    ['AsyncGenerator', 'AsyncIterator', 'AsyncIterable'].some(
                        (name) => name === returnType.details.name
                    )
                ) {
                    // If it's already an AsyncGenerator, AsyncIterator or AsyncIterable,
                    // leave it as is.
                    awaitableReturnType = returnType;
                }
            }
        }

        if (!awaitableReturnType || !isGenerator) {
            // Wrap in a Coroutine, which is a subclass of Awaitable.
            const coroutineType = getTypingType(node, 'Coroutine');
            if (coroutineType && isInstantiableClass(coroutineType)) {
                awaitableReturnType = ClassType.cloneAsInstance(
                    ClassType.cloneForSpecialization(
                        coroutineType,
                        [AnyType.create(), AnyType.create(), returnType],
                        /* isTypeArgumentExplicit */ true
                    )
                );
            } else {
                awaitableReturnType = UnknownType.create();
            }
        }

        return awaitableReturnType;
    }

    function inferFunctionReturnType(node: FunctionNode, isAbstract: boolean): TypeResult | undefined {
        const returnAnnotation = node.returnTypeAnnotation || node.functionAnnotationComment?.returnTypeAnnotation;

        // This shouldn't be called if there is a declared return type, but it
        // can happen if there are unexpected cycles between decorators and
        // classes that they decorate. We'll just return an undefined type
        // in this case.
        if (returnAnnotation) {
            return undefined;
        }

        // Is this type already cached?
        let inferredReturnType = readTypeCache(node.suite, EvaluatorFlags.None);
        let isIncomplete = false;

        if (inferredReturnType) {
            return { type: inferredReturnType, isIncomplete };
        }

        if (functionRecursionMap.has(node.id) || functionRecursionMap.size >= maxInferFunctionReturnRecursionCount) {
            inferredReturnType = UnknownType.create();
            isIncomplete = true;
        } else {
            functionRecursionMap.add(node.id);

            try {
                let functionDecl: FunctionDeclaration | undefined;
                const decl = AnalyzerNodeInfo.getDeclaration(node);
                if (decl) {
                    functionDecl = decl as FunctionDeclaration;
                }

                const functionNeverReturns = !isAfterNodeReachable(node);
                const implicitlyReturnsNone = isAfterNodeReachable(node.suite);

                // Infer the return type based on all of the return statements in the function's body.
                if (AnalyzerNodeInfo.getFileInfo(node).isStubFile) {
                    // If a return type annotation is missing in a stub file, assume
                    // it's an "unknown" type. In normal source files, we can infer the
                    // type from the implementation.
                    inferredReturnType = UnknownType.create();
                } else {
                    if (functionNeverReturns) {
                        // If the function always raises and never returns, assume a "NoReturn" type.
                        // Skip this for abstract methods which often are implemented with "raise
                        // NotImplementedError()".
                        if (isAbstract || methodAlwaysRaisesNotImplemented(functionDecl)) {
                            inferredReturnType = UnknownType.create();
                        } else {
                            inferredReturnType = NeverType.createNoReturn();
                        }
                    } else {
                        const inferredReturnTypes: Type[] = [];
                        if (functionDecl?.returnStatements) {
                            functionDecl.returnStatements.forEach((returnNode) => {
                                if (isNodeReachable(returnNode)) {
                                    if (returnNode.returnExpression) {
                                        const returnTypeResult = getTypeOfExpression(returnNode.returnExpression);
                                        if (returnTypeResult.isIncomplete) {
                                            isIncomplete = true;
                                        }

                                        let returnType = returnTypeResult.type;

                                        // If the return type includes an instance of a class with isEmptyContainer
                                        // set, clear that because we don't want this flag to "leak" into the
                                        // inferred return type.
                                        returnType = mapSubtypes(returnType, (subtype) => {
                                            if (isClassInstance(subtype) && subtype.isEmptyContainer) {
                                                return ClassType.cloneForSpecialization(
                                                    subtype,
                                                    subtype.typeArguments,
                                                    !!subtype.isTypeArgumentExplicit,
                                                    subtype.includeSubclasses,
                                                    subtype.tupleTypeArguments,
                                                    /* isEmptyContainer */ false
                                                );
                                            }
                                            return subtype;
                                        });

                                        inferredReturnTypes.push(returnType);
                                    } else {
                                        inferredReturnTypes.push(NoneType.createInstance());
                                    }
                                }
                            });
                        }

                        if (!functionNeverReturns && implicitlyReturnsNone) {
                            inferredReturnTypes.push(NoneType.createInstance());
                        }

                        inferredReturnType = combineTypes(inferredReturnTypes);

                        // Remove any unbound values since those would generate an exception
                        // before being returned.
                        inferredReturnType = removeUnbound(inferredReturnType);
                    }

                    // Is it a generator?
                    if (functionDecl?.isGenerator) {
                        const inferredYieldTypes: Type[] = [];
                        let useAwaitableGenerator = false;

                        if (functionDecl.yieldStatements) {
                            functionDecl.yieldStatements.forEach((yieldNode) => {
                                if (isNodeReachable(yieldNode)) {
                                    if (yieldNode.nodeType === ParseNodeType.YieldFrom) {
                                        const iteratorTypeResult = getTypeOfExpression(yieldNode.expression);
                                        if (
                                            isClassInstance(iteratorTypeResult.type) &&
                                            ClassType.isBuiltIn(iteratorTypeResult.type, 'Coroutine')
                                        ) {
                                            // Handle old-style (pre-await) Coroutines.
                                            inferredYieldTypes.push();
                                            useAwaitableGenerator = true;
                                        } else {
                                            const yieldType = getTypeOfIterator(
                                                iteratorTypeResult,
                                                /* isAsync */ false,
                                                yieldNode
                                            )?.type;
                                            inferredYieldTypes.push(yieldType ?? UnknownType.create());
                                        }
                                    } else {
                                        if (yieldNode.expression) {
                                            const yieldType = getTypeOfExpression(yieldNode.expression).type;
                                            inferredYieldTypes.push(yieldType ?? UnknownType.create());
                                        } else {
                                            inferredYieldTypes.push(NoneType.createInstance());
                                        }
                                    }
                                }
                            });
                        }

                        if (inferredYieldTypes.length === 0) {
                            inferredYieldTypes.push(NoneType.createInstance());
                        }
                        const inferredYieldType = combineTypes(inferredYieldTypes);

                        // Inferred yield types need to be wrapped in a Generator or
                        // AwaitableGenerator to produce the final result.
                        const generatorType = getTypingType(
                            node,
                            useAwaitableGenerator ? 'AwaitableGenerator' : 'Generator'
                        );

                        if (generatorType && isInstantiableClass(generatorType)) {
                            const typeArgs: Type[] = [];

                            if (useAwaitableGenerator) {
                                typeArgs.push(AnyType.create());
                            }

                            typeArgs.push(
                                inferredYieldType,
                                NoneType.createInstance(),
                                isNever(inferredReturnType) ? NoneType.createInstance() : inferredReturnType
                            );

                            inferredReturnType = ClassType.cloneAsInstance(
                                ClassType.cloneForSpecialization(
                                    generatorType,
                                    typeArgs,
                                    /* isTypeArgumentExplicit */ true
                                )
                            );
                        } else {
                            inferredReturnType = UnknownType.create();
                        }
                    }
                }

                writeTypeCache(node.suite, { type: inferredReturnType, isIncomplete }, EvaluatorFlags.None);
            } finally {
                functionRecursionMap.delete(node.id);
            }
        }

        return inferredReturnType ? { type: inferredReturnType, isIncomplete } : undefined;
    }

    // Determines whether the function consists only of a "raise" statement
    // and the exception type raised is a NotImplementedError. This is commonly
    // used for abstract methods that
    function methodAlwaysRaisesNotImplemented(functionDecl?: FunctionDeclaration): boolean {
        if (
            !functionDecl ||
            !functionDecl.isMethod ||
            functionDecl.returnStatements ||
            functionDecl.yieldStatements ||
            !functionDecl.raiseStatements
        ) {
            return false;
        }

        for (const raiseStatement of functionDecl.raiseStatements) {
            if (!raiseStatement.typeExpression || raiseStatement.valueExpression) {
                return false;
            }
            const raiseType = getTypeOfExpression(raiseStatement.typeExpression).type;
            const classType = isInstantiableClass(raiseType)
                ? raiseType
                : isClassInstance(raiseType)
                ? raiseType
                : undefined;
            if (!classType || !ClassType.isBuiltIn(classType, 'NotImplementedError')) {
                return false;
            }
        }

        return true;
    }

    function evaluateTypesForForStatement(node: ForNode): void {
        if (isTypeCached(node)) {
            return;
        }

        const iteratorTypeResult = getTypeOfExpression(node.iterableExpression);
        const iteratedType =
            getTypeOfIterator(iteratorTypeResult, !!node.isAsync, node.iterableExpression)?.type ??
            UnknownType.create();

        assignTypeToExpression(
            node.targetExpression,
            iteratedType,
            !!iteratorTypeResult.isIncomplete,
            node.targetExpression
        );

        writeTypeCache(
            node,
            { type: iteratedType, isIncomplete: !!iteratorTypeResult.isIncomplete },
            EvaluatorFlags.None
        );
    }

    function evaluateTypesForExceptStatement(node: ExceptNode): void {
        // This should be called only if the except node has a target exception.
        assert(node.typeExpression !== undefined);

        if (isTypeCached(node)) {
            return;
        }

        const exceptionTypeResult = getTypeOfExpression(node.typeExpression!);
        const exceptionTypes = exceptionTypeResult.type;

        function getExceptionType(exceptionType: Type, errorNode: ExpressionNode) {
            exceptionType = makeTopLevelTypeVarsConcrete(exceptionType);

            if (isAnyOrUnknown(exceptionType)) {
                return exceptionType;
            }

            if (isInstantiableClass(exceptionType)) {
                return ClassType.cloneAsInstance(exceptionType);
            }

            if (isClassInstance(exceptionType)) {
                const iterableType =
                    getTypeOfIterator(
                        { type: exceptionType, isIncomplete: exceptionTypeResult.isIncomplete },
                        /* isAsync */ false,
                        errorNode
                    )?.type ?? UnknownType.create();

                return mapSubtypes(iterableType, (subtype) => {
                    if (isAnyOrUnknown(subtype)) {
                        return subtype;
                    }

                    return UnknownType.create();
                });
            }

            return UnknownType.create();
        }

        let targetType = mapSubtypes(exceptionTypes, (subType) => {
            // If more than one type was specified for the exception, we'll receive
            // a specialized tuple object here.
            const tupleType = getSpecializedTupleType(subType);
            if (tupleType && tupleType.tupleTypeArguments) {
                const entryTypes = tupleType.tupleTypeArguments.map((t) => {
                    return getExceptionType(t.type, node.typeExpression!);
                });
                return combineTypes(entryTypes);
            }

            return getExceptionType(subType, node.typeExpression!);
        });

        // If this is an except group, wrap the exception type in an BaseExceptionGroup.
        if (node.isExceptGroup) {
            targetType = getBuiltInObject(node, 'BaseExceptionGroup', [targetType]);
        }

        if (node.name) {
            assignTypeToExpression(node.name, targetType, /* isIncomplete */ false, node.name);
        }

        writeTypeCache(node, { type: targetType }, EvaluatorFlags.None);
    }

    function evaluateTypesForWithStatement(node: WithItemNode): void {
        if (isTypeCached(node)) {
            return;
        }

        const exprTypeResult = getTypeOfExpression(node.expression);
        let exprType = exprTypeResult.type;
        const isAsync = node.parent && node.parent.nodeType === ParseNodeType.With && !!node.parent.isAsync;

        if (isOptionalType(exprType)) {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportOptionalContextManager,
                DiagnosticRule.reportOptionalContextManager,
                Localizer.Diagnostic.noneNotUsableWith(),
                node.expression
            );
            exprType = removeNoneFromUnion(exprType);
        }

        // Verify that the target has an __enter__ or __aenter__ method defined.
        const enterMethodName = isAsync ? '__aenter__' : '__enter__';
        const scopedType = mapSubtypes(exprType, (subtype) => {
            subtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isAnyOrUnknown(subtype)) {
                return subtype;
            }

            const additionalHelp = new DiagnosticAddendum();

            if (isClass(subtype)) {
                let enterType = getTypeOfMagicMethodReturn(
                    subtype,
                    [],
                    enterMethodName,
                    node.expression,
                    /* inferenceContext */ undefined
                );

                if (enterType) {
                    // For "async while", an implicit "await" is performed.
                    if (isAsync) {
                        enterType = getTypeOfAwaitable(enterType, node.expression);
                    }

                    return enterType;
                }

                if (!isAsync) {
                    if (
                        getTypeOfMagicMethodReturn(
                            subtype,
                            [],
                            '__aenter__',
                            node.expression,
                            /* inferenceContext */ undefined
                        )
                    ) {
                        additionalHelp.addMessage(Localizer.DiagnosticAddendum.asyncHelp());
                    }
                }
            }

            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeNotUsableWith().format({ type: printType(subtype), method: enterMethodName }) +
                    additionalHelp.getString(),
                node.expression
            );
            return UnknownType.create();
        });

        // Verify that the target has an __exit__ or __aexit__ method defined.
        const exitMethodName = isAsync ? '__aexit__' : '__exit__';
        doForEachSubtype(exprType, (subtype) => {
            subtype = makeTopLevelTypeVarsConcrete(subtype);

            if (isAnyOrUnknown(subtype)) {
                return;
            }

            if (isClass(subtype)) {
                const anyArg: TypeResult = { type: AnyType.create() };
                const exitType = getTypeOfMagicMethodReturn(
                    subtype,
                    [anyArg, anyArg, anyArg],
                    exitMethodName,
                    node.expression,
                    /* inferenceContext */ undefined
                );

                if (exitType) {
                    return;
                }
            }

            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
            addDiagnostic(
                fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                DiagnosticRule.reportGeneralTypeIssues,
                Localizer.Diagnostic.typeNotUsableWith().format({ type: printType(subtype), method: exitMethodName }),
                node.expression
            );
        });

        if (node.target) {
            assignTypeToExpression(node.target, scopedType, !!exprTypeResult.isIncomplete, node.target);
        }

        writeTypeCache(node, { type: scopedType, isIncomplete: !!exprTypeResult.isIncomplete }, EvaluatorFlags.None);
    }

    function evaluateTypesForImportAs(node: ImportAsNode): void {
        if (isTypeCached(node)) {
            return;
        }

        let symbolNameNode: NameNode;
        if (node.alias) {
            // The symbol name is defined by the alias.
            symbolNameNode = node.alias;
        } else {
            // There was no alias, so we need to use the first element of
            // the name parts as the symbol.
            symbolNameNode = node.module.nameParts[0];
        }

        if (!symbolNameNode) {
            // This can happen in certain cases where there are parse errors.
            return;
        }

        // Look up the symbol to find the alias declaration.
        let symbolType = getAliasedSymbolTypeForName(node, symbolNameNode.value) ?? UnknownType.create();

        // Is there a cached module type associated with this node? If so, use
        // it instead of the type we just created.
        const cachedModuleType = readTypeCache(node, EvaluatorFlags.None) as ModuleType;
        if (cachedModuleType && isModule(cachedModuleType) && symbolType) {
            if (isTypeSame(symbolType, cachedModuleType)) {
                symbolType = cachedModuleType;
            }
        }

        assignTypeToNameNode(symbolNameNode, symbolType, /* isIncomplete */ false, /* ignoreEmptyContainers */ false);

        writeTypeCache(node, { type: symbolType }, EvaluatorFlags.None);
    }

    function evaluateTypesForImportFromAs(node: ImportFromAsNode): void {
        if (isTypeCached(node)) {
            return;
        }

        const aliasNode = node.alias || node.name;
        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        // If this is a redundant form of an import, assume it is an intentional
        // export and mark the symbol as accessed.
        if (node.alias?.value === node.name.value) {
            const symbolInScope = lookUpSymbolRecursive(node, node.name.value, /* honorCodeFlow */ true);
            if (symbolInScope) {
                setSymbolAccessed(fileInfo, symbolInScope.symbol, node);
            }
        }

        let symbolType = getAliasedSymbolTypeForName(node, aliasNode.value);
        if (!symbolType) {
            const parentNode = node.parent as ImportFromNode;
            assert(parentNode && parentNode.nodeType === ParseNodeType.ImportFrom);
            assert(!parentNode.isWildcardImport);

            const importInfo = AnalyzerNodeInfo.getImportInfo(parentNode.module);
            if (importInfo && importInfo.isImportFound && !importInfo.isNativeLib) {
                const resolvedPath = importInfo.resolvedPaths[importInfo.resolvedPaths.length - 1];

                const importLookupInfo = importLookup(resolvedPath);
                let reportError = false;

                // If we were able to resolve the import, report the error as
                // an unresolved symbol.
                if (importLookupInfo) {
                    reportError = true;

                    // Handle PEP 562 support for module-level __getattr__ function,
                    // introduced in Python 3.7.
                    if (fileInfo.executionEnvironment.pythonVersion >= PythonVersion.V3_7 || fileInfo.isStubFile) {
                        const getAttrSymbol = importLookupInfo.symbolTable.get('__getattr__');
                        if (getAttrSymbol) {
                            const getAttrType = getEffectiveTypeOfSymbol(getAttrSymbol);
                            if (isFunction(getAttrType)) {
                                symbolType = getFunctionEffectiveReturnType(getAttrType);
                                reportError = false;
                            }
                        }
                    }
                } else if (!resolvedPath) {
                    // This corresponds to the "from . import a" form.
                    reportError = true;
                }

                if (reportError) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.importSymbolUnknown().format({ name: node.name.value }),
                        node.name
                    );
                }
            }

            if (!symbolType) {
                symbolType = UnknownType.create();
            }
        }

        assignTypeToNameNode(aliasNode, symbolType, /* isIncomplete */ false, /* ignoreEmptyContainers */ false);
        writeTypeCache(node, { type: symbolType }, EvaluatorFlags.None);
    }

    function evaluateTypesForMatchStatement(node: MatchNode): void {
        if (isTypeCached(node)) {
            return;
        }

        const subjectTypeResult = getTypeOfExpression(node.subjectExpression);
        let subjectType = subjectTypeResult.type;

        // Apply negative narrowing for each of the cases that doesn't have a guard statement.
        for (const caseStatement of node.cases) {
            if (!caseStatement.guardExpression) {
                subjectType = narrowTypeBasedOnPattern(
                    evaluatorInterface,
                    subjectType,
                    caseStatement.pattern,
                    /* isPositiveTest */ false
                );
            }
        }

        writeTypeCache(
            node,
            { type: subjectType, isIncomplete: !!subjectTypeResult.isIncomplete },
            EvaluatorFlags.None
        );
    }

    function evaluateTypesForCaseStatement(node: CaseNode): void {
        if (isTypeCached(node)) {
            return;
        }

        if (!node.parent || node.parent.nodeType !== ParseNodeType.Match) {
            fail('Expected parent of case statement to be match statement');
            return;
        }

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        const subjectTypeResult = getTypeOfExpression(node.parent.subjectExpression);
        let subjectType = subjectTypeResult.type;

        // Apply negative narrowing for each of the cases prior to the current one
        // except for those that have a guard expression.
        for (const caseStatement of node.parent.cases) {
            if (caseStatement === node) {
                if (fileInfo.diagnosticRuleSet.reportUnnecessaryComparison !== 'none') {
                    checkForUnusedPattern(evaluatorInterface, node.pattern, subjectType);
                }
                break;
            }

            if (!caseStatement.guardExpression) {
                subjectType = narrowTypeBasedOnPattern(
                    evaluatorInterface,
                    subjectType,
                    caseStatement.pattern,
                    /* isPositiveTest */ false
                );
            }
        }

        // Determine if the pre-narrowed subject type contains an object.
        let subjectIsObject = false;
        doForEachSubtype(makeTopLevelTypeVarsConcrete(subjectType), (subtype) => {
            if (isClassInstance(subtype) && ClassType.isBuiltIn(subtype, 'object')) {
                subjectIsObject = true;
            }
        });

        // Apply positive narrowing for the current case statement.
        subjectType = narrowTypeBasedOnPattern(
            evaluatorInterface,
            subjectType,
            node.pattern,
            /* isPositiveTest */ true
        );

        assignTypeToPatternTargets(
            evaluatorInterface,
            subjectType,
            !!subjectTypeResult.isIncomplete,
            subjectIsObject,
            node.pattern
        );

        writeTypeCache(
            node,
            { type: subjectType, isIncomplete: !!subjectTypeResult.isIncomplete },
            EvaluatorFlags.None
        );
    }

    function evaluateTypesForImportFrom(node: ImportFromNode): void {
        if (isTypeCached(node)) {
            return;
        }

        // Use the first element of the name parts as the symbol.
        const symbolNameNode = node.module.nameParts[0];

        // Look up the symbol to find the alias declaration.
        let symbolType = getAliasedSymbolTypeForName(node, symbolNameNode.value);
        if (!symbolType) {
            return;
        }

        // Is there a cached module type associated with this node? If so, use
        // it instead of the type we just created.
        const cachedModuleType = readTypeCache(node, EvaluatorFlags.None) as ModuleType;
        if (cachedModuleType && isModule(cachedModuleType) && symbolType) {
            if (isTypeSame(symbolType, cachedModuleType)) {
                symbolType = cachedModuleType;
            }
        }

        assignTypeToNameNode(symbolNameNode, symbolType, /* isIncomplete */ false, /* ignoreEmptyContainers */ false);

        writeTypeCache(node, { type: symbolType }, EvaluatorFlags.None);
    }

    function evaluateTypesForTypeAnnotationNode(node: TypeAnnotationNode) {
        // If this node is part of an assignment statement, use specialized
        // logic that performs bidirectional inference and assignment
        // type narrowing.
        if (node.parent?.nodeType === ParseNodeType.Assignment) {
            evaluateTypesForAssignmentStatement(node.parent);
        } else {
            const annotationType = getTypeOfAnnotation(node.typeAnnotation, {
                isVariableAnnotation: true,
                allowFinal: ParseTreeUtils.isFinalAllowedForAssignmentTarget(node.valueExpression),
                allowClassVar: ParseTreeUtils.isClassVarAllowedForAssignmentTarget(node.valueExpression),
            });

            writeTypeCache(node.valueExpression, { type: annotationType }, EvaluatorFlags.None);
        }
    }

    function getAliasedSymbolTypeForName(
        node: ImportAsNode | ImportFromAsNode | ImportFromNode,
        name: string
    ): Type | undefined {
        const symbolWithScope = lookUpSymbolRecursive(node, name, /* honorCodeFlow */ true);
        if (!symbolWithScope) {
            return undefined;
        }

        // Normally there will be at most one decl associated with the import node, but
        // there can be multiple in the case of the "from .X import X" statement. In such
        // case, we want to choose the last declaration.
        const filteredDecls = symbolWithScope.symbol
            .getDeclarations()
            .filter(
                (decl) => ParseTreeUtils.isNodeContainedWithin(node, decl.node) && decl.type === DeclarationType.Alias
            );
        let aliasDecl = filteredDecls.length > 0 ? filteredDecls[filteredDecls.length - 1] : undefined;

        // If we didn't find an exact match, look for any alias associated with
        // this symbol. In cases where we have multiple ImportAs nodes that share
        // the same first-part name (e.g. "import asyncio" and "import asyncio.tasks"),
        // we may not find the declaration associated with this node.
        if (!aliasDecl) {
            aliasDecl = symbolWithScope.symbol.getDeclarations().find((decl) => decl.type === DeclarationType.Alias);
        }

        if (!aliasDecl) {
            return undefined;
        }

        assert(aliasDecl.type === DeclarationType.Alias);

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

        // Try to resolve the alias while honoring external visibility.
        const resolvedAliasInfo = resolveAliasDeclarationWithInfo(
            aliasDecl,
            /* resolveLocalNames */ true,
            /* allowExternallyHiddenAccess */ fileInfo.isStubFile
        );

        if (!resolvedAliasInfo) {
            return undefined;
        }

        if (!resolvedAliasInfo.declaration) {
            return evaluatorOptions.evaluateUnknownImportsAsAny ? AnyType.create() : UnknownType.create();
        }

        if (node.nodeType === ParseNodeType.ImportFromAs) {
            if (resolvedAliasInfo.isPrivate) {
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportPrivateUsage,
                    DiagnosticRule.reportPrivateUsage,
                    Localizer.Diagnostic.privateUsedOutsideOfModule().format({
                        name: node.name.value,
                    }),
                    node.name
                );
            }

            if (resolvedAliasInfo.privatePyTypedImporter) {
                const diag = new DiagnosticAddendum();
                if (resolvedAliasInfo.privatePyTypedImported) {
                    diag.addMessage(
                        Localizer.DiagnosticAddendum.privateImportFromPyTypedSource().format({
                            module: resolvedAliasInfo.privatePyTypedImported,
                        })
                    );
                }
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportPrivateImportUsage,
                    DiagnosticRule.reportPrivateImportUsage,
                    Localizer.Diagnostic.privateImportFromPyTypedModule().format({
                        name: node.name.value,
                        module: resolvedAliasInfo.privatePyTypedImporter,
                    }) + diag.getString(),
                    node.name
                );
            }
        }

        return getInferredTypeOfDeclaration(symbolWithScope.symbol, aliasDecl);
    }

    // In some cases, an expression must be evaluated in the context of another
    // expression or statement that contains it. This contextual evaluation
    // allows for bidirectional type evaluation.
    function evaluateTypesForExpressionInContext(node: ExpressionNode): void {
        // Check for a couple of special cases where the node is a NameNode but
        // is technically not part of an expression. We'll handle these here so
        // callers don't need to include special-case logic.
        if (node.nodeType === ParseNodeType.Name && node.parent) {
            if (node.parent.nodeType === ParseNodeType.Function && node.parent.name === node) {
                getTypeOfFunction(node.parent);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.Class && node.parent.name === node) {
                getTypeOfClass(node.parent);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.ImportFromAs) {
                evaluateTypesForImportFromAs(node.parent);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.ImportAs) {
                evaluateTypesForImportAs(node.parent);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.TypeAlias && node.parent.name === node) {
                getTypeOfTypeAlias(node.parent);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.Global || node.parent.nodeType === ParseNodeType.Nonlocal) {
                // For global and nonlocal statements, allow forward references so
                // we don't use code flow during symbol lookups.
                getTypeOfExpression(node, EvaluatorFlags.AllowForwardReferences);
                return;
            }

            if (node.parent.nodeType === ParseNodeType.ModuleName) {
                // A name within a module name isn't an expression,
                // so there's nothing we can evaluate here.
                return;
            }
        }

        // If the expression is part of a type annotation, we need to evaluate
        // it with special evaluation flags.
        const annotationNode = ParseTreeUtils.getParentAnnotationNode(node);
        if (annotationNode) {
            // Annotations need to be evaluated with specialized evaluation flags.
            const annotationParent = annotationNode.parent;
            assert(annotationParent !== undefined);

            if (annotationParent.nodeType === ParseNodeType.Assignment) {
                if (annotationNode === annotationParent.typeAnnotationComment) {
                    getTypeOfAnnotation(annotationNode, {
                        isVariableAnnotation: true,
                        allowFinal: ParseTreeUtils.isFinalAllowedForAssignmentTarget(annotationParent.leftExpression),
                        allowClassVar: ParseTreeUtils.isClassVarAllowedForAssignmentTarget(
                            annotationParent.leftExpression
                        ),
                    });
                } else {
                    evaluateTypesForAssignmentStatement(annotationParent);
                }
                return;
            }

            if (annotationParent.nodeType === ParseNodeType.TypeAnnotation) {
                evaluateTypesForTypeAnnotationNode(annotationParent);
                return;
            }

            if (
                annotationParent.nodeType === ParseNodeType.Function &&
                annotationNode === annotationParent.returnTypeAnnotation
            ) {
                getTypeOfAnnotation(annotationNode, {
                    associateTypeVarsWithScope: true,
                });
                return;
            }

            getTypeOfAnnotation(annotationNode, {
                isVariableAnnotation: annotationNode.parent?.nodeType === ParseNodeType.TypeAnnotation,
                allowUnpackedTuple:
                    annotationParent.nodeType === ParseNodeType.Parameter &&
                    annotationParent.category === ParameterCategory.VarArgList,
                allowUnpackedTypedDict:
                    annotationParent.nodeType === ParseNodeType.Parameter &&
                    annotationParent.category === ParameterCategory.VarArgDictionary,
            });
            return;
        }

        // See if the expression is part of a pattern used in a case statement.
        const possibleCaseNode = ParseTreeUtils.getParentNodeOfType(node, ParseNodeType.Case);
        if (possibleCaseNode) {
            const caseNode = possibleCaseNode as CaseNode;
            if (ParseTreeUtils.isNodeContainedWithin(node, caseNode.pattern)) {
                evaluateTypesForCaseStatement(caseNode);
                return;
            }
        }

        // Scan up the parse tree until we find a node that doesn't
        // require any context to be evaluated.
        let nodeToEvaluate: ExpressionNode = node;
        let flags = EvaluatorFlags.None;

        while (true) {
            // If we're within an argument node in a call or index expression, skip
            // all of the nodes between because the entire argument expression
            // needs to be evaluated contextually.
            const argumentNode = ParseTreeUtils.getParentNodeOfType(nodeToEvaluate, ParseNodeType.Argument);
            if (argumentNode && argumentNode !== nodeToEvaluate) {
                assert(argumentNode.parent !== undefined);

                if (
                    argumentNode.parent.nodeType === ParseNodeType.Call ||
                    argumentNode.parent.nodeType === ParseNodeType.Index
                ) {
                    nodeToEvaluate = argumentNode.parent;
                    continue;
                }

                if (argumentNode.parent.nodeType === ParseNodeType.Class) {
                    // If this is an argument node within a class declaration,
                    // evaluate the full class declaration node.
                    getTypeOfClass(argumentNode.parent);
                    return;
                }
            }

            let parent = nodeToEvaluate.parent;
            if (!parent) {
                break;
            }

            // If this is the target of an assignment expression, evaluate the
            // assignment expression node instead.
            if (parent.nodeType === ParseNodeType.AssignmentExpression && nodeToEvaluate === parent.name) {
                nodeToEvaluate = parent;
                continue;
            }

            // The left expression of a call or member access expression is not contextual.
            if (parent.nodeType === ParseNodeType.Call || parent.nodeType === ParseNodeType.MemberAccess) {
                if (nodeToEvaluate === parent.leftExpression) {
                    // Handle the special case where the LHS is a call to super().
                    if (
                        nodeToEvaluate.nodeType === ParseNodeType.Call &&
                        nodeToEvaluate.leftExpression.nodeType === ParseNodeType.Name &&
                        nodeToEvaluate.leftExpression.value === 'super'
                    ) {
                        nodeToEvaluate = parent;
                        continue;
                    }

                    flags = EvaluatorFlags.DoNotSpecialize;
                    break;
                }
            } else if (parent.nodeType === ParseNodeType.Index) {
                // The base expression of an index expression is not contextual.
                if (nodeToEvaluate === parent.baseExpression) {
                    flags = EvaluatorFlags.DoNotSpecialize;
                    break;
                }
            } else if (parent.nodeType === ParseNodeType.StringList && nodeToEvaluate === parent.typeAnnotation) {
                // Forward-declared type annotation expressions need to be be evaluated
                // in context so they have the appropriate flags set. Most of these cases
                // will have been detected above when calling getParentAnnotationNode,
                // but TypeAlias expressions are not handled there.
                nodeToEvaluate = parent;
                continue;
            }

            if (!isExpressionNode(parent)) {
                // If we've hit a non-expression node, we generally want to
                // stop. However, there are a few special "pass through"
                // node types that we can skip over to get to a known
                // expression node.
                if (
                    parent.nodeType === ParseNodeType.DictionaryKeyEntry ||
                    parent.nodeType === ParseNodeType.DictionaryExpandEntry ||
                    parent.nodeType === ParseNodeType.ListComprehensionFor ||
                    parent.nodeType === ParseNodeType.ListComprehensionIf
                ) {
                    assert(parent.parent !== undefined && isExpressionNode(parent.parent));
                    parent = parent.parent;
                } else if (parent.nodeType === ParseNodeType.Parameter) {
                    assert(parent.parent !== undefined);

                    // Parameters are contextual for lambdas.
                    if (parent.parent.nodeType === ParseNodeType.Lambda) {
                        parent = parent.parent;
                    } else {
                        break;
                    }
                } else {
                    break;
                }
            }

            nodeToEvaluate = parent;
        }

        const parent = nodeToEvaluate.parent!;
        assert(parent !== undefined);

        switch (parent.nodeType) {
            case ParseNodeType.Del: {
                verifyDeleteExpression(nodeToEvaluate);
                return;
            }

            case ParseNodeType.TypeParameter: {
                // If this is the name node within a type parameter list, see if it's a type alias
                // definition. If so, we need to evaluate the type alias contextually.
                if (
                    nodeToEvaluate === parent.name &&
                    parent.parent?.nodeType === ParseNodeType.TypeParameterList &&
                    parent.parent.parent?.nodeType === ParseNodeType.TypeAlias
                ) {
                    getTypeOfTypeAlias(parent.parent.parent);
                    return;
                }
                break;
            }

            case ParseNodeType.TypeAlias: {
                getTypeOfTypeAlias(parent);
                return;
            }

            case ParseNodeType.Decorator: {
                if (parent.parent?.nodeType === ParseNodeType.Class) {
                    getTypeOfClass(parent.parent);
                } else if (parent.parent?.nodeType === ParseNodeType.Function) {
                    getTypeOfFunction(parent.parent);
                }
                return;
            }

            case ParseNodeType.Parameter: {
                if (nodeToEvaluate !== parent.defaultValue) {
                    evaluateTypeOfParameter(parent);
                    return;
                }
                break;
            }

            case ParseNodeType.Argument: {
                if (nodeToEvaluate === parent.name) {
                    // A name used to specify a named parameter in an argument isn't an
                    // expression, so there's nothing we can evaluate here.
                    return;
                }

                if (parent.parent?.nodeType === ParseNodeType.Class) {
                    // A class argument must be evaluated in the context of the class declaration.
                    getTypeOfClass(parent.parent);
                    return;
                }
                break;
            }

            case ParseNodeType.Return: {
                // Return expressions must be evaluated in the context of the expected return type.
                if (parent.returnExpression) {
                    const enclosingFunctionNode = ParseTreeUtils.getEnclosingFunction(node);
                    const declaredReturnType = enclosingFunctionNode
                        ? getFunctionDeclaredReturnType(enclosingFunctionNode)
                        : undefined;
                    getTypeOfExpression(
                        parent.returnExpression,
                        EvaluatorFlags.None,
                        makeInferenceContext(declaredReturnType)
                    );
                    return;
                }
                break;
            }

            case ParseNodeType.TypeAnnotation: {
                evaluateTypesForTypeAnnotationNode(parent);
                return;
            }

            case ParseNodeType.Assignment: {
                evaluateTypesForAssignmentStatement(parent);
                return;
            }
        }

        if (nodeToEvaluate.nodeType === ParseNodeType.TypeAnnotation) {
            evaluateTypesForTypeAnnotationNode(nodeToEvaluate);
            return;
        }

        getTypeOfExpression(nodeToEvaluate, flags);
    }

    function evaluateTypeOfParameter(node: ParameterNode): void {
        // If this parameter has no name, we have nothing to do.
        if (!node.name) {
            return;
        }

        // We need to handle lambdas differently from functions because
        // the former never have parameter type annotations but can
        // be inferred, whereas the latter sometimes have type annotations
        // but cannot be inferred.
        const parent = node.parent!;
        if (parent.nodeType === ParseNodeType.Lambda) {
            evaluateTypesForExpressionInContext(parent);
            return;
        }

        assert(parent.nodeType === ParseNodeType.Function);
        const functionNode = parent as FunctionNode;

        const paramIndex = functionNode.parameters.findIndex((param) => param === node);
        const typeAnnotation = ParseTreeUtils.getTypeAnnotationForParameter(functionNode, paramIndex);

        if (typeAnnotation) {
            const param = functionNode.parameters[paramIndex];
            const annotatedType = getTypeOfParameterAnnotation(
                typeAnnotation,
                functionNode.parameters[paramIndex].category
            );

            const adjType = transformVariadicParamType(
                node,
                node.category,
                adjustParameterAnnotatedType(param, annotatedType)
            );

            writeTypeCache(node.name!, { type: adjType }, EvaluatorFlags.None);
            return;
        }

        const containingClassNode = ParseTreeUtils.getEnclosingClass(functionNode, /* stopAtFunction */ true);
        const classInfo = containingClassNode ? getTypeOfClass(containingClassNode) : undefined;

        if (
            classInfo &&
            ClassType.isPseudoGenericClass(classInfo?.classType) &&
            functionNode.name.value === '__init__'
        ) {
            const typeParamName = getPseudoGenericTypeVarName(node.name.value);
            const paramType = classInfo.classType.details.typeParameters.find(
                (param) => param.details.name === typeParamName
            );

            if (paramType) {
                writeTypeCache(node.name!, { type: paramType }, EvaluatorFlags.None);
                return;
            }
        }

        // See if the function is a method in a child class. We may be able to
        // infer the type of the parameter from a method of the same name in
        // a parent class if it has an annotated type.
        const functionFlags = getFunctionFlagsFromDecorators(functionNode, /* isInClass */ true);
        const inferredParamType = inferParameterType(functionNode, functionFlags, paramIndex, classInfo?.classType);

        writeTypeCache(
            node.name!,
            { type: transformVariadicParamType(node, node.category, inferredParamType ?? UnknownType.create()) },
            EvaluatorFlags.None
        );
    }

    // Evaluates the types that are assigned within the statement that contains
    // the specified parse node. In some cases, a broader statement may need to
    // be evaluated to provide sufficient context for the type. Evaluated types
    // are written back to the type cache for later retrieval.
    function evaluateTypesForStatement(node: ParseNode): void {
        initializedBasicTypes(node);

        let curNode: ParseNode | undefined = node;

        while (curNode) {
            switch (curNode.nodeType) {
                case ParseNodeType.Assignment: {
                    // See if the assignment is part of a chain of assignments. If so,
                    // evaluate the entire chain.
                    const isInAssignmentChain =
                        curNode.parent &&
                        (curNode.parent.nodeType === ParseNodeType.Assignment ||
                            curNode.parent.nodeType === ParseNodeType.AssignmentExpression ||
                            curNode.parent.nodeType === ParseNodeType.AugmentedAssignment) &&
                        curNode.parent.rightExpression === curNode;
                    if (!isInAssignmentChain) {
                        evaluateTypesForAssignmentStatement(curNode);
                        return;
                    }
                    break;
                }

                case ParseNodeType.TypeAlias: {
                    getTypeOfTypeAlias(curNode);
                    return;
                }

                case ParseNodeType.AssignmentExpression: {
                    getTypeOfExpression(curNode);
                    return;
                }

                case ParseNodeType.AugmentedAssignment: {
                    evaluateTypesForAugmentedAssignment(curNode);
                    return;
                }

                case ParseNodeType.Class: {
                    getTypeOfClass(curNode);
                    return;
                }

                case ParseNodeType.Parameter: {
                    evaluateTypeOfParameter(curNode);
                    return;
                }

                case ParseNodeType.Lambda: {
                    evaluateTypesForExpressionInContext(curNode);
                    return;
                }

                case ParseNodeType.Function: {
                    getTypeOfFunction(curNode);
                    return;
                }

                case ParseNodeType.For: {
                    evaluateTypesForForStatement(curNode);
                    return;
                }

                case ParseNodeType.Except: {
                    evaluateTypesForExceptStatement(curNode);
                    return;
                }

                case ParseNodeType.WithItem: {
                    evaluateTypesForWithStatement(curNode);
                    return;
                }

                case ParseNodeType.ListComprehensionFor: {
                    const listComprehension = curNode.parent as ListComprehensionNode;
                    assert(listComprehension.nodeType === ParseNodeType.ListComprehension);
                    if (curNode === listComprehension.expression) {
                        evaluateTypesForExpressionInContext(listComprehension);
                    } else {
                        // Evaluate the individual iterations starting with the first
                        // up to the curNode.
                        for (const forIfNode of listComprehension.forIfNodes) {
                            evaluateListComprehensionForIf(forIfNode);
                            if (forIfNode === curNode) {
                                break;
                            }
                        }
                    }
                    return;
                }

                case ParseNodeType.ImportAs: {
                    evaluateTypesForImportAs(curNode);
                    return;
                }

                case ParseNodeType.ImportFromAs: {
                    evaluateTypesForImportFromAs(curNode);
                    return;
                }

                case ParseNodeType.ImportFrom: {
                    evaluateTypesForImportFrom(curNode);
                    return;
                }

                case ParseNodeType.Case: {
                    evaluateTypesForCaseStatement(curNode);
                    return;
                }
            }

            curNode = curNode.parent;
        }

        fail('Unexpected statement');
        return undefined;
    }

    // Helper function for cases where we need to evaluate the types
    // for a subtree so we can determine the type of one of the subnodes
    // within that tree. If the type cannot be determined (because it's part
    // of a cyclical dependency), the function returns undefined.
    function evaluateTypeForSubnode(subnode: ParseNode, callback: () => void): TypeResult | undefined {
        // If the type cache is already populated with a complete type,
        // don't bother doing additional work.
        let cacheEntry = readTypeCacheEntry(subnode);
        if (cacheEntry && !cacheEntry.typeResult.isIncomplete) {
            return cacheEntry.typeResult;
        }

        callback();
        cacheEntry = readTypeCacheEntry(subnode);
        if (cacheEntry) {
            return cacheEntry.typeResult;
        }

        return undefined;
    }

    function getCodeFlowAnalyzerForNode(nodeId: number) {
        let analyzer = codeFlowAnalyzerCache.get(nodeId);

        if (!analyzer) {
            // Allocate a new code flow analyzer.
            analyzer = codeFlowEngine.createCodeFlowAnalyzer();
            codeFlowAnalyzerCache.set(nodeId, analyzer);
        }

        return analyzer;
    }

    // Attempts to determine the type of the reference expression at the
    // point in the code. If the code flow analysis has nothing to say
    // about that expression, it returns un undefined type. Normally
    // flow analysis starts from the reference node, but startNode can be
    // specified to override this in a few special cases (functions and
    // lambdas) to support analysis of captured variables.
    function getFlowTypeOfReference(
        reference: CodeFlowReferenceExpressionNode,
        targetSymbolId: number,
        typeAtStart: Type,
        startNode?: FunctionNode | LambdaNode,
        options?: FlowNodeTypeOptions
    ): FlowNodeTypeResult {
        // See if this execution scope requires code flow for this reference expression.
        const referenceKey = createKeyForReference(reference);
        const executionNode = ParseTreeUtils.getExecutionScopeNode(startNode?.parent ?? reference);
        const codeFlowExpressions = AnalyzerNodeInfo.getCodeFlowExpressions(executionNode);

        if (
            !codeFlowExpressions ||
            (!codeFlowExpressions.has(referenceKey) && !codeFlowExpressions.has(wildcardImportReferenceKey))
        ) {
            return { type: undefined, isIncomplete: false };
        }

        if (checkCodeFlowTooComplex(reference)) {
            return { type: undefined, isIncomplete: true };
        }

        // Is there an code flow analyzer cached for this execution scope?
        let analyzer: CodeFlowAnalyzer | undefined;

        if (isNodeInReturnTypeInferenceContext(executionNode)) {
            // If we're performing the analysis within a temporary
            // context of a function for purposes of inferring its
            // return type for a specified set of arguments, use
            // a temporary analyzer that we'll use only for this context.
            analyzer = getCodeFlowAnalyzerForReturnTypeInferenceContext();
        } else {
            analyzer = getCodeFlowAnalyzerForNode(executionNode.id);
        }

        const flowNode = AnalyzerNodeInfo.getFlowNode(startNode ?? reference);
        if (flowNode === undefined) {
            return { type: undefined, isIncomplete: false };
        }

        return analyzer.getTypeFromCodeFlow(flowNode!, reference, targetSymbolId, typeAtStart, options);
    }

    // Specializes the specified (potentially generic) class type using
    // the specified type arguments, reporting errors as appropriate.
    // Returns the specialized type and a boolean indicating whether
    // the type indicates a class type (true) or an object type (false).
    function createSpecializedClassType(
        classType: ClassType,
        typeArgs: TypeResultWithNode[] | undefined,
        flags: EvaluatorFlags,
        errorNode: ParseNode
    ): TypeResult {
        // Handle the special-case classes that are not defined
        // in the type stubs.
        if (ClassType.isSpecialBuiltIn(classType)) {
            const aliasedName = classType.aliasName || classType.details.name;
            switch (aliasedName) {
                case 'Callable': {
                    return { type: createCallableType(typeArgs, errorNode) };
                }

                case 'Never': {
                    if (typeArgs && typeArgs.length > 0) {
                        addError(
                            Localizer.Diagnostic.typeArgsExpectingNone().format({ name: 'Never' }),
                            typeArgs[0].node
                        );
                    }
                    return { type: NeverType.createNever() };
                }

                case 'NoReturn': {
                    if (typeArgs && typeArgs.length > 0) {
                        addError(
                            Localizer.Diagnostic.typeArgsExpectingNone().format({ name: 'NoReturn' }),
                            typeArgs[0].node
                        );
                    }
                    return { type: NeverType.createNoReturn() };
                }

                case 'Optional': {
                    return { type: createOptionalType(classType, errorNode, typeArgs, flags) };
                }

                case 'Type': {
                    // PEP 484 says that Type[Any] should be considered
                    // equivalent to type.
                    if (
                        typeArgs?.length === 1 &&
                        isAnyOrUnknown(typeArgs[0].type) &&
                        typeClassType &&
                        isInstantiableClass(typeClassType)
                    ) {
                        return { type: typeClassType };
                    }

                    let typeType = createSpecialType(classType, typeArgs, 1);
                    if (isInstantiableClass(typeType)) {
                        typeType = explodeGenericClass(typeType);
                    }
                    return { type: typeType };
                }

                case 'ClassVar': {
                    return { type: createClassVarType(classType, errorNode, typeArgs, flags) };
                }

                case 'Protocol': {
                    return {
                        type: createSpecialType(
                            classType,
                            typeArgs,
                            /* paramLimit */ undefined,
                            /* allowParamSpec */ true
                        ),
                    };
                }

                case 'Tuple': {
                    return { type: createSpecialType(classType, typeArgs, /* paramLimit */ undefined) };
                }

                case 'Union': {
                    return { type: createUnionType(classType, errorNode, typeArgs, flags) };
                }

                case 'Generic': {
                    return { type: createGenericType(classType, errorNode, typeArgs, flags) };
                }

                case 'Final': {
                    return { type: createFinalType(classType, errorNode, typeArgs, flags) };
                }

                case 'Annotated': {
                    return createAnnotatedType(errorNode, typeArgs);
                }

                case 'Concatenate': {
                    return { type: createConcatenateType(errorNode, classType, typeArgs) };
                }

                case 'TypeGuard':
                case 'StrictTypeGuard': {
                    return { type: createTypeGuardType(errorNode, classType, typeArgs, flags) };
                }

                case 'Unpack': {
                    return { type: createUnpackType(classType, errorNode, typeArgs, flags) };
                }

                case 'Required':
                case 'NotRequired': {
                    return createRequiredType(classType, errorNode, aliasedName === 'Required', typeArgs, flags);
                }

                case 'Self': {
                    return { type: createSelfType(classType, errorNode, typeArgs) };
                }

                case 'LiteralString': {
                    return { type: createSpecialType(classType, typeArgs, 0) };
                }
            }
        }

        const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
        if (
            fileInfo.isStubFile ||
            fileInfo.executionEnvironment.pythonVersion >= PythonVersion.V3_9 ||
            isAnnotationEvaluationPostponed(AnalyzerNodeInfo.getFileInfo(errorNode)) ||
            (flags & EvaluatorFlags.AllowForwardReferences) !== 0
        ) {
            // Handle "type" specially, since it needs to act like "Type"
            // in Python 3.9 and newer.
            if (ClassType.isBuiltIn(classType, 'type') && typeArgs) {
                // PEP 484 says that type[Any] should be considered
                // equivalent to type.
                if (typeArgs.length === 1 && isAnyOrUnknown(typeArgs[0].type)) {
                    return { type: classType };
                }

                const typeClass = getTypingType(errorNode, 'Type');
                if (typeClass && isInstantiableClass(typeClass)) {
                    let typeType = createSpecialType(
                        typeClass,
                        typeArgs,
                        1,
                        /* allowParamSpec */ undefined,
                        /* isCallable */ true
                    );

                    if (isInstantiableClass(typeType)) {
                        typeType = explodeGenericClass(typeType);
                    }

                    return { type: typeType };
                }
            }

            // Handle "tuple" specially, since it needs to act like "Tuple"
            // in Python 3.9 and newer.
            if (isTupleClass(classType)) {
                return {
                    type: createSpecialType(
                        classType,
                        typeArgs,
                        /* paramLimit */ undefined,
                        /* allowParamSpec */ undefined,
                        /* isCallable */ true
                    ),
                };
            }
        }

        let typeArgCount = typeArgs ? typeArgs.length : 0;

        // Make sure the argument list count is correct.
        const typeParameters = ClassType.isPseudoGenericClass(classType) ? [] : ClassType.getTypeParameters(classType);

        // If there are no type parameters or args, the class is already specialized.
        // No need to do any more work.
        if (typeParameters.length === 0 && typeArgCount === 0) {
            return { type: classType };
        }

        const variadicTypeParamIndex = typeParameters.findIndex((param) => isVariadicTypeVar(param));

        if (typeArgs) {
            let minTypeArgCount = typeParameters.length;
            const firstNonDefaultParam = typeParameters.findIndex((param) => !!param.details.defaultType);

            if (firstNonDefaultParam >= 0) {
                minTypeArgCount = firstNonDefaultParam;
            }

            // Classes that accept inlined type dict type args allow only one.
            if (typeArgs[0].inlinedTypeDict) {
                if (typeArgs.length > 1) {
                    addDiagnostic(
                        fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeArgsTooMany().format({
                            name: classType.aliasName || classType.details.name,
                            expected: 1,
                            received: typeArgCount,
                        }),
                        typeArgs[1].node
                    );
                }

                return { type: typeArgs[0].inlinedTypeDict };
            } else if (typeArgCount > typeParameters.length) {
                if (!ClassType.isPartiallyEvaluated(classType) && !ClassType.isTupleClass(classType)) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                    if (typeParameters.length === 0) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeArgsExpectingNone().format({
                                name: classType.aliasName || classType.details.name,
                            }),
                            typeArgs[typeParameters.length].node
                        );
                    } else if (typeParameters.length !== 1 || !isParamSpec(typeParameters[0])) {
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeArgsTooMany().format({
                                name: classType.aliasName || classType.details.name,
                                expected: typeParameters.length,
                                received: typeArgCount,
                            }),
                            typeArgs[typeParameters.length].node
                        );
                    }

                    typeArgCount = typeParameters.length;
                }
            } else if (typeArgCount < minTypeArgCount) {
                const fileInfo = AnalyzerNodeInfo.getFileInfo(errorNode);
                addDiagnostic(
                    fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                    DiagnosticRule.reportGeneralTypeIssues,
                    Localizer.Diagnostic.typeArgsTooFew().format({
                        name: classType.aliasName || classType.details.name,
                        expected: minTypeArgCount,
                        received: typeArgCount,
                    }),
                    typeArgs.length > 0 ? typeArgs[0].node.parent! : errorNode
                );
            }

            typeArgs.forEach((typeArg, index) => {
                if (index === variadicTypeParamIndex) {
                    // The types that make up the tuple that maps to the variadic
                    // type variable have already been validated when the tuple
                    // object was created in adjustTypeArgumentsForVariadicTypeVar.
                    if (isClassInstance(typeArg.type) && isTupleClass(typeArg.type)) {
                        return;
                    }

                    if (isVariadicTypeVar(typeArg.type)) {
                        validateVariadicTypeVarIsUnpacked(typeArg.type, typeArg.node);
                        return;
                    }
                }

                const typeParam = index < typeParameters.length ? typeParameters[index] : undefined;
                const isParamSpecTarget = typeParam?.details.isParamSpec;

                validateTypeArg(typeArg, {
                    allowParamSpec: true,
                    allowTypeArgList: isParamSpecTarget,
                });
            });
        }

        // Handle ParamSpec arguments and fill in any missing type arguments with Unknown.
        let typeArgTypes: Type[] = [];
        const fullTypeParams = ClassType.getTypeParameters(classType);

        // PEP 612 says that if the class has only one type parameter consisting
        // of a ParamSpec, the list of arguments does not need to be enclosed in
        // a list. We'll handle that case specially here.
        if (fullTypeParams.length === 1 && fullTypeParams[0].details.isParamSpec && typeArgs) {
            if (
                typeArgs.every(
                    (typeArg) => !isEllipsisType(typeArg.type) && !typeArg.typeList && !isParamSpec(typeArg.type)
                )
            ) {
                if (
                    typeArgs.length !== 1 ||
                    !isInstantiableClass(typeArgs[0].type) ||
                    !ClassType.isBuiltIn(typeArgs[0].type, 'Concatenate')
                ) {
                    // Package up the type arguments into a typeList.
                    typeArgs = [
                        {
                            type: UnknownType.create(),
                            node: typeArgs[0].node,
                            typeList: typeArgs,
                        },
                    ];
                }
            } else if (typeArgs.length > 1) {
                const paramSpecTypeArg = typeArgs.find((typeArg) => isParamSpec(typeArg.type));
                if (paramSpecTypeArg) {
                    addError(Localizer.Diagnostic.paramSpecContext(), paramSpecTypeArg.node);
                }

                const listTypeArg = typeArgs.find((typeArg) => !!typeArg.typeList);
                if (listTypeArg) {
                    addError(Localizer.Diagnostic.typeArgListNotAllowed(), listTypeArg.node);
                }
            }
        }

        const typeVarContext = new TypeVarContext(classType.details.typeVarScopeId);

        fullTypeParams.forEach((typeParam, index) => {
            if (typeArgs && index < typeArgs.length) {
                if (typeParam.details.isParamSpec) {
                    const typeArg = typeArgs[index];
                    const functionType = FunctionType.createSynthesizedInstance('', FunctionTypeFlags.ParamSpecValue);
                    TypeBase.setSpecialForm(functionType);

                    if (isEllipsisType(typeArg.type)) {
                        FunctionType.addDefaultParameters(functionType);
                        functionType.details.flags |= FunctionTypeFlags.SkipArgsKwargsCompatibilityCheck;
                        typeArgTypes.push(functionType);
                        typeVarContext.setTypeVarType(typeParam, convertTypeToParamSpecValue(functionType));
                        return;
                    }

                    if (typeArg.typeList) {
                        typeArg.typeList!.forEach((paramType, paramIndex) => {
                            FunctionType.addParameter(functionType, {
                                category: ParameterCategory.Simple,
                                name: `__p${paramIndex}`,
                                isNameSynthesized: true,
                                type: convertToInstance(paramType.type),
                                hasDeclaredType: true,
                            });
                        });
                        typeArgTypes.push(functionType);
                        typeVarContext.setTypeVarType(typeParam, convertTypeToParamSpecValue(functionType));
                        return;
                    }

                    if (isInstantiableClass(typeArg.type) && ClassType.isBuiltIn(typeArg.type, 'Concatenate')) {
                        const concatTypeArgs = typeArg.type.typeArguments;
                        if (concatTypeArgs && concatTypeArgs.length > 0) {
                            concatTypeArgs.forEach((typeArg, index) => {
                                if (index === concatTypeArgs.length - 1) {
                                    if (isParamSpec(typeArg)) {
                                        functionType.details.paramSpec = typeArg;
                                    }
                                } else {
                                    FunctionType.addParameter(functionType, {
                                        category: ParameterCategory.Simple,
                                        name: `__p${index}`,
                                        isNameSynthesized: true,
                                        hasDeclaredType: true,
                                        type: typeArg,
                                    });
                                }
                            });
                        }

                        typeArgTypes.push(functionType);
                        return;
                    }
                }

                const typeArgType = convertToInstance(typeArgs[index].type);
                typeArgTypes.push(typeArgType);
                typeVarContext.setTypeVarType(typeParam, typeArgType);
                return;
            }

            const solvedDefaultType = applySolvedTypeVars(typeParam, typeVarContext, { unknownIfNotFound: true });
            typeArgTypes.push(solvedDefaultType);
            if (isParamSpec(typeParam)) {
                typeVarContext.setTypeVarType(typeParam, convertTypeToParamSpecValue(solvedDefaultType));
            } else {
                typeVarContext.setTypeVarType(typeParam, solvedDefaultType);
            }
        });

        typeArgTypes = typeArgTypes.map((typeArgType, index) => {
            if (index < typeArgCount) {
                const diag = new DiagnosticAddendum();
                let adjustedTypeArgType = applyTypeArgToTypeVar(typeParameters[index], typeArgType, diag);

                // Determine if the variance must match.
                if (adjustedTypeArgType && (flags & EvaluatorFlags.EnforceTypeVarVarianceConsistency) !== 0) {
                    const destType = typeParameters[index];
                    const declaredVariance = destType.details.declaredVariance;

                    if (!isVarianceOfTypeArgumentCompatible(adjustedTypeArgType, declaredVariance)) {
                        diag.addMessage(
                            Localizer.DiagnosticAddendum.varianceMismatchForClass().format({
                                typeVarName: printType(adjustedTypeArgType),
                                className: classType.details.name,
                            })
                        );
                        adjustedTypeArgType = undefined;
                    }
                }

                if (adjustedTypeArgType) {
                    typeArgType = adjustedTypeArgType;
                } else {
                    // Avoid emitting this error for a partially-constructed class.
                    if (!isClassInstance(typeArgType) || !ClassType.isPartiallyEvaluated(typeArgType)) {
                        const fileInfo = AnalyzerNodeInfo.getFileInfo(typeArgs![index].node);
                        addDiagnostic(
                            fileInfo.diagnosticRuleSet.reportGeneralTypeIssues,
                            DiagnosticRule.reportGeneralTypeIssues,
                            Localizer.Diagnostic.typeVarAssignmentMismatch().format({
                                type: printType(typeArgType),
                                name: TypeVarType.getReadableName(typeParameters[index]),
                            }) + diag.getString(),
                            typeArgs![index].node
                        );
                    }
                }
            }

            return typeArgType;
        });

        // If the class is partially constructed and doesn't yet have
        // type parameters, assume that the number and types of supplied type
        // arguments are correct.
        if (typeArgs && classType.details.typeParameters.length === 0 && ClassType.isPartiallyEvaluated(classType)) {
            typeArgTypes = typeArgs.map((t) => convertToInstance(t.type));
        }

        const specializedClass = ClassType.cloneForSpecialization(classType, typeArgTypes, typeArgs !== undefined);

        return { type: specializedClass };
    }

    function getTypeOfArgument(arg: FunctionArgument): TypeResult {
        if (arg.typeResult) {
            return { type: arg.typeResult.type, isIncomplete: arg.typeResult.isIncomplete };
        }

        if (!arg.valueExpression) {
            // We shouldn't ever get here, but just in case.
            return { type: UnknownType.create() };
        }

        // If there was no defined type provided, there should always
        // be a value expression from which we can retrieve the type.
        return getTypeOfExpression(arg.valueExpression);
    }

    // This function is like getTypeOfArgument except that it is
    // used in cases where the argument is expected to be a type
    // and therefore follows the normal rules of types (e.g. they
    // can be forward-declared in stubs, etc.).
    function getTypeOfArgumentExpectingType(arg: FunctionArgument): TypeResult {
        if (arg.typeResult) {
            return { type: arg.typeResult.type, isIncomplete: arg.typeResult.isIncomplete };
        }

        // If there was no defined type provided, there should always
        // be a value expression from which we can retrieve the type.
        return getTypeOfExpressionExpectingType(arg.valueExpression!);
    }

    function getTypeOfExpressionExpectingType(node: ExpressionNode, options?: ExpectedTypeOptions): TypeResult {
        let flags =
            EvaluatorFlags.ExpectingType | EvaluatorFlags.EvaluateStringLiteralAsType | EvaluatorFlags.DisallowClassVar;

        const fileInfo = AnalyzerNodeInfo.getFileInfo(node);
        if (fileInfo.isStubFile) {
            flags |= EvaluatorFlags.AllowForwardReferences;
        } else {
            flags |= EvaluatorFlags.InterpreterParsesStringLiteral;
        }

        if (!options?.allowFinal) {
            flags |= EvaluatorFlags.DisallowFinal;
        }

        if (options?.allowRequired) {
            flags |= EvaluatorFlags.AllowRequired | EvaluatorFlags.ExpectingTypeAnnotation;
        }

        if (options?.allowUnpackedTuple) {
            flags |= EvaluatorFlags.AllowUnpackedTupleOrTypeVarTuple;
        } else {
            flags |= EvaluatorFlags.DisallowTypeVarTuple;
        }

        if (!options?.allowParamSpec) {
            flags |= EvaluatorFlags.DisallowParamSpec;
        }

        return getTypeOfExpression(node, flags);
    }

    function getBuiltInType(node: ParseNode, name: string): Type {
        const scope = ScopeUtils.getScopeForNode(node);
        if (scope) {
            const builtInScope = ScopeUtils.getBuiltInScope(scope);
            const nameType = builtInScope.lookUpSymbol(name);
            if (nameType) {
                return getEffectiveTypeOfSymbol(nameType);
            }
        }

        return UnknownType.create();
    }

    function getBuiltInObject(node: ParseNode, name: string, typeArguments?: Type[]) {
        const nameType = getBuiltInType(node, name);
        if (isInstantiableClass(nameType)) {
            let classType = nameType;
            if (typeArguments) {
                classType = ClassType.cloneForSpecialization(
                    classType,
                    typeArguments,
                    /* isTypeArgumentExplicit */ typeArguments !== undefined
                );
            }

            return ClassType.cloneAsInstance(classType);
        }

        return nameType;
    }

    function lookUpSymbolRecursive(
        node: ParseNode,
        name: string,
        honorCodeFlow: boolean,
        preferGlobalScope = false
    ): SymbolWithScope | undefined {
        const scope = ScopeUtils.getScopeForNode(node);
        let symbolWithScope = scope?.lookUpSymbolRecursive(name);
        const scopeType = scope?.type ?? ScopeType.Module;

        // Functions and list comprehensions don't allow access to implicitly
        // aliased symbols in outer scopes if they haven't yet been assigned
        // within the local scope.
        const scopeTypeHonorsCodeFlow = scopeType !== ScopeType.Function && scopeType !== ScopeType.ListComprehension;

        if (symbolWithScope && honorCodeFlow && scopeTypeHonorsCodeFlow) {
            // Filter the declarations based on flow reachability.
            const reachableDecl = symbolWithScope.symbol.getDeclarations().find((decl) => {
                if (decl.type !== DeclarationType.Alias && decl.type !== DeclarationType.Intrinsic) {
                    // Is the declaration in the same execution scope as the "usageNode" node?
                    const usageScope = ParseTreeUtils.getExecutionScopeNode(node);
                    const declNode =
                        decl.type === DeclarationType.Class ||
                        decl.type === DeclarationType.Function ||
                        decl.type === DeclarationType.TypeAlias
                            ? decl.node.name
                            : decl.node;
                    const declScope = ParseTreeUtils.getExecutionScopeNode(declNode);
                    if (usageScope === declScope) {
                        if (!isFlowPathBetweenNodes(declNode, node)) {
                            // If there was no control flow path from the usage back
                            // to the source, see if the usage node is reachable by
                            // any path.
                            const flowNode = AnalyzerNodeInfo.getFlowNode(node);
                            const isReachable =
                                flowNode &&
                                codeFlowEngine.isFlowNodeReachable(
                                    flowNode,
                                    /* sourceFlowNode */ undefined,
                                    /* ignoreNoReturn */ true
                                );
                            return !isReachable;
                        }
                    }
                }
                return true;
            });

            // If none of the declarations are reachable from the current node,
            // search for the symbol in outer scopes.
            if (!reachableDecl) {
                if (symbolWithScope.scope.type !== ScopeType.Function) {
                    let nextScopeToSearch = symbolWithScope.scope.parent;
                    const isOutsideCallerModule =
                        symbolWithScope.isOutsideCallerModule || symbolWithScope.scope.type === ScopeType.Module;
                    let isBeyondExecutionScope =
                        symbolWithScope.isBeyondExecutionScope || symbolWithScope.scope.isIndependentlyExecutable();

                    if (symbolWithScope.scope.type === ScopeType.Class) {
                        // There is an odd documented behavior for classes in that
                        // symbol resolution skips to the global scope rather than
                        // the next scope in the chain.
                        const globalScopeResult = symbolWithScope.scope.getGlobalScope();
                        nextScopeToSearch = globalScopeResult.scope;
                        if (globalScopeResult.isBeyondExecutionScope) {
                            isBeyondExecutionScope = true;
                        }
                    }

                    if (nextScopeToSearch) {
                        symbolWithScope = nextScopeToSearch.lookUpSymbolRecursive(
                            name,
                            isOutsideCallerModule,
                            isBeyondExecutionScope
                        );
                    } else {
                        symbolWithScope = undefined;
                    }
                } else {
                    symbolWithScope = undefined;
                }
            }
        }

        // PEP 563 indicates that if a forward reference can be resolved in the module
        // scope (or, by implication, in the builtins scope), it should prefer that
        // resolution over local resolutions.
        if (symbolWithScope && preferGlobalScope) {
            let curSymbolWithScope: SymbolWithScope | undefined = symbolWithScope;
            while (
                curSymbolWithScope.scope.type !== ScopeType.Module &&
                curSymbolWithScope.scope.type !== ScopeType.Builtin &&
                curSymbolWithScope.scope.parent
            ) {
                curSymbolWithScope = curSymbolWithScope.scope.parent.lookUpSymbolRecursive(
                    name,
                    curSymbolWithScope.isOutsideCallerModule,
                    curSymbolWithScope.isBeyondExecutionScope || curSymbolWithScope.scope.isIndependentlyExecutable()
                );
                if (!curSymbolWithScope) {
                    break;
                }
            }

            if (
                curSymbolWithScope?.scope.type === ScopeType.Module ||
                curSymbolWithScope?.scope.type === ScopeType.Builtin
            ) {
                symbolWithScope = curSymbolWithScope;
            }
        }

        return symbolWithScope;
    }

    // Disables recording of errors and warnings.
    function suppressDiagnostics<T>(node: ParseNode, callback: () => T) {
        suppressedNodeStack.push(node);

        try {
            const result = callback();
            suppressedNodeStack.pop();
            return result;
        } catch (e) {
            // We don't use finally here because the TypeScript debugger doesn't
            // handle finally well when single stepping.
            suppressedNodeStack.pop();
            throw e;
        }
    }

    // Disables recording of errors and warnings and disables
    // any caching of types, under the assumption that we're
    // performing speculative evaluations.
    function useSpeculativeMode<T>(speculativeNode: ParseNode, callback: () => T, allowCacheRetention = true) {
        speculativeTypeTracker.enterSpeculativeContext(speculativeNode, allowCacheRetention);

        try {
            const result = callback();
            speculativeTypeTracker.leaveSpeculativeContext();
            return result;
        } catch (e) {
            // We don't use finally here because the TypeScript debugger doesn't
            // handle finally well when single stepping.
            speculativeTypeTracker.leaveSpeculativeContext();
            throw e;
        }
    }

    function disableSpeculativeMode(callback: () => void) {
        const stack = speculativeTypeTracker.disableSpeculativeMode();

        try {
            callback();
            speculativeTypeTracker.enableSpeculativeMode(stack);
        } catch (e) {
            // We don't use finally here because the TypeScript debugger doesn't
            // handle finally well when single stepping.
            speculativeTypeTracker.enableSpeculativeMode(stack);
            throw e;
        }
    }

    function getDeclarationFromFunctionNamedParameter(type: FunctionType, paramName: string): Declaration | undefined {
        if (isFunction(type)) {
            if (type.details.declaration) {
                const functionDecl = type.details.declaration;
                if (functionDecl.type === DeclarationType.Function) {
                    const functionNode = functionDecl.node;
                    const functionScope = AnalyzerNodeInfo.getScope(functionNode);
                    if (functionScope) {
                        const paramSymbol = functionScope.lookUpSymbol(paramName)!;
                        if (paramSymbol) {
                            return paramSymbol
                                .getDeclarations()
                                .find((decl) => decl.type === DeclarationType.Parameter);
                        }
                    }
                }
            }
        }

        return undefined;
    }

    // In general, string nodes don't have any declarations associated with them, but
    // we need to handle the special case of string literals used as keys within a
    // dictionary expression where those keys are associated with a known TypedDict.
    function getDeclarationsForStringNode(node: StringNode): Declaration[] | undefined {
        const declarations: Declaration[] = [];
        const expectedType = getExpectedType(node)?.type;

        if (expectedType) {
            doForEachSubtype(expectedType, (subtype) => {
                // If the expected type is a TypedDict then the node is either a key expression
                // or a single entry in a set. We then need to check that the value of the node
                // is a valid entry in the TypedDict to avoid resolving declarations for
                // synthesized symbols such as 'get'.
                if (isClassInstance(subtype) && ClassType.isTypedDictClass(subtype)) {
                    const entry = subtype.details.typedDictEntries?.get(node.value);
                    if (entry) {
                        const symbol = lookUpObjectMember(subtype, node.value)?.symbol;

                        if (symbol) {
                            appendArray(declarations, symbol.getDeclarations());
                        }
                    }
                }
            });
        }

        return declarations.length === 0 ? undefined : declarations;
    }

    function getAliasFromImport(node: NameNode): NameNode | undefined {
        if (
            node.parent &&
            node.parent.nodeType === ParseNodeType.ImportFromAs &&
            node.parent.alias &&
            node === node.parent.name
        ) {
            return node.parent.alias;
        }
        return undefined;
    }

    function getDeclarationsForNameNode(node: NameNode, skipUnreachableCode = true): Declaration[] | undefined {
        if (skipUnreachableCode && AnalyzerNodeInfo.isCodeUnreachable(node)) {
            return undefined;
        }

        const declarations: Declaration[] = [];

        // If the node is part of a "from X import Y as Z" statement and the node
        // is the "Y" (non-aliased) name, we need to look up the alias symbol
        // since the non-aliased name is not in the symbol table.
        const alias = getAliasFromImport(node);
        if (alias) {
            const scope = ScopeUtils.getScopeForNode(node);
            if (scope) {
                // Look up the alias symbol.
                const symbolInScope = scope.lookUpSymbolRecursive(alias.value);
                if (symbolInScope) {
                    // The alias could have more decls that don't refer to this import. Filter
                    // out the one(s) that specifically associated with this import statement.
                    const declsForThisImport = symbolInScope.symbol.getDeclarations().filter((decl) => {
                        return decl.type === DeclarationType.Alias && decl.node === node.parent;
                    });

                    appendArray(declarations, getDeclarationsWithUsesLocalNameRemoved(declsForThisImport));
                }
            }
        } else if (
            node.parent &&
            node.parent.nodeType === ParseNodeType.MemberAccess &&
            node === node.parent.memberName
        ) {
            let baseType = getType(node.parent.leftExpression);
            if (baseType) {
                baseType = makeTopLevelTypeVarsConcrete(baseType);
                const memberName = node.parent.memberName.value;
                doForEachSubtype(baseType, (subtype) => {
                    let symbol: Symbol | undefined;

                    subtype = makeTopLevelTypeVarsConcrete(subtype);

                    if (isInstantiableClass(subtype)) {
                        // Try to find a member that has a declared type. If so, that
                        // overrides any inferred types.
                        let member = lookUpClassMember(subtype, memberName, ClassMemberLookupFlags.DeclaredTypesOnly);
                        if (!member) {
                            member = lookUpClassMember(subtype, memberName);
                        }

                        if (!member) {
                            const metaclass = subtype.details.effectiveMetaclass;
                            if (metaclass && isInstantiableClass(metaclass)) {
                                member = lookUpClassMember(metaclass, memberName);
                            }
                        }

                        if (member) {
                            symbol = member.symbol;
                        }
                    } else if (isClassInstance(subtype)) {
                        // Try to find a member that has a declared type. If so, that
                        // overrides any inferred types.
                        let member = lookUpObjectMember(subtype, memberName, ClassMemberLookupFlags.DeclaredTypesOnly);
                        if (!member) {
                            member = lookUpObjectMember(subtype, memberName);
                        }
                        if (member) {
                            symbol = member.symbol;
                        }
                    } else if (isModule(subtype)) {
                        symbol = ModuleType.getField(subtype, memberName);
                    }

                    if (symbol) {
                        // By default, report only the declarations that have type annotations.
                        // If there are none, then report all of the unannotated declarations,
                        // which includes every assignment of that symbol.
                        const typedDecls = symbol.getTypedDeclarations();
                        if (typedDecls.length > 0) {
                            appendArray(declarations, typedDecls);
                        } else {
                            appendArray(declarations, symbol.getDeclarations());
                        }
                    }
                });
            }
        } else if (node.parent && node.parent.nodeType === ParseNodeType.ModuleName) {
            const namePartIndex = node.parent.nameParts.findIndex((part) => part === node);
            const importInfo = AnalyzerNodeInfo.getImportInfo(node.parent);
            if (
                namePartIndex >= 0 &&
                importInfo &&
                !importInfo.isNativeLib &&
                namePartIndex < importInfo.resolvedPaths.length
            ) {
                if (importInfo.resolvedPaths[namePartIndex]) {
                    evaluateTypesForStatement(node);

                    // Synthesize an alias declaration for this name part. The only
                    // time this case is used is for IDE services such as
                    // the find all references, hover provider and etc.
                    declarations.push(createSynthesizedAliasDeclaration(importInfo.resolvedPaths[namePartIndex]));
                }
            }
        } else if (node.parent && node.parent.nodeType === ParseNodeType.Argument && node === node.parent.name) {
            // The target node is the name in a named argument. We need to determine whether
            // the corresponding named parameter can be determined from the context.
            const argNode = node.parent;
            const paramName = node.value;
            if (argNode.parent && argNode.parent.nodeType === ParseNodeType.Call) {
                const baseType = getType(argNode.parent.leftExpression);

                if (baseType) {
                    if (isFunction(baseType) && baseType.details.declaration) {
                        const paramDecl = getDeclarationFromFunctionNamedParameter(baseType, paramName);
                        if (paramDecl) {
                            declarations.push(paramDecl);
                        }
                    } else if (isOverloadedFunction(baseType)) {
                        baseType.overloads.forEach((f) => {
                            const paramDecl = getDeclarationFromFunctionNamedParameter(f, paramName);
                            if (paramDecl) {
                                declarations.push(paramDecl);
                            }
                        });
                    } else if (isInstantiableClass(baseType)) {
                        const initMethodType = getTypeOfObjectMember(
                            argNode.parent.leftExpression,
                            ClassType.cloneAsInstance(baseType),
                            '__init__',
                            { method: 'get' },
                            /* diag */ undefined,
                            MemberAccessFlags.SkipObjectBaseClass
                        )?.type;

                        if (initMethodType && isFunction(initMethodType)) {
                            const paramDecl = getDeclarationFromFunctionNamedParameter(initMethodType, paramName);
                            if (paramDecl) {
                                declarations.push(paramDecl);
                            } else if (ClassType.isDataClass(baseType) || ClassType.isTypedDictClass(baseType)) {
                                const lookupResults = lookUpClassMember(baseType, paramName);
                                if (lookupResults) {
                                    appendArray(declarations, lookupResults.symbol.getDeclarations());
                                }
                            }
                        }
                    }
                }
            }
        } else {
            const fileInfo = AnalyzerNodeInfo.getFileInfo(node);

            // Determine if this node is within a quoted type annotation.
            const isWithinTypeAnnotation = ParseTreeUtils.isWithinTypeAnnotation(
                node,
                !isAnnotationEvaluationPostponed(AnalyzerNodeInfo.getFileInfo(node))
            );
            const allowForwardReferences = isWithinTypeAnnotation || fileInfo.isStubFile;

            let symbol: Symbol | undefined;
            const typeParamSymbol = AnalyzerNodeInfo.getTypeParameterSymbol(node);
            if (typeParamSymbol) {
                symbol = typeParamSymbol;
            } else {
                const symbolWithScope = lookUpSymbolRecursive(
                    node,
                    node.value,
                    !allowForwardReferences,
                    isWithinTypeAnnotation
                );

                symbol = symbolWithScope?.symbol;
            }

            if (symbol) {
                appendArray(declarations, symbol.getDeclarations());
            }
        }

        return declarations;
    }

    function getTypeForDeclaration(declaration: Declaration): DeclaredSymbolTypeInfo {
        switch (declaration.type) {
            case DeclarationType.Intrinsic: {
                if (declaration.intrinsicType === 'Any') {
                    return { type: AnyType.create() };
                }

                if (declaration.intrinsicType === 'class') {
                    const classNode = ParseTreeUtils.getEnclosingClass(declaration.node) as ClassNode;
                    const classTypeInfo = getTypeOfClass(classNode);
                    return { type: classTypeInfo?.classType };
                }

                const strType = getBuiltInObject(declaration.node, 'str');
                const intType = getBuiltInObject(declaration.node, 'int');
                if (isClassInstance(intType) && isClassInstance(strType)) {
                    if (declaration.intrinsicType === 'str') {
                        return { type: strType };
                    }

                    if (declaration.intrinsicType === 'str | None') {
                        return { type: combineTypes([strType, NoneType.createInstance()]) };
                    }

                    if (declaration.intrinsicType === 'int') {
                        return { type: intType };
                    }

                    if (declaration.intrinsicType === 'Iterable[str]') {
                        const iterableType = getBuiltInType(declaration.node, 'Iterable');
                        if (isInstantiableClass(iterableType)) {
                            return {
                                type: ClassType.cloneAsInstance(
                                    ClassType.cloneForSpecialization(
                                        iterableType,
                                        [strType],
                                        /* isTypeArgumentExplicit */ true
                                    )
                                ),
                            };
                        }
                    }

                    if (declaration.intrinsicType === 'Dict[str, Any]') {
                        const dictType = getBuiltInType(declaration.node, 'dict');
                        if (isInstantiableClass(dictType)) {
                            return {
                                type: ClassType.cloneAsInstance(
                                    ClassType.cloneForSpecialization(
                                        dictType,
                                        [strType, AnyType.create()],
                                        /* isTypeArgumentExplicit */ true
                                    )
                                ),
                            };
                        }
                    }
                }

                return { type: UnknownType.create() };
            }

            case DeclarationType.Class: {
                const classTypeInfo = getTypeOfClass(declaration.node);
                return { type: classTypeInfo?.decoratedType };
            }

            case DeclarationType.SpecialBuiltInClass: {
                return { type: getTypeOfAnnotation(declaration.node.typeAnnotation) };
            }

            case DeclarationType.Function: {
                const functionTypeInfo = getTypeOfFunction(declaration.node);
                return { type: functionTypeInfo?.decoratedType };
            }

            case DeclarationType.TypeAlias: {
                return { type: getTypeOfTypeAlias(declaration.node) };
            }

            case DeclarationType.Parameter: {
                let typeAnnotationNode = declaration.node.typeAnnotation || declaration.node.typeAnnotationComment;

                // If there wasn't an annotation, see if the parent function
                // has a function-level annotation comment that provides
                // this parameter's annotation type.
                if (!typeAnnotationNode) {
                    if (declaration.node.parent?.nodeType === ParseNodeType.Function) {
                        const functionNode = declaration.node.parent;
                        if (
                            functionNode.functionAnnotationComment &&
                            !functionNode.functionAnnotationComment.isParamListEllipsis
                        ) {
                            const paramIndex = functionNode.parameters.findIndex((param) => param === declaration.node);
                            typeAnnotationNode = ParseTreeUtils.getTypeAnnotationForParameter(functionNode, paramIndex);
                        }
                    }
                }

                if (typeAnnotationNode) {
                    const declaredType = getTypeOfParameterAnnotation(typeAnnotationNode, declaration.node.category);

                    return {
                        type: transformVariadicParamType(
                            declaration.node,
                            declaration.node.category,
                            adjustParameterAnnotatedType(declaration.node, declaredType)
                        ),
                    };
                }

                return { type: undefined };
            }

            case DeclarationType.TypeParameter: {
                return { type: getTypeOfTypeParameter(declaration.node) };
            }

            case DeclarationType.Variable: {
                const typeAnnotationNode = declaration.typeAnnotationNode;

                if (typeAnnotationNode) {
                    let declaredType: Type | undefined;

                    if (declaration.isRuntimeTypeExpression) {
                        declaredType = convertToInstance(
                            getTypeOfExpressionExpectingType(typeAnnotationNode, {
                                allowFinal: true,
                                allowRequired: true,
                            }).type
                        );
                    } else {
                        const declNode =
                            declaration.isDefinedByMemberAccess &&
                            declaration.node.parent?.nodeType === ParseNodeType.MemberAccess
                                ? declaration.node.parent
                                : declaration.node;
                        declaredType = getTypeOfAnnotation(typeAnnotationNode, {
                            isVariableAnnotation: true,
                            allowClassVar: ParseTreeUtils.isClassVarAllowedForAssignmentTarget(declNode),
                            allowFinal: ParseTreeUtils.isFinalAllowedForAssignmentTarget(declNode),
                            allowRequired: ParseTreeUtils.isRequiredAllowedForAssignmentTarget(declNode),
                        });
                    }

                    if (declaredType) {
                        // Apply enum transform if appropriate.
                        if (declaration.node.nodeType === ParseNodeType.Name) {
                            declaredType =
                                transformTypeForPossibleEnumClass(
                                    evaluatorInterface,
                                    declaration.node,
                                    () => declaredType!
                                ) ?? declaredType;
                        }

                        if (isClassInstance(declaredType) && ClassType.isBuiltIn(declaredType, 'TypeAlias')) {
                            return { type: undefined, isTypeAlias: true };
                        }

                        return { type: declaredType };
                    }
                }

                return { type: undefined };
            }

            case DeclarationType.Alias: {
                return { type: undefined };
            }
        }
    }

    function getTypeOfTypeParameter(node: TypeParameterNode): TypeVarType {
        // Is this type already cached?
        const cachedTypeVarType = readTypeCache(node.name, EvaluatorFlags.None) as FunctionType;
        if (cachedTypeVarType && isTypeVar(cachedTypeVarType)) {
            return cachedTypeVarType;
        }

        let typeVar = TypeVarType.createInstantiable(node.name.value);
        typeVar.details.isTypeParamSyntax = true;

        if (node.typeParamCategory === TypeParameterCategory.TypeVarTuple) {
            typeVar.details.isVariadic = true;
        } else if (node.typeParamCategory === TypeParameterCategory.ParamSpec) {
            typeVar.details.isParamSpec = true;
        }

        // Cache the value before we evaluate the bound or the default type in
        // case it refers to itself in a circular manner.
        writeTypeCache(node, { type: typeVar }, /* flags */ undefined);
        writeTypeCache(node.name, { type: typeVar }, /* flags */ undefined);

        if (node.boundExpression) {
            if (node.boundExpression.nodeType === ParseNodeType.Tuple) {
                const constraints = node.boundExpression.expressions.map((constraint) => {
                    const constraintType = getTypeOfExpressionExpectingType(constraint).type;

                    if (requiresSpecialization(constraintType, /* ignorePseudoGeneric */ true)) {
                        addError(Localizer.Diagnostic.typeVarBoundGeneric(), constraint);
                    }

                    return convertToInstance(constraintType);
                });

                if (constraints.length < 2) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node.boundExpression).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarSingleConstraint(),
                        node.boundExpression
                    );
                } else if (node.typeParamCategory === TypeParameterCategory.TypeVar) {
                    typeVar.details.constraints = constraints;
                }
            } else {
                const boundType = getTypeOfExpressionExpectingType(node.boundExpression).type;

                if (requiresSpecialization(boundType, /* ignorePseudoGeneric */ true)) {
                    addError(Localizer.Diagnostic.typeVarConstraintGeneric(), node.boundExpression);
                }

                if (node.typeParamCategory === TypeParameterCategory.TypeVar) {
                    typeVar.details.boundType = convertToInstance(boundType);
                }
            }
        }

        if (node.defaultExpression) {
            if (node.typeParamCategory === TypeParameterCategory.ParamSpec) {
                typeVar.details.defaultType = getParamSpecDefaultType(node.defaultExpression);
            } else if (node.typeParamCategory === TypeParameterCategory.TypeVarTuple) {
                typeVar.details.defaultType = getTypeVarTupleDefaultType(node.defaultExpression);
            } else {
                typeVar.details.defaultType = convertToInstance(
                    getTypeOfExpressionExpectingType(node.defaultExpression).type
                );
            }
        }

        // If a default is provided, make sure it is compatible with the bound
        // or constraint.
        if (typeVar.details.defaultType && node.defaultExpression) {
            const typeVarContext = new TypeVarContext(WildcardTypeVarScopeId);
            const concreteDefaultType = applySolvedTypeVars(typeVar.details.defaultType, typeVarContext, {
                unknownIfNotFound: true,
            });

            if (typeVar.details.boundType) {
                if (!assignType(typeVar.details.boundType, concreteDefaultType)) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarDefaultBoundMismatch(),
                        node.defaultExpression
                    );
                }
            } else if (typeVar.details.constraints.length > 0) {
                if (!typeVar.details.constraints.some((constraint) => isTypeSame(constraint, concreteDefaultType))) {
                    addDiagnostic(
                        AnalyzerNodeInfo.getFileInfo(node).diagnosticRuleSet.reportGeneralTypeIssues,
                        DiagnosticRule.reportGeneralTypeIssues,
                        Localizer.Diagnostic.typeVarDefaultConstraintMismatch(),
                        node.defaultExpression
                    );
                }
            }
        }

        // Associate the type variable with the owning scope.
        const scopeNode = ParseTreeUtils.getTypeVarScopeNode(node);
        if (scopeNode) {
            let scopeType: TypeVarScopeType;
            if (scopeNode.nodeType === ParseNodeType.Class) {
                scopeType = TypeVarScopeType.Class;

                // Set the variance to "auto" for class-scoped TypeVars.
                typeVar.details.declaredVariance = Variance.Auto;
            } else if (scopeNode.nodeType === ParseNodeType.Function) {
                scopeType = TypeVarScopeType.Function;
            } else {
                assert(scopeNode.nodeType === ParseNodeType.TypeAlias);
                scopeType = TypeVarScopeType.TypeAlias;
            }

            typeVar = TypeVarType.cloneForScopeId(
                typeVar,
                getScopeIdForNode(scopeNode.nodeType === ParseNodeType.TypeAlias ? scopeNode.name : scopeNode),
                scopeNode.name.value,
                scopeType
            );
        }

        return typeVar;
    }

    function getInferredTypeOfDeclaration(symbol: Symbol, decl: Declaration): Type | undefined {
        const resolvedDecl = resolveAliasDeclaration(
            decl,
            /* resolveLocalNames */ true,
            /* allowExternallyHiddenAccess */ AnalyzerNodeInfo.getFileInfo(decl.node).isStubFile
        );

        // We couldn't resolve the alias. Substitute an unknown
        // type in this case.
        if (!resolvedDecl) {
            return evaluatorOptions.evaluateUnknownImportsAsAny ? AnyType.create() : UnknownType.create();
        }

        function applyLoaderActionsToModuleType(
            moduleType: ModuleType,
            loaderActions: ModuleLoaderActions,
            importLookup: ImportLookup
        ): Type {
            if (loaderActions.path && loaderActions.loadSymbolsFromPath) {
                const lookupResults = importLookup(loaderActions.path);
                if (lookupResults) {
                    moduleType.fields = lookupResults.symbolTable;
                    moduleType.docString = lookupResults.docString;
                } else {
                    // Note that all module attributes that are not found in the
                    // symbol table should be treated as Any or Unknown rather than
                    // as an error.
                    moduleType.notPresentFieldType = evaluatorOptions.evaluateUnknownImportsAsAny
                        ? AnyType.create()
                        : UnknownType.create();
                }
            }

            if (loaderActions.implicitImports) {
                loaderActions.implicitImports.forEach((implicitImport, name) => {
                    // Recursively apply loader actions.
                    let symbolType: Type;

                    if (implicitImport.isUnresolved) {
                        symbolType = UnknownType.create();
                    } else {
                        const moduleName = moduleType.moduleName ? moduleType.moduleName + '.' + name : '';
                        const importedModuleType = ModuleType.create(moduleName, implicitImport.path);
                        symbolType = applyLoaderActionsToModuleType(importedModuleType, implicitImport, importLookup);
                    }

                    const importedModuleSymbol = Symbol.createWithType(SymbolFlags.None, symbolType);
                    moduleType.loaderFields.set(name, importedModuleSymbol);
                });
            }

            return moduleType;
        }

        // If the resolved declaration is still an alias, the alias
        // is pointing at a module, and we need to synthesize a
        // module type.
        if (resolvedDecl.type === DeclarationType.Alias) {
            // Build a module type that corresponds to the declaration and
            // its associated loader actions.
            let moduleName = resolvedDecl.moduleName;
            if (decl.type === DeclarationType.Alias) {
                if (decl.symbolName) {
                    moduleName += '.' + decl.symbolName;
                }

                // If the module name is relative to the current file, use that
                // file's module name as a reference.
                if (moduleName.startsWith('.')) {
                    const fileInfo = AnalyzerNodeInfo.getFileInfo(decl.node);
                    const nameParts = fileInfo.moduleName.split('.');
                    moduleName = moduleName.substr(1);

                    while (moduleName.startsWith('.') && nameParts.length > 0) {
                        moduleName = moduleName.substr(1);
                        nameParts.pop();
                    }

                    moduleName = nameParts.join('.') + '.' + moduleName;
                }
            }
            const moduleType = ModuleType.create(moduleName, resolvedDecl.path);
            if (resolvedDecl.symbolName && resolvedDecl.submoduleFallback) {
                return applyLoaderActionsToModuleType(moduleType, resolvedDecl.submoduleFallback, importLookup);
            } else {
                return applyLoaderActionsToModuleType(moduleType, resolvedDecl, importLookup);
            }
        }

        const declaredType = getTypeForDeclaration(resolvedDecl);
        if (declaredType.type) {
            return declaredType.type;
        }

        // If this is part of a "py.typed" package, don't fall back on type inference
        // unless it's marked Final, is a constant, or is a declared type alias.
        const fileInfo = AnalyzerNodeInfo.getFileInfo(resolvedDecl.node);
        let isUnambiguousType = !fileInfo.isInPyTypedPackage || fileInfo.isStubFile;

        // If this is a py.typed package, determine if this is a case where an unannotated
        // variable is considered "unambiguous" because all type checkers are almost
        // guaranteed to infer its type the same.
        if (!isUnambiguousType) {
            if (resolvedDecl.type === DeclarationType.Variable) {
                // Special-case variables within an enum class. These are effectively
                // constants, so we'll treat them as unambiguous.
                const enclosingClass = ParseTreeUtils.getEnclosingClass(resolvedDecl.node, /* stopAtFunction */ true);
                if (enclosingClass) {
                    const classTypeInfo = getTypeOfClass(enclosingClass);
                    if (classTypeInfo && ClassType.isEnumClass(classTypeInfo.classType)) {
                        isUnambiguousType = true;
                    }
                }

                // Special-case constants, which are treated as unambiguous.
                if (isFinalVariableDeclaration(resolvedDecl) || resolvedDecl.isConstant) {
                    isUnambiguousType = true;
                }

                // Special-case calls to certain built-in type functions.
                if (resolvedDecl.inferredTypeSource?.nodeType === ParseNodeType.Call) {
                    const baseTypeResult = getTypeOfExpression(
                        resolvedDecl.inferredTypeSource.leftExpression,
                        EvaluatorFlags.DoNotSpecialize
                    );
                    const callType = baseTypeResult.type;

                    const exemptBuiltins = [
                        'TypeVar',
                        'ParamSpec',
                        'TypeVarTuple',
                        'TypedDict',
                        'NamedTuple',
                        'NewType',
                    ];

                    if (isInstantiableClass(callType) && ClassType.isBuiltIn(callType, exemptBuiltins)) {
                        isUnambiguousType = true;
                    } else if (
                        isFunction(callType) &&
                        exemptBuiltins.some((name) => callType.details.builtInName === name)
                    ) {
                        isUnambiguousType = true;
                    }
                }
            }
        }

        // If the resolved declaration had no defined type, use the
        // inferred type for this node.
        if (resolvedDecl.type === DeclarationType.Parameter) {
            return evaluateTypeForSubnode(resolvedDecl.node.name!, () => {
                evaluateTypeOfParameter(resolvedDecl.node);
            })?.type;
        }

        if (resolvedDecl.type === DeclarationType.Variable && resolvedDecl.inferredTypeSource) {
            const isTypeAlias =
                isExplicitTypeAliasDeclaration(resolvedDecl) || isPossibleTypeAliasOrTypedDict(resolvedDecl);

            // If this is a type alias, evaluate types for the entire assignment
            // statement rather than just the RHS of the assignment.
            const typeSource =
                isTypeAlias && resolvedDecl.inferredTypeSource.parent
                    ? resolvedDecl.inferredTypeSource.parent
                    : resolvedDecl.inferredTypeSource;
            let inferredType = evaluateTypeForSubnode(resolvedDecl.node, () => {
                evaluateTypesForStatement(typeSource);
            })?.type;

            if (inferredType && resolvedDecl.node.nodeType === ParseNodeType.Name) {
                // See if this is an enum member. If so, we need to handle it as a special case.
                const enumMemberType = transformTypeForPossibleEnumClass(evaluatorInterface, resolvedDecl.node, () => {
                    return (
                        evaluateTypeForSubnode(resolvedDecl.inferredTypeSource!, () => {
                            evaluateTypesForStatement(resolvedDecl.inferredTypeSource!);
                        })?.type ?? UnknownType.create()
                    );
                });
                if (enumMemberType) {
                    inferredType = enumMemberType;
                }
            }

            if (inferredType && isTypeAlias && resolvedDecl.typeAliasName) {
                // If this was a speculative type alias, it becomes a real type alias only
                // in the event that its inferred type is instantiable or explicitly Any
                // (but not an ellipsis).
                if (
                    TypeBase.isInstantiable(inferredType) &&
                    !isUnknown(inferredType) &&
                    !isEllipsisType(inferredType)
                ) {
                    inferredType = transformTypeForTypeAlias(
                        inferredType,
                        resolvedDecl.typeAliasName,
                        resolvedDecl.node
                    );

                    isUnambiguousType = true;
                }
            }

            // Determine whether we need to mark the annotation as ambiguous.
            if (inferredType && fileInfo.isInPyTypedPackage && !fileInfo.isStubFile) {
                if (!isUnambiguousType) {
                    // See if this particular inference can be considered "unambiguous".
                    // Any symbol that is assigned more than once is considered ambiguous.
                    if (isUnambiguousInference(symbol, decl, inferredType)) {
                        isUnambiguousType = true;
                    }
                }

                if (!isUnambiguousType) {
                    inferredType = TypeBase.cloneForAmbiguousType(inferredType);
                }
            }

            return inferredType;
        }

        return undefined;
    }

    // Applies some heuristics to determine whether it's likely that all Python
    // type checkers will infer the same type.
    function isUnambiguousInference(symbol: Symbol, decl: Declaration, inferredType: Type): boolean {
        const nonSlotsDecls = symbol.getDeclarations().filter((decl) => {
            return decl.type !== DeclarationType.Variable || !decl.isInferenceAllowedInPyTyped;
        });

        // Any symbol with more than one assignment is considered ambiguous.
        if (nonSlotsDecls.length > 1) {
            return false;
        }

        if (decl.type !== DeclarationType.Variable) {
            return false;
        }

        // If there are no non-slots declarations, don't mark the inferred type as ambiguous.
        if (nonSlotsDecls.length === 0) {
            return true;
        }

        // TypeVar definitions don't require a declaration.
        if (isTypeVar(inferredType)) {
            return true;
        }

        let assignmentNode: AssignmentNode | undefined;

        const parentNode = decl.node.parent;
        if (parentNode) {
            // Is this a simple assignment (x = y) or an assignment of an instance variable (self.x = y)?
            if (parentNode.nodeType === ParseNodeType.Assignment) {
                assignmentNode = parentNode;
            } else if (
                parentNode.nodeType === ParseNodeType.MemberAccess &&
                parentNode.parent?.nodeType === ParseNodeType.Assignment
            ) {
                assignmentNode = parentNode.parent;
            }
        }

        if (!assignmentNode) {
            return false;
        }

        const assignedType = getTypeOfExpression(assignmentNode.rightExpression).type;

        // Assume that literal values will always result in the same inferred type.
        if (isClassInstance(assignedType) && isLiteralType(assignedType)) {
            return true;
        }

        // If the assignment is a simple name corresponding to an unambiguous
        // type, we'll assume the resulting variable will receive the same
        // unambiguous type.
        if (assignmentNode.rightExpression.nodeType === ParseNodeType.Name && !TypeBase.isAmbiguous(assignedType)) {
            return true;
        }

        return false;
    }

    // If the specified declaration is an alias declaration that points to a symbol,
    // it resolves the alias and looks up the symbol, then returns the first declaration
    // associated with that symbol. It does this recursively if necessary. If a symbol
    // lookup fails, undefined is returned. If resolveLocalNames is true, the method
    // resolves aliases through local renames ("as" clauses found in import statements).
    function resolveAliasDeclaration(
        declaration: Declaration,
        resolveLocalNames: boolean,
        allowExternallyHiddenAccess = false
    ): Declaration | undefined {
        return resolveAliasDeclarationUtil(importLookup, declaration, resolveLocalNames, allowExternallyHiddenAccess)
            ?.declaration;
    }

    function resolveAliasDeclarationWithInfo(
        declaration: Declaration,
        resolveLocalNames: boolean,
        allowExternallyHiddenAccess = false
    ): ResolvedAliasInfo | undefined {
        return resolveAliasDeclarationUtil(importLookup, declaration, resolveLocalNames, allowExternallyHiddenAccess);
    }

    // Returns the type of the symbol. If the type is explicitly declared, that type
    // is returned. If not, the type is inferred from assignments to the symbol. All
    // assigned types are evaluated and combined into a union. If a "usageNode"
    // node is specified, only declarations that are outside of the current execution
    // scope or that are reachable (as determined by code flow analysis) are considered.
    // This helps in cases where there are cyclical dependencies between symbols.
    function getEffectiveTypeOfSymbol(symbol: Symbol): Type {
        return getEffectiveTypeOfSymbolForUsage(symbol).type;
    }

    function getEffectiveTypeOfSymbolForUsage(
        symbol: Symbol,
        usageNode?: NameNode,
        useLastDecl = false
    ): EffectiveTypeResult {
        let declaredTypeInfo: DeclaredSymbolTypeInfo | undefined;

        // If there's a declared type, it takes precedence over inferred types.
        if (symbol.hasTypedDeclarations()) {
            declaredTypeInfo = getDeclaredTypeOfSymbol(symbol, usageNode);

            const declaredType = declaredTypeInfo?.type;
            const hasMetadata = !!declaredTypeInfo.isTypeAlias;

            if (declaredType || !hasMetadata) {
                let isIncomplete = false;

                if (declaredType) {
                    if (isFunction(declaredType) && FunctionType.isPartiallyEvaluated(declaredType)) {
                        isIncomplete = true;
                    } else if (isClass(declaredType) && ClassType.isPartiallyEvaluated(declaredType)) {
                        isIncomplete = true;
                    }
                }

                const typedDecls = symbol.getTypedDeclarations();
                const result: EffectiveTypeResult = {
                    type: declaredType ?? UnknownType.create(),
                    isIncomplete,
                    includesVariableDecl: typedDecls.some((decl) => decl.type === DeclarationType.Variable),
                    includesIllegalTypeAliasDecl: !typedDecls.every((decl) => isPossibleTypeAliasDeclaration(decl)),
                    isRecursiveDefinition: !declaredType,
                };

                return result;
            }
        }

        // Look in the inferred type cache to see if we've computed this already.
        let cacheEntries = effectiveTypeCache.get(symbol.id);
        const usageNodeId = usageNode ? usageNode.id : undefined;
        const effectiveTypeCacheKey = `${usageNodeId === undefined ? '.' : usageNodeId.toString()}${
            useLastDecl ? '*' : ''
        }`;

        if (cacheEntries) {
            const result = cacheEntries.get(effectiveTypeCacheKey);
            if (result) {
                if (!result.isIncomplete) {
                    return result;
                }
            }
        }

        // Infer the type.
        const typesToCombine: Type[] = [];
        const decls = symbol.getDeclarations();
        let isIncomplete = false;
        let sawPendingEvaluation = false;
        let includesVariableDecl = false;
        let includesSpeculativeResult = false;

        let declIndexToConsider: number | undefined;

        // Limit the number of declarations to explore.
        if (decls.length > maxDeclarationsToUseForInference) {
            const result: EffectiveTypeResult = {
                type: UnknownType.create(),
                isIncomplete: false,
                includesVariableDecl: false,
                includesIllegalTypeAliasDecl: !decls.every((decl) => isPossibleTypeAliasDeclaration(decl)),
                isRecursiveDefinition: false,
            };

            addToEffectiveTypeCache(result);
            return result;
        }

        // If the caller has requested that we use only the last decl, we
        // will use only the last one, but we'll ignore decls that are in
        // except clauses.
        if (useLastDecl) {
            decls.forEach((decl, index) => {
                if (!decl.isInExceptSuite) {
                    declIndexToConsider = index;
                }
            });
        } else {
            // Handle the case where there are multiple imports — one of them in
            // a try block and one or more in except blocks. In this case, we'll
            // use the one in the try block rather than the excepts.
            if (decls.length > 1 && decls.every((decl) => decl.type === DeclarationType.Alias)) {
                const nonExceptDecls = decls.filter(
                    (decl) => decl.type === DeclarationType.Alias && !decl.isInExceptSuite
                );
                if (nonExceptDecls.length === 1) {
                    declIndexToConsider = decls.findIndex((decl) => decl === nonExceptDecls[0]);
                }
            }
        }

        let sawExplicitTypeAlias = false;
        decls.forEach((decl, index) => {
            let considerDecl = declIndexToConsider === undefined || index === declIndexToConsider;

            // If we have already seen an explicit type alias, do not consider
            // additional decls. This can happen if multiple TypeAlias declarations
            // are provided -- normally an error, but it can happen in stdlib stubs
            // if the user sets the pythonPlatform to "All".
            if (sawExplicitTypeAlias) {
                considerDecl = false;
            }

            // If the symbol is explicitly marked as a ClassVar, consider only the
            // declarations that assign to it from within the class body, not through
            // a member access expression.
            if (symbol.isClassVar() && decl.type === DeclarationType.Variable && decl.isDefinedByMemberAccess) {
                considerDecl = false;
            }

            if (usageNode !== undefined) {
                if (decl.type !== DeclarationType.Alias) {
                    // Is the declaration in the same execution scope as the "usageNode" node?
                    const usageScope = ParseTreeUtils.getExecutionScopeNode(usageNode);
                    const declScope = ParseTreeUtils.getExecutionScopeNode(decl.node);
                    if (usageScope === declScope) {
                        if (!isFlowPathBetweenNodes(decl.node, usageNode)) {
                            considerDecl = false;
                        }
                    }
                }
            }

            if (considerDecl) {
                const resolvedDecl =
                    resolveAliasDeclaration(
                        decl,
                        /* resolveLocalNames */ true,
                        /* allowExternallyHiddenAccess */ AnalyzerNodeInfo.getFileInfo(decl.node).isStubFile
                    ) ?? decl;

                const isExplicitTypeAlias = isExplicitTypeAliasDeclaration(decl);
                const isTypeAlias = isExplicitTypeAlias || isPossibleTypeAliasOrTypedDict(decl);

                if (isExplicitTypeAlias) {
                    sawExplicitTypeAlias = true;
                }

                // If this is a type alias, evaluate it outside of the recursive symbol
                // resolution check so we can evaluate the full assignment statement.
                if (
                    isTypeAlias &&
                    resolvedDecl.type === DeclarationType.Variable &&
                    resolvedDecl.inferredTypeSource?.parent?.nodeType === ParseNodeType.Assignment
                ) {
                    evaluateTypesForAssignmentStatement(resolvedDecl.inferredTypeSource.parent);
                }

                if (pushSymbolResolution(symbol, decl)) {
                    try {
                        let type = getInferredTypeOfDeclaration(symbol, decl);

                        if (!popSymbolResolution(symbol)) {
                            isIncomplete = true;
                        }

                        if (type) {
                            if (resolvedDecl.type === DeclarationType.Variable) {
                                // Exempt typing.pyi, which uses variables to define some
                                // special forms like Any.
                                const fileInfo = AnalyzerNodeInfo.getFileInfo(resolvedDecl.node);
                                if (!fileInfo.isTypingStubFile) {
                                    includesVariableDecl = true;
                                }

                                let isConstant = false;
                                if (resolvedDecl.type === DeclarationType.Variable) {
                                    if (resolvedDecl.isConstant || isFinalVariableDeclaration(resolvedDecl)) {
                                        isConstant = true;
                                    }
                                }

                                // Treat enum values declared within an enum class as though they are const even
                                // though they may not be named as such.
                                if (
                                    isClassInstance(type) &&
                                    ClassType.isEnumClass(type) &&
                                    isDeclInEnumClass(evaluatorInterface, resolvedDecl)
                                ) {
                                    isConstant = true;
                                }

                                // If the symbol is constant, we can retain the literal
                                // value. Otherwise, strip literal values to widen the type.
                                if (TypeBase.isInstance(type) && !isExplicitTypeAlias && !isConstant) {
                                    type = stripLiteralValue(type);
                                }
                            }
                            typesToCombine.push(type);

                            if (speculativeTypeTracker.isSpeculative(decl.node)) {
                                includesSpeculativeResult = true;
                            }
                        } else {
                            isIncomplete = true;
                        }
                    } catch (e: any) {
                        // Clean up the stack before rethrowing.
                        popSymbolResolution(symbol);
                        throw e;
                    }
                } else {
                    if (resolvedDecl.type === DeclarationType.Class) {
                        const classTypeInfo = getTypeOfClass(resolvedDecl.node);
                        if (classTypeInfo?.decoratedType) {
                            typesToCombine.push(classTypeInfo.decoratedType);
                        }
                    }

                    isIncomplete = true;

                    // Note that at least one decl could not be evaluated because
                    // it was already in the process of being evaluated.
                    sawPendingEvaluation = true;
                }
            }
        });

        if (typesToCombine.length > 0) {
            // How many times have we already attempted to evaluate this declaration already?
            const evaluationAttempts = (cacheEntries?.get(effectiveTypeCacheKey)?.evaluationAttempts ?? 0) + 1;

            // Ignore the pending evaluation flag if we've already attempted the
            // type evaluation many times because this probably means there's a
            // cyclical dependency that cannot be broken.
            isIncomplete = sawPendingEvaluation && evaluationAttempts < maxEffectiveTypeEvaluationAttempts;

            const result: EffectiveTypeResult = {
                type: combineTypes(typesToCombine),
                isIncomplete,
                includesVariableDecl,
                includesIllegalTypeAliasDecl: !decls.every((decl) => isPossibleTypeAliasDeclaration(decl)),
                isRecursiveDefinition: false,
                evaluationAttempts,
            };

            if (!includesSpeculativeResult) {
                addToEffectiveTypeCache(result);
            }

            return result;
        }

        return {
            type: UnboundType.create(),
            isIncomplete,
            includesVariableDecl,
            includesIllegalTypeAliasDecl: !decls.every((decl) => isPossibleTypeAliasDeclaration(decl)),
            isRecursiveDefinition: false,
        };

        function addToEffectiveTypeCache(result: EffectiveTypeResult) {
            // Add the entry to the cache so we don't need to compute it next time.
            if (!cacheEntries) {
                cacheEntries = new Map<string, EffectiveTypeResult>();
                effectiveTypeCache.set(symbol.id, cacheEntries);
            }

            cacheEntries.set(effectiveTypeCacheKey, result);
        }
    }

    // If a declaration has an explicit type (e.g. a variable with an annotation),
    // this function evaluates the type and returns it. If the symbol has no
    // explicit declared type, its type will need to be inferred instead. In some
    // cases, non-type information (such as Final or ClassVar attributes) may be
    // provided, but type inference is still required. In such cases, the attributes
    // are returned as flags.
    function getDeclaredTypeOfSymbol(symbol: Symbol, usageNode?: NameNode): DeclaredSymbolTypeInfo {
        const synthesizedType = symbol.getSynthesizedType();
        if (synthesizedType) {
            return { type: synthesizedType };
        }

        let typedDecls = symbol.getTypedDeclarations();

        if (typedDecls.length === 0) {
            // There was no declaration with a defined type.
            return { type: undefined };
        }

        // If there is more than one typed decl, filter out any that are not
        // reachable from the usage node (if specified). This can happen in
        // cases where a property symbol is redefined to add a setter, deleter,
        // etc.
        if (usageNode && typedDecls.length > 1) {
            const filteredTypedDecls = typedDecls.filter((decl) => {
                if (decl.type !== DeclarationType.Alias) {
                    // Is the declaration in the same execution scope as the "usageNode" node?
                    const usageScope = ParseTreeUtils.getExecutionScopeNode(usageNode);
                    const declScope = ParseTreeUtils.getExecutionScopeNode(decl.node);

                    if (usageScope === declScope) {
                        if (!isFlowPathBetweenNodes(decl.node, usageNode, /* allowSelf */ false)) {
                            return false;
                        }
                    }
                }
                return true;
            });

            if (filteredTypedDecls.length === 0) {
                return { type: UnboundType.create() };
            }

            typedDecls = filteredTypedDecls;
        }

        // Start with the last decl. If that's already being resolved,
        // use the next-to-last decl, etc. This can happen when resolving
        // property methods. Often the setter method is defined in reference to
        // the initial property, which defines the getter method with the same
        // symbol name.
        let declIndex = typedDecls.length - 1;
        while (declIndex >= 0) {
            const decl = typedDecls[declIndex];

            // If there's a partially-constructed type that is allowed
            // for recursive symbol resolution, return it as the resolved type.
            const partialType = getSymbolResolutionPartialType(symbol, decl);
            if (partialType) {
                return { type: partialType };
            }

            if (getIndexOfSymbolResolution(symbol, decl) < 0) {
                if (pushSymbolResolution(symbol, decl)) {
                    try {
                        const declaredTypeInfo = getTypeForDeclaration(decl);

                        // If there was recursion detected, don't use this declaration.
                        // The exception is it's a class declaration because getTypeOfClass
                        // handles recursion by populating a partially-created class type
                        // in the type cache. This exception is required to handle the
                        // circular dependency between the "type" and "object" classes in
                        // builtins.pyi (since "object" is a "type" and "type" is an "object").
                        if (popSymbolResolution(symbol) || decl.type === DeclarationType.Class) {
                            return declaredTypeInfo;
                        }
                    } catch (e: any) {
                        // Clean up the stack before rethrowing.
                        popSymbolResolution(symbol);
                        throw e;
                    }
                }
            }

            declIndex--;
        }

        return { type: undefined };
    }

    function inferReturnTypeIfNecessary(type: Type) {
        if (isFunction(type)) {
            getFunctionEffectiveReturnType(type);
        } else if (isOverloadedFunction(type)) {
            type.overloads.forEach((overload) => {
                getFunctionEffectiveReturnType(overload);
            });
        }
    }

    // Returns the return type of the function. If the type is explicitly provided in
    // a type annotation, that type is returned. If not, an attempt is made to infer
    // the return type. If a list of args is provided, the inference logic may take
    // into account argument types to infer the return type.
    function getFunctionEffectiveReturnType(
        type: FunctionType,
        args?: ValidateArgTypeParams[],
        inferTypeIfNeeded = true
    ) {
        const specializedReturnType = FunctionType.getSpecializedReturnType(type);
        if (specializedReturnType) {
            return adjustCallableReturnType(specializedReturnType);
        }

        if (inferTypeIfNeeded) {
            return getFunctionInferredReturnType(type, args);
        }

        return UnknownType.create();
    }

    function getFunctionInferredReturnType(type: FunctionType, args?: ValidateArgTypeParams[]) {
        let returnType: Type | undefined;
        let isIncomplete = false;
        let analyzeUnannotatedFunctions = true;

        // Don't attempt to infer the return type for a stub file.
        if (FunctionType.isStubDefinition(type)) {
            return UnknownType.create();
        }

        // Don't infer the return type for an overloaded function (unless it's synthesized,
        // which is needed for proper operation of the __get__ method in properties).
        if (FunctionType.isOverloaded(type) && !FunctionType.isSynthesizedMethod(type)) {
            return UnknownType.create();
        }

        // If the return type has already been lazily evaluated,
        // don't bother computing it again.
        if (type.inferredReturnType) {
            returnType = type.inferredReturnType;
        } else {
            // Don't bother inferring the return type of __init__ because it's
            // always None.
            if (FunctionType.isInstanceMethod(type) && type.details.name === '__init__') {
                returnType = NoneType.createInstance();
            } else if (type.details.declaration) {
                const functionNode = type.details.declaration.node;
                analyzeUnannotatedFunctions =
                    AnalyzerNodeInfo.getFileInfo(functionNode).diagnosticRuleSet.analyzeUnannotatedFunctions;

                // Skip return type inference if we are in "skip unannotated function" mode.
                if (analyzeUnannotatedFunctions && !checkCodeFlowTooComplex(functionNode.suite)) {
                    const codeFlowComplexity = AnalyzerNodeInfo.getCodeFlowComplexity(functionNode);

                    // For very complex functions that have no annotated parameter types,
                    // don't attempt to infer the return type because it can be extremely
                    // expensive.
                    const parametersAreAnnotated =
                        type.details.parameters.length <= 1 ||
                        type.details.parameters.some((param) => param.hasDeclaredType);

                    if (parametersAreAnnotated || codeFlowComplexity < maxReturnTypeInferenceCodeFlowComplexity) {
                        // Temporarily disable speculative mode while we
                        // lazily evaluate the return type.
                        let returnTypeResult: TypeResult | undefined;
                        disableSpeculativeMode(() => {
                            returnTypeResult = inferFunctionReturnType(
                                functionNode,
                                FunctionType.isAbstractMethod(type)
                            );
                        });

                        returnType = returnTypeResult?.type;
                        if (returnTypeResult?.isIncomplete) {
                            isIncomplete = true;
                        }

                        // Do we need to wrap this in an awaitable?
                        if (returnType && FunctionType.isWrapReturnTypeInAwait(type)) {
                            returnType = createAwaitableReturnType(
                                functionNode,
                                returnType,
                                !!type.details.declaration?.isGenerator
                            );
                        }
                    }
                }
            }

            if (!returnType) {
                returnType = UnknownType.create();
            }

            // Cache the type for next time.
            if (!isIncomplete) {
                type.inferredReturnType = returnType;
            }
        }

        // If the type is partially unknown and the function has one or more unannotated
        // params, try to analyze the function with the provided argument types and
        // attempt to do a better job at inference.
        if (
            !isIncomplete &&
            analyzeUnannotatedFunctions &&
            isPartlyUnknown(returnType) &&
            FunctionType.hasUnannotatedParams(type) &&
            !FunctionType.isStubDefinition(type) &&
            !FunctionType.isPyTypedDefinition(type) &&
            args
        ) {
            const contextualReturnType = getFunctionInferredReturnTypeUsingArguments(type, args);
            if (contextualReturnType) {
                returnType = contextualReturnType;
            }
        }

        return returnType;
    }

    function getFunctionInferredReturnTypeUsingArguments(
        type: FunctionType,
        args: ValidateArgTypeParams[]
    ): Type | undefined {
        let contextualReturnType: Type | undefined;

        if (!type.details.declaration) {
            return undefined;
        }
        const functionNode = type.details.declaration.node;
        const codeFlowComplexity = AnalyzerNodeInfo.getCodeFlowComplexity(functionNode);

        if (codeFlowComplexity >= maxReturnCallSiteTypeInferenceCodeFlowComplexity) {
            return undefined;
        }

        // If an arg hasn't been matched to a specific named parameter,
        // it's an unpacked value that corresponds to multiple parameters.
        // That's an edge case that we don't handle here.
        if (args.some((arg) => !arg.paramName)) {
            return undefined;
        }

        // Detect recurrence. If a function invokes itself either directly
        // or indirectly, we won't attempt to infer contextual return
        // types any further.
        if (returnTypeInferenceContextStack.some((context) => context.functionNode === functionNode)) {
            return undefined;
        }

        const functionType = getTypeOfFunction(functionNode);
        if (!functionType) {
            return undefined;
        }

        // Very complex functions with many arguments can take a long time to analyze,
        // so we'll use a heuristic and avoiding this inference technique for any
        // call site that involves too many arguments.
        if (args.length > maxReturnTypeInferenceArgumentCount) {
            return undefined;
        }

        // Don't explore arbitrarily deep in the call graph.
        if (returnTypeInferenceContextStack.length >= maxReturnTypeInferenceStackSize) {
            return undefined;
        }

        const paramTypes: Type[] = [];
        let isResultFromCache = false;

        // Suppress diagnostics because we don't want to generate errors.
        suppressDiagnostics(functionNode, () => {
            // Allocate a new temporary type cache for the context of just
            // this function so we can analyze it separately without polluting
            // the main type cache.
            const prevTypeCache = returnTypeInferenceTypeCache;
            returnTypeInferenceContextStack.push({
                functionNode,
                codeFlowAnalyzer: codeFlowEngine.createCodeFlowAnalyzer(),
            });

            try {
                returnTypeInferenceTypeCache = new Map<number, TypeCacheEntry>();

                let allArgTypesAreUnknown = true;
                functionNode.parameters.forEach((param, index) => {
                    if (param.name) {
                        let paramType: Type | undefined;
                        const arg = args.find((arg) => param.name!.value === arg.paramName);

                        if (arg && arg.argument.valueExpression) {
                            paramType = getTypeOfExpression(arg.argument.valueExpression).type;
                            if (!isUnknown(paramType)) {
                                allArgTypesAreUnknown = false;
                            }
                        } else if (param.defaultValue) {
                            paramType = getTypeOfExpression(param.defaultValue).type;
                            if (!isUnknown(paramType)) {
                                allArgTypesAreUnknown = false;
                            }
                        } else if (index === 0) {
                            // If this is an instance or class method, use the implied
                            // parameter type for the "self" or "cls" parameter.
                            if (
                                FunctionType.isInstanceMethod(functionType.functionType) ||
                                FunctionType.isClassMethod(functionType.functionType)
                            ) {
                                if (functionType.functionType.details.parameters.length > 0) {
                                    if (functionNode.parameters[0].name) {
                                        paramType = functionType.functionType.details.parameters[0].type;
                                    }
                                }
                            }
                        }

                        if (!paramType) {
                            paramType = UnknownType.create();
                        }

                        paramTypes.push(paramType);
                        writeTypeCache(param.name, { type: paramType }, EvaluatorFlags.None);
                    }
                });

                // Don't bother trying to determine the contextual return
                // type if none of the argument types are known.
                if (!allArgTypesAreUnknown) {
                    // See if the return type is already cached. If so, skip the
                    // inference step, which is potentially very expensive.
                    const cacheEntry = functionType.functionType.callSiteReturnTypeCache?.find((entry) => {
                        return (
                            entry.paramTypes.length === paramTypes.length &&
                            entry.paramTypes.every((t, i) => isTypeSame(t, paramTypes[i]))
                        );
                    });

                    if (cacheEntry) {
                        contextualReturnType = cacheEntry.returnType;
                        isResultFromCache = true;
                    } else {
                        contextualReturnType = inferFunctionReturnType(
                            functionNode,
                            FunctionType.isAbstractMethod(type)
                        )?.type;
                    }
                }
            } finally {
                returnTypeInferenceContextStack.pop();
                returnTypeInferenceTypeCache = prevTypeCache;
            }
        });

        if (contextualReturnType) {
            contextualReturnType = removeUnbound(contextualReturnType);

            // Do we need to wrap this in an awaitable?
            if (FunctionType.isWrapReturnTypeInAwait(type) && !isNever(contextualReturnType)) {
                contextualReturnType = createAwaitableReturnType(
                    functionNode,
                    contextualReturnType,
                    !!type.details.declaration?.isGenerator
                );
            }

            if (!isResultFromCache) {
                // Cache the resulting type.
                if (!functionType.functionType.callSiteReturnTypeCache) {
                    functionType.functionType.callSiteReturnTypeCache = [];
                }
                if (functionType.functionType.callSiteReturnTypeCache.length >= maxCallSiteReturnTypeCacheSize) {
                    functionType.functionType.callSiteReturnTypeCache =
                        functionType.functionType.callSiteReturnTypeCache.slice(1);
                }
                functionType.functionType.callSiteReturnTypeCache.push({
                    paramTypes,
                    returnType: contextualReturnType,
                });
            }

            return contextualReturnType;
        }

        return undefined;
    }

    function getFunctionDeclaredReturnType(node: FunctionNode): Type | undefined {
        const functionTypeInfo = getTypeOfFunction(node)!;
        if (!functionTypeInfo) {
            // We hit a recursive dependency.
            return AnyType.create();
        }

        // Ignore this check for abstract methods, which often
        // don't actually return any value.
        if (FunctionType.isAbstractMethod(functionTypeInfo.functionType)) {
            return AnyType.create();
        }

        if (FunctionType.isGenerator(functionTypeInfo.functionType)) {
            return getDeclaredGeneratorReturnType(functionTypeInfo.functionType);
        }

        return functionTypeInfo.functionType.details.declaredReturnType;
    }

    function getTypeOfMember(member: ClassMember): Type {
        if (isInstantiableClass(member.classType)) {
            return partiallySpecializeType(
                getEffectiveTypeOfSymbol(member.symbol),
                member.classType,
                /* selfClass */ undefined,
                typeClassType ?? UnknownType.create()
            );
        }
        return UnknownType.create();
    }

    function getTypeOfMemberInternal(member: ClassMember, selfClass: ClassType | undefined): TypeResult | undefined {
        if (isInstantiableClass(member.classType)) {
            const typeResult = getEffectiveTypeOfSymbolForUsage(member.symbol);

            if (typeResult) {
                // If the type is a function or overloaded function, infer
                // and cache the return type if necessary. This needs to be done
                // prior to specializing.
                inferReturnTypeIfNecessary(typeResult.type);

                return {
                    type: partiallySpecializeType(typeResult.type, member.classType, selfClass),
                    isIncomplete: !!typeResult.isIncomplete,
                };
            }
        }

        return undefined;
    }

    function assignClass(
        destType: ClassType,
        srcType: ClassType,
        diag: DiagnosticAddendum | undefined,
        destTypeVarContext: TypeVarContext | undefined,
        srcTypeVarContext: TypeVarContext | undefined,
        flags: AssignTypeFlags,
        recursionCount: number,
        reportErrorsUsingObjType: boolean
    ): boolean {
        // If the source or dest types are partially evaluated (i.e. they are in the
        // process of being constructed), assume they are assignable rather than risk
        // emitting false positives.
        if (ClassType.isHierarchyPartiallyEvaluated(destType) || ClassType.isHierarchyPartiallyEvaluated(srcType)) {
            return true;
        }

        // Handle typed dicts. They also use a form of structural typing for type
        // checking, as defined in PEP 589.
        if (
            ClassType.isTypedDictClass(destType) &&
            ClassType.isTypedDictClass(srcType) &&
            !ClassType.isSameGenericClass(destType, srcType)
        ) {
            if (
                !assignTypedDictToTypedDict(
                    evaluatorInterface,
                    destType,
                    srcType,
                    diag,
                    destTypeVarContext,
                    flags,
                    recursionCount
                )
            ) {
                return false;
            }

            if (ClassType.isFinal(destType) !== ClassType.isFinal(srcType)) {
                diag?.addMessage(
                    Localizer.DiagnosticAddendum.typedDictFinalMismatch().format({
                        sourceType: printType(convertToInstance(srcType)),
                        destType: printType(convertToInstance(destType)),
                    })
                );
                return false;
            }

            // If invariance is being enforced, the two TypedDicts must be assignable to each other.
            if ((flags & AssignTypeFlags.EnforceInvariance) !== 0) {
                return assignTypedDictToTypedDict(
                    evaluatorInterface,
                    srcType,
                    destType,
                    /* diag */ undefined,
                    /* typeVarContext */ undefined,
                    flags,
                    recursionCount
                );
            }

            return true;
        }

        // Handle special-case type promotions.
        const promotionList = typePromotions.get(destType.details.fullName);
        if (
            promotionList &&
            promotionList.some((srcName) =>
                srcType.details.mro.some((mroClass) => isClass(mroClass) && srcName === mroClass.details.fullName)
            )
        ) {
            if ((flags & AssignTypeFlags.EnforceInvariance) === 0) {
                return true;
            }
        }

        // Is it a structural type (i.e. a protocol)? If so, we need to
        // perform a member-by-member check.
        const inheritanceChain: InheritanceChain = [];
        const isDerivedFrom = ClassType.isDerivedFrom(srcType, destType, inheritanceChain);

        // Use the slow path for protocols if the dest doesn't explicitly
        // derive from the source. We also need to use this path if we're
        // testing to see if the metaclass matches the protocol.
        if (ClassType.isProtocolClass(destType) && !isDerivedFrom) {
            if (
                !assignClassToProtocol(
                    evaluatorInterface,
                    destType,
                    srcType,
                    diag?.createAddendum(),
                    destTypeVarContext,
                    srcTypeVarContext,
                    flags,
                    /* treatSourceAsInstantiable */ false,
                    recursionCount
                )
            ) {
                diag?.addMessage(
                    Localizer.DiagnosticAddendum.protocolIncompatible().format({
                        sourceType: printType(convertToInstance(srcType)),
                        destType: printType(convertToInstance(destType)),
                    })
                );
                return false;
            }

            return true;
        }

        if ((flags & AssignTypeFlags.EnforceInvariance) === 0 || ClassType.isSameGenericClass(srcType, destType)) {
            if (isDerivedFrom) {
                assert(inheritanceChain.length > 0);

                if (
                    assignClassWithTypeArgs(
                        destType,
                        srcType,
                        inheritanceChain,
                        diag?.createAddendum(),
                        destTypeVarContext,
                        srcTypeVarContext,
                        flags,
                        recursionCount
                    )
                ) {
                    return true;
                }
            }
        }

        // Everything is assignable to an object.
        if (ClassType.isBuiltIn(destType, 'object')) {
            if ((flags & AssignTypeFlags.EnforceInvariance) === 0) {
                return true;
            }
        }

        const destErrorType = reportErrorsUsingObjType ? ClassType.cloneAsInstance(destType) : destType;
        const srcErrorType = reportErrorsUsingObjType ? ClassType.cloneAsInstance(srcType) : srcType;

        let destErrorTypeText = printType(destErrorType);
        let srcErrorTypeText = printType(srcErrorType);

        // If the text is the same, use the fully-qualified name rather than the short name.
        if (destErrorTypeText === srcErrorTypeText && destType.details.fullName && srcType.details.fullName) {
            destErrorTypeText = destType.details.fullName;
            srcErrorTypeText = srcType.details.fullName;
        }

        diag?.addMessage(
            Localizer.DiagnosticAddendum.typeIncompatible().format({
                sourceType: srcErrorTypeText,
                destType: destErrorTypeText,
            })
        );
        return false;
    }

    // This function is used to validate or infer the variance of type
    // parameters within a class. If ignoreBaseClassVariance is set to false,
    // the type parameters for the base class are honored. This is useful for
    // variance inference (PEP 695). For validation of protocol variance, we
    // want to ignore the variance for all base classes in the class hierarchy.
    function assignClassToSelf(
        destType: ClassType,
        srcType: ClassType,
        ignoreBaseClassVariance = true,
        recursionCount = 0
    ): boolean {
        assert(ClassType.isSameGenericClass(destType, srcType));
        assert(destType.details.typeParameters.length > 0);

        const diag = new DiagnosticAddendum();
        const typeVarContext = new TypeVarContext();
        let isAssignable = true;

        destType.details.fields.forEach((symbol, name) => {
            if (isAssignable && symbol.isClassMember() && !symbol.isIgnoredForProtocolMatch()) {
                const memberInfo = lookUpClassMember(srcType, name);
                assert(memberInfo !== undefined);

                let destMemberType = getDeclaredTypeOfSymbol(symbol)?.type;
                if (destMemberType) {
                    const srcMemberType = getTypeOfMember(memberInfo!);
                    destMemberType = partiallySpecializeType(destMemberType, destType);

                    // Properties require special processing.
                    if (
                        isClassInstance(destMemberType) &&
                        ClassType.isPropertyClass(destMemberType) &&
                        isClassInstance(srcMemberType) &&
                        ClassType.isPropertyClass(srcMemberType)
                    ) {
                        if (
                            !assignProperty(
                                evaluatorInterface,
                                ClassType.cloneAsInstantiable(destMemberType),
                                ClassType.cloneAsInstantiable(srcMemberType),
                                destType,
                                srcType,
                                diag,
                                typeVarContext,
                                /* selfTypeVarContext */ undefined,
                                recursionCount
                            )
                        ) {
                            isAssignable = false;
                        }
                    } else {
                        const primaryDecl = symbol.getDeclarations()[0];
                        // Class and instance variables that are mutable need to
                        // enforce invariance.
                        const flags =
                            primaryDecl?.type === DeclarationType.Variable && !isFinalVariableDeclaration(primaryDecl)
                                ? AssignTypeFlags.EnforceInvariance
                                : AssignTypeFlags.Default;
                        if (
                            !assignType(
                                destMemberType,
                                srcMemberType,
                                diag,
                                typeVarContext,
                                /* srcTypeVarContext */ undefined,
                                flags,
                                recursionCount
                            )
                        ) {
                            isAssignable = false;
                        }
                    }
                }
            }
        });

        // Now handle generic base classes.
        destType.details.baseClasses.forEach((baseClass) => {
            if (
                isInstantiableClass(baseClass) &&
                !ClassType.isBuiltIn(baseClass, 'object') &&
                !ClassType.isBuiltIn(baseClass, 'Protocol') &&
                !ClassType.isBuiltIn(baseClass, 'Generic') &&
                baseClass.details.typeParameters.length > 0
            ) {
                const specializedDestBaseClass = specializeForBaseClass(destType, baseClass);
                const specializedSrcBaseClass = specializeForBaseClass(srcType, baseClass);

                if (!ignoreBaseClassVariance) {
                    specializedDestBaseClass.details.typeParameters.forEach((param, index) => {
                        if (
                            !param.details.isParamSpec &&
                            !param.details.isVariadic &&
                            !param.details.isSynthesized &&
                            specializedSrcBaseClass.typeArguments &&
                            index < specializedSrcBaseClass.typeArguments.length &&
                            specializedDestBaseClass.typeArguments &&
                            index < specializedDestBaseClass.typeArguments.length
                        ) {
                            const paramVariance = param.details.declaredVariance;
                            if (isTypeVar(specializedSrcBaseClass.typeArguments[index])) {
                                if (paramVariance === Variance.Invariant || paramVariance === Variance.Contravariant) {
                                    isAssignable = false;
                                }
                            } else if (isTypeVar(specializedDestBaseClass.typeArguments[index])) {
                                if (paramVariance === Variance.Invariant || paramVariance === Variance.Covariant) {
                                    isAssignable = false;
                                }
                            }
                        }
                    });
                }

                if (
                    isAssignable &&
                    !assignClassToSelf(
                        specializedDestBaseClass,
                        specializedSrcBaseClass,
                        ignoreBaseClassVariance,
                        recursionCount
                    )
                ) {
                    isAssignable = false;
                }
            }
        });

        return isAssignable;
    }

    function assignTupleTypeArgs(
        destType: ClassType,
        srcType: ClassType,
        diag: DiagnosticAddendum | undefined,
        typeVarContext: TypeVarContext | undefined,
        flags: AssignTypeFlags,
        recursionCount: number
    ) {
        const destTypeArgs = [...(destType.tupleTypeArguments ?? [])];
        const srcTypeArgs = [...(srcType.tupleTypeArguments ?? [])];

        const destVariadicIndex = destTypeArgs.findIndex((t) => isVariadicTypeVar(t.type));
        const destUnboundedIndex = destTypeArgs.findIndex((t) => t.isUnbounded);
        const srcUnboundedIndex = srcTypeArgs.findIndex((t) => t.isUnbounded);

        // If the source is unbounded, expand the unbounded argument to try
        // to make the source and dest arg counts match.
        if (srcUnboundedIndex >= 0) {
            const typeToReplicate = srcTypeArgs.length > 0 ? srcTypeArgs[srcUnboundedIndex].type : AnyType.create();

            while (srcTypeArgs.length < destTypeArgs.length) {
                srcTypeArgs.splice(srcUnboundedIndex, 0, { type: typeToReplicate, isUnbounded: true });
            }
        }

        // If the dest is unbounded or contains a variadic, determine which
        // source args map to the unbounded or variadic arg.
        if (destUnboundedIndex >= 0 || destVariadicIndex >= 0) {
            // If there's a variadic within the destination, package up the corresponding
            // source arguments into a tuple.
            const srcArgsToCapture = srcTypeArgs.length - destTypeArgs.length + 1;
            if (srcArgsToCapture >= 0) {
                if (destVariadicIndex >= 0) {
                    if (tupleClassType && isInstantiableClass(tupleClassType)) {
                        const removedArgs = srcTypeArgs.splice(destVariadicIndex, srcArgsToCapture);

                        // Package up the remaining type arguments into a tuple object.
                        const variadicTuple = convertToInstance(
                            specializeTupleClass(
                                tupleClassType,
                                removedArgs.map((typeArg) => {
                                    return { type: typeArg.type, isUnbounded: typeArg.isUnbounded };
                                }),
                                /* isTypeArgumentExplicit */ true,
                                /* isUnpackedTuple */ true
                            )
                        );
                        srcTypeArgs.splice(destVariadicIndex, 0, {
                            type: variadicTuple,
                            isUnbounded: false,
                        });
                    }
                } else {
                    const removedArgTypes = srcTypeArgs.splice(destUnboundedIndex, srcArgsToCapture).map((t) => {
                        if (isTypeVar(t.type) && isUnpackedVariadicTypeVar(t.type) && !t.type.isVariadicInUnion) {
                            return TypeVarType.cloneForUnpacked(t.type, /* isInUnion */ true);
                        }
                        return t.type;
                    });
                    srcTypeArgs.splice(destUnboundedIndex, 0, {
                        type: removedArgTypes.length > 0 ? combineTypes(removedArgTypes) : AnyType.create(),
                        isUnbounded: false,
                    });
                }
            }
        }

        if (srcTypeArgs.length === destTypeArgs.length) {
            for (let argIndex = 0; argIndex < srcTypeArgs.length; argIndex++) {
                const entryDiag = diag?.createAddendum();

                if (
                    !assignType(
                        destTypeArgs[argIndex].type,
                        srcTypeArgs[argIndex].type,
                        entryDiag?.createAddendum(),
                        typeVarContext,
                        /* srcTypeVarContext */ undefined,
                        flags,
                        recursionCount
                    )
                ) {
                    if (entryDiag) {
                        entryDiag.addMessage(
                            Localizer.DiagnosticAddendum.tupleEntryTypeMismatch().format({
                                entry: argIndex + 1,
                            })
                        );
                    }
                    return false;
                }
            }
        } else {
            if (srcUnboundedIndex < 0) {
                diag?.addMessage(
                    Localizer.DiagnosticAddendum.tupleSizeMismatch().format({
                        expected: destTypeArgs.length,
                        received: srcTypeArgs.length,
                    })
                );

                return false;
            }
        }

        return true;
    }

    // Determines whether the specified type can be assigned to the
    // specified inheritance chain, taking into account its type arguments.
    function assignClassWithTypeArgs(
        destType: ClassType,
        srcType: ClassType,
        inheritanceChain: InheritanceChain,
        diag: DiagnosticAddendum | undefined,
        destTypeVarContext: TypeVarContext | undefined,
        srcTypeVarContext: TypeVarContext | undefined,
        flags: AssignTypeFlags,
        recursionCount: number
    ): boolean {
        let curSrcType = srcType;
        let curDestTypeVarContext = destTypeVarContext;
        let effectiveFlags = flags;

        inferTypeParameterVarianceForClass(destType);

        effectiveFlags |= AssignTypeFlags.SkipSolveTypeVars;

        if (!destTypeVarContext) {
            curDestTypeVarContext = new TypeVarContext(getTypeVarScopeId(destType));
            effectiveFlags &= ~AssignTypeFlags.SkipSolveTypeVars;
        } else {
            // If we're using the caller's type var context, don't solve the
            // type vars in this pass. We'll do this after we're done looping
            // through the inheritance chain.
            effectiveFlags |= AssignTypeFlags.SkipSolveTypeVars;
        }

        for (let ancestorIndex = inheritanceChain.length - 1; ancestorIndex >= 0; ancestorIndex--) {
            const ancestorType = inheritanceChain[ancestorIndex];

            const curSrcTypeVarContext = new TypeVarContext(getTypeVarScopeId(curSrcType));

            // If we've hit an "unknown", all bets are off, and we need to assume
            // that the type is assignable.
            if (isUnknown(ancestorType)) {
                return true;
            }

            // If we've hit an 'object', it's assignable.
            if (ClassType.isBuiltIn(ancestorType, 'object')) {
                return true;
            }

            // If this isn't the first time through the loop, specialize
            // for the next ancestor in the chain.
            if (ancestorIndex < inheritanceChain.length - 1) {
                curSrcType = specializeForBaseClass(curSrcType, ancestorType);
            }

            // Handle built-in types that support arbitrary numbers
            // of type parameters like Tuple.
            if (ancestorIndex === 0 && destType.tupleTypeArguments && curSrcType.tupleTypeArguments) {
                return assignTupleTypeArgs(destType, curSrcType, diag, curDestTypeVarContext, flags, recursionCount);
            }

            // If there are no type parameters on this class, we're done.
            const ancestorTypeParams = ClassType.getTypeParameters(ancestorType);
            if (ancestorTypeParams.length === 0) {
                continue;
            }

            // If the dest type isn't specialized, there are no type args to validate.
            if (!ancestorType.typeArguments) {
                return true;
            }

            // Validate that the type arguments match.
            if (
                !verifyTypeArgumentsAssignable(
                    ancestorType,
                    curSrcType,
                    diag,
                    curDestTypeVarContext,
                    curSrcTypeVarContext,
                    effectiveFlags,
                    recursionCount
                )
            ) {
                return false;
            }

            // Allocate a new type var map for the next time through the loop.
            curDestTypeVarContext = new TypeVarContext(getTypeVarScopeId(ancestorType));
            effectiveFlags &= ~AssignTypeFlags.SkipSolveTypeVars;
        }

        if (destType.typeArguments) {
            // If the dest type is specialized, make sure the specialized source
            // type arguments are assignable to the dest type arguments.
            if (
                !verifyTypeArgumentsAssignable(
                    destType,
                    curSrcType,
                    diag,
                    destTypeVarContext,
                    srcTypeVarContext,
                    flags,
                    recursionCount
                )
            ) {
                return false;
            }
        } else if (
            destTypeVarContext &&
            destType.details.typeParameters.length > 0 &&
            curSrcType.typeArguments &&
            !destTypeVarContext.isLocked()
        ) {
            // Populate the typeVar map with type arguments of the source.
            const srcTypeArgs = curSrcType.typeArguments;
            for (let i = 0; i < destType.details.typeParameters.length; i++) {
                const typeArgType = i < srcTypeArgs.length ? srcTypeArgs[i] : UnknownType.create();
                destTypeVarContext.setTypeVarType(
                    destType.details.typeParameters[i],
                    /* narrowBound */ undefined,
                    /* narrowBoundNoLiterals */ undefined,
                    typeArgType
                );
            }
        }

        return true;
    }

    function getGetterTypeFromProperty(propertyClass: ClassType, inferTypeIfNeeded: boolean): Type | undefined {
        if (!ClassType.isPropertyClass(propertyClass)) {
            return undefined;
        }

        const fgetSymbol = propertyClass.details.fields.get('fget');

        if (fgetSymbol) {
            const fgetType = getDeclaredTypeOfSymbol(fgetSymbol)?.type;
            if (fgetType && isFunction(fgetType)) {
                return getFunctionEffectiveReturnType(fgetType, /* args */ undefined, inferTypeIfNeeded);
            }
        }

        return undefined;
    }

    function verifyTypeArgumentsAssignable(
        destType: ClassType,
        srcType: ClassType,
        diag: DiagnosticAddendum | undefined,
        destTypeVarContext: TypeVarContext | undefined,
        srcTypeVarContext: TypeVarContext | undefined,
        flags: AssignTypeFlags,
        recursionCount: number
    ): boolean {
        assert(ClassType.isSameGenericClass(destType, srcType));

        inferTypeParameterVarianceForClass(destType);

        const destTypeParams = ClassType.getTypeParameters(destType);
        let destTypeArgs: Type[];
        let srcTypeArgs: Type[] | undefined;

        // If either source or dest type arguments are missing, they are
        // treated as "Any", so they are assumed to be assignable.
        if (!destType.typeArguments || !srcType.typeArguments) {
            return true;
        }

        if (ClassType.isTupleClass(destType)) {
            destTypeArgs = destType.tupleTypeArguments?.map((t) => t.type) ?? [];
            srcTypeArgs = srcType.tupleTypeArguments?.map((t) => t.type);
        } else {
            destTypeArgs = destType.typeArguments!;
            srcTypeArgs = srcType.typeArguments;
        }

        if (srcTypeArgs) {
            for (let srcArgIndex = 0; srcArgIndex < srcTypeArgs.length; srcArgIndex++) {
                const srcTypeArg = srcTypeArgs[srcArgIndex];

                // In most cases, the number of type args should match the number
                // of type arguments, but there are a few special cases where this
                // isn't true (e.g. assigning a Tuple[X, Y, Z] to a tuple[W]).
                const destArgIndex = srcArgIndex >= destTypeArgs.length ? destTypeArgs.length - 1 : srcArgIndex;
                const destTypeArg = destArgIndex >= 0 ? destTypeArgs[destArgIndex] : UnknownType.create();
                const destTypeParam = destArgIndex < destTypeParams.length ? destTypeParams[destArgIndex] : undefined;
                const assignmentDiag = new DiagnosticAddendum();

                if (!destTypeParam || TypeVarType.getVariance(destTypeParam) === Variance.Covariant) {
                    if (
                        !assignType(
                            destTypeArg,
                            srcTypeArg,
                            assignmentDiag,
                            destTypeVarContext,
                            srcTypeVarContext,
                            flags | AssignTypeFlags.RetainLiteralsForTypeVar,
                            recursionCount
                        )
                    ) {
                        if (diag) {
                            if (destTypeParam) {
                                const childDiag = diag.createAddendum();
                                childDiag.addMessage(
                                    Localizer.DiagnosticAddendum.typeVarIsCovariant().format({
                                        name: TypeVarType.getReadableName(destTypeParam),
                                    })
                                );
                                childDiag.addAddendum(assignmentDiag);
                            } else {
                                