// SPDX-License-Identifier: Apache-2.0

#include <scratchcpp/variable.h>

#include "variable_p.h"

using namespace libscratchcpp;

/*! Constructs Variable. */
Variable::Variable(const std::string &id, const std::string &name, const Value &value, bool isCloudVariable) :
    Entity(id),
    impl(spimpl::make_unique_impl<VariablePrivate>(name, value, isCloudVariable))
{
}

/*! Returns the name of the variable. */
const std::string &Variable::name() const
{
    return impl->name;
}

/*! Returns the value. */
const Value &Variable::value() const
{
    return impl->value;
}

/*! Returns a pointer to the value. */
Value *Variable::valuePtr()
{
    return &impl->value;
}

/*! Sets the value. */
void Variable::setValue(const Value &value)
{
    impl->value = value;
}

/*! Returns true if the variable is a cloud variable. */
bool Variable::isCloudVariable() const
{
    return impl->isCloudVariable;
}

/*! Toggles whether the variable is a cloud variable. */
void Variable::setIsCloudVariable(bool isCloudVariable)
{
    impl->isCloudVariable = isCloudVariable;
}

/*! Returns the sprite or stage this variable belongs to. */
Target *Variable::target() const
{
    return impl->target;
}

/*! Sets the sprite or stage this variable belongs to. */
void Variable::setTarget(Target *target)
{
    impl->target = target;
}

/*! Returns the monitor of this variable. */
Monitor *Variable::monitor() const
{
    return impl->monitor;
}

/*! Sets the monitor of this variable. */
void Variable::setMonitor(Monitor *monitor)
{
    impl->monitor = monitor;
}

/*! Creates a copy of the variable. */
std::shared_ptr<Variable> Variable::clone()
{
    return std::make_shared<Variable>(id(), impl->name, impl->value, impl->isCloudVariable);
}
