import { generate } from '../../../lib/generator.js';
import mockFs from 'mock-fs';
import { dirname, resolve } from 'node:path';
import { fileURLToPath } from 'node:url';
import { readFileSync } from 'node:fs';
import { jest } from '@jest/globals';

const __dirname = dirname(fileURLToPath(import.meta.url));

const PATH_NODE_MODULES = resolve(__dirname, '..', '..', '..', 'node_modules');

describe('generate (configs list)', function () {
  describe('basic', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              recommended: {},
            }
          };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.');
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('with --ignore-config', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              foo: {},
              recommended: {},
            }
          };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.', { ignoreConfig: ['foo'] });
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('with --config-format', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              recommended: {},
            }
          };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.', { configFormat: 'prefix-name' });
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('with configs not defined in alphabetical order', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              recommended: {},
              foo: {},
            }
          };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.');
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('when a config exports a description', function () {
    describe('property=description', function () {
      beforeEach(function () {
        mockFs({
          'package.json': JSON.stringify({
            name: 'eslint-plugin-test',
            exports: 'index.js',
            type: 'module',
          }),

          'index.js': `
            export default {
              rules: {
                'no-foo': {
                  meta: { docs: { description: 'Description of no-foo.' }, },
                  create(context) {}
                },
              },
              configs: {
                foo: {},
                recommended: { description: 'This config has the recommended rules...' },
              }
            };`,

          'README.md': `## Rules
  ## Configs
  <!-- begin auto-generated configs list -->
  <!-- end auto-generated configs list -->`,

          'docs/rules/no-foo.md': '',

          // Needed for some of the test infrastructure to work.
          node_modules: mockFs.load(PATH_NODE_MODULES),
        });
      });

      afterEach(function () {
        mockFs.restore();
        jest.resetModules();
      });

      it('generates the documentation', async function () {
        await generate('.');
        expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
      });
    });

    describe('property=meta.description', function () {
      beforeEach(function () {
        mockFs({
          'package.json': JSON.stringify({
            name: 'eslint-plugin-test',
            exports: 'index.js',
            type: 'module',
          }),

          'index.js': `
            export default {
              rules: {
                'no-foo': {
                  meta: { docs: { description: 'Description of no-foo.' }, },
                  create(context) {}
                },
              },
              configs: {
                foo: {},
                recommended: { meta: { description: 'This config has the recommended rules...' } },
              }
            };`,

          'README.md': `## Rules
  ## Configs
  <!-- begin auto-generated configs list -->
  <!-- end auto-generated configs list -->`,

          'docs/rules/no-foo.md': '',

          // Needed for some of the test infrastructure to work.
          node_modules: mockFs.load(PATH_NODE_MODULES),
        });
      });

      afterEach(function () {
        mockFs.restore();
        jest.resetModules();
      });

      it('generates the documentation', async function () {
        await generate('.');
        expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
      });
    });

    describe('property=meta.docs.description', function () {
      beforeEach(function () {
        mockFs({
          'package.json': JSON.stringify({
            name: 'eslint-plugin-test',
            exports: 'index.js',
            type: 'module',
          }),

          'index.js': `
            export default {
              rules: {
                'no-foo': {
                  meta: { docs: { description: 'Description of no-foo.' }, },
                  create(context) {}
                },
              },
              configs: {
                foo: {},
                recommended: { meta: { docs: { description: 'This config has the recommended rules...' } } },
              }
            };`,

          'README.md': `## Rules
  ## Configs
  <!-- begin auto-generated configs list -->
  <!-- end auto-generated configs list -->`,

          'docs/rules/no-foo.md': '',

          // Needed for some of the test infrastructure to work.
          node_modules: mockFs.load(PATH_NODE_MODULES),
        });
      });

      afterEach(function () {
        mockFs.restore();
        jest.resetModules();
      });

      it('generates the documentation', async function () {
        await generate('.');
        expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
      });
    });
  });

  describe('when a config description needs to be escaped in table', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              recommended: { description: 'Foo|Bar' },
            }
          };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.');
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('when there are no configs', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
         };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.');
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });

  describe('when all configs are ignored', function () {
    beforeEach(function () {
      mockFs({
        'package.json': JSON.stringify({
          name: 'eslint-plugin-test',
          exports: 'index.js',
          type: 'module',
        }),

        'index.js': `
          export default {
            rules: {
              'no-foo': {
                meta: { docs: { description: 'Description of no-foo.' }, },
                create(context) {}
              },
            },
            configs: {
              recommended: {},
            }
         };`,

        'README.md': `## Rules
## Configs
<!-- begin auto-generated configs list -->
<!-- end auto-generated configs list -->`,

        'docs/rules/no-foo.md': '',

        // Needed for some of the test infrastructure to work.
        node_modules: mockFs.load(PATH_NODE_MODULES),
      });
    });

    afterEach(function () {
      mockFs.restore();
      jest.resetModules();
    });

    it('generates the documentation', async function () {
      await generate('.', { ignoreConfig: ['recommended'] });
      expect(readFileSync('README.md', 'utf8')).toMatchSnapshot();
    });
  });
});
